// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>
#include <ContentTools/Common/Filters/FilterScene/RemoveScale/hctRemoveScaleFilter.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>

// included for matrix polar decomposition util
#include <Common/Base/Math/Matrix/hkMatrixDecomposition.h>

hctRemoveScaleFilterDesc g_removeScaleDesc;

hctRemoveScaleFilter::hctRemoveScaleFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner)
{

}

hctRemoveScaleFilter::~hctRemoveScaleFilter()
{

}

static void _SafelyDecompose( hkMatrix4& full, hkTransform& transform )
{
    // polar decompose the transform
    hkMatrixDecomposition::Decomposition decomposition;
    hkMatrixDecomposition::decomposeMatrix( full, decomposition );
    if( !decomposition.m_hasSkew )
    {
        transform.set( decomposition.m_rotation, decomposition.m_translation );
    }
    else
    {
        HK_WARN_ALWAYS( 0xabbaf0d1, "Matrix Decomposition Failed! There is skew present" );

        // set to origin with identity
        transform.setIdentity();
    }
}


static void _burnNodeScaleRecursive( hkxNode* node )
{
    if( !node ) return;

    if( node->m_keyFrames.getSize() > 0 )
    {
        // Decompose the first keyframe
        hkMatrix4 worldFromNode;
        hkMatrixDecomposition::Decomposition decomposition;
        hkMatrixDecomposition::decomposeMatrix( node->m_keyFrames[0], decomposition );
        if( decomposition.m_hasScale || decomposition.m_hasSkew )
        {
            // Get the scale/skew variations
            hkMatrix4 scaleSkewInverse;
            hkMatrix4 scaleSkewInverseTranspose;

            {
                scaleSkewInverse.setInverse(decomposition.m_scaleAndSkew,hkSimdReal_Eps);
                scaleSkewInverseTranspose = scaleSkewInverse;
                scaleSkewInverseTranspose.transpose();
            }


            // Remove the scale/skew from this node's keyframes
            int i, j;
            for( i=0; i<node->m_keyFrames.getSize(); ++i )
            {
                node->m_keyFrames[i].mul( scaleSkewInverse );
            }

            // Add the scale/skew into the child node keyframes
            for( i=0; i<node->m_children.getSize(); ++i )
            {
                hkxNode* childNode = node->m_children[i];
                for( j=0; j<childNode->m_keyFrames.getSize(); ++j )
                {
                    hkMatrix4 kf( childNode->m_keyFrames[j] );
                    childNode->m_keyFrames[j].setMul( decomposition.m_scaleAndSkew, kf );
                }
            }


            // Burn the scale/skew into the mesh if there is one
            hkxMesh* mesh = HK_NULL;
            {
                if(hkxMesh* m = node->hasA<hkxMesh>())
                {
                    mesh = m;
                }
                else
                {
                    // EXP-742 : The mesh can be in a skin binding
                    if (hkxSkinBinding* b = node->hasA<hkxSkinBinding>())
                    {
                        mesh = b->m_mesh;
                    }
                }
            }

            if (mesh)
            {
                scaleSkewInverseTranspose.resetFourthRow();

                for( int sectionIndex=0; sectionIndex<mesh->m_sections.getSize(); ++sectionIndex)
                {
                    hkxVertexBuffer* vb = mesh->m_sections[sectionIndex]->m_vertexBuffer;
                    const hkxVertexDescription& desc = vb->getVertexDesc();
                    const hkxVertexDescription::ElementDecl* posDecl = desc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
                    const hkxVertexDescription::ElementDecl* normDecl = desc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
                    const hkxVertexDescription::ElementDecl* tangentDecl = desc.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
                    const hkxVertexDescription::ElementDecl* binormalDecl = desc.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);

                    char* posBuf = static_cast<char*>( posDecl? vb->getVertexDataPtr(*posDecl) : HK_NULL );
                    char* normBuf = static_cast<char*>( normDecl? vb->getVertexDataPtr(*normDecl) : HK_NULL );
                    char* tangentBuf = static_cast<char*>( tangentDecl? vb->getVertexDataPtr(*tangentDecl) : HK_NULL );
                    char* binormalBuf = static_cast<char*>( binormalDecl? vb->getVertexDataPtr(*binormalDecl) : HK_NULL );

                    int posStride = posDecl? posDecl->m_byteStride : 0;
                    int normStride = normDecl? normDecl->m_byteStride : 0;
                    int tangentStride = tangentDecl? tangentDecl->m_byteStride : 0;
                    int binormalStride = binormalDecl? binormalDecl->m_byteStride : 0;

                    int numVertices = vb->getNumVertices();
                    for( int vertexIndex=0; vertexIndex<numVertices; ++vertexIndex)
                    {
                        // Vertex
                        {
                            hkVector4* vec = (hkVector4*)(posBuf);
                            decomposition.m_scaleAndSkew.transformPosition(*vec, *vec);
                        }

                        // Normal
                        if( normDecl )
                        {
                            hkVector4* vec = (hkVector4*)(normBuf);
                            scaleSkewInverseTranspose.transformDirection(*vec, *vec);
                            vec->normalize3();
                        }

                        // Tangent
                        if( tangentDecl)
                        {
                            hkVector4* vec = (hkVector4*)(tangentBuf);
                            scaleSkewInverseTranspose.transformDirection(*vec, *vec);
                            vec->normalize3();
                        }

                        // Binormal
                        if( binormalDecl)
                        {
                            hkVector4* vec = (hkVector4*)(binormalBuf);
                            scaleSkewInverseTranspose.transformDirection(*vec, *vec);
                            vec->normalize3();
                        }

                        posBuf += posStride;
                        normBuf += normStride;
                        tangentBuf += tangentStride;
                        binormalBuf += binormalStride;
                    }
                }
            }
        }
    }

    // Recurse over the children of the node
    for( int i=0; i<node->m_children.getSize(); ++i )
    {
        _burnNodeScaleRecursive( node->m_children[i] );
    }
}


void hctRemoveScaleFilter::process( hkRootLevelContainer& data )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if( scenePtr == HK_NULL )
    {
        HK_WARN_ALWAYS (0xabbaa5f0, "No scene data found");
        return;
    }
    hkxScene& scene = *scenePtr;

    // we don't need to register any classes as we are not adding any extra types
    // that are not 'known' already.

    // Push the scale out of the scene graph, starting from the root node
    _burnNodeScaleRecursive( scene.m_rootNode );

    // Remove the scale from the world bind pose in the blended mesh infos
    for( int i=0; i<scene.m_skinBindings.getSize(); ++i )
    {
        hkxSkinBinding* skin = scene.m_skinBindings[i];
        for (int mi=0; mi < skin->m_bindPose.getSize(); ++mi)
        {
            hkMatrix4& worldT = skin->m_bindPose[mi];
            hkTransform noScaleWorld;

            // polar decompose the transform
            _SafelyDecompose( worldT, noScaleWorld );

            worldT.set( noScaleWorld );
        }

        hkMatrix4& worldT_skin = skin->m_initSkinTransform;
        hkTransform noScaleWorld_skin;

        // polar decompose the transform
        _SafelyDecompose( worldT_skin, noScaleWorld_skin );

        worldT_skin.set( noScaleWorld_skin );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
