// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>

#include <ContentTools/Common/Filters/FilterScene/PlatformWriter/hctPlatformWriterFilter.h>
#include <ContentTools/Common/Filters/Common/Utils/hctFilterUtils.h>

#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#include <Common/Base/Container/String/hkUtf8.h>

#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Graph/hkxNode.h>

#include <Common/SceneData/Environment/hkxEnvironment.h>

#include <shlwapi.h>
#include <tchar.h>
#include <stdio.h>

extern HINSTANCE hInstance;

inline void _fillCombo( HWND combo )
{
    hkArrayView<const hkReflect::TypeCopier::Target> targets = hkReflect::TypeCopier::getTargetStrings();
    for (int i=0; i < targets.getSize(); ++i)
    {
        SendMessage( combo, CB_ADDSTRING, 0, (LPARAM)targets[i].m_description);
    }
}

struct _FilterInfo
{
    hctPlatformWriterFilter* m_filter;
    const hkRootLevelContainer* m_originalContents;
};

void _convertFilenameExtension( hkStringOld& filename, const char* toExt )
{
    // Remove trailing .hkx, if any
    if (const char* dot = hkString::strRchr(filename.cString(), '.'))
    {
        int nameLen = hkLosslessCast<int>(dot - filename.cString());
        filename.setAsSubstr(0, nameLen);
        filename += toExt;
    }
}

void _convertFilename( HWND hWnd, _FilterInfo* filterInfo, const char* toExt  )
{
    hkStringPtr str;
    // Get the filename from the edit control.
    if (hctGetDlgItemHkStringPtr(hWnd, IDC_EDIT_FILENAME, str))
    {
        filterInfo->m_filter->m_filename = str;
    }
    _convertFilenameExtension( filterInfo->m_filter->m_filename, toExt );
    // Set text back
    SetDlgItemTextW( hWnd, IDC_EDIT_FILENAME, hkUtf8::WideFromUtf8(filterInfo->m_filter->m_filename.cString()) );
}

INT_PTR CALLBACK hkFilterPlatformWriterDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message)
    {
        case WM_INITDIALOG:
        {
            _FilterInfo* filterInfo = (_FilterInfo*)lParam;
            if (!filterInfo)
                return FALSE;
            //HK_ASSERT_NO_MSG(0x7ff5fb91, filterInfo);
            HK_ASSERT_NO_MSG(0x6dd976d0, filterInfo->m_filter);

            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later
            filterInfo->m_filter->m_optionsDialog = hWnd;
            hctPlatformWriterOptions& filterOptions = filterInfo->m_filter->m_options;

            // Enable combo box only if format is native.
            EnableWindow(GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO), filterOptions.m_format == hctPlatformWriterOptions::FORMAT_NATIVE);

            // Enable correct radio button.
            CheckDlgButton(hWnd, IDC_FORMAT_TAGFILE, filterOptions.m_format == hctPlatformWriterOptions::FORMAT_PORTABLE);
            CheckDlgButton(hWnd, IDC_FORMAT_PACKFILE, filterOptions.m_format == hctPlatformWriterOptions::FORMAT_NATIVE);
            CheckDlgButton(hWnd, IDC_FORMAT_TEXTXML, filterOptions.m_format == hctPlatformWriterOptions::FORMAT_XML);

            // If no filename has been set, display the default.
            if( filterInfo->m_filter->m_filename.getLength() == 0 )
            {
                const char* extension = "";
                switch (filterOptions.m_format)
                {
                    case hctPlatformWriterOptions::FORMAT_PORTABLE: extension = "hkt"; break;
                    case hctPlatformWriterOptions::FORMAT_NATIVE: extension = "hkx"; break;
                    case hctPlatformWriterOptions::FORMAT_XML: extension = "xml"; break;
                }
                filterInfo->m_filter->m_filename.printf( ".\\$(asset)_$(configuration).%s", extension );
            }
            SetDlgItemTextW(hWnd, IDC_EDIT_FILENAME, hkUtf8::WideFromUtf8(filterInfo->m_filter->m_filename.cString()));

            // Init combo box
            HWND hwndCombo = GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO);
            _fillCombo(hwndCombo);
            // Set cur sel
            int curPresetIndex = hctPlatformWriterOptions::findTargetIndex( filterOptions.m_target );
            curPresetIndex = curPresetIndex >= 0 ? curPresetIndex : 0; // default to win32
            SendMessage(hwndCombo, CB_SETCURSEL, (WPARAM)curPresetIndex, 0 );

            CheckDlgButton(hWnd, IDC_CB_SAVE_ENVIRONMENT, filterOptions.m_saveEnvironmentData);

            // Initialize Tool Tips
            {
                CreateToolTip(IDC_EDIT_FILENAME, hWnd, hInstance, "The filename specifies where to dump the stream contents. NOTE: The path for the filename is always stored relative to the original asset. This means that if the asset is moved the output will be placed in a sensible relative location. This makes transferring assets between machine and batch processing far easier.\nAny instances of '$(asset)' found in the filename will be replaced with the asset's name (the file extension is removed). Similarly, any instances of '$(configuration)' will be replaced with the configuration's name. If an object was selected in the modeler before exporting, then a '$(selected)' string is also available. Of course, you may define your own environment variables within the modeler. Use the $(variableName:defaultValue) format if you want to use a default value whenever the variable is not defined.");
                CreateToolTip(IDC_PLATFORM_PRESET_COMBO, hWnd, hInstance, "This allows you to specify the binary format for a target platform. The binary target is both platform and compiler dependent (since some compilers do different types of optimization).");
                // todo.rt this should create a type compendium
                CreateToolTip(IDC_CB_SAVE_ENVIRONMENT, hWnd, hInstance, "This option will include any defined environment variables in the output file. This is usually not required since the environment variables are generally used within the filter pipeline only.");
                CreateToolTip(IDC_FORMAT_TEXTXML, hWnd, hInstance, "This option will save the file into human readable XML format.");
                CreateToolTip(IDC_FORMAT_TAGFILE, hWnd, hInstance, "This option will save the file into a minimal-sized binary format.");
                CreateToolTip(IDC_FORMAT_PACKFILE, hWnd, hInstance, "This option will save the file into binary format optimized for in-place load on specific platforms.");
            }

            return TRUE; // did handle it
        }
        case WM_COMMAND: // UI Changes
        {
            _FilterInfo* filterInfo = reinterpret_cast<_FilterInfo*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA));

            if (!filterInfo)
                return FALSE;
            //HK_ASSERT_NO_MSG(0x1540aec0, filterInfo);
            HK_ASSERT_NO_MSG(0x4d4640b7, filterInfo->m_filter);
            hctPlatformWriterOptions& filterOptions = filterInfo->m_filter->m_options;

            switch (LOWORD(wParam))
            {
                case IDC_FORMAT_TAGFILE:
                {
                    filterOptions.m_format = hctPlatformWriterOptions::FORMAT_PORTABLE;

                    // Convert from .hkx to .hkt
                    _convertFilename( hWnd, filterInfo, ".hkt" );

                    EnableWindow(GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO), false);

                    break;
                }
                case IDC_FORMAT_PACKFILE:
                {
                    HWND hwndCombo = GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO);
                    LRESULT ir = SendMessage(hwndCombo, CB_GETCURSEL, 0, 0);
                    filterOptions.m_format = hctPlatformWriterOptions::FORMAT_NATIVE;
                    filterOptions.m_target = hctPlatformWriterOptions::getTargetName(ir);

                    EnableWindow(GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO), true);

                    // Convert from .hkx to .hkt
                    _convertFilename( hWnd, filterInfo, ".hkx" );

                    break;
                }
                case IDC_PLATFORM_PRESET_COMBO:
                {
                    HWND hwndCombo = GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO);
                    LRESULT ir = SendMessage(hwndCombo, CB_GETCURSEL, 0, 0);
                    filterOptions.m_target = hctPlatformWriterOptions::getTargetName(ir);
                    HK_ASSERT_NO_MSG(0x41f8521b, filterOptions.m_format == hctPlatformWriterOptions::FORMAT_NATIVE);
                    filterInfo->m_filter->updateOptions();
                    break;
                }
                case IDC_FORMAT_TEXTXML:
                {
                    filterOptions.m_format = hctPlatformWriterOptions::FORMAT_XML;

                    _convertFilename(hWnd, filterInfo, ".xml");

                    EnableWindow(GetDlgItem(hWnd, IDC_PLATFORM_PRESET_COMBO), false);
                    break;
                }
                case IDC_CB_SAVE_ENVIRONMENT:
                {
                    filterOptions.m_saveEnvironmentData = IsDlgButtonChecked(hWnd, IDC_CB_SAVE_ENVIRONMENT) == TRUE;
                    break;
                }
                case IDC_BROWSE_FILENAME:
                {
                    hkStringPtr filename;
                    hctGetDlgItemHkStringPtr(hWnd, IDC_EDIT_FILENAME, filename);

                    // Get the filename from the edit control.
                    if ( filename.getLength() > 0 )
                    {
                        filterInfo->m_filter->m_filename = filename;
                        filterInfo->m_filter->m_filename = filterInfo->m_filter->m_filename.replace( '/', '\\' );

                        // Check whether m_filename is relative or not.
                        if( PathIsRelativeW(hkUtf8::WideFromUtf8(filterInfo->m_filter->m_filename.cString())) )
                        {
                            // It's relative to the asset, so append the asset path in order to open the dialog in the correct location.
                            hkStringOld fullFileName = filterInfo->m_filter->m_assetFolder + filterInfo->m_filter->m_filename;
                            filename = fullFileName.cString();
                        }
                        else
                        {
                            // It's absolute, so copy it back into the filename buffer as is (to update any '/'->'\' changes).
                            filename = filterInfo->m_filter->m_filename.cString();
                        }
                    }

                    filterInfo->m_filter->updateOptions();

                    OPENFILENAMEW op;
                    hkString::memSet( &op, 0, sizeof(OPENFILENAMEW) );
                    op.lStructSize = sizeof(OPENFILENAMEW);
                    op.hwndOwner = hWnd;
                    op.lpstrFilter = hctFilterUtils::s_DialogFileExtensions;

                    hkUtf8::WideFromUtf8 assetFolderWide(filterInfo->m_filter->m_assetFolder.cString());
                    hkUtf8::WideFromUtf8 filenameWide(filename);

                    wchar_t filenameBuffer[1024];
                    wcsncpy_s(filenameBuffer, filenameWide, 1023);

                    if ( PathIsRootW( filenameWide ) )
                    {
                        op.lpstrInitialDir = filenameWide;
                    }
                    else
                    {
                        op.lpstrInitialDir = assetFolderWide;
                        op.lpstrFile = filenameBuffer;
                    }

                    if ( filterOptions.m_format == hctPlatformWriterOptions::FORMAT_XML )
                    {
                        op.lpstrDefExt = L"xml";
                        op.nFilterIndex = 4;
                    }
                    else
                    {
                        if( filterOptions.m_format == hctPlatformWriterOptions::FORMAT_PORTABLE)
                        {
                            op.lpstrDefExt = L"hkt";
                            op.nFilterIndex = 3;
                        }
                        else
                        {
                            op.lpstrDefExt = L"hkx";
                            op.nFilterIndex = 2;
                        }
                    }

                    op.nMaxFile = 1023;
                    op.Flags = OFN_ENABLESIZING | OFN_EXPLORER | OFN_NOREADONLYRETURN | OFN_PATHMUSTEXIST;
                    if ( GetSaveFileNameW(&op) )
                    {
                        hctFilterUtils::getReducedFilename( op.lpstrFile, filterInfo->m_filter->m_assetFolder, filterInfo->m_filter->m_filename );
                        SetDlgItemTextW(hWnd, IDC_EDIT_FILENAME, hkUtf8::WideFromUtf8(filterInfo->m_filter->m_filename.cString()));
                    }
                }
            }

            return TRUE;
        }
        case WM_DESTROY:
        {
            _FilterInfo* filterInfo = reinterpret_cast<_FilterInfo*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA));

            if (filterInfo)
            {
                delete filterInfo;
                // notify window about released user data
                SetWindowLongPtr(hWnd, GWLP_USERDATA, NULL);
            }

            return TRUE;
        }
    }
    return FALSE; //didn't handle it / didn't do much with it
}


HWND hctPlatformWriterFilter::showOptions(HWND owner)
{
    if (m_optionsDialog)
        hideOptions();

    // Try to guess where is the "main" folder for the asset, from the environment or scene data objects
    hctFilterUtils::getAssetFolder(*m_filterManager->getOriginalContents(), m_assetFolder);

    _FilterInfo* fi = new _FilterInfo;
    fi->m_originalContents = getFilterManager()->getOriginalContents();
    fi->m_filter = this;

    m_optionsDialog = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_PLATFORM_WRITER_DIALOG),
        owner, hkFilterPlatformWriterDialogProc, (LPARAM) fi );

    return m_optionsDialog;
}

void hctPlatformWriterFilter::updateOptions()
{
    if (m_optionsDialog)
    {
        HWND hwndCombo = GetDlgItem(m_optionsDialog, IDC_PLATFORM_PRESET_COMBO);
        LRESULT ir = SendMessage(hwndCombo, CB_GETCURSEL, 0, 0);
        m_options.m_target = hctPlatformWriterOptions::getTargetName(ir);

        if (IsDlgButtonChecked(m_optionsDialog, IDC_FORMAT_TAGFILE) != FALSE)
        {
            m_options.m_format = hctPlatformWriterOptions::FORMAT_PORTABLE;
        }
        else if (IsDlgButtonChecked(m_optionsDialog, IDC_FORMAT_PACKFILE) != FALSE)
        {
            m_options.m_format = hctPlatformWriterOptions::FORMAT_NATIVE;
        }
        else
        {
            HK_ASSERT_NO_MSG(0x56eb32c0, IsDlgButtonChecked(m_optionsDialog, IDC_FORMAT_TEXTXML) != FALSE);
            m_options.m_format = hctPlatformWriterOptions::FORMAT_XML;
        }

        m_options.m_saveEnvironmentData = IsDlgButtonChecked(m_optionsDialog, IDC_CB_SAVE_ENVIRONMENT) != FALSE;

        wchar_t text[1024];
        GetDlgItemTextW(m_optionsDialog, IDC_EDIT_FILENAME, text, HK_COUNT_OF(text));
        hctFilterUtils::getReducedFilename( text, m_assetFolder, m_filename );
    }
}

void hctPlatformWriterFilter::hideOptions()
{
    updateOptions();
    if (m_optionsDialog)
    {
        DestroyWindow(m_optionsDialog);
    }
    m_optionsDialog = NULL;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
