// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>
#include <ContentTools/Common/Filters/FilterScene/FindInstances/hctFindInstancesFilter.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.scene.findinstances"
#include <Common/Base/System/Log/hkLog.hxx>


hctFindInstancesFilterDesc g_findInstancesDesc;

hctFindInstancesFilter::hctFindInstancesFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner)
{

}

hctFindInstancesFilter::~hctFindInstancesFilter()
{

}

static void _shallowMeshCompare( const hkxScene& scene, hkArray< hkArray<hkxMesh*> >& setsOut )
{
    setsOut.clear();

    hkArray<hkxMesh*> remainingMeshes(scene.m_meshes.getSize());
    for( int i=0; i<scene.m_meshes.getSize(); ++i )
    {
        remainingMeshes[i] = scene.m_meshes[i];
    }

    while( remainingMeshes.getSize() > 0 )
    {
        // Move the first mesh into a new set
        hkArray<hkxMesh*> set(1,remainingMeshes[0]);
        remainingMeshes.removeAt(0);

        // Look for any loosely matching meshes
        hkxMesh* meshA = set[0];
        for( int i=0; i<remainingMeshes.getSize(); ++i )
        {
            hkxMesh* meshB = remainingMeshes[i];

            // Skip if already instanced
            if( meshA == meshB ) continue;

            // Need same number of sections
            if( meshA->m_sections.getSize() != meshB->m_sections.getSize() ) continue;

            // Need same number of vertices in each section
            int k;
            for( k=0; k<meshA->m_sections.getSize(); ++k )
            {
                if( meshA->m_sections[k]->m_vertexBuffer->getNumVertices()
                    != meshB->m_sections[k]->m_vertexBuffer->getNumVertices())
                {
                    break;
                }
            }
            if( k != meshA->m_sections.getSize() )
            {
                continue;
            }

            // Store it in the set
            set.pushBack( meshB );
            remainingMeshes.removeAtAndCopy(i--);
        }

        // Need at least 2 matches
        if( set.getSize() >= 2 )
        {
            setsOut.pushBack( set );
        }
    }
}


static inline void _roundOff( float& val )
{
    float product = 100 * val;
    int whole = (int)product;
    float fraction = product - whole;
    if( 2*fraction >= 1 )
    {
        val = (whole + 1)/100.0f;
    }
    else
    {
        val = whole/100.0f;
    }
}

static void _roundOffMeshFloats( hkxMesh* mesh )
{
    for( int i=0; i<mesh->m_sections.getSize(); ++i )
    {
        hkxVertexBuffer* vb = mesh->m_sections[i]->m_vertexBuffer;
        const hkxVertexDescription& vd = vb->getVertexDesc();

        // Gather a set of offsets to floating point values
        hkArray<void*> floatOffsets;
        hkArray<int> floatStrides;
        for (int vdi=0; vdi < vd.m_decls.getSize(); ++vdi)
        {
            const hkxVertexDescription::ElementDecl& decl = vd.m_decls[vdi];
            if( decl.m_type == hkxVertexDescription::HKX_DT_FLOAT )
            {
                for( int k=0; k< decl.m_numElements; ++k ) { floatOffsets.pushBack( hkAddByteOffset(vb->getVertexDataPtr(decl), (k*4)) ); floatStrides.pushBack(decl.m_byteStride); }
            }
        }

        // Roundoff the floats for each vertex
        int numVerts = vb->getNumVertices();
        for( int j=0; j< numVerts; ++j )
        {
            for( int k=0; k< floatOffsets.getSize(); ++k )
            {
                _roundOff( *(float*)( hkAddByteOffset(floatOffsets[k], j*floatStrides[k]) ) );
            }
        }

        // Note: any mesh material should be instanced by the exporter already
        // so we don't need to go and roundoff the floats there
    }
}


static void _deepMeshCompare( const hkArray<hkxMesh*>& meshes, hkArray< hkArray<hkxMesh*> >& setsOut )
{
    setsOut.clear();
    //XXXck.todo: need to refactor for new Packfile date
#if 0

    // Deep copy each mesh in the set
    hkArray< hkResource* > deepMeshData; deepMeshData.setSize( meshes.getSize() );
    hkArray<int> remainingIndices( meshes.getSize() );
    for( int i=0; i<meshes.getSize(); ++i )
    {
        hctFilterUtils::deepCopyObject( meshes[i], &hkxMeshClass, deepMeshData[i] );
        remainingIndices[i] = i;
    }

    // Find any subsets which have equal binary length
    hkArray< hkArray<int> > subsets;
    while( remainingIndices.getSize() > 0 )
    {
        // Move first index into a new set
        hkArray<int> subset(1,remainingIndices[0]);
        remainingIndices.removeAt(0);


        // Look for other indices with same binary size
        hkArray< hkResource* >& dataA = deepMeshData[ subset[0] ];

        for( int i=0; i<remainingIndices.getSize(); ++i )
        {
            hkArray<hkResource*>& dataB = deepMeshData[ remainingIndices[i] ];

            if( dataA.getSize() != dataB.getSize() )
            {
                continue;
            }

            subset.pushBack( remainingIndices[i] );
            remainingIndices.removeAtAndCopy(i--);
        }


        // Need at least 2 matches
        if( subset.getSize() >= 2 )
        {
            subsets.pushBack( subset );
        }
    }

    // Deep compare the subsets
    for( int i=0; i<subsets.getSize(); ++i )
    {
        hkArray<int> subset;
        subset.insertAt( 0, subsets[i].begin(), subsets[i].getSize() );
        int binarySize = deepMeshData[ subset[0] ].getSize();

        // Fixup the data by extracting each mesh into a common memory space to equate pointers,
        // and rounding off float values to avoid numerical error
        hkArray<hkResource*> tempMeshData( binarySize );
        for( int j=0; j<subset.getSize(); ++j )
        {
            hkxMesh* mesh = (hkxMesh*)hctFilterUtils::deepCopyObject( meshes[subset[j]], &hkxMeshClass, tempMeshData );
            _roundOffMeshFloats( mesh );
            deepMeshData[ subset[j] ].clear();
            deepMeshData[ subset[j] ].insertAt( 0, tempMeshData.begin(), tempMeshData.getSize() );
        }

        // Gather any matching binary sets
        while( subset.getSize() > 0 )
        {
            int indexA = subset[0];
            hkArray<hkxMesh*> meshSet(1, meshes[indexA] );
            subset.removeAt(0);

            for( int j=0; j<subset.getSize(); ++j )
            {
                int indexB = subset[j];
                if( hkString::memCmp( (void*)deepMeshData[indexA].begin(), (void*)deepMeshData[indexB].begin(), binarySize ) == 0 )
                {
                    meshSet.pushBack( meshes[indexB] );
                    subset.removeAtAndCopy(j--);
                }
            }

            // Need at least 2 matches
            if( meshSet.getSize() >= 2 )
            {
                setsOut.pushBack( meshSet );
            }
        }
    }

    //XXXX remove refs on the packfile data
#endif
}


static void _updateAllInstancedMeshes( hkxNode* node, const hkPointerMap<hkxMesh*, hkxMesh*>& instanceMap )
{
    if( hkxMesh* mesh = node->hasA<hkxMesh>() )
    {
        if( instanceMap.hasKey( mesh ) )
        {
            hkxMesh* meshInMap;
            instanceMap.get( mesh, &meshInMap );
            node->m_object = meshInMap;
        }
    }

    for (int c=0; c < node->m_children.getSize(); ++c)
    {
        _updateAllInstancedMeshes( node->m_children[c], instanceMap );
    }
}

void hctFindInstancesFilter::process( hkRootLevelContainer& data )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if( scenePtr == HK_NULL )
    {
        HK_WARN_ALWAYS (0xabbaa5f1, "No scene data found");
        return;
    }
    hkxScene& scene = *scenePtr;


    // Create an instance map
    hkPointerMap<hkxMesh*, hkxMesh*> instanceMap;
    {
        // Shallow compare all meshes into loosely matching sets
        hkArray< hkArray<hkxMesh*> > meshSets;
        _shallowMeshCompare( scene, meshSets );

        // Binary compare the elements in each subset
        for( int i=0; i<meshSets.getSize(); ++i )
        {
            hkArray< hkArray<hkxMesh*> > meshSubsets;
            _deepMeshCompare( meshSets[i], meshSubsets );

            for( int j=0; j<meshSubsets.getSize(); ++j )
            {
                // Map all instances to the first one
                hkArray<hkxMesh*>& instancedMeshes = meshSubsets[j];
                for( int k=1; k<instancedMeshes.getSize(); ++k )
                {
                    instanceMap.insert( instancedMeshes[k], instancedMeshes[0] );
                }
            }
        }
    }

    // Update the scene data using the instance map
    if( instanceMap.getSize() > 0 )
    {
        // Alter any instanced meshes in the node graph
        _updateAllInstancedMeshes( scene.m_rootNode, instanceMap );

        // Alter any instanced meshes in the skin bindings
        for( int i=0; i<scene.m_skinBindings.getSize(); ++i )
        {
            hkxMesh* mesh = scene.m_skinBindings[i]->m_mesh;
            if( instanceMap.hasKey( mesh ) )
            {
                instanceMap.get( mesh, &mesh );
            }
        }

        // Remove any instanced meshes from the scene container
        hkArray< hkRefPtr<hkxMesh> >& meshes = scene.m_meshes;
        for( int i=meshes.getSize()-1; i>=0; --i )
        {
            if( instanceMap.hasKey( meshes[i] ) )
            {
                meshes.removeAtAndCopy(i);
            }
        }

        Log_Info( "{} duplicate meshes found and replaced by instances", instanceMap.getSize() );
    }
    else
    {
        Log_Info( "No duplicate meshes found" );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
