// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>

// GDI+ to make the image handling simple
#pragma warning(push)
#pragma warning(disable: 4458) // declaration of 'nativeCap' hides class member
#include <gdiplus.h>
#pragma warning(pop)
using namespace Gdiplus;
#pragma comment(lib, "gdiplus.lib")

#include <ContentTools/Common/Filters/FilterScene/ConvertTexturesToPng/hctConvertTexturesToPNGFilter.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>

#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>

#include <tchar.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.scene.texturetopng"
#include <Common/Base/System/Log/hkLog.hxx>


int _GetEncoderClsid(const WCHAR* format, CLSID* pClsid)
{
   UINT  num = 0;          // number of image encoders
   UINT  size = 0;         // size of the image encoder array in bytes

   ImageCodecInfo* pImageCodecInfo = NULL;

   GetImageEncodersSize(&num, &size);
   if(size == 0)
      return -1;  // Failure

   pImageCodecInfo = (ImageCodecInfo*)(malloc(size));
   if(pImageCodecInfo == NULL)
      return -1;  // Failure

   GetImageEncoders(num, size, pImageCodecInfo);

   for(UINT j = 0; j < num; ++j)
   {
      if( wcscmp(pImageCodecInfo[j].MimeType, format) == 0 )
      {
         *pClsid = pImageCodecInfo[j].Clsid;
         free(pImageCodecInfo);
         return j;  // Success
      }
   }

   free(pImageCodecInfo);
   return -1;  // Failure
}


hctConvertTexturesToPNGFilterDesc g_convertTexturesToPNGDesc;

hctConvertTexturesToPNGFilter::hctConvertTexturesToPNGFilter(const hctFilterManagerInterface* owner)
: hctFilterInterface (owner),
  m_optionsDialog(HK_NULL)
{
    // initial defaults
    m_options.m_enable = false;
    m_options.m_u = 0;
    m_options.m_v = 0;
}

hctConvertTexturesToPNGFilter::~hctConvertTexturesToPNGFilter()
{

}

static inline hkUint32 hkRoundUpPow2(hkUint32 n)
{
    n--;
    n |= n >> 1;
    n |= n >> 2;
    n |= n >> 4;
    n |= n >> 8;
    n |= n >> 16;
    n++;
    return n;
}

static inline hkUint32 hkRoundDownPow2(hkUint32 n)
{
    return hkRoundUpPow2( n - n/2 + 1);
}

static void recursivelyFindSceneMaterials(hkxMaterial* current, hkArray<hkxMaterial*>& sceneMaterials)
{
    if (current != HK_NULL)
    {
        sceneMaterials.pushBack(current);
    }

    for (int i=0; i< current->m_subMaterials.getSize(); i++)
    {
        recursivelyFindSceneMaterials(current->m_subMaterials[i], sceneMaterials);
    }
}

static void findSceneMaterials(hkxScene& scene, hkArray<hkxMaterial*>& sceneMaterials)
{
        for (int cim = 0; cim < scene.m_materials.getSize(); ++cim)
        {
            recursivelyFindSceneMaterials( scene.m_materials[cim], sceneMaterials) ;
        }
}

void hctConvertTexturesToPNGFilter::process( hkRootLevelContainer& data  )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL)
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "No scene data found");
        return;
    }

    hkxScene& scene = *scenePtr;

    hkArray<hkRefPtr<hkxTextureFile> > doneTextures;
    hkArray<hkRefPtr<hkxTextureInplace> > replacementTextures;

    if (scene.m_materials.getSize() > 0)
    {
        ULONG_PTR gdiplusToken;
        GdiplusStartupInput gdiplusStartupInput;
        GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);

        hkArray<hkxMaterial*> sceneMaterials;
        findSceneMaterials(scene, sceneMaterials);

        for (int cim = 0; cim < sceneMaterials.getSize(); ++cim)
        {
            hkxMaterial* material = sceneMaterials[cim];
            // all external textures (can extend to inplace aswell later on)
            for (int net = 0; net < material->m_stages.getSize(); ++net)
            {
                if ( !(material->m_stages[net].m_texture) )
                {
                    continue;
                }

                hkxTextureFile* textureFile = hkReflect::exactMatchDynCast<hkxTextureFile>( material->m_stages[net].m_texture);
                if (!textureFile)
                {
                    continue;
                }

                hkxTextureInplace* pTi = HK_NULL;
                int doneIdx = doneTextures.indexOf(textureFile);
                if ( doneIdx >= 0)
                {
                    // done already
                    pTi = replacementTextures[doneIdx];
                    if (pTi) pTi->addReference();
                }
                else
                {

                    // add so we don't try again
                    doneTextures.pushBack(textureFile);
                    replacementTextures.pushBack(HK_NULL);

                    hkStringOld filename = textureFile->m_filename;
                    hkStringOld filenameLowerCase =  filename.asLowerCase();

                    pTi = new hkxTextureInplace();

                    // types we don't convert / don't need to convert, but can imbed
                    bool isDDS = filenameLowerCase.endsWith("dds");
                    bool isTGA = filenameLowerCase.endsWith("tga");

                    const char* assetFolder = hctFilterUtils::getEnvironmentVariable(data, "assetFolder");
                    hkStringOld fullPath;
                    if (assetFolder && (filename.beginsWith(".") || (filename.lastIndexOf(':')<0) ) && (!filename.beginsWith("\\")))
                    {
                        fullPath.printf("%s/%s", assetFolder, filename.cString());
                    }
                    else
                    {
                        fullPath = filename;
                    }
                    const char* fn = fullPath.cString();

                    if ( isDDS || isTGA )
                    {
                        hkArray<char>::Temp pFile;
                        if( hkLoadUtil(fn).toArray(pFile) )
                        {
                            if (pFile.getSize()< 1)
                            {
                                HK_WARN_ALWAYS(0xabba7ed3, "Couldn't load file: " << filename.cString());
                                delete pTi;
                                continue;
                            }

                            pTi->m_data.clear();
                            pTi->m_data.append( reinterpret_cast<hkUint8*>(pFile.begin()), pFile.getSize());
                            HK_WARN_ALWAYS(0xabba7e90, "Texture was not resized or converted: " << filename.cString() ); // EXP-2487 Warn user
                            Log_Info( "Embedded raw texture file: {}", filename.cString() );
                        }
                        else
                        {
                            HK_WARN_ALWAYS(0xabba66d5, "Texture file " << filename.cString() << " not found.");
                            pTi->removeReference();
                            continue;
                        }
                    }
                    else // try to convert to png
                    {

                        int nLen = MultiByteToWideChar(CP_ACP, 0,fn, -1, NULL, NULL);
                        LPWSTR lpszFilenameW = new WCHAR[nLen];
                        MultiByteToWideChar(CP_ACP, 0, fn, -1, lpszFilenameW, nLen);
                        // use it to call OLE here

                        Log_Info( "Converting {}", fullPath.cString() );

                        Bitmap image( lpszFilenameW );
                        delete [] lpszFilenameW;

                        // Render to a proper sized image
                        Bitmap* bm = &image;
                        if (image.GetLastStatus()!=Ok)
                        {
                            HK_WARN_ALWAYS(0xabba7ed3, "Couldn't load texture file: " << filename.cString());
                            pTi->removeReference();
                            continue;
                        }

                        hkUint32 rw = hkRoundDownPow2( bm->GetWidth());
                        hkUint32 rh = hkRoundDownPow2( bm->GetHeight());

                        if( m_options.m_enable )
                        {
                            rw = m_options.m_u;
                            rh = m_options.m_v;
                        }

                        // texture resizing/reformatting
                        Bitmap* resized = HK_NULL;
                        if ( (rw != bm->GetWidth()) || (rh != bm->GetHeight()) ||
                             (bm->GetPixelFormat() == PixelFormat1bppIndexed) ||
                             (bm->GetPixelFormat() == PixelFormat4bppIndexed) ||
                             (bm->GetPixelFormat() == PixelFormat8bppIndexed) )
                        {
                            Log_Info( "Resizing ({},{}) to ({},{})", bm->GetWidth(), bm->GetHeight(), rw, rh );
                            resized = new Bitmap( rw, rh, PixelFormat32bppARGB);
                            Graphics* g = Graphics::FromImage(resized);
                            g->Clear(Color(255,255,255,255));
                            //g->SetCompositingMode(CompositingModeSourceCopy);
                            g->SetInterpolationMode(InterpolationModeHighQuality);
                            RectF rect( 0, 0, (Gdiplus::REAL)rw, (Gdiplus::REAL)rh );
                            g->DrawImage(bm, rect,0,0,(Gdiplus::REAL)bm->GetWidth(), (Gdiplus::REAL)bm->GetHeight(), UnitPixel);
                            delete g;

                            bm = resized; // use the resized version
                        }

                        // Get encoder class id for png compression
                        // for other compressions use
                        //    image/bmp
                        //    image/jpeg
                        //    image/gif
                        //    image/tiff

                        CLSID pngClsid;
                        _GetEncoderClsid(L"image/png", &pngClsid);

                        // Setup encoder parameters
                        EncoderParameters encoderParameters;
                        encoderParameters.Count = 1;
                        encoderParameters.Parameter[0].Guid = EncoderQuality;
                        encoderParameters.Parameter[0].Type = EncoderParameterValueTypeLong;
                        encoderParameters.Parameter[0].NumberOfValues = 1;

                        // setup compression level
                        ULONG quality = 100;
                        encoderParameters.Parameter[0].Value = &quality;

                        // Create stream with 0 size
                        IStream* pIStream    = NULL;
                        if(CreateStreamOnHGlobal(NULL, TRUE, (LPSTREAM*)&pIStream) != S_OK)
                        {
                            //  AfxMessageBox(_T("Failed to create stream on global memory!"));
                            pTi->removeReference();
                            continue;
                        }

                        //  Save the image to the stream
                        Status SaveStatus = bm->Save(pIStream, &pngClsid, &encoderParameters);
                        if (resized) delete resized;

                        if(SaveStatus != Ok)
                        {
                            // this should free global memory used by the stream
                            // according to MSDN
                            pIStream->Release();
                        //  AfxMessageBox(_T("Failed to save to stream!"));
                            pTi->removeReference();
                            continue;
                        }

                        // get the size of the stream
                        ULARGE_INTEGER ulnSize;
                        LARGE_INTEGER lnOffset;
                        lnOffset.QuadPart = 0;
                        if(pIStream->Seek(lnOffset, STREAM_SEEK_END, &ulnSize) != S_OK)
                        {
                            pIStream->Release();
                        //  AfxMessageBox(_T("Failed to get the size of the stream!"));
                            pTi->removeReference();
                            continue;
                        }

                        // now move the pointer to the beginning of the file
                        if(pIStream->Seek(lnOffset, STREAM_SEEK_SET, NULL) != S_OK)
                        {
                            pIStream->Release();
                        //  AfxMessageBox(_T("Failed to move the file pointer to "
                        //      "the beginning of the stream!"));
                            pTi->removeReference();
                            continue;
                        }

                        int buffSize = (int)ulnSize.QuadPart;
                        pTi->m_data.setSize(buffSize);
                        unsigned char* pBuff = pTi->m_data.begin();

                        // Read the stream directly into the buffer
                        ULONG ulBytesRead;
                        if(pIStream->Read(pBuff, (ULONG)ulnSize.QuadPart, &ulBytesRead) != S_OK)
                        {
                            pIStream->Release();
                            pTi->removeReference();
                            continue;
                        }

                        Log_Info( "Converted and embedded {}", filename.cString() );

                        // Free memory used by the stream
                        pIStream->Release();
                    }

                    // make our new texture

                    pTi->m_originalFilename = textureFile->m_originalFilename? textureFile->m_originalFilename : textureFile->m_filename;
                    pTi->m_name = textureFile->m_name;

                    if (isTGA)
                    {
                        pTi->m_fileType[0] = 'T'; pTi->m_fileType[1] = 'G'; pTi->m_fileType[2] = 'A';
                    }
                    else if (isDDS)
                    {
                        pTi->m_fileType[0] = 'D'; pTi->m_fileType[1] = 'D'; pTi->m_fileType[2] = 'S';
                    }
                    else
                    {
                        pTi->m_fileType[0] = 'P'; pTi->m_fileType[1] = 'N'; pTi->m_fileType[2] = 'G';
                    }
                    pTi->m_fileType[3] = '\0';

                    replacementTextures[replacementTextures.getSize()-1] = pTi;
                }

                // Replace it!
                if (pTi)
                {
                    material->m_stages[net].m_texture = pTi;
                    pTi->removeReference();
                }
            }

        } // all mats
        GdiplusShutdown(gdiplusToken);
    }
}


void hctConvertTexturesToPNGFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctConvertTexturesToPNGOptions* options = hctFilterUtils::getNativeOptions<hctConvertTexturesToPNGOptions>(optVar))
    {
        m_options = *options;
        delete options;
    }
}

void hctConvertTexturesToPNGFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
