// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/Filters/FilterScene/AlignSceneToNode/hctAlignSceneToNodeFilter.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/AlignScene/hkAlignSceneToNodeOptions.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.scene.aligntonode"
#include <Common/Base/System/Log/hkLog.hxx>


hctAlignSceneToNodeFilterDesc g_alignSceneToNodeDesc;

hctAlignSceneToNodeFilter::hctAlignSceneToNodeFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner),
    m_optionsDialog(NULL), m_fillingControls (false), m_doNotRefreshEdit(false)
{
}

hctAlignSceneToNodeFilter::~hctAlignSceneToNodeFilter()
{
}

void hctAlignSceneToNodeFilter::similarity( hkMatrix4& result, const hkMatrix4& T, const hkMatrix4& M )
{
    hkMatrix4 tmp;
    tmp.setMul( M, T );

    hkMatrix4 Ti;   Ti.setInverse( T, hkSimdReal::fromFloat(1.0e-3f) );

    // T^-1 * M * T
    result.setMul( Ti, tmp );
}

void hctAlignSceneToNodeFilter::process( class hkRootLevelContainer& data )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL)
    {
        HK_WARN_ALWAYS (0xabbaa5f0, "No scene data found");
        return;
    }

    // Ensure the scene root exists
    if ( scenePtr->m_rootNode == HK_NULL )
    {
        HK_WARN_ALWAYS (0xabbaa5f0, "No scene data found");
        return;
    }

    // Ensure that the root node has a keyframe
    if ( scenePtr->m_rootNode->m_keyFrames.getSize() <= 0 )
    {
        HK_WARN_ALWAYS(0xabbaa5e4, "Node had no keyframe data " << scenePtr->m_rootNode->m_name );
        return;
    }

    // Ensure that a node name was specified
    if ( hkString::strLen( m_options.getNodeName() ) <= 0 )
    {
        HK_WARN_ALWAYS(0xabbaa5e4, "No node name specified.  Please choose a node." );
        return;
    }

    // Find the desired node
    hkxNode* refFrameNode = scenePtr->findNodeByName( m_options.getNodeName() );
    if (refFrameNode == HK_NULL)
    {
        HK_WARN_ALWAYS(0xabbaa5e4, "Could not find node: " << m_options.getNodeName() );
        return;
    }

    // Ensure that the chosen node has a keyframe
    if ( refFrameNode->m_keyFrames.getSize() <= 0 )
    {
        HK_WARN_ALWAYS(0xabbaa5e4, "Node had no keyframe data " << m_options.getNodeName() );
        return;
    }

    hkResult res;

    // Find the transform of the current node at the desired frame
    hkMatrix4 worldFromNode;
    res = scenePtr->getWorldFromNodeTransform (refFrameNode, worldFromNode, m_options.m_keyframe );
    if ( res.isFailure() )
    {
        HK_WARN_ALWAYS(0xabbaa5e4, "Keyframe " << m_options.m_keyframe << " is out of bounds for node " << m_options.getNodeName() );
        return;
    }

    // Work with a copy transform
    hkMatrix4 xform;
    xform = worldFromNode;

    // Modify the transform of all children of the root
    // (Done this way as other filters may assume that the root node has an identity transform)
    // Decompose
    if ( m_options.needToDecompose() )
    {
        hkMatrixDecomposition::Decomposition decomp;
        hkMatrixDecomposition::decomposeMatrix( xform, decomp );

        hkMatrix4 mat4;
        mat4.setIdentity();

        // Rotation
        if ( m_options.m_transformRotation )
        {
            hkTransform rotation; rotation.set(decomp.m_rotation, hkVector4::getZero());
            hkMatrix4 m4temp; m4temp.set4x4ColumnMajor((hkReal*) &rotation);

            mat4.mul( m4temp);
        }

        // Scale
        if ( m_options.m_transformScale && decomp.m_hasScale )
        {
            hkMatrix4 scale;
            scale.setDiagonal( decomp.m_scale( 0 ), decomp.m_scale( 1 ), decomp.m_scale( 2 ), 1.0f );
            mat4.mul( scale );
        }

        // Skew
        if ( m_options.m_transformSkew && decomp.m_hasSkew )
        {
            mat4.mul( decomp.m_skew );
        }

        // Recompose
        xform.setZero();
        xform.setCols( mat4.getColumn( 0 ), mat4.getColumn( 1 ), mat4.getColumn( 2 ), decomp.m_translation );

        if ( !m_options.m_transformPositionX ) xform( 0, 3 ) = 0.0f;
        if ( !m_options.m_transformPositionY ) xform( 1, 3 ) = 0.0f;
        if ( !m_options.m_transformPositionZ ) xform( 2, 3 ) = 0.0f;

        xform.resetFourthRow();
    }

    // Invert the transform
    // Do this *after* decomposition or the translation and rotation will get mixed
    if ( m_options.m_invert )
    {
        xform.setInverse(xform, hkSimdReal::fromFloat(1.0e-3f) );
    }

    // Apply a similarity transform (accounts for being a child of the root)
    // It's likely guaranteed that the root transform was identity, but do this just in case
    // This code only accounts for the case of a static scene transform
    similarity( xform, scenePtr->m_rootNode->m_keyFrames[ 0 ], xform );

    // Modify each child of the root
    for ( hkInt32 c = 0; c < scenePtr->m_rootNode->m_children.getSize(); c++ )
    {
        hkxNode* child = scenePtr->m_rootNode->m_children[ c ];

        // For each child keyframe
        for ( hkInt32 f = 0; f < child->m_keyFrames.getSize(); f++ )
        {
            hkMatrix4& keyframe = child->m_keyFrames[ f ];
            hkMatrix4 tmp = keyframe;

            // Transform the keyframe
            keyframe.setMul( xform, tmp );
        }
    }

    // Add the alignment info to the rootLevelContainer
    {
        // Create a new options class instance with a copy of the current options
        hkAlignSceneToNodeOptions* options = new hkAlignSceneToNodeOptions;

        // Copy over the results
        {
            options->m_invert = m_options.m_invert;
            options->m_transformPositionX = m_options.m_transformPositionX;
            options->m_transformPositionY = m_options.m_transformPositionY;
            options->m_transformPositionZ = m_options.m_transformPositionZ;
            options->m_transformRotation = m_options.m_transformRotation;
            options->m_transformScale = m_options.m_transformScale;
            options->m_keyframe = m_options.m_keyframe;

            options->setNodeName( m_options.getNodeName() );
        }

        // Assign to the end of the array
        data.m_namedVariants.expandOne().set( "Align Scene To Node Options", options);

        options->removeReference();
    }

    Log_Info( "Successfully aligned node: {}", m_options.getNodeName() );
}

void hctAlignSceneToNodeFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctAlignSceneToNodeOptions* options = hctFilterUtils::getNativeOptions<hctAlignSceneToNodeOptions>(optVar))
    {
        m_options = *options;
        delete options;
    }
}

void hctAlignSceneToNodeFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
