// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>

#include <ContentTools/Common/Filters/FilterPhysics2012/OptimizeShapeHierarchy/hctOptimizeShapeHierarchyFilter.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>

extern HINSTANCE hInstance;

INT_PTR CALLBACK _optimizeShapeHierarchyDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    // We store a pointer to the filter associated with this dialog using Get/SetWindowLongPtr()
    hctOptimizeShapeHierarchyFilter* filter = reinterpret_cast<hctOptimizeShapeHierarchyFilter*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            filter = (hctOptimizeShapeHierarchyFilter*)lParam;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            filter->m_optionsDialog = hWnd;

            filter->setControlsFromData();

            // Initialize Tool Tips
            {
                CreateToolTip(IDC_CB_Share, hWnd, hInstance, "If enabled, the filter identifies identical (or near-identical) shape structures in one or multiple rigid bodies and changes the hierarchy so such shapes are shared. It replaces \"cloned\" shapes or shapes structures with \"instanced\" ones.");
                CreateToolTip(IDC_ED_TOLERANCE, hWnd, hInstance, "Specifies a tolerance used to decide whether two shapes should be considered identical (and thus replaced with one, shared shape).");
                CreateToolTip(IDC_CB_Permute, hWnd, hInstance, "If enabled, the filter will check for permutations of vertices/triangles/planes when comparing hkpSimpleMeshShape or hkpConvexVerticesShape objects. This will possibly detect more instanced shapes but due to the increased amount of comparisons it will take longer time. When the option is off vertices, triangles and planes are compared using the same order.");
                CreateToolTip(IDC_CB_CollapseTransforms, hWnd, hInstance, "If enabled, the filter attempts to collapse transform (and convex transform or convex translate) shapes into their child shape. For example, a transform shape on top of a capsule shape can be collapsed into a new capsule shape with transformed vertices.");
                CreateToolTip(IDC_RB_AlwaysCollapse, hWnd, hInstance, "Transform shapes will be collapsed even if they point to a shared shape.");
                CreateToolTip(IDC_RB_NeverCollapse, hWnd, hInstance, "Transform shapes won't be collapsed into shared shapes.");
                CreateToolTip(IDC_RB_CollapseIfLessThan, hWnd, hInstance, "Transform shapes will not be collapsed if they point to a shape shared by a set number of shapes/rigid bodies or more. Specifying a number <=2 is equivalent to Never Collapse.");
                CreateToolTip(IDC_ED_THRESHOLDNUM, hWnd, hInstance, "Transform shapes will not be collapsed if they point to a shape shared by this many number of shapes/rigid bodies or more. Specifying a number <=2 is equivalent to Never Collapse.");
                CreateToolTip(IDC_CB_Propagate, hWnd, hInstance, "If enabled, the filter will propagate any transform shapes applied to list shapes into the children of the list. This may reduce the depth on some of the branches if the transform can be collapsed in any of the children - however if the transform cannot be collapsed into the children this may actually increase the total number of shapes.");
            }

            return TRUE; // did handle it
        }

        case WM_COMMAND: // UI Changes
            {
                filter->enableControls();
                break;
            }
    }
    return FALSE; //didn't handle it
}



void hctOptimizeShapeHierarchyFilter::setDataFromControls()
{
    // Ensure the options we store match the options shown in the UI
    if (!m_optionsDialog) return;

    // Share shapes stuff
    {
        m_options.m_shareShapes = IsDlgButtonChecked(m_optionsDialog, IDC_CB_Share) == TRUE;
        m_options.m_permuteDetect = IsDlgButtonChecked(m_optionsDialog, IDC_CB_Permute) == TRUE;

        hctLocaleScope scope;

        TCHAR text[256];
        GetWindowText(GetDlgItem(m_optionsDialog, IDC_ED_TOLERANCE), text, 256);
        m_options.m_shareTolerance = (float)atof(text);
    }

    // Collapse transforms stuff
    {
        m_options.m_collapseTransforms = IsDlgButtonChecked(m_optionsDialog, IDC_CB_CollapseTransforms) == TRUE;

        m_options.m_collapseBehaviourType = hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_ALWAYS;

        if (IsDlgButtonChecked(m_optionsDialog, IDC_RB_AlwaysCollapse))
        {
            m_options.m_collapseBehaviourType = hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_ALWAYS;
        }
        else if (IsDlgButtonChecked(m_optionsDialog, IDC_RB_NeverCollapse))
        {
            m_options.m_collapseBehaviourType = hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_NEVER;
        }
        else if (IsDlgButtonChecked(m_optionsDialog, IDC_RB_CollapseIfLessThan))
        {
            m_options.m_collapseBehaviourType = hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_THRESHOLD;
        }

        TCHAR text[256];
        GetWindowText(GetDlgItem(m_optionsDialog, IDC_ED_THRESHOLDNUM), text, 256);
        m_options.m_collapseThreshold = atoi(text);

        m_options.m_propagate = IsDlgButtonChecked(m_optionsDialog, IDC_CB_Propagate) == TRUE;

    }
}

void hctOptimizeShapeHierarchyFilter::setControlsFromData()
{
    if (!m_optionsDialog) return;


    // Share Shapes stuff
    {
        // Sharing behaviour
        CheckDlgButton(m_optionsDialog, IDC_CB_Share, m_options.m_shareShapes);
        CheckDlgButton(m_optionsDialog, IDC_CB_Permute, m_options.m_permuteDetect);

        hctLocaleScope scope;

        // "tolerance" edit box
        hkStringOld s;
        s.printf("%f", m_options.m_shareTolerance);
        SetDlgItemText(m_optionsDialog, IDC_ED_TOLERANCE, s.cString());
    }

    // Collapse transforms stuff
    {
        // Collapse behaviour
        CheckDlgButton(m_optionsDialog, IDC_CB_CollapseTransforms, m_options.m_collapseTransforms);
        // "threshold" edit box


        const int firstRB = IDC_RB_AlwaysCollapse;
        const int lastRB = IDC_RB_CollapseIfLessThan;
        BOOL res;
        switch (m_options.m_collapseBehaviourType)
        {
            case hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_ALWAYS:
                res =CheckRadioButton(m_optionsDialog, firstRB, lastRB, IDC_RB_AlwaysCollapse);
                break;
            case hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_NEVER:
                res =CheckRadioButton(m_optionsDialog, firstRB, lastRB, IDC_RB_NeverCollapse);
                break;
            case hctOptimizeShapeHierarchyOptions::COLLAPSE_OPTIONS_THRESHOLD:
                res = CheckRadioButton(m_optionsDialog, firstRB, lastRB, IDC_RB_CollapseIfLessThan);
                break;

            default:
                break;

        }

        hkStringOld s;
        s.printf("%d", m_options.m_collapseThreshold);
        SetDlgItemText(m_optionsDialog, IDC_ED_THRESHOLDNUM, s.cString());

        CheckDlgButton(m_optionsDialog, IDC_CB_Propagate, m_options.m_propagate);
    }

    enableControls();
}

void hctOptimizeShapeHierarchyFilter::enableControls()
{
    const BOOL shareEnabled = IsDlgButtonChecked(m_optionsDialog, IDC_CB_Share);
    {
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_LAB_Tolerance), shareEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_ED_TOLERANCE), shareEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_CB_Permute), shareEnabled);
    }

    const BOOL collapseEnabled = IsDlgButtonChecked(m_optionsDialog, IDC_CB_CollapseTransforms);
    const BOOL thresholdEnabled = IsDlgButtonChecked(m_optionsDialog,IDC_RB_CollapseIfLessThan);
    {
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_GB_SharedShapes), collapseEnabled);

        EnableWindow(GetDlgItem(m_optionsDialog, IDC_RB_AlwaysCollapse), collapseEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_RB_NeverCollapse), collapseEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_RB_CollapseIfLessThan), collapseEnabled);

        EnableWindow(GetDlgItem(m_optionsDialog, IDC_ED_THRESHOLDNUM), collapseEnabled && thresholdEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_CB_Propagate), collapseEnabled);
        EnableWindow(GetDlgItem(m_optionsDialog, IDC_LAB_References), collapseEnabled);
    }

}


HWND hctOptimizeShapeHierarchyFilter::showOptions(HWND owner)
{
    if (m_optionsDialog)
    {
        hideOptions();
    }

    m_optionsDialog = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_OPTIMIZE_SHAPE_DIALOG),
                        owner, _optimizeShapeHierarchyDialogProc, (LPARAM) this );

    return m_optionsDialog;
}

void hctOptimizeShapeHierarchyFilter::hideOptions()
{
    // Update any changes before we close UI
    setDataFromControls();

    if (m_optionsDialog)
    {
        DestroyWindow(m_optionsDialog);
    }

    m_optionsDialog = NULL;

}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
