// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateWorld/hctCreateWorldFilter.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>

#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.h>
#include <Physics2012/Collide/Filter/Group/hkpGroupFilterSetup.h>

#include <stdlib.h>

extern HINSTANCE hInstance;
#define NUM_LAYERS 32 // can't really be more than 32 without changing from hkUint32 below anyway

static void _fillComboWithEnum(HWND combo, const hk::Presets& enums, const char* removeStart, int initialVal, int* ignoreValues = HK_NULL)
{
    SendMessage(combo, CB_RESETCONTENT, 0, 0);
    int startLen = removeStart? hkString::strLen(removeStart) : 0;
    int curSel = 0;
    int addedEntries = 0;
    for (int ei=0; ei < enums.getNumPresets(); ++ei)
    {
        hkReflect::IntVar intVar(enums.getPreset(ei));
        HK_ASSERT_NO_MSG(0x6e1715ad, intVar);
        int value = intVar.getValue().convertTo<int>();

        int iv=0;
        while (ignoreValues && (ignoreValues[iv] != -1))
        {
            if (ignoreValues[iv] == value)
            {
                iv = -1;
                break;
            }
            iv++;
        }
        if (iv == -1)
            continue;

        hkStringOld entryName = enums.getPresetName(ei);
        if (startLen > 0)
        {
            entryName = entryName.substr(startLen);
        }

        // ignore standard Havok enum 0 and max values:
        if (entryName.endsWith("INVALID") ||
            entryName.endsWith("MAX_ID") )
            continue;

        // add to list
        SendMessage(combo, CB_ADDSTRING, 0, (LPARAM)entryName.cString() );

        // is this the one we want to select?
        if (value == initialVal)
            curSel = addedEntries;

        SendMessage(combo, CB_SETITEMDATA, addedEntries, value );

        addedEntries++;
    }

    SendMessage(combo, CB_SETCURSEL, curSel, 0 );
}

static int _getComboEnumValue(HWND combo)
{
    // get the current selection
    int sel = (int)SendMessage(combo, CB_GETCURSEL, 0, 0);
    if (sel != CB_ERR)
    {
        return (int)SendMessage(combo, CB_GETITEMDATA, sel, 0);
    }
    return 0;
}


static UINT deactivationIDs[] =
{
    IDC_EDIT_DEACTIVATION_LOW,
    IDC_EDIT_DEACTIVATION_HIGH,
    IDC_STATIC_DEACTIVATION_LOW,
    IDC_STATIC_DEACTIVATION_HIGH
};
static UINT numDeactivationIDs = sizeof(deactivationIDs)/sizeof(UINT);

// All IDs based on the create world check box state (so we grat them)
static UINT createWorldIDs[] =
{
    IDC_EDIT_GRAVITY_X            ,
    IDC_EDIT_GRAVITY_Y            ,
    IDC_EDIT_GRAVITY_Z            ,
    IDC_COMBO_SIMULATION_TYPE     ,
    IDC_CHECK_ENABLE_DEACTIVATION ,
    IDC_EDIT_DEACTIVATION_LOW     ,
    IDC_EDIT_DEACTIVATION_HIGH    ,
    IDC_STATIC_DEACTIVATION_LOW   ,
    IDC_STATIC_DEACTIVATION_HIGH  ,
    IDC_COMBO_SOLVER_TYPE         ,
    IDC_EDIT_COLLISION_TOL        ,
    IDC_COMBO_COLLISION_FILTER    ,
    IDC_BUTTON_DEFINE             ,
    IDC_CHECK_AUTO_CONSTRAINED    ,
    IDC_EDIT_AUTO_LAYER           ,
    IDC_STATIC_GRAVITY            ,
    IDC_STATIC_SIM                ,
    IDC_STATIC_SOLVER             ,
    IDC_STATIC_ADV_OPTIONS        ,
    IDC_STATIC_COL_TOL            ,
    IDC_STATIC_LAYER              ,
    IDC_STATIC_FILTERING
};
static UINT numCreateWorldIDs = sizeof(createWorldIDs)/sizeof(UINT);

static int SimTypesIgnored[] =
{
    hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED,
    -1
};

static void _enableGroupBasedOnCheck(HWND dlg, UINT checkBox, UINT* ids, int numIds)
{
    BOOL enabled = IsDlgButtonChecked(dlg, checkBox) && IsWindowEnabled(GetDlgItem(dlg, checkBox));
    for (int i=0; i < numIds; ++i)
    {
        EnableWindow(GetDlgItem(dlg, ids[i]), enabled);
    }
}

struct GroupFilterData
{
    hkUint32 curLayer;
    hkUint32 tempLayerState[NUM_LAYERS];
};

static void _reflectLayerState( HWND dlgWnd, hkUint32 curLayerState )
{
    UINT32 checkID = IDC_CHECK_LAYER1;
    hkUint32 bitfield = 2; // 1 == layer0

    for (int li=1; li < NUM_LAYERS; ++li)
    {
        BOOL enabled = curLayerState & bitfield;
        CheckDlgButton(dlgWnd, checkID, enabled);

        // shift to next layer
        bitfield = bitfield << 1;
        checkID++;
    }
}

static void _getLayerState( HWND dlgWnd, hkUint32 curLayer, hkUint32 allLayerState[NUM_LAYERS] )
{
    UINT32 checkID = IDC_CHECK_LAYER1;
    hkUint32 bitfield = 2; // 1 == layer0, which we skip

    hkUint32& curLayerState = allLayerState[curLayer];
    hkUint32 curLayerBitfield = 1 << curLayer;
    hkUint32 notCurLayerBitfield = ~curLayerBitfield;

    curLayerState = 1; // always enable layer0 (bit 0)
    for (int li=1; li < NUM_LAYERS; ++li)
    {
        BOOL enabled = IsDlgButtonChecked(dlgWnd, checkID);
        hkUint32& otherLayerState = allLayerState[li];
        if (enabled)
        {
            curLayerState |= bitfield;

            // as symmetric, if we enable between li and curlayer we
            // must also enable curlayer and li:
            otherLayerState |= curLayerBitfield;
        }
        else
        {
            // we disable the li and curlayer in li's state too:
            otherLayerState &= notCurLayerBitfield;
        }

        // shift to next layer
        bitfield = bitfield << 1;
        checkID++;
    }
}


INT_PTR CALLBACK hkFilterCreateWorldGroupFilterDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    GroupFilterData* filterData = reinterpret_cast<GroupFilterData*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;

    switch(message)
    {
    case WM_INITDIALOG:
        {
            filterData = (GroupFilterData*)lParam;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            // fill the combo
            HWND combo = GetDlgItem(hWnd, IDC_COMBO_LAYER);
            for (int ci=1; ci < NUM_LAYERS; ++ci)
            {
                // add to list
                hkStringOld s; s.printf("%d", ci);
                SendMessage(combo, CB_ADDSTRING, 0, (LPARAM)s.cString());
                SendMessage(combo, CB_SETITEMDATA, ci-1, ci );
            }
            SendMessage(combo, CB_SETCURSEL, filterData->curLayer - 1, 0 ); // select entry 0 (layer 1)

            // reflect current options for layer 1 in the dlg
            _reflectLayerState( hWnd, filterData->tempLayerState[filterData->curLayer] );

            return TRUE;
        }

    case WM_COMMAND:
        {
            switch ( LOWORD(wParam) )
            {
                case IDC_COMBO_LAYER:
                {
                    switch HIWORD(wParam)
                    {
                        case CBN_SELCHANGE:
                        {
                            HWND combo = (HWND)lParam;
                            int sel = (int)SendMessage(combo, CB_GETCURSEL, 0, 0);
                            if (sel != CB_ERR)
                            {
                                int layer = (int)SendMessage(combo, CB_GETITEMDATA, sel, 0);

                                // get old layer info out first
                                _getLayerState( hWnd, filterData->curLayer, filterData->tempLayerState);

                                // set in new layer
                                filterData->curLayer = layer;
                                _reflectLayerState( hWnd, filterData->tempLayerState[filterData->curLayer] );
                            }
                            return TRUE;
                        }
                    }
                    break;
                }

                case IDC_BUTTON_USEDEFAULTS:
                {
                    hkpGroupFilter *f = new hkpGroupFilter();
                    hkpGroupFilterSetup::setupGroupFilter( f );
                    hkString::memCpy( filterData->tempLayerState, f->m_collisionLookupTable, NUM_LAYERS * sizeof(hkUint32) );
                    _reflectLayerState( hWnd, filterData->tempLayerState[filterData->curLayer] );
                    f->removeReference();
                    return TRUE;
                }

                case IDOK:
                    // reflect current state back into the group layers in the options
                    _getLayerState( hWnd, filterData->curLayer, filterData->tempLayerState);
                    EndDialog(hWnd, TRUE);
                    return TRUE;

                case IDCANCEL:
                    EndDialog(hWnd, FALSE);
                    return TRUE;
            }
        }
    }

    return FALSE;
}
static void _updateFilterActiveOptions(HWND hWnd, HWND combo)
{
    int sel = (int)SendMessage(combo, CB_GETCURSEL, 0, 0);
    if (sel != CB_ERR)
    {
        int value = (int)SendMessage(combo, CB_GETITEMDATA, sel, 0);
        BOOL enabled = (value == hctCreateWorldOptions::COLLISION_OPTIONS_GROUP_FILTER);
        EnableWindow(GetDlgItem(hWnd, IDC_BUTTON_DEFINE), enabled);
        EnableWindow(GetDlgItem(hWnd, IDC_CHECK_AUTO_CONSTRAINED), enabled);
        EnableWindow(GetDlgItem(hWnd, IDC_EDIT_AUTO_LAYER), enabled);
        EnableWindow(GetDlgItem(hWnd, IDC_STATIC_LAYER), enabled);
    }
}

INT_PTR CALLBACK hkFilterCreateWorldDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctCreateWorldFilter* filter = reinterpret_cast<hctCreateWorldFilter*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            filter = (hctCreateWorldFilter*)lParam;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            {
                hctLocaleScope scope;

                hkStringOld s;
                hkpWorldCinfo* ci = &(filter->m_options.m_worldCinfo);

                // Gravity
                s.printf("%.3f", (float)( ci->m_gravity.getSimdAt(0) ));
                SetDlgItemText(hWnd, IDC_EDIT_GRAVITY_X, s.cString());
                s.printf("%.3f", (float)( ci->m_gravity.getSimdAt(1) ));
                SetDlgItemText(hWnd, IDC_EDIT_GRAVITY_Y, s.cString());
                s.printf("%.3f", (float)( ci->m_gravity.getSimdAt(2) ));
                SetDlgItemText(hWnd, IDC_EDIT_GRAVITY_Z, s.cString());

                // Sim type
                _fillComboWithEnum(GetDlgItem(hWnd, IDC_COMBO_SIMULATION_TYPE), hkReflect::getPresetsOf<hkpWorldCinfo::SimulationType>(), "SIMULATION_TYPE_", (int)(ci->m_simulationType), SimTypesIgnored );

                // Solver
                _fillComboWithEnum(GetDlgItem(hWnd, IDC_COMBO_SOLVER_TYPE), hkReflect::getPresetsOf<hkpWorldCinfo::SolverType>(), "SOLVER_TYPE_", (int)(filter->m_options.m_solverType) );

                // Collision Filter
                _fillComboWithEnum(GetDlgItem(hWnd, IDC_COMBO_COLLISION_FILTER), hkReflect::getPresetsOf<hctCreateWorldOptions::CollisionFilterOptionsType>(), "COLLISION_OPTIONS_", (int)(filter->m_options.m_collisionFilterType) );

                CheckDlgButton(hWnd, IDC_CHECK_AUTO_CONSTRAINED, filter->m_options.m_autoGroupRBS);
                s.printf("%d", filter->m_options.m_autoGroupRBLayer );
                SetDlgItemText(hWnd, IDC_EDIT_AUTO_LAYER, s.cString());

                // Deactivation
                CheckDlgButton(hWnd, IDC_CHECK_ENABLE_DEACTIVATION, ci->m_enableDeactivation);
                _enableGroupBasedOnCheck(hWnd, IDC_CHECK_ENABLE_DEACTIVATION, deactivationIDs, numDeactivationIDs);

                // Collision Tol
                s.printf("%.3f", (float)( ci->m_collisionTolerance ));
                SetDlgItemText(hWnd, IDC_EDIT_COLLISION_TOL, s.cString());

                // Create World or not
                CheckDlgButton(hWnd, IDC_CHECK_CREATE_WORLD, filter->m_options.m_createWorld);
                _enableGroupBasedOnCheck(hWnd, IDC_CHECK_CREATE_WORLD, createWorldIDs, numCreateWorldIDs);

                // System groups
                CheckDlgButton(hWnd, IDC_RADIO_GROUP_SINGLE, filter->m_options.m_systemGrouping == hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_SINGLE);
                CheckDlgButton(hWnd, IDC_RADIO_GROUP_CONSTRAINED, filter->m_options.m_systemGrouping == hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_CONSTRAINED_HIERARCHIES);

                _updateFilterActiveOptions(hWnd, GetDlgItem(hWnd, IDC_COMBO_COLLISION_FILTER));
            }

            // Initialize Tool Tips
            {
                CreateToolTip(IDC_CHECK_CREATE_WORLD, hWnd, hInstance, "If this box is checked, a world setup will be added to the asset. Please consult the Havok Dynamics chapter for details on these parameters and how they may affect the simulation.");
                CreateToolTip(IDC_EDIT_GRAVITY_X, hWnd, hInstance, "Set X component of gravity vector");
                CreateToolTip(IDC_EDIT_GRAVITY_Y, hWnd, hInstance, "Set Y component of gravity vector");
                CreateToolTip(IDC_EDIT_GRAVITY_Z, hWnd, hInstance, "Set Z component of gravity vector");
                CreateToolTip(IDC_COMBO_SIMULATION_TYPE, hWnd, hInstance, "");
                CreateToolTip(IDC_COMBO_SOLVER_TYPE, hWnd, hInstance, "Used to specify the speed/stiffness balance of the core solver. Each constraint will be solved like a very stiff spring, which gives the solver a 'soft' feeling. So the goal of the solver is to make the constraint appear hard.\nITERS refers to the number of iterations, SOFT/MEDIUM/HARD refers to the dampening.");
                CreateToolTip(IDC_CHECK_ENABLE_DEACTIVATION, hWnd, hInstance, "");
                CreateToolTip(IDC_EDIT_COLLISION_TOL, hWnd, hInstance, "This value sets the collision tolerance.");
                CreateToolTip(IDC_EDIT_DEACTIVATION_LOW, hWnd, hInstance, "The low deactivation period.");
                CreateToolTip(IDC_EDIT_DEACTIVATION_HIGH, hWnd, hInstance, "The high deactivation period.");
                CreateToolTip(IDC_COMBO_LandscapeWelding, hWnd, hInstance, "Collision filters are used in order to determine which collisions between rigid bodies should be ignored. Many algorithms (collision filters) are provided in the Havok Physics SDK to do so. The options here allow you to choose between three options.");
                CreateToolTip(IDC_CHECK_AUTO_CONSTRAINED, hWnd, hInstance, "If checked, IDs will be assigned to rigid bodies so the GROUP_FILTER will disable collisions between constrained rigid bodies, as the CONSTRAINT_FILTER does (but at a smaller cost). You can also define an specific layer ID for all the rigid bodies in the current asset, as well as defining which layers should collide with which other layer. Again, check the Physics SDK documentation on more details about how this filter works and how it interprets the different rigid body IDs.");
                CreateToolTip(IDC_BUTTON_DEFINE, hWnd, hInstance, "Allows definition of a specific layer ID for all the rigid bodies in the current asset, as well as defining which layers should collide with which other layer.");
                CreateToolTip(IDC_EDIT_AUTO_LAYER, hWnd, hInstance, "");
                CreateToolTip(IDC_RADIO_GROUP_SINGLE, hWnd, hInstance, "If checked, this option will put all rigid bodies and constraints in a single system.");
                CreateToolTip(IDC_RADIO_GROUP_CONSTRAINED, hWnd, hInstance, "If checked, this option will split the rigid bodies and constraints currently in the scene into multiple Physics Systems by looking at how rigid bodies are connected between them and whether they are fixed or not.");
            }

            return TRUE; // did handle it
        }

        case WM_COMMAND: // UI Changes
        {
            switch ( LOWORD(wParam) )
            {
                case IDC_CHECK_CREATE_WORLD:
                    _enableGroupBasedOnCheck(hWnd, IDC_CHECK_CREATE_WORLD, createWorldIDs, numCreateWorldIDs);
                    _updateFilterActiveOptions(hWnd, GetDlgItem(hWnd, IDC_COMBO_COLLISION_FILTER));
                    // no break, update deactive too
                case IDC_CHECK_ENABLE_DEACTIVATION:
                    _enableGroupBasedOnCheck(hWnd, IDC_CHECK_ENABLE_DEACTIVATION, deactivationIDs, numDeactivationIDs);
                    break;

                case IDC_COMBO_COLLISION_FILTER:
                    switch HIWORD(wParam)
                    {
                        case CBN_SELCHANGE:
                        {
                            _updateFilterActiveOptions(hWnd, (HWND)lParam);
                            break;
                        }
                    }
                    break;

                case IDC_BUTTON_DEFINE:
                    {
                        GroupFilterData fd;
                        fd.curLayer = 1;
                        hkString::memCpy( fd.tempLayerState, filter->m_options.m_groupFilterLayerTable, NUM_LAYERS * sizeof(hkUint32) );
                        BOOL res = (BOOL)DialogBoxParamW(hInstance, MAKEINTRESOURCEW(IDD_GROUPFILTER_DIALOG), hWnd, hkFilterCreateWorldGroupFilterDialogProc, (LPARAM)&fd);
                        if (res) // save the filter info back into the options
                        {
                            hkString::memCpy( filter->m_options.m_groupFilterLayerTable, fd.tempLayerState, NUM_LAYERS * sizeof(hkUint32) );
                        }
                        return TRUE;
                    }
            }
            break;
        }
    }

    return FALSE; //didn't handle it / didn't do much with it
}

HWND hctCreateWorldFilter::showOptions(HWND owner)
{
    if (m_optionsDialog)
        hideOptions();

    // figure out some per modeller defaults
    if (!m_haveSetOptions)
    {
        hkxScene* s = getFilterManager()->getOriginalContents()->findObject<hkxScene>();
        if (s && s->m_modeller)
        {
            hkStringOld modeller = (hkStringOld(s->m_modeller)).asLowerCase();
            if (modeller.beginsWith("3ds"))
                m_options.m_worldCinfo.m_gravity.set(0,0,-9.81f);
            else
                m_options.m_worldCinfo.m_gravity.set(0,-9.81f,0);
        }
    }

    m_optionsDialog = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_CREATE_WORLD_DIALOG),
        owner, hkFilterCreateWorldDialogProc, (LPARAM) this );

    return m_optionsDialog;
}

void hctCreateWorldFilter::updateOptions()
{
    if (m_optionsDialog)
    {
        char tempText[128];

        hkpWorldCinfo& ci = m_options.m_worldCinfo;

        hctLocaleScope scope;

        // Gravity
        GetDlgItemText(m_optionsDialog, IDC_EDIT_GRAVITY_X, tempText, 128);
        float gx = (float)atof(tempText);
        GetDlgItemText(m_optionsDialog, IDC_EDIT_GRAVITY_Y, tempText, 128);
        float gy = (float)atof(tempText);
        GetDlgItemText(m_optionsDialog, IDC_EDIT_GRAVITY_Z, tempText, 128);
        float gz = (float)atof(tempText);
        ci.m_gravity.set(gx, gy, gz);

        // Sim type
        ci.m_simulationType = (hkpWorldCinfo::SimulationType)_getComboEnumValue(GetDlgItem(m_optionsDialog, IDC_COMBO_SIMULATION_TYPE) );

        // Solver
        m_options.m_solverType = _getComboEnumValue(GetDlgItem(m_optionsDialog, IDC_COMBO_SOLVER_TYPE) );
        if ((m_options.m_solverType <= hkpWorldCinfo::SOLVER_TYPE_INVALID) ||
            (m_options.m_solverType >= hkpWorldCinfo::SOLVER_TYPE_MAX_ID))
            m_options.m_solverType = hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM;

        ci.setupSolverInfo( (hkpWorldCinfo::SolverType)m_options.m_solverType );

        // Collision Filter
        m_options.m_collisionFilterType = (hctCreateWorldOptions::CollisionFilterOptionsType)_getComboEnumValue(GetDlgItem(m_optionsDialog, IDC_COMBO_COLLISION_FILTER) );
        m_options.m_autoGroupRBS = IsDlgButtonChecked(m_optionsDialog, IDC_CHECK_AUTO_CONSTRAINED)? true : false;
        if ( m_options.m_autoGroupRBS )
        {
            GetDlgItemText(m_optionsDialog, IDC_EDIT_AUTO_LAYER, tempText, 128);
            m_options.m_autoGroupRBLayer = (int)atoi(tempText);
        }

        // Deactivation
        ci.m_enableDeactivation = IsDlgButtonChecked(m_optionsDialog, IDC_CHECK_ENABLE_DEACTIVATION)? true : false;

        // Collision Tol
        GetDlgItemText(m_optionsDialog, IDC_EDIT_COLLISION_TOL, tempText, 128);
        ci.m_collisionTolerance = (float)atof(tempText);

        // Create world
        m_options.m_createWorld = IsDlgButtonChecked(m_optionsDialog, IDC_CHECK_CREATE_WORLD)? true : false;

        // System grouping
        if ( IsDlgButtonChecked(m_optionsDialog, IDC_RADIO_GROUP_SINGLE) )
            m_options.m_systemGrouping = hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_SINGLE;
        else
            m_options.m_systemGrouping = hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_CONSTRAINED_HIERARCHIES;
    }
}

void hctCreateWorldFilter::hideOptions()
{
    updateOptions();

    if (m_optionsDialog)
    {
        DestroyWindow(m_optionsDialog);
    }

    m_optionsDialog = NULL;
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
