// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateWorld/hctCreateWorldFilter.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.h>
#include <Physics2012/Collide/Filter/Group/hkpGroupFilterSetup.h>

#include <Physics2012/Dynamics/Collide/Filter/Constraint/hkpConstraintCollisionFilter.h>
#include <Physics2012/Utilities/Collide/Filter/GroupFilter/hkpGroupFilterUtil.h>


hctFilterCreateWorldDesc g_createWorldDesc;

#define NUM_LAYERS 32


hctCreateWorldFilter::hctCreateWorldFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner),
    m_optionsDialog (NULL)
{
    m_haveSetOptions = false;

    m_options.m_solverType = hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM;
    m_options.m_collisionFilterType = hctCreateWorldOptions::COLLISION_OPTIONS_CONSTRAINT_FILTER;

    m_options.m_autoGroupRBS = false;
    m_options.m_autoGroupRBLayer = 2; // dynamic
    m_options.m_createWorld = true;
    m_options.m_systemGrouping = hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_CONSTRAINED_HIERARCHIES;

    // Temporarily create a group filter to query the collision lookup table
    hkpGroupFilter* f = new hkpGroupFilter();
    hkpGroupFilterSetup::setupGroupFilter( f );

    hkString::memCpy( m_options.m_groupFilterLayerTable, f->m_collisionLookupTable, NUM_LAYERS * sizeof(hkUint32) );

    f->removeReference();
}

hctCreateWorldFilter::~hctCreateWorldFilter()
{

}

void hctCreateWorldFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctCreateWorldOptions* options = hctFilterUtils::getNativeOptions<hctCreateWorldOptions>(optVar))
    {
        m_options = *options;
        m_haveSetOptions = true;
        delete options;
    }
    else
    {
        m_haveSetOptions = false;
    }
}

void hctCreateWorldFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

static void _mergeSystem(hkpPhysicsSystem* to, hkpPhysicsSystem* from)
{
    const hkArray<hkpRigidBody*>& rbs = from->getRigidBodies();
    int nrb = rbs.getSize();
    for (int r=0; r < nrb; ++r)
    {
        to->addRigidBody(rbs[r]);
    }

    const hkArray<hkpPhantom*>& phantoms = from->getPhantoms();
    int np = phantoms.getSize();
    for (int p=0; p < np; ++p)
    {
        to->addPhantom(phantoms[p]);
    }

    const hkArray<hkpConstraintInstance*>& constraints = from->getConstraints();
    int nc = constraints.getSize();
    for (int c=0; c < nc; ++c)
    {
        to->addConstraint(constraints[c]);
    }

    const hkArray<hkpAction*>& actions = from->getActions();
    int na = actions.getSize();
    for (int a=0; a < na; ++a)
    {
        to->addAction(actions[a]);
    }
}

static void _findAllConstraints(hkpPhysicsData* physicsPtr, hkArray<hkpConstraintInstance*>& allConstraints)
{
    const hkArray<hkpPhysicsSystem*>& ps = physicsPtr->getPhysicsSystems();
    for (int psi=0; psi < ps.getSize(); ++psi)
    {
        if (ps[psi]->getConstraints().getSize() > 0)
        {
            allConstraints.insertAt( allConstraints.getSize(), ps[psi]->getConstraints().begin(), ps[psi]->getConstraints().getSize() );
        }
    }
}

static void _copyCinfo( hkpWorldCinfo* to, const hkpWorldCinfo& from )
{
    to->m_gravity                                            = from.m_gravity;
    to->m_broadPhaseQuerySize                                = from.m_broadPhaseQuerySize;
    to->m_contactRestingVelocity                             = from.m_contactRestingVelocity;
    to->m_broadPhaseType                                     = from.m_broadPhaseType;
    to->m_broadPhaseBorderBehaviour                          = from.m_broadPhaseBorderBehaviour;
    to->m_mtPostponeAndSortBroadPhaseBorderCallbacks         = from.m_mtPostponeAndSortBroadPhaseBorderCallbacks;
    to->m_broadPhaseWorldAabb                                = from.m_broadPhaseWorldAabb;
    to->m_collisionTolerance                                 = from.m_collisionTolerance;
    to->m_collisionFilter                                    = from.m_collisionFilter;
    to->m_convexListFilter                                   = from.m_convexListFilter;
    to->m_expectedMaxLinearVelocity                          = from.m_expectedMaxLinearVelocity;
    to->m_sizeOfToiEventQueue                                = from.m_sizeOfToiEventQueue;
    to->m_expectedMinPsiDeltaTime                            = from.m_expectedMinPsiDeltaTime;
    to->m_memoryWatchDog                                     = from.m_memoryWatchDog;
    to->m_broadPhaseNumMarkers                               = from.m_broadPhaseNumMarkers;
    to->m_contactPointGeneration                             = from.m_contactPointGeneration;
    to->m_allowToSkipConfirmedCallbacks                      = from.m_allowToSkipConfirmedCallbacks;
    to->m_solverTau                                          = from.m_solverTau;
    to->m_solverDamp                                         = from.m_solverDamp;
    to->m_solverIterations                                   = from.m_solverIterations;
    to->m_solverMicrosteps                                   = from.m_solverMicrosteps;
    to->m_forceCoherentConstraintOrderingInSolver            = from.m_forceCoherentConstraintOrderingInSolver;
    to->m_snapCollisionToConvexEdgeThreshold                 = from.m_snapCollisionToConvexEdgeThreshold;
    to->m_snapCollisionToConcaveEdgeThreshold                = from.m_snapCollisionToConcaveEdgeThreshold;
    to->m_enableToiWeldRejection                             = from.m_enableToiWeldRejection;
    to->m_enableDeprecatedWelding                            = from.m_enableDeprecatedWelding;
    to->m_iterativeLinearCastEarlyOutDistance                = from.m_iterativeLinearCastEarlyOutDistance;
    to->m_iterativeLinearCastMaxIterations                   = from.m_iterativeLinearCastMaxIterations;
    to->m_deactivationNumInactiveFramesSelectFlag0           = from.m_deactivationNumInactiveFramesSelectFlag0;
    to->m_deactivationNumInactiveFramesSelectFlag1           = from.m_deactivationNumInactiveFramesSelectFlag1;
    to->m_deactivationIntegrateCounter                       = from.m_deactivationIntegrateCounter;
    to->m_shouldActivateOnRigidBodyTransformChange           = from.m_shouldActivateOnRigidBodyTransformChange;
    to->m_deactivationReferenceDistance                      = from.m_deactivationReferenceDistance;
    to->m_toiCollisionResponseRotateNormal                   = from.m_toiCollisionResponseRotateNormal;
    to->m_maxSectorsPerMidphaseCollideTask                   = from.m_maxSectorsPerMidphaseCollideTask;
    to->m_maxSectorsPerNarrowphaseCollideTask                = from.m_maxSectorsPerNarrowphaseCollideTask;
    to->m_processToisMultithreaded                           = from.m_processToisMultithreaded;
    to->m_maxEntriesPerToiMidphaseCollideTask                = from.m_maxEntriesPerToiMidphaseCollideTask;
    to->m_maxEntriesPerToiNarrowphaseCollideTask             = from.m_maxEntriesPerToiNarrowphaseCollideTask;
    to->m_maxNumToiCollisionPairsSinglethreaded              = from.m_maxNumToiCollisionPairsSinglethreaded;
    to->m_numToisTillAllowedPenetrationSimplifiedToi         = from.m_numToisTillAllowedPenetrationSimplifiedToi;
    to->m_numToisTillAllowedPenetrationToi                   = from.m_numToisTillAllowedPenetrationToi;
    to->m_numToisTillAllowedPenetrationToiHigher             = from.m_numToisTillAllowedPenetrationToiHigher;
    to->m_numToisTillAllowedPenetrationToiForced             = from.m_numToisTillAllowedPenetrationToiForced;
    to->m_enableDeactivation                                 = from.m_enableDeactivation;
    to->m_simulationType                                     = from.m_simulationType;
    to->m_enableSimulationIslands                            = from.m_enableSimulationIslands;
    to->m_minDesiredIslandSize                               = from.m_minDesiredIslandSize;
    to->m_processActionsInSingleThread                       = from.m_processActionsInSingleThread;
    to->m_frameMarkerPsiSnap                                 = from.m_frameMarkerPsiSnap;
    to->m_fireCollisionCallbacks                             = from.m_fireCollisionCallbacks;
}

void hctCreateWorldFilter::process( hkRootLevelContainer& data  )
{
    // Find the scene in the root level container
    // create a new cinfo
    {
        hkpPhysicsData* physicsPtr = data.findObject<hkpPhysicsData>();
        if (physicsPtr == HK_NULL)
        {
            HK_WARN_ALWAYS(0xabba8bcc, "No physics data found.");
            HK_WARN_ALWAYS(0xabba8bcc, "The Create World filter should be placed after rigid bodies and constraints have been created.");

            physicsPtr = new hkpPhysicsData();
            data.m_namedVariants.expandOne().set("Physics Data", physicsPtr ); // virtual
            physicsPtr->removeReference();
        }

        hkpWorldCinfo* ci = HK_NULL;
        if (m_options.m_createWorld)
        {
            ci = new hkpWorldCinfo; // tracked
            _copyCinfo( ci, m_options.m_worldCinfo ); // copy our settings

            // create a collision filter?
            if (m_options.m_collisionFilterType == hctCreateWorldOptions::COLLISION_OPTIONS_CONSTRAINT_FILTER)
            {
                ci->m_collisionFilter = new hkpConstraintCollisionFilter;
            }
            else if (m_options.m_collisionFilterType == hctCreateWorldOptions::COLLISION_OPTIONS_GROUP_FILTER)
            {
                hkpGroupFilter* gf = new hkpGroupFilter;
                ci->m_collisionFilter = gf;
                hkString::memCpy( gf->m_collisionLookupTable, m_options.m_groupFilterLayerTable, NUM_LAYERS * sizeof(hkUint32) );
            }

            physicsPtr->setWorldCinfo(ci);
            if (ci->m_collisionFilter) ci->m_collisionFilter->removeReference(); // our ref
            ci->removeReference();
        }

        const hkArray<hkpPhysicsSystem*>& phySystems = physicsPtr->getPhysicsSystems();
        int numSystems = phySystems.getSize();

        const bool emptyWorld = numSystems == 0;

        // Warn if the user wanted us to assign collision groups but there are no rbs
        if (emptyWorld && m_options.m_autoGroupRBS)
        {
            HK_WARN_ALWAYS(0xabba761a, "No rigid bodies in the world. Can't auto assign collision group IDs.");
        }

        // Warn if the user wanted us to split physics systems but there are none
        if (emptyWorld && (m_options.m_systemGrouping == hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_CONSTRAINED_HIERARCHIES))
        {
            HK_WARN_ALWAYS(0xabba761a, "No rigid bodies in the world. Can't group objects in physics systems.");
        }

        if (numSystems > 0)
        {
            // auto layer constrained bodies
            if (m_options.m_autoGroupRBS)
            {
                    hkArray<hkpConstraintInstance*> allConstraints;
                    _findAllConstraints(physicsPtr, allConstraints);
                    if (allConstraints.getSize() > 0)
                    {
                        hkpGroupFilterUtil::disableCollisionsBetweenConstraintBodies(allConstraints.begin(), allConstraints.getSize(), m_options.m_autoGroupRBLayer);
                    }
            }

            // regrouping
            switch (m_options.m_systemGrouping)
            {
            case hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_SINGLE:
                {
                    if (numSystems == 1)
                        break;

                    // merge them all together
                    hkpPhysicsSystem* toS = phySystems[0];
                    for (int si=numSystems; si > 0; --si)
                    {
                        _mergeSystem(toS, phySystems[si]);
                        physicsPtr->removePhysicsSystem(si);
                    }

                    break;
                }

            case hctCreateWorldOptions::SYSTEM_GROUPING_OPTIONS_CONSTRAINED_HIERARCHIES:
                {
                    hkpPhysicsSystem* sourceSystem = phySystems[0];
                    if (numSystems > 1)
                    {
                        for (int si=numSystems-1; si > 0; --si)
                        {
                            _mergeSystem(sourceSystem, phySystems[si]);
                            physicsPtr->removePhysicsSystem(si);
                        }
                    }

                    hkpPhysicsData::SplitPhysicsSystemsOutput split;

                    // split into 3 systems: fixed, moving and constrained
                    hkpPhysicsData::splitPhysicsSystems(sourceSystem, split);

                    // remove source system
                    physicsPtr->removePhysicsSystem(0);

                    if (split.m_unconstrainedFixedBodies)
                    {
                        physicsPtr->addPhysicsSystem(split.m_unconstrainedFixedBodies);
                        split.m_unconstrainedFixedBodies->removeReference();
                    }
                    if (split.m_unconstrainedKeyframedBodies)
                    {
                        physicsPtr->addPhysicsSystem(split.m_unconstrainedKeyframedBodies);
                        split.m_unconstrainedKeyframedBodies->removeReference();
                    }
                    if (split.m_unconstrainedMovingBodies)
                    {
                        physicsPtr->addPhysicsSystem(split.m_unconstrainedMovingBodies);
                        split.m_unconstrainedMovingBodies->removeReference();
                    }
                    if (split.m_phantoms)
                    {
                        physicsPtr->addPhysicsSystem(split.m_phantoms);
                        split.m_phantoms->removeReference();
                    }

                    for (int cs=0; cs < split.m_constrainedSystems.getSize(); ++cs)
                    {
                        physicsPtr->addPhysicsSystem(split.m_constrainedSystems[cs]);
                        split.m_constrainedSystems[cs]->removeReference();
                    }

                    break;
                }

            default:
                break;
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
