// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateRigidBodies/hctNamedMeshMaterialsUtil.h>

#include <Common/SceneData/Mesh/Channels/hkxTriangleSelectionChannel.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/Base/Container/PointerMap/hkMap.h>

static hkBool HK_CALL isTriangleSelectionClassName( const char* name )
{
    return ( hkString::strCmp( hkReflect::getName<hkxTriangleSelectionChannel>(), name ) == 0 );
}


hkBool HK_CALL hctNamedMeshMaterialsUtil::isUserChannelForMaterials( const hkxMesh::UserChannelInfo& userChannelInfo, const hkStringPtr& prefix )
{
    return isTriangleSelectionClassName( userChannelInfo.m_className ) && isPrefixForMaterials( userChannelInfo.m_name, prefix );
}

hkBool HK_CALL hctNamedMeshMaterialsUtil::isPrefixForMaterials( const hkStringPtr& name, const hkStringPtr& prefix )
{
    return ( 0 == hkString::strNcmp( name, prefix, hkString::strLen( prefix ) ) );
}


void HK_CALL hctNamedMeshMaterialsUtil::getMaterialTable( const hkxMesh* mesh, hkArray<hkpNamedMeshMaterial>& materialTableOut, MappingTable& mappingOut, const hkStringPtr& prefix, hkBool useTriangleSelection)
{
    if( useTriangleSelection )
    {
        //
        //  Traverse the userChannelInfos, looking for those that correspond to materials.
        //
        int triangleSelectionIndex = 0;
        for( int i = 0; i < mesh->m_userChannelInfos.getSize(); ++i )
        {
            HK_ASSERT_NO_MSG(0x15948514, mesh->m_userChannelInfos[i]);
            const hkxMesh::UserChannelInfo& userChannelInfo = *mesh->m_userChannelInfos[i];

            // Is the userChannel a triangle selection?
            if ( isTriangleSelectionClassName( userChannelInfo.m_className ) )
            {
                // Does the prefix match what the user specified?
                if( isPrefixForMaterials( userChannelInfo.m_name, prefix ) )
                {
                    if( materialTableOut.getSize() == 0 )
                    {
                        // Add a default material name for triangles that are not selected.
                        hkpNamedMeshMaterial namedMaterial;
                        materialTableOut.pushBack( namedMaterial );
                    }

                    // Store the name without the prefix.
                    hkpNamedMeshMaterial namedMaterial( ( hkStringOld( userChannelInfo.m_name ).substr( hkString::strLen( prefix ) ) ).cString() );
                    materialTableOut.pushBack( namedMaterial );
                    // Map the material index to the appropriate triangle selection index
                    mappingOut.pushBack( triangleSelectionIndex );
                }

                // Keep track of the triangle selection index. Note: Might not be the same as "i" since not all user channels are triangle selections.
                ++triangleSelectionIndex;
            }
        }
    }
    else
    {
        //
        // Traverse the mesh sections, looking for those that contain materials
        //
        int backupMaterialIndex = -1;
        const char * backupMaterialName = "UnassignedMaterial";
        for (int si = 0; si < mesh->m_sections.getSize() ; ++si)
        {
            HK_ASSERT_NO_MSG(0x1b58a649, mesh->m_sections[si]);
            //HK_ASSERT_NO_MSG(0x63ad6b2e, mesh->m_sections[si]->m_material);

            const hkxMaterial* material = mesh->m_sections[si]->m_material;
            if(material)
            {
                // Store the name without the prefix.
                hkpNamedMeshMaterial namedMaterial( material->m_name );
                materialTableOut.pushBack( namedMaterial );
                mappingOut.pushBack( materialTableOut.getSize() -1 );
            }
            else
            {
                // Warn and create a new default material
                if (backupMaterialIndex == -1)
                {
                    HK_WARN_ALWAYS(0xabba5095, "At least one section does not have a material."
                    " Creating a new NamedMaterial " << " '" << backupMaterialName << "' for this section");
                    hkpNamedMeshMaterial backupMaterial(backupMaterialName);
                    materialTableOut.pushBack(backupMaterial);
                    backupMaterialIndex = materialTableOut.getSize() -1;
                }
                mappingOut.pushBack(backupMaterialIndex);
            }
        }
    }
}


hkBool HK_CALL hctNamedMeshMaterialsUtil::getTriangleSelectionTableFromSection( const hkxMeshSection* section, TriangleSelectionTable& triSelectionChannels )
{
    int numSelectedTriangles = 0;
    {
        for( int uci = 0 ; uci < section->m_userChannels.getSize(); ++uci )
        {
            const hkRefVariant& pInfo = section->m_userChannels[uci];
            if( isTriangleSelectionClassName( pInfo.getType()->getName() ) )
            {
                hkxTriangleSelectionChannel* tSel = (hkxTriangleSelectionChannel*)pInfo.val();
                hkArray<hkInt32> triSel( tSel->m_selectedTriangles.begin(), tSel->m_selectedTriangles.getSize(), tSel->m_selectedTriangles.getSize());
                triSelectionChannels.pushBack( triSel );
                numSelectedTriangles += tSel->m_selectedTriangles.getSize();
            }
        }
    }
    return ( numSelectedTriangles > 0 );
}


int HK_CALL hctNamedMeshMaterialsUtil::getMaterialForTriangle( int triangleIndex, const TriangleSelectionTable& triSelectionChannels, const MappingTable& mapping, hkBool joinMaterials, hkArray<hkpNamedMeshMaterial>& materialTable, hkBool& overlap )
{
    int foundMaterialIndex = -1;
    const int numTriChannels = triSelectionChannels.getSize();
    for( int channelIndex = 0; channelIndex < numTriChannels; ++channelIndex )
    {
        const int materialIndex = mapping.indexOf( channelIndex );
        // Only consider channels which correspond to materials.
        if ( materialIndex != -1 )
        {
            // Is the triangle in the selection?
            if( triSelectionChannels[channelIndex].indexOf( triangleIndex ) != -1 )
            {
                // Is this the first material we've encountered for this triangle?
                if( foundMaterialIndex == -1 )
                {
                    foundMaterialIndex = materialIndex;
                }
                else if ( joinMaterials )
                {
                    // Create a concatenated material.
                    hkpNamedMeshMaterial& currentMaterial = materialTable[foundMaterialIndex + 1];
                    hkpNamedMeshMaterial& thisMaterial = materialTable[materialIndex + 1];
                    hkpNamedMeshMaterial concatenatedMaterial( ( hkStringOld( currentMaterial.m_name.cString() ) + MESH_MATERIAL_SEPARATOR + thisMaterial.m_name.cString() ).cString() );
                    // Check if this concatenated material already exists.
                    const int cmatIndex = materialTable.indexOf( concatenatedMaterial );
                    if ( cmatIndex != -1 )
                    {
                        foundMaterialIndex = cmatIndex - 1;
                    }
                    else
                    {
                        foundMaterialIndex = materialTable.getSize() - 1;
                        materialTable.pushBack( concatenatedMaterial );
                    }
                    overlap = true;
                }
                else
                {
                    overlap = true;
                    break;
                }
            }
        }
    }
    // If namedMaterialIndex = -1, then the triangle was not selected and so will be set to default material index (0)
    return foundMaterialIndex + 1;
}


void HK_CALL hctNamedMeshMaterialsUtil::getNamedMaterials( const hkxMesh* mesh, hkArray<hkpNamedMeshMaterial>& materialTableOut, hkArray<hkUint16>& indicesOut, hkBool& overlap, const hkStringPtr& prefix, hkBool joinMaterials, hkBool useTriangleSelection )
{
    overlap = false;

    MappingTable mapping;

    getMaterialTable( mesh, materialTableOut, mapping, prefix, useTriangleSelection );

    if ( materialTableOut.getSize() == 0 )
    {
        return;
    }

    //  Have materials so check each section to see if it contains any of them
    for (int si = 0; si < mesh->m_sections.getSize() ; ++si)
    {
        hkxMeshSection* section = mesh->m_sections[si];

        HK_ASSERT( 0x50b7a93b, mesh->m_sections[si]->m_indexBuffers.getSize() == 1, "");

        hkxIndexBuffer* ibuffer = section->m_indexBuffers[0];
        HK_ASSERT( 0x50b7a93b, ibuffer->m_indexType == hkxIndexBuffer::INDEX_TYPE_TRI_LIST, "Only implemented for triangle lists");

        const int numTriangles = ( ibuffer->m_indices16.getSize() > 0 ? ibuffer->m_indices16.getSize() : ibuffer->m_indices32.getSize() ) / 3;

        TriangleSelectionTable materialSelections;
        const hkBool hasSelectedTriangle = useTriangleSelection ? getTriangleSelectionTableFromSection( section, materialSelections ) : ( section->m_material != HK_NULL );

        hkUint16* p = indicesOut.expandBy( numTriangles );
        for( int triIndex = 0 ; triIndex < numTriangles ; ++triIndex, ++p )
        {
            if( hasSelectedTriangle )
            {
                if( useTriangleSelection )
                {
                    *p = hkUint16( getMaterialForTriangle( triIndex, materialSelections, mapping, joinMaterials, materialTableOut, overlap ) );
                }
                else
                {
                    *p = hkUint16( si );
                }
            }
            else
            {
                *p = 0;
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
