// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateRigidBodies/hctCreateRigidBodiesFilter.h>

#include <ContentTools/Common/Filters/Common/Utils/hctLocalFrameUtils.h>

#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <Common/Base/Math/Matrix/hkMatrix4Util.h>

#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>

#include <Physics2012/Collide/Util/ShapeShrinker/hkpShapeShrinker.h>
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Convex/Sphere/hkpSphereShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Misc/Transform/hkpTransformShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTransform/hkpConvexTransformShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/Mesh/hkpMeshMaterial.h>
#include <Physics2012/Collide/Util/ShapeInfo/hkpShapeInfo.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Code/hkpMoppCode.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics2012/Collide/Util/Welding/hkpMeshWeldingUtility.h>
#include <Physics2012/Collide/Shape/Deprecated/CompressedMesh/hkpCompressedMeshShapeBuilder.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Dynamics/Action/hkpUnaryAction.h>
#include <Physics2012/Dynamics/Action/hkpBinaryAction.h>

#include <Common/Internal/ConvexHull/hkGeometryUtility.h>

#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Types/Geometry/LocalFrame/hkLocalFrame.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#include <Physics2012/Dynamics/Common/hkpProperty.h>

#include <Common/SceneData/Material/hkxMaterial.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateRigidBodies/hctNamedMeshMaterialsUtil.h>
#include <Physics2012/Utilities/Collide/ShapeUtils/CreateShape/hkpCreateShapeUtility.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShapeCinfo.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.p12.createbodies"
#include <Common/Base/System/Log/hkLog.hxx>


hctCreateRigidBodiesFilterDesc g_createRigidBodiesDesc;


// Construction info to build a hkpBvCompressedMeshShape with a embedded string table of material names.
class hkpMaterialBvCompressedMeshShapeCinfo : public hkpDefaultBvCompressedMeshShapeCinfo
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SHAPE, hkpMaterialBvCompressedMeshShapeCinfo );

        // Builds the Cinfo structure with a triangle geometry and a list of materials.
        // Each triangle's m_material member is an index into the material list.
        hkpMaterialBvCompressedMeshShapeCinfo( const hkGeometry* geometry , const hkArray<hkpNamedMeshMaterial>& meshNamedMaterials )
        : hkpDefaultBvCompressedMeshShapeCinfo(geometry)
        {
            m_userDataMode =  hkpBvCompressedMeshShape::PER_PRIMITIVE_DATA_STRING_PALETTE;
            for (int i = 0; i < meshNamedMaterials.getSize(); ++ i)
            {
                m_meshNamedMaterials.pushBack( meshNamedMaterials[i] );
            }
        }

        virtual ~hkpMaterialBvCompressedMeshShapeCinfo() {}

        // Adds a convex shape with a material.
        // Default material name is an empty string.
        virtual void addConvexShape( const hkpConvexShape* convexShape, const hkQsTransform& transform = hkQsTransform::getIdentity(), const hkpNamedMeshMaterial& material = hkpNamedMeshMaterial("") )
        {
            hkpDefaultBvCompressedMeshShapeCinfo::addConvexShape(convexShape, transform);
            m_convexNamedMaterials.pushBack(material);
        }

        // Returns the material name of a given triangle.
        // If the material is not in the list provided at construction, this returns an empty string.
        virtual void getTriangleUserString( int triangleIndex, hkStringPtr& stringOut ) const
        {
            int materialIdx = m_geometry->m_triangles[triangleIndex].m_material;
            if (materialIdx >= 0 && materialIdx < m_meshNamedMaterials.getSize())
            {
                stringOut = m_meshNamedMaterials[materialIdx].m_name;
            }
            else
            {
                stringOut = "";
            }
        }

        // Returns the material name of a given convex shape.
        virtual void getConvexShapeUserString( int convexIndex, hkStringPtr& stringOut  ) const
        {
            stringOut = m_convexNamedMaterials[convexIndex].m_name;
        }

    protected:

        hkArray<hkpNamedMeshMaterial> m_meshNamedMaterials;
        hkArray<hkpNamedMeshMaterial> m_convexNamedMaterials;
};


hctCreateRigidBodiesFilter::hctCreateRigidBodiesFilter (const hctFilterManagerInterface* owner)
:   hctFilterInterface(owner)
,   m_optionsDialog(NULL)
{
    // EXP-366
    m_options.m_bestFittingShapes = false;
    // EXP-1081
    m_options.m_mergeMeshDuplicates = true;
    // EXP-955
    m_options.m_listShapesMoppThreshold = 5;
    // EXP-1094
    m_options.m_enableMoppChunks = true;
    // EXP-1148
    m_options.m_spuOptimizedMopps = false;
    // EXP-1148
    m_options.m_spuOptimizedConvex = false;
    // EXP-1136
    m_options.m_landscapeWelding = hctCreateRigidBodiesOptions::WELD_COUNTERCLOCKWISE;
    // HVK-390
    m_options.m_weldOpenEdges = false;
    // HVK-4832
    m_options.m_weldAcrossAllMopps = false;
    // HVK-5218
    m_options.m_markEdgesBadWinding = false;
    // EXP-1333
    m_options.m_collapseShapeOffsetsIntoShapes = true; // We default to true for new filters
    // HKD-25
    m_options.m_defaultConvexRadius = hkConvexShapeDefaultRadius;

    m_options.m_enableAutomaticShapeShrinking = true;
    m_options.m_maxVertexDisplacement = 0.03f;
    m_options.m_relShrinkRadius       = 0.05f;

    // HKD-118
    m_options.m_exportDestructionInformation = true; // We default to true for new filters

    // HVK-4699
    m_options.m_joinOverlappingMeshMaterials = false;

    m_options.m_namedMaterialSource = hctCreateRigidBodiesOptions::MATERIAL_NONE;

    m_options.m_materialPrefix = DEFAULT_MATERIAL_PREFIX;

    // HVK-5079
    m_options.m_quantizationError = 0.001f;

    // HVK-6165
    m_options.m_meshShapeType = hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE;

    m_options.m_wrapWithMopp = true;
}

hctCreateRigidBodiesFilter::~hctCreateRigidBodiesFilter()
{
}

// Tests whether the triangles in a BV Tree have valid (i.e., consistent) winding. This test ignores triangles that are degenerate.
bool _testValidTriangleWinding( const hkpBvTreeShape* bvTreeShape )
{
    hkpShapeBuffer buffer;
    const hkpShapeContainer* theContainer = bvTreeShape->getContainer();

    for( hkpShapeKey shapeKey = theContainer->getFirstKey(); shapeKey != HK_INVALID_SHAPE_KEY; shapeKey = theContainer->getNextKey( shapeKey ) )
    {
        const hkpShape* shape = theContainer->getChildShape(shapeKey, buffer);
        if( ( shape->getType() == hkcdShapeType::TRIANGLE ) &&
            !hkpMeshWeldingUtility::isTriangleWindingValid( shapeKey, bvTreeShape ) )
        {
            return false;
        }
    }

    return true;
}

// Transforms all vertices in a triangle sub-part by the given transformation. This method is a template
// so that the same code can operate on the two different triangle subpart structs (one in hkpExtendedMeshShape,
// and another in hkpSimpleMeshShape). These structs are essentially copy-and-pastes of each other, so they have
// the same member names and such, so that a template can be used to treat them uniformly.
template< class TrianglePartType >
void _transformVertices( TrianglePartType& trianglePart, const hkTransform& transformation )
{
    hkVector4* vertexBase = const_cast< hkVector4* >( reinterpret_cast< const hkVector4* >( trianglePart.m_vertexBase ) );
    int vertexStride = trianglePart.m_vertexStriding;
    int vertexCount = trianglePart.m_numVertices;

    hkVector4 temp;
    for( int i = 0; i != vertexCount; ++i )
    {
        hkVector4& vertex = *vertexBase;
        temp = vertex;
        vertex.setTransformedPos( transformation, temp );
        vertexBase = reinterpret_cast< hkVector4* >( hkAddByteOffset( vertexBase, vertexStride ) );
    }
}


static void _getStridedVertices( const hkxMesh* mesh, const hkMatrix4& vertexTransform, hkStridedVertices& stridedVertsOut, hkArray<hkVector4>& tempStorageOut)
{
    stridedVertsOut.m_numVertices = 0;

    const hkBool32 vertexTransformIsIdentity = vertexTransform.isApproximatelyIdentity(hkSimdReal::fromFloat(1e-3f));

    // gather all together if more than one section
    // or we have scale
    if ( (mesh->m_sections.getSize() > 1) || !vertexTransformIsIdentity )
    {
        for (int si=0; si < mesh->m_sections.getSize(); ++si)
        {
            hkxMeshSection* section = mesh->m_sections[si];
            hkxVertexBuffer* vertices = section->m_vertexBuffer;
            const hkxVertexDescription& vDesc = vertices->getVertexDesc();
            const hkxVertexDescription::ElementDecl* posDecl = vDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
            char* posBuf = static_cast<char*>( vertices->getVertexDataPtr(*posDecl) );

            int bufOffset = tempStorageOut.getSize();
            int numVerts = vertices->getNumVertices();
            tempStorageOut.setSize(bufOffset + numVerts);
            for (int vi=0; vi < numVerts; ++vi)
            {
                hkVector4& vert = *(hkVector4*)( posBuf );
                vertexTransform.transformPosition(vert, tempStorageOut[vi + bufOffset]);

                posBuf += posDecl->m_byteStride;
            }
        }

        if (tempStorageOut.getSize() > 0)
        {
            stridedVertsOut.m_numVertices = tempStorageOut.getSize();
            stridedVertsOut.m_striding = sizeof(hkVector4);
            stridedVertsOut.m_vertices = (float*)tempStorageOut.begin();
        }
    }
    else if (mesh->m_sections.getSize() == 1)
    {
        // can convert directly, no need to copy
        hkxMeshSection* section = mesh->m_sections[0];
        hkxVertexBuffer* vertices = section->m_vertexBuffer;
        const hkxVertexDescription& vDesc = vertices->getVertexDesc();
        const hkxVertexDescription::ElementDecl* posDecl = vDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
        stridedVertsOut.m_numVertices = vertices->getNumVertices();
        stridedVertsOut.m_striding = posDecl->m_byteStride;;
        stridedVertsOut.m_vertices = (float*)( vertices->getVertexDataPtr(*posDecl) );
    }
}

inline static hkxMesh* _findNodeMesh (const hkxNode* node)
{
    if ( !node || (!node->m_object) )
    {
        return HK_NULL;
    }

    if (hkxMesh* theMesh = node->hasA<hkxMesh>())
    {
        return theMesh;
    }

    if (hkxSkinBinding* theSkinBinding = node->hasA<hkxSkinBinding>())
    {
        hkxMesh* theMesh = theSkinBinding->m_mesh;
        return theMesh;
    }

    return HK_NULL;
}


bool hctCreateRigidBodiesFilter::computeAABB(const hkxMesh* mesh, const hkMatrix4& vertexTransform, hkVector4& halfExtentsOut, hkTransform& obbTransformOut)
{
    if (!mesh)
    {
        return false;
    }

    hkArray<hkVector4> tempStorage;
    hkStridedVertices stridedVerts;

    _getStridedVertices( mesh, vertexTransform, stridedVerts, tempStorage );

    if (stridedVerts.m_numVertices > 0)
    {
        hkVector4 minBox (HK_REAL_MAX, HK_REAL_MAX, HK_REAL_MAX);
        hkVector4 maxBox (-HK_REAL_MAX, -HK_REAL_MAX, -HK_REAL_MAX);
        {
            const int floatstriding = stridedVerts.m_striding/hkSizeOf(float);
            for( int i = 0; i < stridedVerts.m_numVertices * 3; i++)
            {
                const int vidx = i / 3; // vertex index (0..numVertices)
                const int component = i % 3; // component (0,1,2 == x,y,z)

                float value = stridedVerts.m_vertices[vidx*floatstriding+component];

                maxBox(component) = hkMath::max2(maxBox(component), value);

                minBox(component) = hkMath::min2(minBox(component), value);

            }
        }

        hkVector4 boxCenter; boxCenter.setInterpolate4(minBox, maxBox, 0.5f);

        obbTransformOut.set(hkQuaternion::getIdentity(), boxCenter);
        halfExtentsOut.setSub4(maxBox, minBox); halfExtentsOut.mul4(0.5f);

        if (halfExtentsOut.length3() > 0.0001f)
        {
            return true;
        }
    }

    return false;
}

bool hctCreateRigidBodiesFilter::computeOBB(const hkxMesh* mesh, const hkMatrix4& vertexTransform, hkVector4& halfExtentsOut, hkTransform& obbTransformOut)
{
    if( !mesh )
    {
        return false;
    }

    // EXP-366 : If option is off, use old calculations (AABB)
    if (!m_options.m_bestFittingShapes)
    {
        return computeAABB(mesh, vertexTransform, halfExtentsOut, obbTransformOut);
    }

    // Use OBB
    hkArray<hkVector4> tempStorage;
    hkStridedVertices stridedVerts;

    _getStridedVertices( mesh, vertexTransform, stridedVerts, tempStorage );

    if (stridedVerts.m_numVertices > 0)
    {
        hkBool wasEnabled = hkError::getInstance().isEnabled( 0x34df5494 );
        hkError::getInstance().setEnabled( 0x34df5494, false );

        hkGeometryUtility::calcObb(stridedVerts, halfExtentsOut, obbTransformOut);

        hkError::getInstance().setEnabled( 0x34df5494, wasEnabled );

        if (halfExtentsOut.length3() > 0.0001f)
        {
            return true;
        }
    }

    return false;
}

static hkpShape* _wrapShapeWithTransform (const hctCreateRigidBodiesFilter::ShapeInfo& shapeInfo, const hkTransform& parentSpace)
{
    // see if we need to wrap the shape in a transform or translate shape
    bool needTransform = !parentSpace.isApproximatelyEqual(shapeInfo.m_decomposedWorldT);
    if (needTransform)
    {
        hkTransform shapeToBody;
        shapeToBody.setMulInverseMul(parentSpace, shapeInfo.m_decomposedWorldT);

        // see if it can make do with a convex translate instead of a transform shape
        hkpShape* ts = HK_NULL;
        if (shapeInfo.m_isConvex)
        {
            hkpConvexShape* cshape = (hkpConvexShape*) shapeInfo.m_shape;

            const bool isSphere = cshape->getType()== hkcdShapeType::SPHERE;

            // do we have translate only?
            if (isSphere || shapeToBody.getRotation().isApproximatelyEqual( hkTransform::getIdentity().getRotation() ))
            {
                ts = new hkpConvexTranslateShape(cshape, shapeToBody.getTranslation());
            }
            else // convex transform required
            {
                ts = new hkpConvexTransformShape(cshape, shapeToBody);
            }
        }
        else // non convex, full transform only supported.
        {
            ts = new hkpTransformShape(shapeInfo.m_shape, shapeToBody);
        }

        shapeInfo.m_shape->removeReference();

        return ts;
    }
    else
    {
        return shapeInfo.m_shape;
    }

}

hkpListShape* hctCreateRigidBodiesFilter::buildListShapeFromConvexShapes( hkArray<hkpConvexShape*>& convexChildShapes )
{
    Log_Info( "Building List Shape with {} children.", convexChildShapes.getSize() );

    hkpShape** shapeArray = static_cast< hkpShape** >( static_cast< void* >( convexChildShapes.begin() ) );
    hkpListShape* listShape = new hkpListShape( shapeArray, convexChildShapes.getSize() );

    for( hkInt32 i = 0; i < convexChildShapes.getSize(); ++i )
    {
        convexChildShapes[i]->removeReference();
    }

    return listShape;
}

hkpStorageExtendedMeshShape* hctCreateRigidBodiesFilter::buildExtendedMeshFromCompound( hkArray<hkpConvexShape*>& convexChildShapes, hkArray<ShapeInfo*>& meshChildShapes, const hkTransform& rbTransform)
{
    Log_Info( "Building Extended Mesh Shape with {} convex shapes and {} concave shapes", convexChildShapes.getSize(), meshChildShapes.getSize() );

    hkpStorageExtendedMeshShape* extendedMeshShape = new hkpStorageExtendedMeshShape();
    extendedMeshShape->m_materialClass = hkReflect::getType<hkpNamedMeshMaterial>();

    // Add all mesh shapes's triangle data to our boy
    for( hkInt32 i = 0; i != meshChildShapes.getSize(); ++i )
    {
        ShapeInfo* childShapeInfo = meshChildShapes[i];
        hkpShape* childShape = childShapeInfo->m_shape;

        hkTransform shapeToBody;
        // child shape transform is worldFromShape
        shapeToBody.setMulInverseMul( rbTransform, childShapeInfo->m_decomposedWorldT );

        int numNonDegenerateTriangles = 0;
        int numTriangles = 0;

        switch( childShape->getType() )
        {

        case hkcdShapeType::EXTENDED_MESH :
            {
                hkpStorageExtendedMeshShape* childMeshShape = static_cast<hkpStorageExtendedMeshShape*> (childShape);

                if(childMeshShape->getRadius() < extendedMeshShape->getRadius())
                {
                    extendedMeshShape->setRadius(childMeshShape->getRadius());
                }

                for( hkInt32 j = 0; j < childMeshShape->getNumTrianglesSubparts(); ++j )
                {
                    hkpExtendedMeshShape::TrianglesSubpart& triangles = childMeshShape->getTrianglesSubpartAt(j);
                    _transformVertices( triangles, shapeToBody );
                    extendedMeshShape->addTrianglesSubpart( triangles );
                    numTriangles += triangles.m_numTriangleShapes;
                }
                // the number of nondegenerate triangles is the total number of validated triangles (getNumChildShapes validates)
                // less the number of shape subparts
                numNonDegenerateTriangles = childMeshShape->getNumChildShapes() - childMeshShape->getNumShapesSubparts();
                break;
            }
        case hkcdShapeType::TRIANGLE_COLLECTION :
            {
                hkpSimpleMeshShape* childMeshShape = static_cast<hkpSimpleMeshShape*> (childShape);

                if(childMeshShape->getRadius() < extendedMeshShape->getRadius())
                {
                    extendedMeshShape->setRadius(childMeshShape->getRadius());
                }

                hkpExtendedMeshShape::TrianglesSubpart tris;
                tris.m_vertexBase = reinterpret_cast<float*> (childMeshShape->m_vertices.begin());
                tris.m_vertexStriding = sizeof(hkVector4);
                tris.m_numVertices = childMeshShape->m_vertices.getSize();
                tris.m_indexBase = childMeshShape->m_triangles.begin();
                tris.m_indexStriding = sizeof(hkpSimpleMeshShape::Triangle);
                tris.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;
                tris.m_numTriangleShapes = childMeshShape->m_triangles.getSize();

                _transformVertices( tris, shapeToBody );

                extendedMeshShape->addTrianglesSubpart(tris);
                numTriangles = tris.m_numTriangleShapes;
                numNonDegenerateTriangles = childMeshShape->getNumChildShapes();
                break;
            }
        default :
            HK_WARN_ALWAYS( 0xabbadaff, "Can't add non-convex shape to the root extended mesh shape." );
            break;
        }

        // Check the shape for degeneracy
        if( numTriangles > numNonDegenerateTriangles  )
        {
            HK_WARN_ALWAYS( 0xabbadafd, "Detected degenerate triangles in shape \"" << childShapeInfo->m_node->m_name << "\". This may also be caused by scene scaling." );
        }

        //childShape->removeReference();
    }

    // Add all convex shapes to our extended mesh shape.
    {
        if( convexChildShapes.getSize() > 0 )
        {
            hkpExtendedMeshShape::ShapesSubpart shapesSP (convexChildShapes.begin(), convexChildShapes.getSize());
            extendedMeshShape->addShapesSubpart(shapesSP);

            for( hkInt32 i = 0; i < convexChildShapes.getSize(); ++i )
            {
                convexChildShapes[i]->removeReference();
            }
        }
    }
    return extendedMeshShape;
}

// Extract the geometry and materials info from an extended mesh shape for compressed meshes building.
// It appends data to the provided geometryOut and material array, thus allowing it to be called for several meshes sequentially.
static void _extractFromExtendedMeshShape( const hkpStorageExtendedMeshShape* meshShape, hkGeometry* geometryOut, hkArray<hkpNamedMeshMaterial>* namedMaterialsOut )
{
        int numMaterials = namedMaterialsOut->getSize();
        int numTriangles = 0;
        hkGeometry geometry;
        for( hkInt32 j = 0; j < meshShape->getNumTrianglesSubparts(); ++j )
        {
            const hkpExtendedMeshShape::TrianglesSubpart& triangles = meshShape->getTrianglesSubpartAt(j);
            geometry.m_vertices.append( (hkVector4*)triangles.m_vertexBase, triangles.m_numVertices );
            geometry.m_triangles.append( (hkGeometry::Triangle*)triangles.m_indexBase, triangles.m_numTriangleShapes );
            // add material data (if existing) and count the materials
            if ( triangles.getNumMaterials() != 0 && triangles.m_materialBase != HK_NULL && triangles.m_materialIndexBase != HK_NULL )
            {
                for ( int k = 0; k < triangles.m_numTriangleShapes; ++k )
                {
                    int index;
                    if ( triangles.m_materialIndexStriding == sizeof(hkUint8) )
                    {
                        index = ((hkUint8*)triangles.m_materialIndexBase)[k];
                    }
                    else
                    {
                        index = ((hkUint16*)triangles.m_materialIndexBase)[k];
                    }
                    geometry.m_triangles[numTriangles + k].m_material = numMaterials + index;
                }
                numMaterials += triangles.getNumMaterials();
                const hkpNamedMeshMaterial* materials = reinterpret_cast<const hkpNamedMeshMaterial*>( triangles.m_materialBase );
                namedMaterialsOut->append( materials, triangles.getNumMaterials() );
            }
            numTriangles += triangles.m_numTriangleShapes;
        }
        geometryOut->appendGeometry(geometry);
}

// Helper function to convert between our options enum and hkpWeldingUtility enum
static hkpWeldingUtility::WeldingType _getWeldingType(hctCreateRigidBodiesOptions::LanscapeWelding welding )
{
    switch (welding)
    {
        case hctCreateRigidBodiesOptions::WELD_COUNTERCLOCKWISE:
            return hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE;

        case hctCreateRigidBodiesOptions::WELD_CLOCKWISE:
            return hkpWeldingUtility::WELDING_TYPE_CLOCKWISE;

        case hctCreateRigidBodiesOptions::WELD_TWO_SIDED:
            return hkpWeldingUtility::WELDING_TYPE_TWO_SIDED;

        case hctCreateRigidBodiesOptions::WELD_NONE:
            return hkpWeldingUtility::WELDING_TYPE_NONE;

        default:
            HK_WARN_ALWAYS(0xabbabef3, "Unknown Welding Type");
            return hkpWeldingUtility::WELDING_TYPE_NONE;
    }
}

hkpBvCompressedMeshShape* hctCreateRigidBodiesFilter::buildBvCompressedMesh( hkArray<hkpConvexShape*>& convexChildShapes, hkArray<ShapeInfo*>& meshChildShapes, const hkTransform& rbTransform , const hkArray<hkpNamedMeshMaterial> & convexNamedMaterials)
{
    Log_Info( "Building BV Compressed Mesh Shape with {} mesh shapes and {} convex shapes.", meshChildShapes.getSize(), convexChildShapes.getSize() );

    // Gather the mesh geometry from all mesh shapes children
    hkGeometry resultGeometry;
    hkArray<hkpNamedMeshMaterial> namedMaterials;
    for ( int i = 0; i < meshChildShapes.getSize(); ++i )
    {
        ShapeInfo* childShapeInfo = meshChildShapes[i];
        hkpShape* childShape = childShapeInfo->m_shape;
        hkpStorageExtendedMeshShape* childMeshShape = static_cast<hkpStorageExtendedMeshShape*> (childShape);

        _extractFromExtendedMeshShape(childMeshShape, &resultGeometry, &namedMaterials);
    }

    // Gather the convex shapes and their transforms
    hkArray<const hkpConvexShape*> convexShapes;
    hkArray<hkQsTransform> transforms;
    for ( int i = 0; i < convexChildShapes.getSize(); ++i )
    {
        const hkpConvexShape * shape = convexChildShapes[i];
        hkQsTransform transform = hkQsTransform::getIdentity();

        // Extract transform from the convex transformed shapes, to feed them to the mesh info directly.
        if ( convexChildShapes[i]->getType() == hkcdShapeType::CONVEX_TRANSFORM )
        {
            hkpConvexTransformShape * transformShape = static_cast<hkpConvexTransformShape *>( convexChildShapes[i] );
            shape = transformShape->getChildShape();
            transform = transformShape->getQsTransform();
        }
        else if ( convexChildShapes[i]->getType() == hkcdShapeType::CONVEX_TRANSLATE )
        {
            hkpConvexTranslateShape * transformShape = static_cast<hkpConvexTranslateShape *>( convexChildShapes[i] );
            shape = transformShape->getChildShape();
            transform.setTranslation( transformShape->getTranslation() );
        }

        convexShapes.pushBack( shape );
        transforms.pushBack( transform );
    }

    if ( m_options.m_namedMaterialSource != hctCreateRigidBodiesOptions::MATERIAL_NONE )
    {
        // Use a custom cinfo which provides string data for the triangles and convex shapes
        hkpMaterialBvCompressedMeshShapeCinfo meshInfo( &resultGeometry, namedMaterials );

        HK_ASSERT( 0x7b9764c2, convexShapes.getSize() == convexNamedMaterials.getSize(), "Inconsistent number of materials for convex shapes" );
        for ( int i = 0; i < convexShapes.getSize(); ++i )
        {
            // Material information for convex shapes is sent to the builder along with the convex shape
            meshInfo.addConvexShape( convexShapes[i],transforms[i], convexNamedMaterials[i] );
        }

        // Add welding info.
        meshInfo.m_weldOpenEdges = m_options.m_weldOpenEdges;
        meshInfo.m_weldingType = _getWeldingType(m_options.m_landscapeWelding);
        return new hkpBvCompressedMeshShape( meshInfo );
    }
    else
    {
        // Use the default cinfo, without user data
        hkpDefaultBvCompressedMeshShapeCinfo meshInfo(&resultGeometry);
        for ( int i = 0; i < convexShapes.getSize() ; ++i )
        {
            meshInfo.addConvexShape( convexShapes[i], transforms[i] );
        }

        // Add welding info.
        meshInfo.m_weldOpenEdges = m_options.m_weldOpenEdges;
        meshInfo.m_weldingType = _getWeldingType(m_options.m_landscapeWelding);
        return new hkpBvCompressedMeshShape( meshInfo );
    }
}

hkpCompressedMeshShape* hctCreateRigidBodiesFilter::buildCompressedMesh( hkArray<hkpConvexShape*>& convexChildShapes, hkArray<ShapeInfo*>& meshChildShapes, const hkTransform& rbTransform)
{
    Log_Info( "Building Compressed Mesh Shape with {} mesh shapes and {} convex shapes.", meshChildShapes.getSize(), convexChildShapes.getSize() );

    // Gather geometries and materials from extended mesh shapes
    hkArray<hkpNamedMeshMaterial> namedMaterials;
    hkArray<hkGeometry> geometries;
    for( hkInt32 i = 0; i != meshChildShapes.getSize(); ++i )
    {
        ShapeInfo* childShapeInfo = meshChildShapes[i];
        hkpShape* childShape = childShapeInfo->m_shape;
        hkpStorageExtendedMeshShape* childMeshShape = static_cast<hkpStorageExtendedMeshShape*>( childShape );

        hkGeometry geometry;
        _extractFromExtendedMeshShape( childMeshShape, &geometry, &namedMaterials );
        geometries.pushBack( geometry );
    }

    int numMaterials = namedMaterials.getSize();

    // initialize the compressed mesh shape builder
    hkpCompressedMeshShapeBuilder builder;
    builder.m_overlapRatio = 0.2f;
    builder.m_stripperPasses = 5000;
    builder.m_weldVertices = false;
    builder.m_TjunctionTolerance = 1e-7f;

    // set the material type
    hkpCompressedMeshShape::MaterialType materialType;
    if ( numMaterials == 0 )
    {
        materialType = hkpCompressedMeshShape::MATERIAL_NONE;
    }
    else if ( numMaterials <= 256 )
    {
        materialType = hkpCompressedMeshShape::MATERIAL_ONE_BYTE_PER_TRIANGLE;
    }
    else if ( numMaterials <= ( 1 << 16) )
    {
        materialType = hkpCompressedMeshShape::MATERIAL_TWO_BYTES_PER_TRIANGLE;
    }
    else
    {
        materialType = hkpCompressedMeshShape::MATERIAL_FOUR_BYTES_PER_TRIANGLE;
    }

    // start building the mesh
    hkpCompressedMeshShape* compressedMeshShape = builder.createMeshShape( m_options.m_quantizationError, materialType );

    // assign the named materials array
    if ( namedMaterials.getSize() != 0 )
    {
        compressedMeshShape->m_namedMaterials = namedMaterials;
        compressedMeshShape->m_meshMaterials = namedMaterials.begin();
        compressedMeshShape->m_numMaterials = (hkUint16)namedMaterials.getSize();
        compressedMeshShape->m_materialStriding = sizeof( hkpNamedMeshMaterial );
    }

    // Add all mesh shapes's triangle data to our boy
    hkArray<hkpShape*> addedShapes;
    hkArray<int> subpartIDs;
    for( hkInt32 i = 0; i != meshChildShapes.getSize(); ++i )
    {
        ShapeInfo* childShapeInfo = meshChildShapes[i];
        hkpShape* childShape = childShapeInfo->m_shape;

        hkTransform shapeToBody;
        // child shape transform is worldFromShape
        shapeToBody.setMulInverseMul( rbTransform, childShapeInfo->m_decomposedWorldT );
        hkMatrix4 transform;
        transform.set( shapeToBody );

        // check if the shape is an instance of another
        int shapeIndex = addedShapes.indexOf( childShape );
        bool isInstance = ( shapeIndex != -1 );
        if ( isInstance )
        {
            builder.addInstance( subpartIDs[shapeIndex], transform, compressedMeshShape );
            continue;
        }

        shapeIndex = addedShapes.getSize();
        addedShapes.pushBack( childShape );
        const int subpartID = builder.beginSubpart( compressedMeshShape );
        subpartIDs.pushBack( subpartID );
        transform.setIdentity();

        builder.addGeometry( geometries[i], transform, compressedMeshShape );

        builder.endSubpart( compressedMeshShape );
        transform.set( shapeToBody );
        builder.addInstance( subpartIDs[shapeIndex], transform, compressedMeshShape );
    }

    
    // 2. Add all convex shapes to to the compressed mesh shape as convex pieces
    {
        for ( int i = 0; i < convexChildShapes.getSize(); ++i )
        {
            const hkpShape* convexShape = convexChildShapes[i];
            hkBool transformVertices = false;
            hkTransform transform; transform.setIdentity();
            if ( convexShape->m_type == hkcdShapeType::CONVEX_TRANSLATE )
            {
                const hkpConvexTranslateShape* convexTranslate = reinterpret_cast<const hkpConvexTranslateShape*>( convexShape );
                convexShape = convexTranslate->getChildShape();
                transformVertices = true;
                transform.setTranslation( convexTranslate->getTranslation() );
            }
            if ( convexShape->m_type == hkcdShapeType::CONVEX_VERTICES )
            {
                const hkpConvexVerticesShape* convexVerticesShape = reinterpret_cast<const hkpConvexVerticesShape*>( convexShape );
                hkArray<hkVector4> vertices;
                convexVerticesShape->getOriginalVertices( vertices );
                if ( transformVertices )
                {
                    for ( int j = 0; j < vertices.getSize(); ++j )
                    {
                        vertices[j].setTransformedPos( transform, vertices[j] );
                    }
                }
                if ( !builder.addConvexPiece( vertices, compressedMeshShape ) )
                {
                    Log_Error( "Failed to add convex piece. The convex hull should have no more than {} vertices and {} faces.",
                        hkpCompressedMeshShape::MAX_CONVEX_VERTICES, hkpCompressedMeshShape::MAX_CONVEX_FACES );
                }
            }
        }
    }
    return compressedMeshShape;
}

hkBool hctCreateRigidBodiesFilter::shouldBuildBvCompressedMesh( const hkxNode* node, const hkArray<hctCreateRigidBodiesFilter::ShapeInfo>& shapes )
{
    if ( m_options.m_meshShapeType != hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE )
    {
        return false;
    }

    int numMeshes = 0;
    for( int i = 0; i < shapes.getSize(); ++i )
    {
        const hkpShapeType type = shapes[i].m_shape->m_type;

        // We accept either meshes or all convex types.
        if ( type == hkcdShapeType::TRIANGLE_COLLECTION || type == hkcdShapeType::EXTENDED_MESH )
        {
            numMeshes++;
        }
        else if( !(shapes[i].m_isConvex) )
        {
            HK_WARN(0xabbae7b1, "'" << node->m_name << "' - Create BV Compressed Mesh failed: At least one shape is not supported by the BV Compressed Mesh Shape. A different shape type will be created instead.");
            return false;
        }
    }

    if ( numMeshes == 0 )
    {
        return false;
    }

    return true;
}
hkBool hctCreateRigidBodiesFilter::shouldBuildCompressedMesh( const hkxNode* node, const hkArray<hctCreateRigidBodiesFilter::ShapeInfo>& shapes )
{
    // Added logic to ensure that all shapes are supported by the Compressed Mesh Shape
    if ( m_options.m_meshShapeType != hctCreateRigidBodiesOptions::COMPRESSED_MESH_SHAPE )
    {
        return false;
    }

    int numMeshes = 0;
    for(int i = 0 ; i < shapes.getSize(); ++i )
    {
        const hkpShapeType type = shapes[i].m_shape->m_type;
        if( type != hkcdShapeType::CONVEX_TRANSLATE && type != hkcdShapeType::CONVEX_VERTICES && type != hkcdShapeType::TRIANGLE_COLLECTION && type != hkcdShapeType::EXTENDED_MESH )
        {
            HK_WARN( 0xabbae7b1, "'" << node->m_name << "' - Create Compressed Mesh failed: At least one shape is not supported by the Compressed Mesh Shape. A different shape type will be created instead." );
            return false;
        }
        else if ( type == hkcdShapeType::TRIANGLE_COLLECTION || type == hkcdShapeType::EXTENDED_MESH )
        {
            numMeshes++;
        }
    }

    if ( numMeshes == 0 )
    {
        return false;
    }

    return true;
}
hkpShape* hctCreateRigidBodiesFilter::gatherShapes(const hkxNode* node, hkArray<hctCreateRigidBodiesFilter::ShapeInfo>& shapes, const hkTransform& rbTransform )
{
    // Assert that we have some shapes
    HK_ASSERT( 0x6526dca6, shapes.getSize() > 0, "No shapes created." );

    // Compressed mesh shape has priority over everything.
    const hkBool useCompressedMesh = shouldBuildCompressedMesh( node, shapes );
    const hkBool useBvCompressedMesh = shouldBuildBvCompressedMesh( node, shapes );
    const hkBool useCompression = (useCompressedMesh || useBvCompressedMesh);

    HK_ASSERT(0x7d6bd340, !(useCompressedMesh && useBvCompressedMesh), "Cannot use two different compressed meshes types at the same time.");

    // If we have a single shape then return the single shape transformed into the rigid body's space
    if( shapes.getSize() == 1 && !useCompression )
    {
        ShapeInfo& shapeInfo = shapes[0];
        hkpShape* shape = shapes[0].m_shape;
        const hkpShapeContainer* shapeContainer = shape->getContainer();

        if( !shapeInfo.m_isConvex && shapeContainer )
        {
            const hkpShapeCollection* shapeCollectionConst = static_cast< const hkpShapeCollection* >( shape );
            hkpShapeCollection* shapeCollection = const_cast< hkpShapeCollection* >( shapeCollectionConst );

            // A MOPP is needed to create welding information.
            hkpShape* moppShape = setupMoppAndWelding( shapeCollection, node, rbTransform );

            // User wants to export MOPP code
            if( ( m_options.m_wrapWithMopp ) && ( shapeContainer->getNumChildShapes() >= m_options.m_listShapesMoppThreshold ) )
            {
                shape = moppShape;
                Log_Info( "Created MOPP Code for {}...", node->m_name );
            }
            else
            {
                shapeCollection->addReference();
                moppShape->removeReference();
            }

            shapeInfo.m_shape = shape;
            return shape;
        }
        else
        {
            return _wrapShapeWithTransform( shapeInfo, rbTransform );
        }
    }

    // Gather the mesh and convex shapes, and the material for each convex shape
    hkArray<ShapeInfo*> meshChildShapes;
    hkArray<hkpConvexShape*> convexChildShapes;
    hkArray<hkpNamedMeshMaterial> convexNamedMaterials;
    {
        meshChildShapes.reserveExactly( shapes.getSize() );
        convexChildShapes.reserveExactly( shapes.getSize() );

        const hkBool useTriangleSelections = ( m_options.m_namedMaterialSource == hctCreateRigidBodiesOptions::MATERIAL_TRIANGLE_SELECTIONS );

        for( hkInt32 i = 0; i < shapes.getSize(); ++i )
        {
            if( shapes[i].m_isConvex )
            {
                hkpShape* childShapeInRigidBodySpace = _wrapShapeWithTransform( shapes[i], rbTransform );

                // Gather convex material info if we want a BV compressed mesh
                if( ( m_options.m_namedMaterialSource != hctCreateRigidBodiesOptions::MATERIAL_NONE ) &&
                    ( m_options.m_meshShapeType == hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE ) )
                {
                    hkArray<hkUint16> convexMaterialIndices;
                    hkArray<hkpNamedMeshMaterial> materials;
                    hkBool overlap = false;
                    hctNamedMeshMaterialsUtil::getNamedMaterials(
                        _findNodeMesh(shapes[i].m_node), materials, convexMaterialIndices, overlap,
                        m_options.m_materialPrefix, m_options.m_joinOverlappingMeshMaterials, useTriangleSelections );

                    if (materials.getSize()> 0 )
                    {
                        // Remove the "default" named material that could be put here by the material extractor.
                        if ( materials[0].m_name == DEFAULT_MESH_MATERIAL_NAME )
                        {
                            materials.removeAtAndCopy(0);
                        }

                        if( materials.getSize() > 1 )
                        {
                            // Found a convex shape with more than one material selection, but we can only keep one material.
                            if ( m_options.m_joinOverlappingMeshMaterials )
                            {
                                // As with meshes, we will create a new material by concatenation of the found material names.
                                hkStringBuf newName( materials[0].m_name );
                                for ( int j = 1; j < materials.getSize(); ++j )
                                {
                                    newName.append( MESH_MATERIAL_SEPARATOR );
                                    newName.append( materials[j].m_name );
                                }

                                hkpNamedMeshMaterial newMaterial( newName.cString() );
                                materials[0] = newMaterial;
                                HK_WARN_ALWAYS( 0xabbaf4ce, "Found several materials in convex shape " << shapes[i].m_node->m_name << ", created new material '" << materials[0].m_name <<"'." );

                            }
                            else
                            {
                                // Keep the first material only.
                                HK_WARN_ALWAYS( 0xabbaf4cf, "Found several materials in convex shape " << shapes[i].m_node->m_name << ", the first material found (" << materials[0].m_name << ") will be used." );
                            }
                        }

                        convexNamedMaterials.pushBack( materials[0] );
                    }
                    else
                    {
                        // Use "empty string" material if no material is found.
                        convexNamedMaterials.pushBack( hkpNamedMeshMaterial( "" ) );
                    }
                }

                convexChildShapes.pushBack( static_cast<hkpConvexShape*>( childShapeInRigidBodySpace ) );
            }
            else
            {
                meshChildShapes.pushBack( &shapes[i] );
            }
        }
    }

    // Build the compound shape
    hkpShape* rootShape = HK_NULL;
    {
        if ( useCompressedMesh )
        {
            rootShape = buildCompressedMesh( convexChildShapes, meshChildShapes, rbTransform );
        }
        else if ( useBvCompressedMesh )
        {
            rootShape = buildBvCompressedMesh( convexChildShapes, meshChildShapes, rbTransform, convexNamedMaterials );

            // As the data are copied over, we can now dereference the shape
            for (int i = 0; i < convexChildShapes.getSize(); i++)
            {
                // Checks if the shape was not shared and delete it from the shared array;
                // (the shared array is not reference-counting). so we will end up with a
                // dangling pointer otherwise.
                for (int j = 0 ; j < m_instancedShapesData.getSize(); ++j)
                {
                    if (m_instancedShapesData[j].m_havokShape.m_shape == convexChildShapes[i]
                    && m_instancedShapesData[j].m_havokShape.m_shape->getReferenceCount() <= 1)
                    {
                        m_instancedShapesData.removeAtAndCopy(j);
                        break;
                    }
                }
                convexChildShapes[i]->removeReference();
            }

            // Same thing for mesh shapes.
            for (int i = 0; i < meshChildShapes.getSize(); i++)
            {
                for (int j = 0 ; j < m_instancedShapesData.getSize(); ++j)
                {
                    if (m_instancedShapesData[j].m_havokShape.m_shape == meshChildShapes[i]->m_shape
                    && m_instancedShapesData[j].m_havokShape.m_shape->getReferenceCount() <= 1)
                    {
                        m_instancedShapesData.removeAtAndCopy(j);
                        break;
                    }
                }

                meshChildShapes[i]->m_shape->removeReference();
            }
            convexChildShapes.clear();
            meshChildShapes.clear();
        }
        else if( meshChildShapes.isEmpty() ) // all are convex
        {
            // If none of the shapes are mesh shapes then build a hkpListShape with each shape transformed into the rigid body's space
            rootShape = buildListShapeFromConvexShapes( convexChildShapes );
        }
        else
        {
            // If we have one or more mesh shapes, build an hkpExtendedMeshShape that contains everything (transformed into the rigid body's space)
            rootShape = buildExtendedMeshFromCompound( convexChildShapes, meshChildShapes, rbTransform );
        }
    }

    // Build a MOPP, and compute welding info, if need be but skipping all the BV compressed meshes (as that contains its own BV)
    {
        if ( rootShape->getType() != hkcdShapeType::BV_COMPRESSED_MESH )
        {
            hkpShapeCollection* shapeCollection = static_cast<hkpShapeCollection*>( rootShape );
            hkpShape* moppShape = setupMoppAndWelding( shapeCollection, node, rbTransform );
            if( m_options.m_wrapWithMopp && shapeCollection->getNumChildShapes() >= m_options.m_listShapesMoppThreshold )
            {
                rootShape = moppShape;
                Log_Info( "Created MOPP code for compound shape {}...", node->m_name );
            }
            else
            {
                shapeCollection->addReference();
                moppShape->removeReference();
            }
        }
    }

    return rootShape;
}

// ShapeType Enum
enum ShapeTypeEnum
{
    ST_INVALID = -1,
    ST_BOX = 0,
    ST_SPHERE,
    ST_CAPSULE,
    ST_CYLINDER,
    ST_HULL,
    ST_MESH
};

hctFilterUtils::EnumItem shapeTypeItems[] =
{
    { ST_BOX,"Box" },
    { ST_SPHERE,"Sphere" },
    { ST_CAPSULE,"Capsule" },
    { ST_CYLINDER,"Cylinder" },
    { ST_HULL,"Hull" },
    { ST_MESH, "Mesh" }
};

hctFilterUtils::Enum g_shapeTypeEnum(hkArrayViewT::make(shapeTypeItems));

void hctCreateRigidBodiesFilter::registerCreatedShape (const class hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, const ShapeInfo& info)
{
    InstancedShapeData instancedShapeData;

    instancedShapeData.m_originalMesh = _findNodeMesh(node);

    const char* shapeTypeStr = HK_NULL;
    shapeAttribs->getStringValue("shapeType", true, shapeTypeStr);
    if (shapeTypeStr != HK_NULL)
    {
        instancedShapeData.m_shapeType = hctFilterUtils::getValueOfName(g_shapeTypeEnum, shapeTypeStr, instancedShapeData.m_shapeType);
    }

    instancedShapeData.m_extraRadius = m_options.m_defaultConvexRadius;
    shapeAttribs->getFloatValue("extraRadius", false,instancedShapeData.m_extraRadius);

    instancedShapeData.m_vertexTransform = vertexTransform;

    instancedShapeData.m_havokShape = info;

    m_instancedShapesData.pushBack(instancedShapeData);
}


bool hctCreateRigidBodiesFilter::lookForInstancedShape (const class hkxNode* node, const hkxAttributeGroup* shapeAttribs,  const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    hkxMesh* mesh = _findNodeMesh(node);
    float extraRadius = m_options.m_defaultConvexRadius;
    shapeAttribs->getFloatValue("extraRadius", false, extraRadius);

    int shapeType = ST_INVALID;
    {
        const char* shapeTypeStr = HK_NULL;
        shapeAttribs->getStringValue("shapeType", true, shapeTypeStr);
        if (shapeTypeStr != HK_NULL)
        {
            shapeType = hctFilterUtils::getValueOfName(g_shapeTypeEnum, shapeTypeStr, ST_INVALID);
        }
    }

    for (int i=0; i<m_instancedShapesData.getSize(); i++)
    {
        const InstancedShapeData& data = m_instancedShapesData[i];

        if  (   (data.m_originalMesh == mesh) &&
                (data.m_shapeType == shapeType) &&
                ((data.m_extraRadius == extraRadius) || (shapeType == ST_SPHERE) || (shapeType==ST_CAPSULE) ) && // extra radius ignored for spheres and capsules
                (data.m_vertexTransform.isApproximatelyEqual(vertexTransform,hkSimdReal::fromFloat(1e-3f)))
            )
        {
            infoOut = data.m_havokShape;
            return true;
        }
    }

    return false;
}


void hctCreateRigidBodiesFilter::makeProperties(const hkxNode* node, const hkxScene* theScene, hkpRigidBody* rb)
{
    const hkxAttributeGroup* propertyAttribs = node->findAttributeGroupByName("hkProperties");
    if (!propertyAttribs) return;

    hkpPropertyValue pvalue;

    for (int n=0; n<propertyAttribs->m_attributes.getSize(); ++n)
    {

        hkxAttribute attribute = propertyAttribs->m_attributes[n];
        const char* propertyName = attribute.m_name.cString();

        hkUint32 key;
        hkpProperty::mapStringToKey( propertyName, key );

        // determine type of property value

        // hkxSparselyAnimatedInt / hkxSparselyAnimatedEnum
        if ( hkReflect::exactMatchDynCast<hkxSparselyAnimatedInt>( attribute.m_value ) ||
             hkReflect::exactMatchDynCast<hkxSparselyAnimatedEnum>( attribute.m_value ) )
        {
            hkxSparselyAnimatedInt* int_property = static_cast<hkxSparselyAnimatedInt*> ( attribute.m_value.val() );

            //take the first integer
            if ( int_property->m_ints.getSize() > 0 )
            {
                int intValue = int_property->m_ints[0];
                pvalue.setInt( intValue );

                rb->addProperty( key, pvalue );
            }
        }

        // hkxAnimatedFloat
        else if ( hkxAnimatedFloat* float_property = hkReflect::exactMatchDynCast<hkxAnimatedFloat>( attribute.m_value ) )
        {
            //take the first float
            if ( float_property->m_floats.getSize() > 0 )
            {
                float floatValue = float_property->m_floats[0];
                pvalue.setReal( (hkReal)floatValue );

                rb->addProperty( key, pvalue );
            }
        }

        // hkxSparselyAnimatedBool
        else if ( hkxSparselyAnimatedBool* bool_property = hkReflect::exactMatchDynCast<hkxSparselyAnimatedBool>( attribute.m_value ) )
        {
            //take the first bool
            if ( bool_property->m_bools.getSize() > 0 )
            {
                if ( bool_property->m_bools[0] )
                    pvalue.setInt(1);
                else
                    pvalue.setInt(0);

                rb->addProperty( key, pvalue );
            }
        }

    }

}

void hctCreateRigidBodiesFilter::makeShapes(const hkxNode* currentNode, const hkxScene* theScene, const hkMatrix4& currentWorldTransform,  const hkTransform& rbTransform, bool collectHkdShapeShapes, bool isDynamic, hkArray<ShapeInfo>& shapes)
{
    ShapeInfo sinfo;
    sinfo.m_shape = HK_NULL;
    sinfo.m_extraShapeTransform.setIdentity();
    sinfo.m_node = currentNode;

    hkBool oldCollapseShapeOffsetsIntoShapes = m_options.m_collapseShapeOffsetsIntoShapes;
    const hkxAttributeGroup* breakableShapeAttributeGroup = currentNode->findAttributeGroupByName("hkdShape");
    if ( breakableShapeAttributeGroup )
    {
        m_options.m_collapseShapeOffsetsIntoShapes = false;
    }


    const hkxAttributeGroup* shapeAttribs = currentNode->findAttributeGroupByName("hkShape");

    // Data used
    int shapeType = ST_INVALID;
    bool failure = false;

    if (shapeAttribs)
    {
        // Shape Type: we use the string enum and convert it to the enum
        {
            const char* shapeTypeStr = HK_NULL;
            shapeAttribs->getStringValue("shapeType", true, shapeTypeStr);
            if (shapeTypeStr != HK_NULL)
            {
                shapeType = hctFilterUtils::getValueOfName(g_shapeTypeEnum, shapeTypeStr, ST_INVALID);
            }
        }

        // Give a warning if the node has no mesh exported
        if (! _findNodeMesh (currentNode) )
        {
            HK_WARN_ALWAYS(0xabbaa237, "No mesh exported with "<<currentNode->m_name<<". Cannot create shape from it.");
            failure = true;
        }

    }


    // Required : shapeType and a mesh
    const bool validShape = (shapeType != ST_INVALID) && (currentNode!=HK_NULL) && (!failure);

    if (validShape)
    {

        // EXP-1333 : Depending on the shape type and user choice in filter, we may place the shape in the space of the rigid body
        hkTransform shapeWorldTransform;
        {
            hkMatrixDecomposition::Decomposition decomposition;
            hkMatrixDecomposition::decomposeMatrix (currentWorldTransform, decomposition);

            if (!m_options.m_collapseShapeOffsetsIntoShapes)
            {
                shapeWorldTransform.set(decomposition.m_rotation, decomposition.m_translation);
            }
            else
            {
                switch ( shapeType )
                {
                    // For bounding volumes we still want to use the local space of the shape, since moving it will change the meaning of "bounding"
                case ST_BOX:
                case ST_SPHERE:
                case ST_CAPSULE:
                case ST_CYLINDER:
                    {
                        shapeWorldTransform.set(decomposition.m_rotation, decomposition.m_translation);
                        break;
                    }

                    // For all other types, the transform can be baked into the vertices
                case ST_MESH:
                case ST_HULL:
                    {
                        shapeWorldTransform = rbTransform;
                        break;
                    }
                }
            }

        }
        sinfo.m_decomposedWorldT = shapeWorldTransform;

        // worldTransform = decomposedT * vertexTransform
        // vertexTransform = inv(decomposedT) * worldTransform

        hkMatrix4 invDecomposedTransform;
        {
            hkMatrix4 decomposedTransform;
            decomposedTransform.set4x4ColumnMajor((hkReal*) &shapeWorldTransform);
            decomposedTransform.resetFourthRow();
            invDecomposedTransform.setInverse(decomposedTransform,hkSimdReal_0);
        }
        hkMatrix4 vertexTransform; vertexTransform.setMul(invDecomposedTransform, currentWorldTransform);

        // EXP-580
        bool alreadyCreated = lookForInstancedShape(currentNode, shapeAttribs, vertexTransform, sinfo);

        if (alreadyCreated)
        {
            // Reuse the shape
            sinfo.m_shape->addReference();

            // The transform in sinfo is unique for each instance - we reset it
            sinfo.m_decomposedWorldT = shapeWorldTransform;

            // and apply the extra shape transform (EXP-1202)
            hkTransform res;
            res.setMul(sinfo.m_decomposedWorldT, sinfo.m_extraShapeTransform);
            sinfo.m_decomposedWorldT = res;
        }
        else
        {
            if ( !m_options.m_enableAutomaticShapeShrinking )
            {
                if ( m_options.m_defaultConvexRadius == 0.0f )
                {
                    HK_WARN_ONCE(0xabbabfd1, "'Default Convex Radius' has been set to 0.0.");
                }
            }
            else
            {
                if ( m_options.m_relShrinkRadius == 0.0f || m_options.m_maxVertexDisplacement == 0.0f )
                {
                    HK_WARN_ONCE(0xabbabfd0, "Automatic Shape Shrinking is disabled (settings result in zero shrinkage).");
                }
            }

            switch ( shapeType )
            {
                case ST_BOX:
                    {
                        createBoxShape(currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }
                case ST_SPHERE:
                    {
                        createSphereShape(currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }

                case ST_CAPSULE:
                    {
                        createCapsuleShape(currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }

                case ST_CYLINDER:
                    {
                        createCylinderShape (currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }

                case ST_MESH:
                    {
                        if (isDynamic)
                        {
                            HK_WARN_ALWAYS (0xabba090f, "Dynamic (not fixed) rigid body with concave (Mesh) Shape ("<<currentNode->m_name<<").");
                            HK_WARN_ALWAYS (0xabba090f, "This is not fully supported, and may slow down the simulation.");
                        }

                        createMeshShape (currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }

                case ST_HULL:
                    {
                        createHullShape(currentNode, shapeAttribs, vertexTransform, sinfo);
                        break;
                    }

                default:
                    break;
            }

            if (sinfo.m_shape)
            {
                registerCreatedShape(currentNode, shapeAttribs, vertexTransform, sinfo);
            }
        }
    }

    // add shape to our list
    if (sinfo.m_shape)
    {
        shapes.pushBack(sinfo);
    }

    //might be a compound shape so continue to children that aren't rbs or breakable children
    hkMatrix4 currentChildTransform;
    for (int c=0; c < currentNode->m_children.getSize(); ++c)
    {
        hkxNode* child = currentNode->m_children[c];
        const hkxAttributeGroup* rbAttribs = child->findAttributeGroupByName("hkRigidBody");
        if ( rbAttribs ) // if not an rb, then see if we can make it a shape.
        {
            continue;
        }

        currentChildTransform.setMul( currentWorldTransform, child->m_keyFrames[0] );

        const hkxAttributeGroup* bsAttribs = child->findAttributeGroupByName("hkdShape");
        if ( bsAttribs )     // if not an breakable shape, then see if we can make it a shape.
        {
            if ( !collectHkdShapeShapes )
            {
                continue;
            }

                // try to collect shapes, if fails, try to continue one level deeper
            int numShapes = shapes.getSize();
            makeShapes(child, theScene, currentChildTransform, rbTransform, false, isDynamic, shapes);
            if ( numShapes != shapes.getSize() )
            {
                continue;   // success, continue
            }
        }
        makeShapes(child, theScene, currentChildTransform, rbTransform, collectHkdShapeShapes, isDynamic, shapes);
    }
    m_options.m_collapseShapeOffsetsIntoShapes = oldCollapseShapeOffsetsIntoShapes;
}

bool hctCreateRigidBodiesFilter::createBoxShape (const hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    hkpCreateShapeUtility::CreateShapeInput input;
    hkpCreateShapeUtility::ShapeInfoOutput  output;

    // Init utility input
    {
        input.m_decomposedWorldT    =   infoOut.m_decomposedWorldT;
        input.m_extraShapeTransform =   infoOut.m_extraShapeTransform;

        input.m_bestFittingShapes               =   m_options.m_bestFittingShapes;
        input.m_defaultConvexRadius             =   m_options.m_defaultConvexRadius;
        input.m_enableAutomaticShapeShrinking   =   m_options.m_enableAutomaticShapeShrinking;
        input.m_maxVertexDisplacement           =   m_options.m_maxVertexDisplacement;
        input.m_relShrinkRadius                 =   m_options.m_relShrinkRadius;

        input.m_extraRadiusOverride =   -1;
        {
            float extraRadiusOverride;
            hkResult res = shapeAttribs->getFloatValue("extraRadius", false, extraRadiusOverride);
            if ( res.isSuccess() )
            {
                input.m_enableAutomaticShapeShrinking   = false;
                input.m_extraRadiusOverride             = extraRadiusOverride;
            }
        }

        input.m_szMeshName  =   node->m_name;
        input.m_vertexTM    =   vertexTransform;
    }

    // Extract mesh vertices
    hkxMesh* mesh = _findNodeMesh(node);
    mesh->collectVertexPositions(input.m_vertices);

    // Call the utility
    hkResult bOk = hkpCreateShapeUtility::createBoxShape(input, output);

    // Fill the infoOut structure
    infoOut.m_decomposedWorldT      = output.m_decomposedWorldT;
    infoOut.m_extraShapeTransform   = output.m_extraShapeTransform;
    infoOut.m_isConvex              = output.m_isConvex;
    infoOut.m_shape                 = output.m_shape;

    // Return result
    return (bOk.isSuccess());
}

bool hctCreateRigidBodiesFilter::createSphereShape (const hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    hkpCreateShapeUtility::CreateShapeInput input;
    hkpCreateShapeUtility::ShapeInfoOutput  output;

    // Init utility input
    {
        input.m_decomposedWorldT    =   infoOut.m_decomposedWorldT;
        input.m_extraShapeTransform =   infoOut.m_extraShapeTransform;

        input.m_bestFittingShapes               =   m_options.m_bestFittingShapes;
        input.m_defaultConvexRadius             =   m_options.m_defaultConvexRadius;
        input.m_enableAutomaticShapeShrinking   =   m_options.m_enableAutomaticShapeShrinking;
        input.m_maxVertexDisplacement           =   m_options.m_maxVertexDisplacement;
        input.m_relShrinkRadius                 =   m_options.m_relShrinkRadius;

        input.m_szMeshName  =   node->m_name;
        input.m_vertexTM    =   vertexTransform;
    }

    // Extract mesh vertices
    hkxMesh* mesh = _findNodeMesh(node);
    mesh->collectVertexPositions(input.m_vertices);

    // Call the utility
    hkResult bOk = hkpCreateShapeUtility::createSphereShape(input, output);

    // Fill the infoOut structure
    infoOut.m_decomposedWorldT      = output.m_decomposedWorldT;
    infoOut.m_extraShapeTransform   = output.m_extraShapeTransform;
    infoOut.m_isConvex              = output.m_isConvex;
    infoOut.m_shape                 = output.m_shape;

    // Return result
    return (bOk.isSuccess());
}

bool hctCreateRigidBodiesFilter::createCapsuleShape (const hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    hkpCreateShapeUtility::CreateShapeInput input;
    hkpCreateShapeUtility::ShapeInfoOutput  output;

    // Init utility input
    {
        input.m_decomposedWorldT    =   infoOut.m_decomposedWorldT;
        input.m_extraShapeTransform =   infoOut.m_extraShapeTransform;

        input.m_bestFittingShapes               =   m_options.m_bestFittingShapes;
        input.m_defaultConvexRadius             =   m_options.m_defaultConvexRadius;
        input.m_enableAutomaticShapeShrinking   =   m_options.m_enableAutomaticShapeShrinking;
        input.m_maxVertexDisplacement           =   m_options.m_maxVertexDisplacement;
        input.m_relShrinkRadius                 =   m_options.m_relShrinkRadius;

        input.m_szMeshName  =   node->m_name;
        input.m_vertexTM    =   vertexTransform;
    }

    // Extract mesh vertices
    hkxMesh* mesh = _findNodeMesh(node);
    mesh->collectVertexPositions(input.m_vertices);

    // Call the utility
    hkResult bOk = hkpCreateShapeUtility::createCapsuleShape(input, output);

    // Fill the infoOut structure
    infoOut.m_decomposedWorldT      = output.m_decomposedWorldT;
    infoOut.m_extraShapeTransform   = output.m_extraShapeTransform;
    infoOut.m_isConvex              = output.m_isConvex;
    infoOut.m_shape                 = output.m_shape;

    // Return result
    return (bOk.isSuccess());
}

bool hctCreateRigidBodiesFilter::createCylinderShape(const hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    hkpCreateShapeUtility::CreateShapeInput input;
    hkpCreateShapeUtility::ShapeInfoOutput  output;

    // Init utility input
    {
        input.m_decomposedWorldT    =   infoOut.m_decomposedWorldT;
        input.m_extraShapeTransform =   infoOut.m_extraShapeTransform;

        input.m_bestFittingShapes               =   m_options.m_bestFittingShapes;
        input.m_defaultConvexRadius             =   m_options.m_defaultConvexRadius;
        input.m_enableAutomaticShapeShrinking   =   m_options.m_enableAutomaticShapeShrinking;
        input.m_maxVertexDisplacement           =   m_options.m_maxVertexDisplacement;
        input.m_relShrinkRadius                 =   m_options.m_relShrinkRadius;

        input.m_extraRadiusOverride =   -1;
        {
            float extraRadiusOverride;
            hkResult res = shapeAttribs->getFloatValue("extraRadius", false, extraRadiusOverride);
            if ( res.isSuccess() )
            {
                input.m_enableAutomaticShapeShrinking   = false;
                input.m_extraRadiusOverride             = extraRadiusOverride;
            }
        }

        input.m_szMeshName  =   node->m_name;
        input.m_vertexTM    =   vertexTransform;
    }

    // Extract mesh vertices
    hkxMesh* mesh = _findNodeMesh(node);
    mesh->collectVertexPositions(input.m_vertices);

    // Call the utility
    hkResult bOk = hkpCreateShapeUtility::createCylinderShape(input, output);

    // Fill the infoOut structure
    infoOut.m_decomposedWorldT      = output.m_decomposedWorldT;
    infoOut.m_extraShapeTransform   = output.m_extraShapeTransform;
    infoOut.m_isConvex              = output.m_isConvex;
    infoOut.m_shape                 = output.m_shape;

    // Return result
    return (bOk.isSuccess());
}

bool hctCreateRigidBodiesFilter::createHullShape (const hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut)
{
    // get vertices from exported mesh
    const hkxMesh*  mesh = _findNodeMesh(node);
    hkArray<hkVector4> tempStorage;
    hkStridedVertices stridedVerts;
    _getStridedVertices( mesh, vertexTransform, stridedVerts, tempStorage);

    // did we make a vertex set?
    if (stridedVerts.m_numVertices > 0)
    {
        bool automaticShrinkingEnabled = m_options.m_enableAutomaticShapeShrinking;

        // Get the convex/extra radius override. If it is available, we assume that this overrides the general settings here in the filter.
        float extraRadiusOverride = -1.0f; // Override range is between 0 and 1, so we are safe to use this for the time being.
        hkResult res = shapeAttribs->getFloatValue("extraRadius", false, extraRadiusOverride); // do not warn if attribute not found; extraRadiusOverride is left unchanged in this case
        if ( res.isSuccess() )
        {
            automaticShrinkingEnabled = false;
        }

        // Create the convex hull
        hkpConvexVerticesShape* cvs = HK_NULL;
        {
            float extraRadius = 0.0f;
            if ( !automaticShrinkingEnabled )
            {
                if ( extraRadiusOverride >= 0.0f )
                {
                    extraRadius = extraRadiusOverride;
                }
                else
                {
                    extraRadius = m_options.m_defaultConvexRadius;
                }
            }

            hkpConvexVerticesShape::BuildConfig config;
            config.m_convexRadius = 0.0f;
            config.m_shrinkByConvexRadius = false;

            
            
            
            
            if ( m_options.m_meshShapeType == hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE )
            {
                config.m_maxVertices = hkpBvCompressedMeshShape::MAX_NUM_VERTICES_PER_HULL;
                config.m_useOptimizedShrinking = true;
                if ( automaticShrinkingEnabled )
                {
                    config.m_maxRelativeShrink = m_options.m_relShrinkRadius;
                    config.m_maxShrinkingVerticesDisplacement = m_options.m_maxVertexDisplacement;
                }
                else
                {
                    config.m_convexRadius = extraRadius;
                    config.m_shrinkByConvexRadius = true;
                }
            }

            cvs = new hkpConvexVerticesShape(stridedVerts, config);
            if ( cvs )
            {
                cvs->setRadius( extraRadius );
            }
        }

        // Post process
        if ( cvs && cvs->getNumCollisionSpheres() > 0)
        {
            bool shrunk = false;

            // Shrink shape if option is enabled (with meaningful values). Disabled with BV compressed mesh (see above)
            if ( automaticShrinkingEnabled && m_options.m_relShrinkRadius != 0.0f && m_options.m_maxVertexDisplacement != 0.0f &&
                 m_options.m_meshShapeType != hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE )
            {
                hkReal maximumConvexRadius = HK_REAL_MAX * 0.1f;
                hkpConvexVerticesShape* shrinkShape = hkpShapeShrinker::shrinkConvexVerticesShape( cvs, maximumConvexRadius, m_options.m_relShrinkRadius, m_options.m_maxVertexDisplacement, node->m_name );
                if ( shrinkShape != HK_NULL )
                {
                    shrunk = true;

                    // Delete original shape
                    cvs->removeReference();

                    // Swap pointers
                    cvs = shrinkShape;
                    shrinkShape = HK_NULL;
                }
                else
                {
                    HK_WARN_ALWAYS(0xabbac982, "'" << node->m_name << "' - Failed to shrink the shape.");
                }
            }

            // Issue some warnings if calculated convex/extra radius is either 0.0 or rather large.
            {
                hkReal convexRadius = cvs->getRadius();

                hkReal meshSize;
                {
                    hkVector4 halfExtents;
                    hkTransform geomFromBox;
                    bool aabb = computeAABB ( mesh, vertexTransform, halfExtents, geomFromBox );
                    if (!aabb)
                    {
                        HK_WARN_ALWAYS(0xabbaa3fc, "Couldn't calculate AABB for node " << node->m_name);
                        return false;
                    }
                    meshSize = halfExtents (halfExtents.getMajorAxis3());
                }

                if ( shrunk && convexRadius == 0.0f )
                {
                    HK_WARN_ALWAYS (0xabbabfd6, "'" << node->m_name << "' - Calculated 'extra radius' for convex hull shape is 0. Performance may be affected.");
                }
                else if ( convexRadius > meshSize )
                {
                    HK_WARN_ALWAYS (0xabbabad7, "'" << node->m_name << "' - Calculated 'extra radius' for convex hull shape is very large (radius " << convexRadius << " > halfExtent " << meshSize << " )");
                }
            }

            infoOut.m_shape = cvs;
            infoOut.m_isConvex = true;

            return true;
        }
        else
        {
            if ( cvs )
            {
                cvs->removeReference();
            }
        }
    }

    HK_WARN_ALWAYS(0xabba9fec, "Couldn't create hull from " << node->m_name);
    return false;
}

hkpShape* hctCreateRigidBodiesFilter::setupMoppAndWelding (hkpShapeCollection* shapeCollection, const hkxNode* node, const hkTransform& rbTransform )
{
    // We remove the "do not use buildCode() at run-time" since we are not really in run-time
    hkBool wasEnabled = hkError::getInstance().isEnabled(0x6e8d163b);
    hkError::getInstance().setEnabled(0x6e8d163b, false); // hkpMoppUtility.cpp:18

    hkpMoppCompilerInput mci;

    // EXP-1094 / EXP-1376 /
    mci.m_enableChunkSubdivision = m_options.m_enableMoppChunks;

    hkpMoppCode* moppCode = hkpMoppUtility::buildCode(shapeCollection, mci);

    hkError::getInstance().setEnabled(0x6e8d163b, wasEnabled);

    // EXP-1136 : Welding Info
    hkpWeldingUtility::WeldingType weldType = _getWeldingType(m_options.m_landscapeWelding);

    if (!moppCode)
    {
        HK_WARN_ALWAYS(0xabba0bb1, "MOPP code creation failed. Welding information could not be computed");
        return shapeCollection;
    }
    else
    {
        hkpMoppBvTreeShape* moppShape = new hkpMoppBvTreeShape(shapeCollection, moppCode);

        if ( weldType != hkpWeldingUtility::WELDING_TYPE_NONE && shapeCollection->m_collectionType != hkpShapeCollection::COLLECTION_LIST )
        {
            if ( weldType == hkpWeldingUtility::WELDING_TYPE_TWO_SIDED )
            {
                HK_WARN_ALWAYS(0xabbacafe, "Welding type is set to 'two-sided' : Note that one-sided welding is recommended.");
            }
            if ( m_options.m_weldAcrossAllMopps )
            {
                hkpMeshWeldingUtility::ShapeInfo info;
                info.m_transform = rbTransform;
                info.m_shape = moppShape;
                m_moppShapes.pushBack( info );
                moppShape->addReference();
            }
            else
            {
                Log_Info( "Computing Welding Information for {}...", node->m_name );
                hkpMeshWeldingUtility::computeWeldingInfo( shapeCollection, moppShape, weldType, m_options.m_weldOpenEdges, m_options.m_markEdgesBadWinding );
            }
        }

        shapeCollection->removeReference();
        moppCode->removeReference();
        return moppShape;
    }

    // doesn't really get here
    // return HK_NULL;
}

void hctCreateRigidBodiesFilter::getGeometryAndMaterials( const hkxNode* node, const hkMatrix4& vertexTransform, hkArray<hkpNamedMeshMaterial>& materialsOut, hkArray<hkUint16>& indicesOut, hkGeometry& geometryOut, hkBool mergeDuplicates )
{
    const hkxMesh* mesh = _findNodeMesh(node);
    hkBool overlap = false;
    if( !(m_options.m_namedMaterialSource == hctCreateRigidBodiesOptions::MATERIAL_NONE) )
    {
        const hkBool useTriangleSelections = m_options.m_namedMaterialSource == hctCreateRigidBodiesOptions::MATERIAL_TRIANGLE_SELECTIONS;
        hctNamedMeshMaterialsUtil::getNamedMaterials( mesh, materialsOut, indicesOut, overlap, m_options.m_materialPrefix, m_options.m_joinOverlappingMeshMaterials, useTriangleSelections );

        if ( materialsOut.getSize() > 0 )
        {
            // If using triangle selections there will be an extra default material for unselected triangles
            const int numMaterials = materialsOut.getSize() - ( useTriangleSelections ? 1 : 0 );
            Log_Info( "Exporting materials (Found {} mesh material(s) in mesh {}).", numMaterials, node->m_name );
            if ( overlap )
            {
                if ( m_options.m_joinOverlappingMeshMaterials )
                {
                    Log_Warning( "Found overlapping materials. (New materials were created to handle this.)" );
                }
                else
                {
                    Log_Warning( "Found overlapping materials. (Triangles in overlaps were assigned the first material encountered.)" );
                }
            }
        }
        else if( useTriangleSelections )
        {
            Log_Error( "No triangle selection materials with prefix \"{}\" found in mesh {}.", m_options.m_materialPrefix, node->m_name );
        }
        else
        {
            Log_Error( "No mesh section materials found in mesh {}.", node->m_name );
        }
    }

    mesh->appendGeometry( geometryOut );

    // Transform vertices
    {
        for (int i=0; i< geometryOut.m_vertices.getSize(); i++)
        {
            vertexTransform.transformPosition( geometryOut.m_vertices[i], geometryOut.m_vertices[i]);
        }
    }

    // EXP-1081
    if ( mergeDuplicates )
    {
        const int originalNumVertices = geometryOut.m_vertices.getSize();
        const int originalNumTriangles = geometryOut.m_triangles.getSize();

        hkArray<int> vertexRemap;
        hkArray<int> triangleRemap;
        hkGeometryUtils::weldVertices( geometryOut, 1e-3f, false, vertexRemap, triangleRemap );

        const int removedVertices = originalNumVertices - geometryOut.m_vertices.getSize();
        const int removedTriangles = originalNumTriangles - geometryOut.m_triangles.getSize();

        if (removedVertices>0)
        {
            Log_Info( "Removed {} duplicated vertices in mesh {}", removedVertices, node->m_name );
        }

        if (removedTriangles>0)
        {
            Log_Info( "Removed {} duplicated triangles in mesh {}", removedTriangles, node->m_name );

            if( materialsOut.getSize() > 0 )
            {
                hkUint16* nonDegenerate = indicesOut.begin();
                for( int i = 0 ; i < indicesOut.getSize() ; i++ )
                {
                    hkUint16 index = indicesOut[i];
                    if( triangleRemap[i] != -1 )
                    {
                        *nonDegenerate++ = index;
                    }
                }
                indicesOut.setSize( (int) ( nonDegenerate - indicesOut.begin() ) );
            }
        }
    }

}
bool hctCreateRigidBodiesFilter::createMeshShape( const class hkxNode* node, const hkxAttributeGroup* shapeAttribs, const hkMatrix4& vertexTransform, ShapeInfo& infoOut )
{
    const hkxMesh* mesh = _findNodeMesh(node);

    bool automaticShrinkingEnabled = m_options.m_enableAutomaticShapeShrinking;

    float extraRadiusOverride = -1.0f; // Override range is between 0 and 1, so we are safe to use this for the time being.

    // Get the convex/extra radius override. If it is available, we assume that this overrides the general settings here in the filter.
    hkResult res = shapeAttribs->getFloatValue("extraRadius", false, extraRadiusOverride); // do not warn if attribute not found; extraRadiusOverride is left unchanged in this case
    if ( res.isSuccess() )
    {
        automaticShrinkingEnabled = false;
    }

    float extraRadius = 0.0f;
    if ( !automaticShrinkingEnabled )
    {
        if ( extraRadiusOverride >= 0.0f )
        {
            extraRadius = extraRadiusOverride;
        }
        else
        {
            extraRadius = m_options.m_defaultConvexRadius;
        }

        // Get an indication of the size of the mesh, use the AABB for that
        hkReal meshSize;
        {
            hkVector4 halfExtents;
            hkTransform geomFromBox;
            bool aabb = computeAABB ( _findNodeMesh(node), vertexTransform, halfExtents, geomFromBox );
            if (!aabb)
            {
                HK_WARN_ALWAYS(0xabbaa3fc, "Couldn't calculate AABB for node "<<node->m_name);
                return false;
            }
            meshSize = halfExtents (halfExtents.getMajorAxis3());
        }

        // Issue some warnings if calculated convex/extra radius is either 0.0 or rather large.
        if ( extraRadius == 0.0f )
        {
            HK_WARN_ALWAYS (0xabbabad4, "'" << node->m_name << "' - Calculated 'extra radius' for mesh shape is 0. Performance may be affected.");
        }
        else if ( extraRadius > meshSize )
        {
            HK_WARN_ALWAYS (0xabbabad3, "'" << node->m_name << "' - Calculated 'extra radius' for mesh shape is very large (radius " << extraRadius << " > halfExtent " << meshSize << " )");
        }
    }
    else
    {
        HK_WARN_ALWAYS(0xabbafad3, "'" << node->m_name << "' - Shape shrinking not supported for mesh shapes. 'Extra Radius' will be 0. Performance may be affected.");
    }

    if (mesh->m_sections.getSize() > 0)
    {
        hkArray<hkpNamedMeshMaterial> namedMaterials;
        hkArray<hkUint16> indices;
        hkGeometry meshGeometry;

        getGeometryAndMaterials( node, vertexTransform, namedMaterials, indices, meshGeometry, m_options.m_mergeMeshDuplicates );

        const hkBool exportNamedMaterials = ( namedMaterials.getSize() > 0 );

        // EXP-1307 : Use extended mesh shape if user asked for it
        switch( m_options.m_meshShapeType )
        {
            // We create an Extended mesh shape first, it will be compressed at the gathering pass if required
            case hctCreateRigidBodiesOptions::EXTENDED_MESH_SHAPE:
            case hctCreateRigidBodiesOptions::COMPRESSED_MESH_SHAPE:
            case hctCreateRigidBodiesOptions::BV_COMPRESSED_MESH_SHAPE:
            {
                hkpStorageExtendedMeshShape* extendedMeshShape = new hkpStorageExtendedMeshShape(extraRadius);

                hkpExtendedMeshShape::TrianglesSubpart tris;

                tris.m_vertexBase = reinterpret_cast<float*> (meshGeometry.m_vertices.begin());
                tris.m_vertexStriding = sizeof(hkVector4);
                tris.m_numVertices = meshGeometry.m_vertices.getSize();
                tris.m_indexBase = meshGeometry.m_triangles.begin();

                tris.m_indexStriding = sizeof(hkGeometry::Triangle);
                tris.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;
                tris.m_numTriangleShapes = meshGeometry.m_triangles.getSize();

                hkArray<hkUint8>    indices8;

                if ( exportNamedMaterials )
                {
                    //
                    //  Setup triangle subpart named material.
                    //

                    extendedMeshShape->m_materialClass = hkReflect::getType<hkpNamedMeshMaterial>();
                    if( indices.getSize() > 255 )
                    {
                        tris.m_materialIndexBase = indices.begin();
                        tris.m_materialIndexStriding = sizeof(hkUint16);
                    }
                    else
                    {
                        indices8.setSize( indices.getSize() );
                        for( int i = 0 ; i < indices.getSize() ; i ++ )
                        {
                            indices8[i] = static_cast<hkUint8>( indices[i] );
                        }

                        tris.m_materialIndexBase = indices8.begin();
                        tris.m_materialIndexStriding = sizeof(hkUint8);
                    }

                    tris.m_materialBase = (const hkpMeshMaterial*)namedMaterials.begin();
                    tris.setNumMaterials((hkUint16)namedMaterials.getSize());
                    tris.m_materialStriding = sizeof(hkpNamedMeshMaterial);
                }

                extendedMeshShape->addTrianglesSubpart(tris);

                infoOut.m_shape = extendedMeshShape;

                // We don't put MOPP or welding yet, as we may add shapes to it
                break;
            }

            case hctCreateRigidBodiesOptions::SIMPLE_MESH_SHAPE:
            {
                //Convert hkGeometry into hkpSimpleMeshShape
                hkpSimpleMeshShape* simpleMeshShape = new hkpSimpleMeshShape;

                // EXP-444
                simpleMeshShape->setRadius( extraRadius );

                simpleMeshShape->m_vertices = meshGeometry.m_vertices;
                simpleMeshShape->m_triangles.setSize(  meshGeometry.m_triangles.getSize() );

                if( exportNamedMaterials )
                {
                    Log_Warning( "{} - Cannot export named mesh materials with simple mesh shapes, so only the indices are exported.", node->m_name );
                    if( indices.getSize() > 255 )
                    {
                        HK_WARN_ALWAYS( 0xabbaec0a , "" << node->m_name << " - Too many named mesh materials (>255) found for a simple mesh shape.");
                    }
                    else
                    {
                        simpleMeshShape->m_materialIndices.setSize( simpleMeshShape->m_triangles.getSize() );
                        for( int ti=0; ti < simpleMeshShape->m_triangles.getSize() ; ++ti )
                        {
                            simpleMeshShape->m_materialIndices[ti] = (hkUint8)indices[ti];
                        }
                    }
                }

                for (int ti=0; ti<meshGeometry.m_triangles.getSize(); ++ti)
                {
                    hkpSimpleMeshShape::Triangle triangle;
                    triangle.m_a = meshGeometry.m_triangles[ti].m_a;
                    triangle.m_b = meshGeometry.m_triangles[ti].m_b;
                    triangle.m_c = meshGeometry.m_triangles[ti].m_c;
                    simpleMeshShape->m_triangles[ti] = triangle;
                }

                infoOut.m_shape = simpleMeshShape;
                break;
            }

            default:
            {
                // Should not get here!
                HK_ASSERT_NO_MSG( 0x71c10bcd, false );
                break;
            }
        }
        HK_ASSERT_NO_MSG(0x57b3c1fa, infoOut.m_shape);

        infoOut.m_isConvex = false;
    }

    return true;
}

// Advanced attribute enums (to match strings defined in hkCommonAttributeData.xml)
hctFilterUtils::EnumItem qualityTypeItems[] =
{
    { HK_COLLIDABLE_QUALITY_FIXED, "Fixed" },
    { HK_COLLIDABLE_QUALITY_KEYFRAMED, "Keyframed" },
    { HK_COLLIDABLE_QUALITY_DEBRIS, "Debris" },
    { HK_COLLIDABLE_QUALITY_DEBRIS_SIMPLE_TOI, "Debris Simple TOI" },
    { HK_COLLIDABLE_QUALITY_MOVING, "Moving" },
    { HK_COLLIDABLE_QUALITY_CRITICAL, "Critical" },
    { HK_COLLIDABLE_QUALITY_BULLET, "Bullet" },
    { HK_COLLIDABLE_QUALITY_USER, "User" },
    { HK_COLLIDABLE_QUALITY_KEYFRAMED_REPORTING, "Keyframed Reporting" }
};
hctFilterUtils::EnumItem solverDeactivationItems[] =
{
    { hkpRigidBodyCinfo::SOLVER_DEACTIVATION_OFF, "Off" },
    { hkpRigidBodyCinfo::SOLVER_DEACTIVATION_LOW, "Low" },
    { hkpRigidBodyCinfo::SOLVER_DEACTIVATION_MEDIUM, "Medium" },
    { hkpRigidBodyCinfo::SOLVER_DEACTIVATION_HIGH, "High" }
};
hctFilterUtils::EnumItem rigidBodyDeactivatorTypeItems[] =
{
    { 1, "Spatial" },
    { 0, "Never" }
};
hctFilterUtils::Enum g_qualityTypeEnum(hkArrayViewT::make(qualityTypeItems));
hctFilterUtils::Enum g_solverDeactivationEnum(hkArrayViewT::make(solverDeactivationItems));
hctFilterUtils::Enum g_rigidBodyDeactivatorTypeEnum(hkArrayViewT::make(rigidBodyDeactivatorTypeItems));


hkpRigidBody* hctCreateRigidBodiesFilter::makeRigidBody(const hkxNode* node, const hkxScene* theScene, const hkxAttributeGroup& group, const hkMatrix4& currentWorldTransform,  hkArray<ShapeInfo>& shapeInfosOut, bool& hkdShapesCollected, hkStringMap<hkLocalFrameGroup*>& groupsCreated )
{

    // The constructor of hkpRigidBodyCinfo sets defaults. We (possibly) override them from the attributes

    hkpRigidBodyCinfo cinfo;

    group.getFloatValue("mass", true, cinfo.m_mass);
    group.getFloatValue("friction", false,cinfo.m_friction);
    group.getFloatValue("restitution", false,cinfo.m_restitution);
    group.getVectorValue("linearVelocity", false, cinfo.m_linearVelocity);
    group.getVectorValue("angularVelocity", false, cinfo.m_angularVelocity);

    // optional (advanced) attributes

    const char* qualityTypeStr = HK_NULL;
    bool overrideQualityType = ( group.getStringValue("qualityType", false, qualityTypeStr).isSuccess() );
    if (qualityTypeStr != HK_NULL)
    {
        int qualityType = hctFilterUtils::getValueOfName(g_qualityTypeEnum, qualityTypeStr, 0);
        if ( overrideQualityType )
        {
            cinfo.m_qualityType = static_cast<enum hkpCollidableQualityType>(qualityType);
        }
    }

    const char* solverDeactivationStr = HK_NULL;
    bool overrideSolverDeactivation = ( group.getStringValue("solverDeactivation", false, solverDeactivationStr).isSuccess() );
    if (solverDeactivationStr != HK_NULL)
    {
        int solverDeactivation = hctFilterUtils::getValueOfName(g_solverDeactivationEnum, solverDeactivationStr, 0);
        if ( overrideSolverDeactivation )
        {
            cinfo.m_solverDeactivation = static_cast<enum hkpRigidBodyCinfo::SolverDeactivation>(solverDeactivation);
        }
    }

    const char* rigidBodyDeactivatorTypeStr = HK_NULL;
    bool overrideRigidBodyDeactivatorType = ( group.getStringValue("rigidBodyDeactivatorType", false, rigidBodyDeactivatorTypeStr).isSuccess() );
    if (rigidBodyDeactivatorTypeStr != HK_NULL)
    {
        int rigidBodyDeactivatorType = hctFilterUtils::getValueOfName(g_rigidBodyDeactivatorTypeEnum, rigidBodyDeactivatorTypeStr, 0);
        if ( overrideRigidBodyDeactivatorType )
        {
            cinfo.m_enableDeactivation = rigidBodyDeactivatorType != 0;
        }
    }

    int collisionFilterInfo;
    bool overrideCollisionFilterInfo = (group.getIntValue("collisionFilterInfo", false, collisionFilterInfo).isSuccess());
    if ( overrideCollisionFilterInfo )
    {
        cinfo.m_collisionFilterInfo = collisionFilterInfo;
    }

    hkReal linearDamping;
    bool overrideLinearDamping = (group.getFloatValue("linearDamping", false, linearDamping).isSuccess());
    if ( overrideLinearDamping )
    {
        cinfo.m_linearDamping = linearDamping;
    }

    hkReal angularDamping;
    bool overrideAngularDamping = (group.getFloatValue("angularDamping", false, angularDamping).isSuccess());
    if ( overrideAngularDamping )
    {
        cinfo.m_angularDamping = angularDamping;
    }

    hkReal maxAngularVelocity;
    bool overrideMaxAngularVelocity = ( group.getFloatValue("maxAngularVelocity", false, maxAngularVelocity).isSuccess());
    if ( overrideMaxAngularVelocity )
    {
        cinfo.m_maxAngularVelocity = maxAngularVelocity;
    }

    hkReal maxLinearVelocity;
    bool overrideMaxLinearVelocity = ( group.getFloatValue("maxLinearVelocity", false, maxLinearVelocity).isSuccess());
    if ( overrideMaxLinearVelocity )
    {
        cinfo.m_maxLinearVelocity = maxLinearVelocity;
    }

    hkReal allowedPenetrationDepth;
    bool overrideAllowedPenetrationDepth = ( group.getFloatValue("allowedPenetrationDepth", false, allowedPenetrationDepth).isSuccess());
    if ( overrideAllowedPenetrationDepth )
    {
        cinfo.m_allowedPenetrationDepth = allowedPenetrationDepth;
    }

    // given our list of shapes, convert to
    // a single shape pointer
    hkMatrixDecomposition::Decomposition decomposition;
    hkMatrixDecomposition::decomposeMatrix(currentWorldTransform, decomposition);
    cinfo.m_rotation = decomposition.m_rotation;
    cinfo.m_position = decomposition.m_translation;

    const hkTransform rbTransform (decomposition.m_rotation, decomposition.m_translation);

    // recurse down to make shapes.

    const bool isDynamic = (cinfo.m_mass != 0.0f);

    // Collect simple physics shapes only. Child Breakable Shapes will be ignored.
    hkdShapesCollected = false;
    makeShapes(node, theScene, currentWorldTransform, rbTransform, false, isDynamic, shapeInfosOut);

    if ( shapeInfosOut.isEmpty() )
    {
        // Root node is no simple physics shape (but could be a Breakable Shape) and doesn't have any simple child
        // physics shapes (but could have child Breakable Shapes).

        // Now collect child Breakable Shapes.
        hkdShapesCollected = true;
        makeShapes(node, theScene, currentWorldTransform, rbTransform, true, isDynamic, shapeInfosOut);

        // If we have shapes now that means there's no root physics shape but valid child breakable shapes. In this
        // case it should be a Compound Breakable Shape.
    }

    if (shapeInfosOut.getSize() == 0)
    {
        HK_WARN_ALWAYS(0xabbaf331, "'" << node->m_name << "' - No shapes found. Cannot create Rigid Body.");
        return HK_NULL;
    }

    cinfo.m_shape = gatherShapes(node, shapeInfosOut, rbTransform);

    if (!cinfo.m_shape)
    {
        return HK_NULL;
    }

    /* Center of mass and inertial tensor */
    hkMatrix3 defaultTensor;
    hkVector4 defaultCOM;

    if (cinfo.m_mass > 0)
    {
        hkMassProperties res;
        hkpInertiaTensorComputer::computeShapeVolumeMassProperties(cinfo.m_shape, cinfo.m_mass, res);
        defaultTensor = res.m_inertiaTensor;
        defaultCOM = res.m_centerOfMass;
    }
    else
    {
        hkMatrix3Util::_setDiagonal( 1.0, 1.0f, 1.0f, defaultTensor);
        defaultCOM.setZero4();
    }

    // Center of mass -> it's node-local, we need to convert to rigid body-local
    {
        hkVector4 comLocal(0,0,0);
        const bool overrideCOM = (group.getVectorValue("centerOfMass", false, comLocal).isSuccess());

        if (overrideCOM)
        {
            comLocal(3)=1.0f;

            // No vector multiply in matrix4, use matrix4 multiply
            // todo : cast it to hktransform
            hkMatrix4 comNodeLocalMatrix;
            comNodeLocalMatrix.setIdentity();
            comNodeLocalMatrix.getColumn(3) = comLocal;

            const hkMatrix4& worldFromNode = currentWorldTransform;
            hkMatrix4 comWorldMatrix;
            comWorldMatrix.setMul( worldFromNode, comNodeLocalMatrix);

            hkVector4 comWorld = comWorldMatrix.getColumn(3);

            const hkTransform worldFromRigidBody = hkTransform(cinfo.m_rotation, cinfo.m_position);

            cinfo.m_centerOfMass.setTransformedInversePos(worldFromRigidBody, comWorld);
        }
        else
        {
            cinfo.m_centerOfMass = defaultCOM;
        }
    }

    // Inertial tensor
    {
        // convert inertia diagonal to 3x3 tensor
        hkVector4 tensorDiagonal(1,1,1);
        bool overrideTensor = (group.getVectorValue("inertiaTensor", false, tensorDiagonal).isSuccess());

        if (tensorDiagonal.length3()<0.001f)
        {
            HK_WARN_ALWAYS( 0xabba36fa, "Invalid tensor. Using computed one." );
            overrideTensor = false;
        }

        if (overrideTensor)
        {
            // TODO : handle changes in local transform (due to scale or scene transform)

            tensorDiagonal.setAbs4( tensorDiagonal );
            const hkReal normalizeScaleFactor = pow( 1 / (tensorDiagonal(0) * tensorDiagonal(1) * tensorDiagonal(2) ), 0.33333f );
            hkVector4 defaultTensorDiagonal( defaultTensor(0,0), defaultTensor(1,1), defaultTensor(2,2) );
            tensorDiagonal.mul4( normalizeScaleFactor );
            tensorDiagonal.mul4( defaultTensorDiagonal );

            hkMatrix3Util::_setDiagonal( tensorDiagonal(0), tensorDiagonal(1), tensorDiagonal(2), cinfo.m_inertiaTensor );
        }
        else
        {
            cinfo.m_inertiaTensor = defaultTensor;
        }

        hkBool scaleTensor;
        if ( ( group.getBoolValue( "scaleInertiaTensor", false, scaleTensor ).isSuccess() ) && scaleTensor )
        {
            float inertiaTensorScale;
            if ( group.getFloatValue( "inertiaTensorScale", false, inertiaTensorScale ).isSuccess() )
            {
                cinfo.m_inertiaTensor.mul( inertiaTensorScale );
            }
        }
    }

    /*
    ** Consistency checks for mass vs. motion vs. quality
    */
    if (overrideQualityType) // the user specified quality -> takes precedence over mass
    {
        if (cinfo.m_qualityType == HK_COLLIDABLE_QUALITY_FIXED)
        {
            cinfo.m_motionType = hkpMotion::MOTION_FIXED;
            cinfo.m_mass = 0.0f;
        }
        else if ( (cinfo.m_qualityType == HK_COLLIDABLE_QUALITY_KEYFRAMED) || (cinfo.m_qualityType == HK_COLLIDABLE_QUALITY_KEYFRAMED_REPORTING) )
        {
            cinfo.m_motionType = hkpMotion::MOTION_KEYFRAMED;
        }
        else
        {
            // Moving quality type - but if mass was set to zero, set it to fixed and warn
            if (cinfo.m_mass == 0.0f)
            {
                HK_WARN_ALWAYS(0xabba91ea, "Quality specified for rigid body "<<node->m_name<<" is \""<<qualityTypeStr<<"\" but mass is 0.0. Changed to \"Fixed\" quality.");
                cinfo.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;
                cinfo.m_motionType = hkpMotion::MOTION_FIXED;
            }
        }
    }
    else // the user didn't specify a quality, so we base ourselves in the mass only
    {
        if (cinfo.m_mass== 0.0f)
        {
            cinfo.m_motionType = hkpMotion::MOTION_FIXED;
            cinfo.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;
        }
        else
        {
            cinfo.m_motionType = hkpMotion::MOTION_DYNAMIC;
            cinfo.m_qualityType = HK_COLLIDABLE_QUALITY_MOVING;
        }
    }

    hkpRigidBody* rb = new hkpRigidBody(cinfo);
    cinfo.m_shape->removeReference();

    // EXP-501
    // Look for and add any hkpProperty nodes
    makeProperties(node, theScene, rb);

    // add local frames
    rb->m_localFrame.setAndDontIncrementRefCount( hctLocalFrameUtils::createLocalFrames(node, theScene, true, HK_NULL, groupsCreated ) );

    return rb;
}

hkpShapeInfo* hctCreateRigidBodiesFilter::makeShape(const hkxNode* node, const hkxScene* theScene, const hkMatrix4& currentWorldTransform, const hkTransform& lastRbTransform )
{
    // given our list of shapes, convert to
    // a single shape pointer
    hkMatrixDecomposition::Decomposition decomposition;
    hkMatrixDecomposition::decomposeMatrix(currentWorldTransform, decomposition);
    const hkTransform fullShapeTransform (decomposition.m_rotation, decomposition.m_translation);

    // recurse down to make shapes.
    hkArray<ShapeInfo> shapes;
    hkBool isDynamic = true;
    hkBool collecthkdShapes = false;
    makeShapes(node, theScene, currentWorldTransform, fullShapeTransform, false, isDynamic, shapes);
    if ( shapes.isEmpty() )
    {
        collecthkdShapes = true;
        makeShapes(node, theScene, currentWorldTransform, fullShapeTransform, true, isDynamic, shapes);
    }
    if (shapes.getSize() == 0)
    {
        HK_WARN_ALWAYS(0xabbaff31, "'" << node->m_name << "' - No shapes found. Cannot create shape for Breakable Shape.");
        return HK_NULL;
    }

    hkpShape* shape = gatherShapes(node, shapes, fullShapeTransform);

    hkpShapeInfo* si = new hkpShapeInfo();
    si->m_shape = shape;
    si->m_transform.setMulInverseMul( lastRbTransform, fullShapeTransform );;
    si->m_hkdShapesCollected = collecthkdShapes;
    if ( shapes.getSize() > 1 && node->findAttributeGroupByName("hkShape") )    // if we gathered multiple children and our node has a shape, we have a hierarchical compound
    {
        si->m_isHierarchicalCompound = true;
    }
    for (int i =0; i < shapes.getSize(); i++ )
    {
        si->m_childShapeNames.pushBack( shapes[i].m_node->m_name );

        hkTransform relShapeTransform; relShapeTransform.setMulInverseMul( fullShapeTransform, shapes[i].m_decomposedWorldT );
        si->m_childTransforms.pushBack( relShapeTransform );
    }
    return si;
}


void hctCreateRigidBodiesFilter::fillSystem(const hkxNode* node, const hkxScene* theScene, const hkMatrix4& currentWorldTransform, const hkTransform& lastRbTransform, hkpPhysicsSystem* system, hkResourceContainer* resourceContainer, hkStringMap<hkLocalFrameGroup*>& groupsCreated, hkArray<UnresolvedBinaryAction>& unresolvedBinaryAction )
{
    // see if node has any attributes
    bool createdRB = false;
    hkBool oldCollapseShapeOffsetsIntoShapes = m_options.m_collapseShapeOffsetsIntoShapes;
    const hkxAttributeGroup* breakableShapeAttributeGroup = node->findAttributeGroupByName("hkdShape");
    const hkxAttributeGroup* breakableBodyAttributeGroup = node->findAttributeGroupByName("hkdBody");
    if ( breakableShapeAttributeGroup || breakableBodyAttributeGroup )
    {
        m_options.m_collapseShapeOffsetsIntoShapes = false;
    }

    hkTransform rbTransform = lastRbTransform;
    {
        const hkxAttributeGroup* group = node->findAttributeGroupByName("hkRigidBody");

        if ( group && group->m_attributes.getSize()  )
        {
            //have a rigid body!
            hkArray<ShapeInfo> shapeInfos;
            bool hkdShapesCollected = false;
            hkpRigidBody* rb = makeRigidBody(node, theScene, *group, currentWorldTransform, shapeInfos, hkdShapesCollected, groupsCreated);
            if (rb)
            {
                rb->setName( node->m_name ); // should we copy it?
                system->addRigidBody(rb);
                rb->removeReference();
                rbTransform = rb->getTransform();
                createdRB = true;

                if ( resourceContainer )
                {
                    hkResourceContainer* c = resourceContainer->createContainer( node->m_name );
                    /*hkResourceHandle* resourceHandle = */

                    //
                    // add rigid body and shape info
                    //
                    {
                        c->createResource( group->m_name, rb );

                        hkpShapeInfo* si = new hkpShapeInfo();
                        si->m_shape = rb->getCollidable()->getShape();
                        si->m_transform = rbTransform;
                        si->m_hkdShapesCollected = hkdShapesCollected;
                        for (int i =0; i < shapeInfos.getSize(); i++ )
                        {
                            si->m_childShapeNames.pushBack( shapeInfos[i].m_node->m_name );

                            hkTransform relShapeTransform; relShapeTransform.setMulInverseMul( rbTransform, shapeInfos[i].m_decomposedWorldT );
                            si->m_childTransforms.pushBack( relShapeTransform );
                        }

                        if ( shapeInfos.getSize()>1 &&  node->findAttributeGroupByName("hkShape") ) // if we gathered multiple children and our node has a shape, we have a hierarchical compound
                        {
                            si->m_isHierarchicalCompound = true;
                        }

                        c->createResource( hkReflect::getName<hkpShapeInfo>(), si );
                        si->removeReference();
                    }

                    //
                    // add action
                    //
                    {
                        for ( hkResourceHandle* actionH = c->findResourceByName<hkpAction>(HK_NULL, HK_NULL); actionH; actionH = resourceContainer->findResourceByName<hkpAction>(HK_NULL, actionH ) )
                        {
                            if ( hkpUnaryAction* ua = actionH->hasA<hkpUnaryAction>() )
                            {
                                ua->setEntity( rb );
                                actionH->removeExternalLink( "entity" );
                                system->addAction( ua );
                            }
                            else if ( hkpBinaryAction* ba = actionH->hasA<hkpBinaryAction>() )
                            {
                                ba->setEntityA( rb );
                                actionH->removeExternalLink( "entityA" );

                                hkInplaceArray<hkResourceHandle::Link,2> links;
                                actionH->getExternalLinks( links );
                                for ( int i =0; i < links.getSize(); i++ )
                                {
                                    hkResourceHandle::Link& link = links[i];
                                    if ( hkString::strCasecmp(link.m_memberName, "entityB") == 0 )
                                    {
                                        actionH->removeExternalLink( "entityB" );
                                        UnresolvedBinaryAction& uaNew = unresolvedBinaryAction.expandOne();
                                        uaNew.m_action = ba;
                                        uaNew.m_bodyNameForEntityB = link.m_externalId;
                                    }
                                }
                            }
                            else
                            {
                                HK_WARN_ALWAYS( 0xabba45fe, "Unknown action of type '" << actionH->getObjectType()->getName() << "'" );
                            }
                        }
                    }

                }
            }
        }
    }

    if ( !createdRB && resourceContainer )
    {
        if ( breakableShapeAttributeGroup && breakableShapeAttributeGroup->m_attributes.getSize()  )
        {
            //have a child breakable shape!
            hkpShapeInfo* si = makeShape(node, theScene, currentWorldTransform, rbTransform);
            if (si)
            {
                hkResourceContainer* c = resourceContainer->createContainer( node->m_name );
                c->createResource("hkpShapeInfo", si);
                si->removeReference();
            }

            // set a new root
            hkMatrixDecomposition::Decomposition decomposition;
            hkMatrixDecomposition::decomposeMatrix(currentWorldTransform, decomposition);
            rbTransform  = hkTransform(decomposition.m_rotation, decomposition.m_translation);
        }
    }

    //if (!createdRB) // look further down for other rbs?
    //XX Quick test: At the moment we will always search the whole tree for rbs
    {
        hkMatrix4 currentChildTransform;
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            if ( node->m_children[c]->m_keyFrames.getSize() > 0 )
            {
                currentChildTransform.setMul( currentWorldTransform, node->m_children[c]->m_keyFrames[0] );
            }
            else
            {
                currentChildTransform = currentWorldTransform;
            }
            hkResourceContainer* childContainer = resourceContainer;
            if ( resourceContainer && (theScene->m_rootNode != node) )
            {
                childContainer = resourceContainer->createContainer( node->m_name );
            }
            fillSystem(node->m_children[c], theScene, currentChildTransform, rbTransform, system, childContainer, groupsCreated, unresolvedBinaryAction );
        }
    }
    m_options.m_collapseShapeOffsetsIntoShapes = oldCollapseShapeOffsetsIntoShapes;
}

void hctCreateRigidBodiesFilter::process( hkRootLevelContainer& data  )
{

    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    hkpPhysicsData* physicsPtr = data.findObject<hkpPhysicsData>();

    hkResourceContainer* resourceContainer = HK_NULL;
    if ( m_options.m_exportDestructionInformation )
    {
        resourceContainer = reinterpret_cast<hkResourceContainer*>( data.findObjectByName( "Resource Data" ) );
    }

    if (scenePtr == HK_NULL || (scenePtr->m_rootNode == HK_NULL) )
    {
        HK_WARN_ALWAYS(0xabbac3c0, "No scene data (or scene data root hkxNode) found. Can't continue.");
        return;
    }

    bool havePhysicsData = physicsPtr != HK_NULL;
    if (havePhysicsData && (physicsPtr->getPhysicsSystems().getSize() > 0))
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "Some physics data found already, have you already created rigid bodies?");
    }

    {
        if (!physicsPtr)
        {
            physicsPtr = new hkpPhysicsData();
        }
        else
        {
            physicsPtr->addReference();
        }

        hkpPhysicsSystem* system = new hkpPhysicsSystem(); // tracked new

        // search the scene data for nodes with attribs
        // for each one with attribs, see if it is a rb set of attribs
        // convert to rb based on those attribs
        hkStringMap<hkLocalFrameGroup*> groupsCreated;
        hkArray<UnresolvedBinaryAction> unresolvedActions;
        fillSystem(scenePtr->m_rootNode, scenePtr, scenePtr->m_rootNode->m_keyFrames[0], hkTransform::getIdentity(), system, resourceContainer, groupsCreated, unresolvedActions );

        {
            const char* _name  = "Default Physics System";
            system->setName( _name );
        }

        int numBodies = system->getRigidBodies().getSize();
        if ( numBodies > 0)
        {
            // add it to the physics data
            physicsPtr->addPhysicsSystem(system);

            // resolve actions
            {
                for (int i = 0; i < unresolvedActions.getSize(); i++)
                {
                    UnresolvedBinaryAction& ua = unresolvedActions[i];
                    hkpRigidBody* body = physicsPtr->findRigidBodyByName( ua.m_bodyNameForEntityB );
                    if ( body )
                    {
                        ua.m_action->setEntityB( body );
                    }
                    system->addAction( ua.m_action );
                }
            }


            // add the physics data to the world
            if (!havePhysicsData)
            {
                physicsPtr->setWorldCinfo(HK_NULL); // none / default
                data.m_namedVariants.expandOne().set("Physics Data", physicsPtr ); // virtual
            }
        }

        // Report
        if( numBodies == 0 )
        {
            Log_Warning( "No rigid body created." );
        }
        else
        {
            if( !m_options.m_wrapWithMopp )
            {
                Log_Info( "Note: MOPP Code was not requested for the rigid bodies." );
            }

            Log_Info( "Created {} rigid bodies.", numBodies );
        }

        physicsPtr->removeReference();
        system->removeReference();
    }

    {
        hkpWeldingUtility::WeldingType weldType = hkpWeldingUtility::WELDING_TYPE_NONE;
        if (m_moppShapes.getSize() > 0 )
        {
            Log_Info( "Computing Welding Information Across all Shapes..." );

            switch (m_options.m_landscapeWelding)
            {
            case hctCreateRigidBodiesOptions::WELD_COUNTERCLOCKWISE:
                weldType = hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE;
                break;

            case hctCreateRigidBodiesOptions::WELD_CLOCKWISE:
                weldType = hkpWeldingUtility::WELDING_TYPE_CLOCKWISE;
                break;

            case hctCreateRigidBodiesOptions::WELD_TWO_SIDED:
                weldType = hkpWeldingUtility::WELDING_TYPE_TWO_SIDED;
                break;

            case hctCreateRigidBodiesOptions::WELD_NONE:
                weldType = hkpWeldingUtility::WELDING_TYPE_NONE;
                break;

            default:
                HK_WARN_ALWAYS(0xabbabef3, "Unknown Welding Type");
                break;
            }

            if ( weldType == hkpWeldingUtility::WELDING_TYPE_TWO_SIDED )
            {
                HK_WARN_ALWAYS(0xabbacafe, "Welding type is set to 'TWOSIDED' : This feature is deprecated - please use one sided welding instead.");
            }
        }

        for ( int i = 0; i < m_moppShapes.getSize(); ++i )
        {
            const hkpBvTreeShape* shape = m_moppShapes[ i ].m_shape;
            if ( shape->getType() == hkcdShapeType::MOPP )
            {
                const hkpMoppBvTreeShape* mopp = static_cast< const hkpMoppBvTreeShape* >( shape );
                hkpShapeCollection* collection = (hkpShapeCollection*)mopp->getShapeCollection();


                hkpMeshWeldingUtility::computeWeldingInfoMultiShape( m_moppShapes[ i ].m_transform, collection, weldType, m_moppShapes, m_options.m_weldOpenEdges, m_options.m_markEdgesBadWinding );
            }
        }

        for ( int i = 0; i < m_moppShapes.getSize(); ++i )
        {
            m_moppShapes[ i ].m_shape->removeReference();
        }

        m_moppShapes.clearAndDeallocate();
    }

    // Enable warnings again so that they will be displayed on the next pipeline run.
    hkError::getInstance().setEnabled(0xabbabfd1, true);
    hkError::getInstance().setEnabled(0xabbabfd0, true);
}

void hctCreateRigidBodiesFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctCreateRigidBodiesOptions* options = hctFilterUtils::getNativeOptions<hctCreateRigidBodiesOptions>(optVar))
    {
        m_options = *options;
        hctFilterUtils::emptyIfNull(m_options.m_materialPrefix);
        delete options;
    }
}

void hctCreateRigidBodiesFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
