// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateConstraints/hctCreateConstraintsFilter.h>

#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>

#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics/Constraint/Data/Hinge/hkpHingeConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Constraint/Data/Prismatic/hkpPrismaticConstraintData.h>
#include <Physics/Constraint/Data/StiffSpring/hkpStiffSpringConstraintData.h>
#include <Physics/Constraint/Data/Wheel/hkpWheelConstraintData.h>
#include <Physics/Constraint/Data/Fixed/hkpFixedConstraintData.h>
#include <Physics/Constraint/Data/PointToPlane/hkpPointToPlaneConstraintData.h>

#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>
#include <Physics/Constraint/Motor/SpringDamper/hkpSpringDamperConstraintMotor.h>
#include <Physics/Constraint/Motor/Velocity/hkpVelocityConstraintMotor.h>

#include <Physics2012/Dynamics/Constraint/hkpConstraintInstance.h>
#include <Physics2012/Dynamics/Constraint/Breakable/hkpBreakableConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.p12.createconstraints"
#include <Common/Base/System/Log/hkLog.hxx>


hctCreateConstraintsFilterDesc g_createConstraintsDesc;

hctCreateConstraintsFilter::hctCreateConstraintsFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner)
,   m_optionsDialog(NULL)
{
    m_options.m_enableAutoInertiaStabilization      = true;
    m_options.m_stabilizationAmount                 = 1.0f;
    m_options.m_constraintsSolvingMethod            = hkpConstraintAtom::METHOD_STABILIZED;
}

hctCreateConstraintsFilter::~hctCreateConstraintsFilter()
{

}

/*
** ENUMS : We translate strings into this enums so we can switch, etc..
*/

// Constraint Type
enum KnownConstraintTypes
{
    CT_UNKNOWN = -1,
    CT_HINGE = 0, CT_BALL_AND_SOCKET, CT_PRISMATIC, CT_RAGDOLL, CT_STIFF_SPRING, CT_WHEEL, CT_FIXED, CT_POINT_TO_PLANE,
};

hctFilterUtils::EnumItem constraintTypeItems[] =
{
    { CT_HINGE,"hkHingeConstraint" },
    { CT_BALL_AND_SOCKET, "hkBallAndSocketConstraint" },
    { CT_PRISMATIC, "hkPrismaticConstraint" },
    { CT_RAGDOLL, "hkRagdollConstraint" },
    { CT_STIFF_SPRING, "hkStiffSpringConstraint" },
    { CT_WHEEL, "hkWheelConstraint" },
    { CT_FIXED, "hkFixedConstraint" },
    { CT_POINT_TO_PLANE, "hkPointToPlaneConstraint" }
};

hctFilterUtils::Enum g_constraintTypeEnum(hkArrayViewT::make(constraintTypeItems));

hkpConstraintInstance* hctCreateConstraintsFilter::makeConstraint(hkpPhysicsData* thePhysics, const hkxScene* theScene, hkxNode* node, int groupIndex, const hkMatrix4& currentWorldTransform)
{
    hkxAttributeGroup& group = node->m_attributeGroups[groupIndex];

    int constraintType = hctFilterUtils::getValueOfName(g_constraintTypeEnum, group.m_name, CT_UNKNOWN);

    if (constraintType == CT_UNKNOWN)
    {
        return HK_NULL;
    }

    // Start by retrieving the names of the parent and child
    const char* parentName = HK_NULL;
    group.getStringValue("parent", false, parentName);
    const char* childName = HK_NULL;
    group.getStringValue("child", false, childName);

    // Blank names taken as no name
    if( parentName && (hkString::strLen( parentName ) == 0) )
    {
        parentName = HK_NULL;
    }
    if( childName && (hkString::strLen( childName ) == 0) )
    {
        childName = HK_NULL;
    }

    const hkxNode* parentNode = HK_NULL;
    const hkxNode* childNode = HK_NULL;
    hkpRigidBody* parentBody = HK_NULL;
    hkpRigidBody* childBody = HK_NULL;

    if (parentName)
    {
        parentBody = thePhysics->findRigidBodyByName (parentName);
        parentNode = theScene->findNodeByName (parentName);
    }

    if (childName)
    {
        childBody = thePhysics->findRigidBodyByName (childName);
        childNode = theScene->findNodeByName (childName);
    }
    else // if no child, then assume that the child is 'this'
    {
        childBody = thePhysics->findRigidBodyByName(node->m_name);
        childNode = node;
    }

    if (!childBody || !childNode)
    {
        HK_WARN_ALWAYS(0xabba654a, "No child rigid body / node found for constraint in node " << node->m_name );
        return HK_NULL;
    }

    if (parentName && (!parentNode || !parentBody))
    {
        HK_WARN_ALWAYS(0xabbacf53, "Couldn't find parent rigid body / node "<<parentName<<" for constraint in node "<<node->m_name);
        return HK_NULL;
    }

    const bool parentIsWorld = !parentName;

    // Construct the spaces in world
    // Note: we use the "t" prefix for hkTransforms and "m" for hkMatrix4, as we need to convert to/from often enough
    hkTransform tWorldFromParentSpace;
    hkTransform tWorldFromChildSpace;
    {
        // Node transforms
        hkMatrix4 mWorldFromChildNode;
        theScene->getWorldFromNodeTransform(childNode, mWorldFromChildNode);
        hkMatrix4 mWorldFromParentNode;
        if (parentIsWorld)
        {
            mWorldFromParentNode.setIdentity();
        }
        else
        {
            theScene->getWorldFromNodeTransform(parentNode, mWorldFromParentNode);
        }

        // Retrieve attributes
        // Start with defaults
        hkQuaternion childSpaceRotation; childSpaceRotation.setIdentity();
        hkVector4 childSpaceTranslation; childSpaceTranslation.setZero4();
        hkQuaternion parentSpaceRotation; parentSpaceRotation.setIdentity();
        hkVector4 parentSpaceTranslation; parentSpaceTranslation.setZero4();
        hkBool parentSpaceRotationLocked = true;
        hkBool parentSpaceTranslationLocked = true;

        // Ball-Socket an StiffSpring shouldn't warn if it doesn't find rotation attributes
        hkBool needsRotation = ((constraintType != CT_BALL_AND_SOCKET) && (constraintType!=CT_STIFF_SPRING));

        group.getQuaternionValue ("childSpaceRotation", needsRotation, childSpaceRotation);
        group.getVectorValue ("childSpaceTranslation", true, childSpaceTranslation);
        group.getQuaternionValue ("parentSpaceRotation", needsRotation, parentSpaceRotation);
        group.getVectorValue ("parentSpaceTranslation", true, parentSpaceTranslation);
        group.getBoolValue ("parentSpaceRotationLocked", needsRotation, parentSpaceRotationLocked);
        group.getBoolValue ("parentSpaceTranslationLocked", true, parentSpaceTranslationLocked);

        // Now, build the spaces
        // Child Space : the easy one, always relative to child node
        {
            // Convert into matrix4, using temporary hkTransform
            hkMatrix4 mChildNodeFromChildSpace;
            {
                hkTransform tChildNodeFromChildSpace(childSpaceRotation, childSpaceTranslation);
                tChildNodeFromChildSpace.get4x4ColumnMajor((hkReal*)&mChildNodeFromChildSpace);
            }

            hkMatrix4 mWorldFromChildSpaceTemp;

            mWorldFromChildSpaceTemp.setMul(mWorldFromChildNode, mChildNodeFromChildSpace);

            hkMatrixDecomposition::Decomposition decomposition;
            hkMatrixDecomposition::decomposeMatrix(mWorldFromChildSpaceTemp, decomposition);

            tWorldFromChildSpace.set(decomposition.m_rotation, decomposition.m_translation);

        }

        // Parent Space : more complex
        {
            // Rotation component (we'll replace the translation)
            hkQuaternion worldFromParentSpace_Rotation;
            {
                if (parentSpaceRotationLocked)
                {
                    hkQuaternion worldFromChildSpace_Rotation(tWorldFromChildSpace.getRotation());

                    worldFromParentSpace_Rotation.setMul(worldFromChildSpace_Rotation, parentSpaceRotation);
                }
                else
                {
                    // Unlocked : relative to parent node or world
                    hkMatrix4 mParentNodeFromParentSpace_Rotation;
                    {
                        hkTransform tParentNodeFromParentSpace_Rotation (parentSpaceRotation, hkVector4::getZero());
                        tParentNodeFromParentSpace_Rotation.get4x4ColumnMajor((hkReal*) &mParentNodeFromParentSpace_Rotation);
                    }

                    hkMatrix4 mWorldFromParentSpaceTemp;
                    mWorldFromParentSpaceTemp.setMul(mWorldFromParentNode, mParentNodeFromParentSpace_Rotation);

                    hkMatrixDecomposition::Decomposition decomposition;
                    hkMatrixDecomposition::decomposeMatrix(mWorldFromParentSpaceTemp, decomposition);

                    worldFromParentSpace_Rotation = decomposition.m_rotation;

                }
            }

            // Translation component
            hkVector4 worldFromParentSpace_Translation;
            {
                if (parentSpaceTranslationLocked)
                {
                    worldFromParentSpace_Translation.setTransformedPos(tWorldFromChildSpace, parentSpaceTranslation);
                }
                else
                {
                    hkTransform tWorldFromParentNode;
                    tWorldFromParentNode.set4x4ColumnMajor((hkReal*) &mWorldFromParentNode);

                    worldFromParentSpace_Translation.setTransformedPos(tWorldFromParentNode, parentSpaceTranslation);
                }
            }

            tWorldFromParentSpace.set(worldFromParentSpace_Rotation, worldFromParentSpace_Translation);
        }
    }

    // Now we calculate where those spaces are relative to the rigid body transforms. Scale/skew is removed.
    hkTransform tChildBodyFromChildSpace;
    hkTransform tParentBodyFromParentSpace;
    {

        // Child
        {
            const hkTransform& tWorldFromChildBody = childBody->getTransform();

            tChildBodyFromChildSpace.setMulInverseMul(tWorldFromChildBody, tWorldFromChildSpace);
        }

        // Parent
        {
            const hkTransform& tWorldFromParentBody = parentIsWorld ? hkTransform::getIdentity() : parentBody->getTransform();

            tParentBodyFromParentSpace.setMulInverseMul(tWorldFromParentBody, tWorldFromParentSpace);
        }
    }

    // Also, calculate the transformed X,Y,Z basis based on the transform applied to the scene
    hkRotation transformedBasis;
    {
        hkMatrixDecomposition::Decomposition decomp;
        hkMatrixDecomposition::decomposeMatrix (theScene->m_appliedTransform, decomp);

        transformedBasis = decomp.m_basis;
    }

    // Put all the common data in a struct
    ConstraintSpaceData commonData;
    commonData.m_transformedBasis = transformedBasis;
    commonData.m_childSpace = tChildBodyFromChildSpace;
    commonData.m_parentSpace = tParentBodyFromParentSpace;
    commonData.m_childRBTransform = childBody->getTransform();
    commonData.m_parentRBTransform = parentIsWorld ? hkTransform::getIdentity() : parentBody->getTransform();

    hkpConstraintData* constraintData = HK_NULL;
    switch (constraintType)
    {
        case CT_BALL_AND_SOCKET:
            {
                constraintData = createBallAndSocketConstraint (commonData, group );
                break;
            }
        case CT_HINGE:
            {
                constraintData = createHingeConstraint (commonData, group );
                break;
            }
        case CT_RAGDOLL:
            {
                constraintData = createRagdollConstraint (commonData, group );
                break;
            }
        case CT_PRISMATIC:
            {
                constraintData = createPrismaticConstraint (commonData, group );
                break;
            }
        case CT_STIFF_SPRING:
            {

                constraintData = createStiffSpringConstraint (commonData, group);
                break;
            }
        case CT_WHEEL:
            {
                constraintData = createWheelConstraint (commonData, group);
                break;
            }
        case CT_FIXED:
            {
                constraintData = createFixedConstraint(commonData, group);
                break;
            }
        default:
            {
                HK_WARN_ALWAYS (0xabba66b3, "Unsupported Constraint Type "<< group.m_name);
                break;
            }
    }

    if (!constraintData) return HK_NULL;

    hkBool isBreakable = false;
    group.getBoolValue("isBreakable", false, isBreakable);
    if (isBreakable)
    {
        float threshold = 10.0f;
        group.getFloatValue("breakThreshold", true, threshold);

        hkpBreakableConstraintData* breakable = new hkpBreakableConstraintData(constraintData);
        constraintData->removeReference();
        breakable->setThreshold(threshold);

        constraintData = breakable;
    }

    hkpConstraintInstance* constraintInstance = new hkpConstraintInstance (childBody, parentBody, constraintData);
    constraintInstance->setName(node->m_name);
    constraintData->removeReference();

    return constraintInstance;
}

// Motor Type
enum KnownMotorTypes
{
    MT_UNKNOWN = -1,
    MT_NONE = 0, MT_POSITION, MT_VELOCITY, MT_SPRING_DAMPER
};

hctFilterUtils::EnumItem motorTypeItems[] =
{
    { MT_NONE, "None" },
    { MT_POSITION, "Position" },
    { MT_VELOCITY, "Velocity" },
    { MT_SPRING_DAMPER, "Spring Damper" }
};
hctFilterUtils::Enum g_motorTypeEnum(hkArrayViewT::make(motorTypeItems));

hkpConstraintMotor* hctCreateConstraintsFilter::createMotor (const hkxAttributeGroup& attribs)
{
    int motorType = MT_NONE;

    const char* motorTypeStr = HK_NULL;
    attribs.getStringValue("motorType", true, motorTypeStr);
    if (motorTypeStr != HK_NULL)
    {
        motorType = hctFilterUtils::getValueOfName(g_motorTypeEnum, motorTypeStr, MT_UNKNOWN);
    }

    if (motorType==MT_UNKNOWN)
    {
        HK_WARN_ALWAYS (0xabbabb82, "Unknown motor type. Ignoring");
        motorType=MT_NONE;
    }

    hkpConstraintMotor* theMotor = HK_NULL;

    switch (motorType)
    {
        case MT_POSITION:
            {
                theMotor = new hkpPositionConstraintMotor();
                break;
            }
        case MT_SPRING_DAMPER:
            {
                theMotor = new hkpSpringDamperConstraintMotor();
                break;
            }
        case MT_VELOCITY:
            {
                theMotor = new hkpVelocityConstraintMotor();
                break;
            }
        default:
            // leave as NULL
            break;
    }

    return theMotor;
}

hkpConstraintData* hctCreateConstraintsFilter::createBallAndSocketConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    hkpBallAndSocketConstraintData* bsdata = new hkpBallAndSocketConstraintData;

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();
    bsdata->setInBodySpace( pivotChild, pivotParent );

    return bsdata;
}

hkpConstraintData* hctCreateConstraintsFilter::createHingeConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    hkBool isLimited = false;
    attribs.getBoolValue("isLimited", false, isLimited);


    const hkVector4& xAxis = commonData.m_transformedBasis.getColumn(0);
    const hkVector4& zAxis = commonData.m_transformedBasis.getColumn(2);

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();
    hkVector4 axisChild; axisChild.setRotatedDir(commonData.m_childSpace.getRotation(), zAxis);
    hkVector4 axisParent; axisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), zAxis);

    if (isLimited)
    {
        float limitMin = -HK_REAL_PI;
        attribs.getFloatValue("limitMin", true, limitMin);
        float limitMax = HK_REAL_PI;
        attribs.getFloatValue("limitMax", true, limitMax);

        // Make sure they are in the right order, otherwise flip them
        if (limitMin>limitMax)
        {
            hkAlgorithm::swap(limitMin, limitMax);
        }

        // friction
        float maxFrictionTorque = 0.0f;
        attribs.getFloatValue("maxFrictionTorque", false, maxFrictionTorque);

        // motor
        hkpConstraintMotor* theMotor = createMotor(attribs);

        hkpLimitedHingeConstraintData* lhdata = new hkpLimitedHingeConstraintData;
        if (theMotor != HK_NULL)
        {
            lhdata->setMotor(theMotor);
            lhdata->setMotorEnabled(HK_NULL, false);
            lhdata->disableLimits();
            theMotor->removeReference();
        }

        hkVector4 perpChild; perpChild.setRotatedDir(commonData.m_childSpace.getRotation(), xAxis);
        hkVector4 perpParent; perpParent.setRotatedDir(commonData.m_parentSpace.getRotation(), xAxis);

        lhdata->setInBodySpace(pivotChild, pivotParent, axisChild, axisParent, perpChild, perpParent);

        lhdata->setMinAngularLimit(limitMin);
        lhdata->setMaxAngularLimit(limitMax);
        lhdata->setMaxFrictionTorque(maxFrictionTorque);

        return lhdata;
    }
    else
    {
        hkpHingeConstraintData* hdata = new hkpHingeConstraintData;

        hdata->setInBodySpace(pivotChild, pivotParent, axisChild, axisParent);

        return hdata;
    }
}

hkpConstraintData* hctCreateConstraintsFilter::createRagdollConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    const hkVector4& xAxis = commonData.m_transformedBasis.getColumn(0);
    const hkVector4& yAxis = commonData.m_transformedBasis.getColumn(1);

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();
    hkVector4 planeAxisChild; planeAxisChild.setRotatedDir(commonData.m_childSpace.getRotation(), yAxis);
    hkVector4 planeAxisParent; planeAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), yAxis);
    hkVector4 twistAxisChild; twistAxisChild.setRotatedDir(commonData.m_childSpace.getRotation(), xAxis);
    hkVector4 twistAxisParent; twistAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), xAxis);

    const float piOverTwo = HK_REAL_PI*0.5f;

    // Read cone angle. Make sure it is positive and within the right ranges
    float coneAngle = piOverTwo;
    attribs.getFloatValue("coneAngle", true, coneAngle);

    coneAngle = hkMath::fabs(coneAngle);
    hkMath::clamp( coneAngle, 0.0f, HK_REAL_PI );

    // Read plane angles.
    float planeAngleMin = -piOverTwo;
    attribs.getFloatValue("planeAngleMin", true, planeAngleMin);

    float planeAngleMax = piOverTwo;
    attribs.getFloatValue("planeAngleMax", true, planeAngleMax);

    // Max sure the signs are right - the perp Axis (on which these angles are specified) is not
    // transformed by the basis (since it is not passed to the constructor of the constraint), so although the
    // "transform scene" filter may have flipped these angles, we need to restore them.
    planeAngleMin = -hkMath::fabs(planeAngleMin);
    planeAngleMax = hkMath::fabs(planeAngleMax);

    // Read twist values. Make sure they are in the right order, otherwise flip them
    float twistMin = -HK_REAL_PI;
    attribs.getFloatValue("twistMin", true, twistMin);
    float twistMax = HK_REAL_PI;
    attribs.getFloatValue("twistMax", true, twistMax);
    if (twistMin>twistMax)
    {
        hkAlgorithm::swap(twistMin, twistMax);
    }

    // Max friction torque
    float maxFrictionTorque = 0.0f;
    attribs.getFloatValue("maxFrictionTorque", false, maxFrictionTorque);

    // motor
    hkpConstraintMotor* theMotor = createMotor(attribs);

    hkpRagdollConstraintData* data = new hkpRagdollConstraintData;
    if (theMotor != HK_NULL)
    {
        data->setConeMotor(theMotor);
        data->setPlaneMotor(theMotor);
        data->setTwistMotor(theMotor);
        data->setMotorsEnabled(HK_NULL, false);
        theMotor->removeReference();
    }

    data->setInBodySpace(pivotChild, pivotParent, planeAxisChild, planeAxisParent, twistAxisChild, twistAxisParent);
    data->setConeAngularLimit(coneAngle);
    data->setPlaneMinAngularLimit(planeAngleMin);
    data->setPlaneMaxAngularLimit(planeAngleMax);
    data->setTwistMinAngularLimit(twistMin);
    data->setTwistMaxAngularLimit(twistMax);
    data->setMaxFrictionTorque(maxFrictionTorque);

    return data;
}

hkpConstraintData* hctCreateConstraintsFilter::createPrismaticConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    const hkVector4& xAxis = commonData.m_transformedBasis.getColumn(0);
    const hkVector4& zAxis = commonData.m_transformedBasis.getColumn(2);

    float minLinearLimit = -HK_REAL_MAX;
    attribs.getFloatValue("minLinearLimit", false, minLinearLimit);
    float maxLinearLimit = HK_REAL_MAX;
    attribs.getFloatValue("maxLinearLimit", false, maxLinearLimit);

    // Make sure they are in the right order, otherwise flip them
    if (minLinearLimit>maxLinearLimit)
    {
        hkAlgorithm::swap(minLinearLimit, maxLinearLimit);
    }

    // friction
    float maxFrictionForce = 0.0f;
    attribs.getFloatValue("maxFrictionForce", false, maxFrictionForce);

    hkpPrismaticConstraintData* prismaticConstraint = new hkpPrismaticConstraintData;
    // motor
    hkpConstraintMotor* theMotor = createMotor(attribs);

    if (theMotor != HK_NULL)
    {
        prismaticConstraint->setMotor(theMotor);
        prismaticConstraint->setMotorEnabled(HK_NULL, true);
        theMotor->removeReference();
    }

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();
    hkVector4 mainAxisChild; mainAxisChild.setRotatedDir(commonData.m_childSpace.getRotation(), zAxis);
    hkVector4 mainAxisParent; mainAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), zAxis);
    hkVector4 perpAxisChild; perpAxisChild.setRotatedDir(commonData.m_childSpace.getRotation(), xAxis);
    hkVector4 perpAxisParent; perpAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), xAxis);

    prismaticConstraint->setInBodySpace(pivotChild, pivotParent,
                                        mainAxisChild, mainAxisParent,
                                        perpAxisChild, perpAxisParent);

    prismaticConstraint->setMinLinearLimit(minLinearLimit);
    prismaticConstraint->setMaxLinearLimit(maxLinearLimit);
    prismaticConstraint->setMaxFrictionForce(maxFrictionForce);

    return prismaticConstraint;
}

hkpConstraintData* hctCreateConstraintsFilter::createStiffSpringConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    hkpStiffSpringConstraintData* ssdata = new hkpStiffSpringConstraintData;

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();

    // If the user doesn't specify the rest length, we assume use the current distance between pivots
    float restLength = 0.0f;
    const bool restLengthSpecified = attribs.getFloatValue("restLength", false, restLength).isSuccess();
    if (!restLengthSpecified)
    {
        hkVector4 pivotChildWS; pivotChildWS.setTransformedPos(commonData.m_childRBTransform, pivotChild);
        hkVector4 pivotParentWS; pivotParentWS.setTransformedPos(commonData.m_parentRBTransform, pivotParent);
        hkVector4 diff; diff.setSub4(pivotChildWS, pivotParentWS);
        restLength = diff.length3();
    }

    ssdata->setInBodySpace( pivotChild, pivotParent, restLength );

    return ssdata;
}

hkpConstraintData* hctCreateConstraintsFilter::createWheelConstraint (const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    hkpWheelConstraintData* wdata = new hkpWheelConstraintData;

    const hkVector4& pivotChild = commonData.m_childSpace.getTranslation();
    const hkVector4& pivotParent = commonData.m_parentSpace.getTranslation();

    const hkVector4& xAxis = commonData.m_transformedBasis.getColumn(0);
    const hkVector4& zAxis = commonData.m_transformedBasis.getColumn(2);

    hkVector4 axleChild; axleChild.setRotatedDir(commonData.m_childSpace.getRotation(), xAxis);
    hkVector4 axleParent; axleParent.setRotatedDir(commonData.m_parentSpace.getRotation(), xAxis);
    hkVector4 steeringAxisParent; steeringAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), zAxis);

    float suspensionOffset =0.0f;
    attribs.getFloatValue("suspensionOffset", false, suspensionOffset);

    hkRotation offsetRot; offsetRot.setAxisAngle(xAxis, suspensionOffset);
    hkVector4 suspensionAxis; suspensionAxis.setRotatedDir(offsetRot, zAxis);
    hkVector4 suspensionAxisParent; suspensionAxisParent.setRotatedDir(commonData.m_parentSpace.getRotation(), suspensionAxis);

    wdata->setInBodySpace(pivotChild, pivotParent, axleChild, axleParent, suspensionAxisParent, steeringAxisParent);

    float minLimit;
    if (attribs.getFloatValue("suspensionLimitMin", false, minLimit).isSuccess())
    {
        wdata->setSuspensionMinLimit(minLimit);
    }

    float maxLimit;
    if (attribs.getFloatValue("suspensionLimitMax", false, maxLimit).isSuccess())
    {
        wdata->setSuspensionMaxLimit(maxLimit);
    }

    float strength;
    if (attribs.getFloatValue("suspensionStrength", false, strength).isSuccess())
    {
        wdata->setSuspensionStrength(strength);
    }

    float damping;
    if (attribs.getFloatValue("suspensionDamping", false, damping).isSuccess())
    {
        wdata->setSuspensionDamping(damping);
    }

    return wdata;

}

//
//  Creates a Fixed constraint from the given attributes

hkpConstraintData* hctCreateConstraintsFilter::createFixedConstraint(const ConstraintSpaceData& commonData, const hkxAttributeGroup& attribs)
{
    hkpFixedConstraintData* gcd = new hkpFixedConstraintData;

    // Set pivots
    gcd->setInBodySpace(commonData.m_childSpace, commonData.m_parentSpace);

    // Set strengths
    float maxLinImpulse = 0.0f;
    if (attribs.getFloatValue("maxLinearImpulse", false, maxLinImpulse).isSuccess() && maxLinImpulse != 0.f )
    {
        gcd->setMaximumLinearImpulse((hkReal)(maxLinImpulse));
    }
    float maxAngImpulse = 0.0f;
    if ( attribs.getFloatValue("maxAngularImpulse", false, maxAngImpulse).isSuccess() && maxAngImpulse != 0.f )
    {
        gcd->setMaximumAngularImpulse((hkReal)(maxAngImpulse));
    }

    return gcd;
}

void hctCreateConstraintsFilter::convertConstraints(hkpPhysicsData* thePhysics, const hkxScene* theScene, hkxNode* node, const hkMatrix4& currentWorldTransform, hkArray<hkpConstraintInstance*>& constraints )
{
    for (int gi=0; gi < node->m_attributeGroups.getSize(); ++gi)
    {
        const hkxAttributeGroup* group = &node->m_attributeGroups[gi];
        hkStringOld t(group->m_name);
        if (t.asLowerCase().endsWith("constraint"))
        {
            hkpConstraintInstance* c = makeConstraint(thePhysics, theScene, node, gi, currentWorldTransform);
            if (c)
            {
                constraints.pushBack(c);
            }
        }
    }

    hkMatrix4 currentChildTransform;
    for (int c=0; c < node->m_children.getSize(); ++c)
    {
        currentChildTransform.setMul( currentWorldTransform, node->m_children[c]->m_keyFrames[0] );
        convertConstraints(thePhysics, theScene, node->m_children[c], currentChildTransform, constraints);
    }
}

//
//  Stabilizes a physics system. Returns the number of stabilized rigid bodies

static int stabilizePhysicsSystem(hkpPhysicsSystem* physicsSystem, const hkReal stabilizationAmount)
{
    int numStabilized = 0;

    // For each rigid body, gather its attached constraints
    const hkArray<hkpRigidBody*>& bodies = physicsSystem->getRigidBodies();
    const hkArray<hkpConstraintInstance*>& constraints = physicsSystem->getConstraints();
    const int numBodies = bodies.getSize();
    const int numConstraints = constraints.getSize();
    for (int bi = 0; bi < numBodies; bi++)
    {
        //Get body
        hkpRigidBody* body = bodies[bi];

        // Add body constraints
        hkArray<hkpConstraintInstance*> bodyConstraints;
        const int numBodyConstraints = body->getNumConstraints();
        for (int ci = 0; ci < numBodyConstraints; ci++)
        {
            hkpConstraintInstance* cInstance = const_cast<hkpConstraintInstance*>(body->getConstraint(ci));
            if ( bodyConstraints.indexOf(cInstance) < 0 )
            {
                bodyConstraints.pushBack(cInstance);
            }
        }

        // Search for constraints not yet attached to the body
        for (int ci = 0; ci < numConstraints; ci++)
        {
            hkpConstraintInstance* cInstance = const_cast<hkpConstraintInstance*>(constraints[ci]);
            if ( (cInstance->getRigidBodyA() != body) &&
                 (cInstance->getRigidBodyB() != body) )
            {
                continue;   // Constraint not related to current body
            }

            // Constraint on current body, add if necessary
            if ( bodyConstraints.indexOf(cInstance) < 0 )
            {
                bodyConstraints.pushBack(cInstance);
            }
        }

        // We have the body with all its constraints, stabilize it!
        numStabilized += hkpConstraintStabilizationUtil::stabilizeRigidBodyInertia(body, bodyConstraints, stabilizationAmount);
    }

    // Return the number of stabilized constraints
    return numStabilized;
}

void hctCreateConstraintsFilter::process( hkRootLevelContainer& data  )
{
    // Find the scene data and physics data in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL || (scenePtr->m_rootNode == HK_NULL) )
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "No scene data (or scene data root hkxNode) found. Can't continue.");
        return;
    }
    hkpPhysicsData* physicsPtr = data.findObject<hkpPhysicsData>();
    if (physicsPtr == HK_NULL)
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "No physics data found, you need to use CreateRigidBodies before this filter or have bodies already in the input data.");
        return;
    }

    {
        // search the scene data for nodes with attribs
        // for each one with attribs, see if it has constraint attribs
        // convert to constraint based on those attribs
        hkArray<hkpConstraintInstance*> constraints;
        convertConstraints(physicsPtr, scenePtr, scenePtr->m_rootNode, scenePtr->m_rootNode->m_keyFrames[0], constraints );
        int numConstraints = constraints.getSize();

        // find out where to put the constraints
        // For now, just add to the system that holds the primary rigid body

        const hkArray<hkpPhysicsSystem*>& systems = physicsPtr->getPhysicsSystems();
        for (int c=0; c < numConstraints; ++c)
        {
            hkpConstraintInstance* ci = constraints[c];

            // check that this is not a constraint between two fixed rigid bodies, which is illegal.
            // If it is, generate a warning and ignore this constraint.
            if ( ci->getEntityA() && ci->getEntityB() )
            {
                hkpRigidBody* rbA = static_cast<hkpRigidBody*>( ci->getEntityA() );
                hkpRigidBody* rbB = static_cast<hkpRigidBody*>( ci->getEntityB() );

                if ( rbA->isFixed() && rbB->isFixed() )
                {
                    HK_WARN_ALWAYS(0xabbaa5f0, "Found a constraint between two fixed bodies, " << rbA->getName() << " and " << rbB->getName() << ", ignoring." );
                    continue;
                }
            }

            hkpRigidBody* matchRB = (hkpRigidBody*)( ci->getEntityA()? ci->getEntityA() : ci->getEntityB() );
            hkpPhysicsSystem* matchSys = HK_NULL;
            for (int pi=0; pi < systems.getSize(); ++pi)
            {
                hkpPhysicsSystem* sys = systems[pi];
                const hkArray<hkpRigidBody*>& rbs = sys->getRigidBodies();
                for (int ri=0; ri < rbs.getSize(); ++ri)
                {
                    if (rbs[ri] == matchRB)
                    {
                        matchSys = sys;
                        break;
                    }
                }

                if (matchSys )
                    break;
            }

            if (matchSys )
            {
                // This will resize a hkArray that either was from file
                // or should have been allocated with this same tracked mem.
                // Otherwise this will not work:
                matchSys->addConstraint(ci);
            }
            else
            {
                HK_WARN_ALWAYS(0xabba9d1d, "Created a constraint that does not seem to have a system!");
            }
        }

        // Report
        if( numConstraints == 0 )
        {
            HK_WARN_ALWAYS( 0xabba10b0, "No constraints created." );
        }
        else
        {
            Log_Info( "Created {} constraints.", numConstraints );
        }
    }

    const hkArray<hkpPhysicsSystem*>& systems = physicsPtr->getPhysicsSystems();

    // Scale inertia of all bodies
    if ( m_options.m_enableAutoInertiaStabilization && m_options.m_stabilizationAmount > 0.0f )
    {
        int numStabilized = 0;
        int numTotal = 0;
        for (int si = 0 ; si < systems.getSize(); si++)
        {
            hkpPhysicsSystem* physicsSystem = systems[si];
            numTotal += physicsSystem->getRigidBodies().getSize();
            numStabilized += stabilizePhysicsSystem(physicsSystem, m_options.m_stabilizationAmount);
        }
        Log_Info( "Stabilized inertia tensors for {} out of {} rigid bodies.", numStabilized, numTotal );
    }

    // Set solving method
    for (int si = 0 ; si < systems.getSize(); si++)
    {
        hkpPhysicsSystem* physicsSystem = systems[si];
        hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(physicsSystem, m_options.m_constraintsSolvingMethod);
    }
}

void hctCreateConstraintsFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctCreateConstraintsOptions* options = hctFilterUtils::getNativeOptions<hctCreateConstraintsOptions>(optVar))
    {
        m_options = *options;
        delete options;
    }
}

void hctCreateConstraintsFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
