// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateConstraints/hctCreateConstraintsFilter.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>

extern HINSTANCE hInstance;

//
//  Create Constraints filter dialog proc

INT_PTR CALLBACK hkFilterCreateConstraintsDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctCreateConstraintsFilter* filter = reinterpret_cast<hctCreateConstraintsFilter*> ( (hkUlong) GetWindowLongPtr(hWnd, GWLP_USERDATA)) ;

    switch(message)
    {
    case WM_INITDIALOG:
        {
            filter = (hctCreateConstraintsFilter*)lParam;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            hctCreateConstraintsOptions& filterOptions = filter->m_options;

            CheckDlgButton(hWnd, IDC_CB_AutoStabilization, filterOptions.m_enableAutoInertiaStabilization);

            {
                hctLocaleScope scope;

                hkStringOld s;
                s.printf("Factor: %.2f", filterOptions.m_stabilizationAmount);
                SetDlgItemText(hWnd, IDC_STABILIZATION_FACTOR_LABEL, s.cString());

                HWND hwndSlider = GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_SLIDER);
                SendMessage(hwndSlider, TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
                SendMessage(hwndSlider, TBM_SETTICFREQ, (WPARAM)1, 0);
                SendMessage(hwndSlider, TBM_SETPOS, TRUE, hkMath::clamp(int(filterOptions.m_stabilizationAmount*100.0 + 0.5f), 0, 100));

                EnableWindow(GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_LABEL), filterOptions.m_enableAutoInertiaStabilization);
                EnableWindow(GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_SLIDER), filterOptions.m_enableAutoInertiaStabilization);

                // Initialize solving method
                {
                    CheckDlgButton(hWnd, IDC_RB_STABILIZED_METHOD, filterOptions.m_constraintsSolvingMethod == hkpConstraintAtom::METHOD_STABILIZED);
                    CheckDlgButton(hWnd, IDC_RB_OLD_METHOD, filterOptions.m_constraintsSolvingMethod == hkpConstraintAtom::METHOD_OLD);
                }
            }

            // Initialize Tool Tips
            {
                CreateToolTip(IDC_CB_AutoStabilization, hWnd, hInstance, "If enabled, the constrained rigid bodies will have their inverse inertias modified so that attached constraints become more stable.");
                CreateToolTip(IDC_STABILIZATION_FACTOR_SLIDER, hWnd, hInstance, "Amount of inertia stabilization applied, ranges between 0 (no stabilization, inertia tensors are left unchanged) to 1 (full stabilization, inertia tensors are modified to achieve a stable constraint simulation).");
                CreateToolTip(IDC_RB_STABILIZED_METHOD, hWnd, hInstance, "If selected, a block solver will be used to compute the forces that maintain constraints. This is the preferred choice.");
                CreateToolTip(IDC_RB_OLD_METHOD, hWnd, hInstance, "If selected, a sequential solver will be used to compute the forces that maintain constraints. This option is deprecated.");
            }

            return TRUE; // did handle it
        }

    case WM_COMMAND: // UI Changes
        {
            switch ( LOWORD(wParam) )
            {
            case IDC_CB_AutoStabilization:
                {
                    bool autoStabilizationEnabled = IsDlgButtonChecked(hWnd, IDC_CB_AutoStabilization) != 0;
                    EnableWindow(GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_LABEL), autoStabilizationEnabled);
                    EnableWindow(GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_SLIDER), autoStabilizationEnabled);
                }
                break;
            }
        }
        break;

    case WM_HSCROLL:
        if (lParam)
        {
            hkStringOld s;
            HWND hwndSlider = GetDlgItem(hWnd, IDC_STABILIZATION_FACTOR_SLIDER);
            LRESULT ir = SendMessage(hwndSlider, TBM_GETPOS, 0, 0);
            LRESULT mx = SendMessage(hwndSlider, TBM_GETRANGEMAX, 0, 0);
            LRESULT mn = SendMessage(hwndSlider, TBM_GETRANGEMIN, 0, 0);

            {
                hctLocaleScope scope;
                s.printf( "Factor: %.2f", hkMath::clamp( hkReal(ir-mn)/hkReal(mx-mn), 0.f, 1.f ) );
            }

            SetDlgItemText( hWnd, IDC_STABILIZATION_FACTOR_LABEL, s.cString() );
        }
        break;
    }

    return FALSE; // didn't handle it / didn't do much with it
}

//
//  Shows the create constraints dialog

HWND hctCreateConstraintsFilter::showOptions (HWND owner)
{
    if (m_optionsDialog)
    {
        hideOptions();
    }

    m_optionsDialog = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_CREATE_CONSTRAINTS_DIALOG),
        owner, hkFilterCreateConstraintsDialogProc, (LPARAM) this );

    return m_optionsDialog;
}

//
//  Updates the constraints options from the UI

void hctCreateConstraintsFilter::updateOptions()
{
    if (m_optionsDialog)
    {
        m_options.m_enableAutoInertiaStabilization      = IsDlgButtonChecked(m_optionsDialog, IDC_CB_AutoStabilization) != FALSE;
        if ( m_options.m_enableAutoInertiaStabilization )
        {
            HWND hwndSlider = GetDlgItem(m_optionsDialog, IDC_STABILIZATION_FACTOR_SLIDER);
            LRESULT ir = SendMessage(hwndSlider, TBM_GETPOS, 0, 0);
            LRESULT mx = SendMessage(hwndSlider, TBM_GETRANGEMAX, 0, 0);
            LRESULT mn = SendMessage(hwndSlider, TBM_GETRANGEMIN, 0, 0);
            m_options.m_stabilizationAmount = hkMath::clamp( hkReal(ir-mn)/hkReal(mx-mn), 0.f, 1.f );
        }

        // Constraint solving method enum
        {
            if ( IsDlgButtonChecked(m_optionsDialog, IDC_RB_STABILIZED_METHOD) )
                m_options.m_constraintsSolvingMethod = hkpConstraintAtom::METHOD_STABILIZED;
            else
                m_options.m_constraintsSolvingMethod = hkpConstraintAtom::METHOD_OLD;
        }
    }
}

//
//  Hides the create constraints dialog

void hctCreateConstraintsFilter::hideOptions()
{
    updateOptions();

    if (m_optionsDialog)
    {
        DestroyWindow(m_optionsDialog);
    }

    m_optionsDialog = NULL;
}

//
//  END!
//

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
