// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateConstraintChains/hctCreateConstraintChainsFilter.h>

#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/Serialize/Resource/hkResource.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>

extern HINSTANCE hInstance;


/*
** DIALOG : ADD CHAIN
*/

struct _AddChainData
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_EXPORT, _AddChainData);

    const hkpPhysicsSystem* m_physicsSystem;

    hkStringOld m_nameA;
    hkStringOld m_nameB;
};

static void _fillCombo(HWND combo, const hkpPhysicsSystem* system, const char* curSel)
{
    SendMessage(combo, CB_RESETCONTENT, 0, 0);

    int nrbs = system->getRigidBodies().getSize();
    for (int i=0; i < nrbs; ++i)
    {
        hkpRigidBody* rb = system->getRigidBodies()[i];
        if (rb->getName())
        {
            SendMessage(combo, CB_ADDSTRING, 0, (LPARAM) rb->getName());
        }
    }

    SendMessage(combo, WM_SETTEXT, 0, (LPARAM)curSel);
}


INT_PTR CALLBACK _addChainDialogProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    _AddChainData* addChainData = reinterpret_cast<_AddChainData*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            addChainData = (_AddChainData*)lParam;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            HWND a = GetDlgItem(hWnd, IDC_CB_NAMEA);
            HWND b = GetDlgItem(hWnd, IDC_CB_NAMEB);

            _fillCombo(a, addChainData->m_physicsSystem, addChainData->m_nameA.cString());
            _fillCombo(b, addChainData->m_physicsSystem, addChainData->m_nameB.cString());
        }
        break;

        case WM_COMMAND: // UI Changes
        {
            switch ( LOWORD(wParam) )
            {
                case IDOK:
                {
                    HWND a= GetDlgItem(hWnd, IDC_CB_NAMEA);
                    HWND b= GetDlgItem(hWnd, IDC_CB_NAMEB);
                    char t[256];
                    GetWindowText(a, t, 255);
                    addChainData->m_nameA = t;
                    GetWindowText(b, t, 255);
                    addChainData->m_nameB = t;

                    EndDialog(hWnd, TRUE);
                }
                break;

                case IDCANCEL:
                {
                    EndDialog(hWnd, FALSE);
                }
                break;
            }
        }
        break;


        break;
    }
    return FALSE;
}

bool _pickTwoRigidBodiesForChain (HWND owner, const hkpPhysicsSystem* psystem, const hkpRigidBody* &rbA, const hkpRigidBody* &rbB)
{
    _AddChainData addChainData;

    addChainData.m_physicsSystem = psystem;

    INT_PTR result =
        DialogBoxParamW(hInstance, MAKEINTRESOURCEW(IDD_ADD_CHAIN_DIALOG),
        owner, _addChainDialogProc, (LPARAM)&addChainData );

    if (result != TRUE)
    {
        return false;
    }

    // We have the names, find the rigid bodies
    rbA = HK_NULL;
    rbB = HK_NULL;

    for (int i=0; i<psystem->getRigidBodies().getSize(); i++)
    {
        const hkpRigidBody* rb = psystem->getRigidBodies()[i];
        if (hkString::strCmp(rb->getName(), addChainData.m_nameA.cString()) == 0)
        {
            rbA = rb;
        }
        if (hkString::strCmp(rb->getName(), addChainData.m_nameB.cString()) == 0)
        {
            rbB = rb;
        }
    }

    if (!rbA || !rbB)
    {
        return false;
    }

    return true;
}


/*
** LOADING OF DATA FROM PREVIOUS FILTERS
*/


struct _MySimpleMappingData
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_EXPORT, _MySimpleMappingData);

    _MySimpleMappingData()
        : m_filter(HK_NULL), m_filterManager(HK_NULL),
        m_curSystem(HK_NULL), m_optionData(HK_NULL), m_optionTracker(HK_NULL)
    {
    }

    void makeOptionData();
    void rediscoverPtrs();

    hctCreateConstraintChainsFilter* m_filter;
    const hctFilterManagerInterface* m_filterManager;
    hkpPhysicsSystem* m_curSystem;
    HWND m_owner;

    // On the fly pre stage filter run data
    hkRootLevelContainer* m_optionData; // ptr in optionStorage
    hkResource* m_optionTracker;
};

void discoverSystems(const hkRootLevelContainer* data, hkArray<hkpPhysicsSystem*>& systems)
{
    hkpPhysicsData* physicsPtr = data->findObject<hkpPhysicsData>();
    while (physicsPtr)
    {
        const hkArray<hkpPhysicsSystem*>& psystems = physicsPtr->getPhysicsSystems();
        for (int psi = 0; psi < psystems.getSize(); ++psi)
        {
            systems.pushBack(psystems[psi]);
        }
        physicsPtr = data->findObject<hkpPhysicsData>(physicsPtr);
    }
}

hkpPhysicsSystem* findSystem(const hkRootLevelContainer* data, const char* sysName )
{
    if (!sysName)
        return HK_NULL;

    hkArray<hkpPhysicsSystem*> systems;
    discoverSystems(data, systems);

    for (int psi = 0; psi < systems.getSize(); ++psi)
    {
        hkpPhysicsSystem* s = systems[psi];
        if (s && s->getName() && (hkString::strCmp(s->getName(), sysName)==0))
        {
            return s;
        }
    }
    return HK_NULL;
}

void _MySimpleMappingData::makeOptionData()
{
    if (m_optionTracker)
    {
        m_optionTracker->removeReference();
        m_optionTracker = HK_NULL;
    }

    // make our image of the current root inc processing up to our filter
    m_filterManager->getInputContentsToCurrentFilter( &m_optionData, m_optionTracker);

    // find our system
    m_curSystem = findSystem(m_optionData, m_filter->m_physicsSystemName.cString());
}


void _MySimpleMappingData::rediscoverPtrs()
{
    m_curSystem = findSystem( m_optionData, m_filter->m_physicsSystemName.cString() );
}

/*
** MAIN DIALOG PROC (FILTER OPTIONS)
*/

void hctCreateConstraintChainsFilter::fillChainsListBox (HWND hWnd)
{
    ListView_DeleteAllItems(hWnd);

    LVITEM LvItem;
    memset(&LvItem,0,sizeof(LvItem)); // Reset Item Struct
    LvItem.mask=LVIF_TEXT;   // Text Style
    LvItem.cchTextMax = 260; // Max size of test

    const int numChains = hkMath::min2(m_startNames.getSize(), m_endNames.getSize());

    for (int c=0; c < numChains; ++c)
    {
        LvItem.iItem = c;
        LvItem.pszText = (char*)m_startNames[c].cString();
        ListView_InsertItem( hWnd, &LvItem);
        ListView_SetItemText( hWnd, c, 1, (char*)m_endNames[c].cString() );
    }

}

// Returns the given physics system, or the first one if the name is NULL
hkpPhysicsSystem* hctCreateConstraintChainsFilter::getPhysicsSystemGivenName (const hkRootLevelContainer& data, const char* systemName)
{
    hkpPhysicsData* physicsPtr = data.findObject<hkpPhysicsData>();

    if (physicsPtr == HK_NULL)
    {
        MessageBox(m_optionsDlg, "No physics data found", "Create Constraint Chains", MB_OK | MB_ICONWARNING);
        return HK_NULL;
    }

    if (physicsPtr->getPhysicsSystems().getSize()==0)
    {
        MessageBox(m_optionsDlg, "No physics systems found", "Create Constraint Chains", MB_OK | MB_ICONWARNING);
        return HK_NULL;
    }

    if (systemName && (hkString::strLen(systemName)>0))
    {
        hkpPhysicsSystem* theSystem = physicsPtr->findPhysicsSystemByName(systemName);
        if (!theSystem)
        {
            MessageBox(m_optionsDlg, "Can't find named physics system", "Create Constraint Chains", MB_OK | MB_ICONWARNING);
            return HK_NULL;
        }
        return theSystem;
    }
    else
    {
        hkpPhysicsSystem* theSystem = physicsPtr->getPhysicsSystems()[0];
        return theSystem;
    }

}


INT_PTR CALLBACK hkFilterCreateConstraintChainsDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    _MySimpleMappingData* filterData = reinterpret_cast<_MySimpleMappingData*> ( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;
    hctCreateConstraintChainsFilter* filter = filterData? filterData->m_filter : NULL;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            filterData = (_MySimpleMappingData*)lParam;
            filter = filterData->m_filter;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later

            HWND sysNameWnd = GetDlgItem(hWnd, IDC_ED_PhysicsSystemName);
            SetWindowTextW( sysNameWnd, hkUtf8::WideFromUtf8(filter->m_physicsSystemName.cString()));

            // Init the list view
            HWND chainListWnd = GetDlgItem(hWnd, IDC_L_Chains);
            DWORD style = LVS_EX_FULLROWSELECT | LVS_EX_INFOTIP /*| LVS_EX_TRACKSELECT*/;
            SendMessage(chainListWnd,LVM_SETEXTENDEDLISTVIEWSTYLE,0,style); // Set style

            RECT rect;
            LVCOLUMN col;
            memset(&col, 0, sizeof(col));
            col.mask = LVCF_WIDTH | LVCF_TEXT;

            // Chain mappings == 4 cols
            GetClientRect(chainListWnd, &rect);
            col.cx = (rect.right - rect.left) / 2;
            col.pszText = "Start";
            ListView_InsertColumn(chainListWnd, 0, &col);
            col.pszText = "End";
            ListView_InsertColumn(chainListWnd, 1, &col);

            CheckDlgButton(hWnd, IDC_CB_CreateLimitConstraints, filter->m_options.m_createLimitConstraints);
            CheckDlgButton(hWnd, IDC_CB_CloneMotors, filter->m_options.m_cloneMotors);

            filter->fillChainsListBox(chainListWnd);

            // Initialize Tool Tips
            {
                CreateToolTip(IDC_L_Chains, hWnd, hInstance, "This filter allows you to create constraint chains within the filter pipeline. To create a chain constraint chain you must specify the start and end bodies in the chain. See the Physics SDK documentation for more on constraint chains.");
                CreateToolTip(IDC_CB_CreateLimitConstraints, hWnd, hInstance, "Limit constraints use information from the existing powered/non-powered ragdoll/hinge constraints only. No limit constraints are created for any other types of constraints.");
                CreateToolTip(IDC_CB_CloneMotors, hWnd, hInstance, "Should the chains clone motors or reuse the ones from the original constraints.");
            }

            return TRUE; // did handle it
        }

        case WM_COMMAND: // UI Changes
        {
            switch ( LOWORD(wParam) )
            {

                case IDC_B_PickPhysicsSystem:
                    {
                        // Discover systems
                        hkArray<hkpPhysicsSystem*> systems;
                        {
                            if( !filterData->m_optionData )
                            {
                                filterData->makeOptionData();
                            }
                            discoverSystems(filterData->m_optionData, systems);
                        }

                        // Pick a system
                        hkpPhysicsSystem* s = NULL;
                        {
                            hkArray< const char* > items;
                            for( int i=0; i<systems.getSize(); ++i )
                            {
                                items.pushBack( systems[i]->getName() );
                            }
                            int index = filterData->m_filterManager->selectItemFromList( hWnd, "Pick a Physics System", items );
                            if( index >= 0 && index < systems.getSize() )
                            {
                                s = systems[index];
                            }
                        }

                        if (s && s->getName())
                        {
                            filterData->m_curSystem = s;
                            filter->m_physicsSystemName = s->getName();
                            HWND sysNameWnd = GetDlgItem(hWnd, IDC_ED_PhysicsSystemName);
                            SetWindowTextW( sysNameWnd, hkUtf8::WideFromUtf8(s->getName()));
                            filterData->rediscoverPtrs();
                        }
                        return TRUE;
                    }

                case IDC_ED_PhysicsSystemName:
                    {
                        switch (HIWORD(wParam))
                        {
                        case EN_CHANGE:
                            {
                                char textStr[128];
                                GetWindowText((HWND)lParam, textStr, 127);
                                filter->m_physicsSystemName = textStr;
                                return TRUE;
                            }
                        }
                        break;
                    }

                case IDC_B_AddChain:
                    {
                        // Make sure we evaluated the pipeline
                        if (!filterData->m_optionData)
                        {
                            filterData->makeOptionData();
                        }

                        const char* systemName = filter->m_physicsSystemName.cString();
                        hkpPhysicsSystem* theSystem = filter->getPhysicsSystemGivenName(*filterData->m_optionData, systemName);

                        if (theSystem)
                        {
                            const hkpRigidBody* rbStart;
                            const hkpRigidBody* rbEnd;
                            bool ok = _pickTwoRigidBodiesForChain(hWnd, theSystem, rbStart, rbEnd);

                            if (ok)
                            {
                                HWND chainListWnd = GetDlgItem(hWnd, IDC_L_Chains);
                                filter->m_startNames.pushBack(rbStart->getName());
                                filter->m_endNames.pushBack(rbEnd->getName());
                                filter->fillChainsListBox(chainListWnd);
                            }
                        }

                        return TRUE;
                    }
                    break;

                case IDC_B_RemoveChain:
                    {
                        // Remove selected items
                        HWND chainListWnd = GetDlgItem(hWnd, IDC_L_Chains);

                        INT ni = ListView_GetSelectionMark( chainListWnd );

                        while( ni >= 0 )
                        {
                            ListView_DeleteItem(chainListWnd, ni);

                            // We also remove from the filter options
                            filter->m_startNames.removeAtAndCopy(ni);
                            filter->m_endNames.removeAtAndCopy(ni);

                            // Not needed
                            //filter->fillChainsListBox(chainListWnd);

                            ni = ListView_GetNextItem( chainListWnd, -1, LVNI_SELECTED);
                        }


                        return TRUE;
                    }
                    break;

                case IDC_CB_CreateLimitConstraints:
                    {
                        filter->m_options.m_createLimitConstraints = IsDlgButtonChecked(hWnd, IDC_CB_CreateLimitConstraints) == TRUE? true : false;;
                        return TRUE;
                    }

                case IDC_CB_CloneMotors:
                    {
                        filter->m_options.m_cloneMotors = IsDlgButtonChecked(hWnd, IDC_CB_CloneMotors) == TRUE? true : false;;
                        return TRUE;
                    }

            }
        } // Commands
        break;

    }
    return FALSE; //didn't handle it / didn't do much with it
}

HWND hctCreateConstraintChainsFilter::showOptions(HWND owner)
{
    if (m_optionsDlg)
    {
        hideOptions();
    }

    _MySimpleMappingData* md = new _MySimpleMappingData;
    md->m_filter = this;
    md->m_filterManager = m_filterManager;
    md->m_owner = owner;

    m_dialogContext = md;

    m_optionsDlg = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_CREATE_CONSTRAINT_CHAINS_DIALOG),
        owner, hkFilterCreateConstraintChainsDlgProc, (LPARAM) md );

    return m_optionsDlg;
}

void hctCreateConstraintChainsFilter::hideOptions()
{
    if (m_optionsDlg)
    {
        DestroyWindow(m_optionsDlg);
    }

    if (m_dialogContext)
        delete ((_MySimpleMappingData*)m_dialogContext);

    m_dialogContext = NULL;
    m_optionsDlg = NULL;
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
