// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterPhysics2012/hctFilterPhysics.h>
#include <ContentTools/Common/Filters/FilterPhysics2012/CreateConstraintChains/hctCreateConstraintChainsFilter.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstance.h>

#include <Physics2012/Utilities/Constraint/Bilateral/hkpConstraintUtils.h>
#include <Physics2012/Utilities/Constraint/Chain/hkpPoweredChainMapper.h>

#include <Physics/Constraint/Motor/hkpConstraintMotor.h>
#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>

#include <Physics2012/Dynamics/Constraint/hkpConstraintInstance.h>
#include <Physics2012/Dynamics/Constraint/Chain/Powered/hkpPoweredChainData.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.p12.createconstraintchains"
#include <Common/Base/System/Log/hkLog.hxx>


hctCreateConstraintChainsFilterDesc g_createConstraintChainsDesc;

hctCreateConstraintChainsFilter::hctCreateConstraintChainsFilter( const hctFilterManagerInterface* owner)
:   hctFilterInterface( owner ),
    m_optionsDlg(HK_NULL), m_dialogContext(HK_NULL)
{
        // Set defaults
        m_options = hctCreateConstraintChainsOptions ();
        m_physicsSystemName = "";
}

hctCreateConstraintChainsFilter::~hctCreateConstraintChainsFilter()
{

}

class hkpConstraintInstance;
class hkpConstraintMotor;
class hkpPositionConstraintMotor;
class hkpConstraintChainInstance;
class hkpBallSocketChainData;

void hctCreateConstraintChainsFilter::process( hkRootLevelContainer& data  )
{
    // Find the scene in the root level container
    hkpPhysicsData* physicsPtr = data.findObject<hkpPhysicsData>();
    if (physicsPtr == HK_NULL)
    {
        HK_WARN_ALWAYS (0xabbaa5f0, "No physics data found");
        return;
    }

    if (physicsPtr->getPhysicsSystems().getSize()==0)
    {
        HK_WARN_ALWAYS (0xabba898d, "No physics systems found. Can't do anything");
        return;
    }

    // Try to create a powered chain mapper
    hkpPhysicsSystem* theSystem = HK_NULL;
    hkpPoweredChainMapper* poweredChainMapper = HK_NULL;
    {
        if (m_options.m_physicsSystemName && (hkString::strLen(m_options.m_physicsSystemName)>0))
        {
            theSystem = physicsPtr->findPhysicsSystemByName(m_options.m_physicsSystemName);
            if (!theSystem)
            {
                HK_WARN_ALWAYS (0xabbaa3ed, "Can't find physics system named "<<m_options.m_physicsSystemName<<". Will try to use the first system instead");
            }
        }
        if (!theSystem)
        {
            // We know there is at least one, take the first one
            theSystem = physicsPtr->getPhysicsSystems()[0];
            Log_Info( "Using first physics system : {}", theSystem->getName() );
        }

        const int numConstraints = theSystem->getConstraints().getSize();

        // Need at least some constraints
        if ( numConstraints < 1)
        {
            HK_WARN_ALWAYS(0xabba734c, "Selected physics system has no constraints in it.");
            return;
        }

        // Now we have what we need, create the "powered chain mapper"

        hkpPoweredChainMapper::Config theConfig;

        theConfig.m_cloneMotors = m_options.m_cloneMotors;
        theConfig.m_createLimitConstraints = m_options.m_createLimitConstraints;

        // Get list of entities
        const int numChains = hkMath::min2(m_startNames.getSize(), m_endNames.getSize());

        hkArray<hkpPoweredChainMapper::ChainEndpoints> chainEndPointsArray;

        for (int i=0; i<numChains; i++)
        {
            hkpPoweredChainMapper::ChainEndpoints endPoints;
            endPoints.m_start = HK_NULL;
            endPoints.m_end = HK_NULL;

            const char* startName = m_startNames[i].cString();
            const char* endName = m_endNames[i].cString();

            for (int r=0; r<theSystem->getRigidBodies().getSize(); r++)
            {
                hkpRigidBody* rb =theSystem->getRigidBodies()[r];
                const char* rbname = rb->getName();
                if (!rbname) continue;

                // If start not found yet, compare names
                if ((!endPoints.m_start) && (hkString::strCmp(rbname, startName) == 0))
                {
                    endPoints.m_start = rb;
                }

                // If end not found yet, compare names
                if ((!endPoints.m_end) && (hkString::strCmp(rbname, endName) == 0))
                {
                    endPoints.m_end = rb;
                }

                // Both found? finish loop
                if (endPoints.m_start && endPoints.m_end)
                {
                    break;
                }
            }

            // Make sure we got both rigid bodies
            if (!endPoints.m_start)
            {
                HK_WARN_ALWAYS(0xabbaa99d, "Couldn't find rigid body named "<<startName);
            }
            if (!endPoints.m_end)
            {
                HK_WARN_ALWAYS(0xabbaa99d, "Couldn't find rigid body named "<<endName);
            }
            if (!endPoints.m_start || !endPoints.m_end)
            {
                HK_WARN_ALWAYS(0xabbaa99d, "Ignoring chain");
                continue;
            }

            chainEndPointsArray.pushBack(endPoints);
        }

        if (chainEndPointsArray.getSize()>0)
        {
            Log_Info( "Powering all constraints..." );

            hkArray<hkpConstraintInstance*> newConstraints;
            {
                hkpPositionConstraintMotor* defaultMotor = new hkpPositionConstraintMotor( );
                defaultMotor->m_tau = 0.8f;
                defaultMotor->m_maxForce = 100.0f;
                defaultMotor->m_proportionalRecoveryVelocity = 5.0f;
                defaultMotor->m_constantRecoveryVelocity = 0.2f;

                for (int cit=0; cit< theSystem->getConstraints().getSize(); cit++)
                {
                    hkpConstraintInstance* origConstraint = theSystem->getConstraints()[cit];
                    hkpConstraintInstance* powered = hkpConstraintUtils::convertToPowered(origConstraint, defaultMotor, false);
                    if (powered)
                    {
                        newConstraints.pushBack(powered);
                    }
                    else
                    {
                        newConstraints.pushBack(origConstraint);
                    }
                }

                defaultMotor->removeReference();
            }

            // Remove all constraints from the system
            {
                for (int i=theSystem->getConstraints().getSize()-1; i>=0; --i)
                {
                    theSystem->removeConstraint(i);
                }
            }

            // Add the new constraints
            {
                for (int i=0; i<newConstraints.getSize(); ++i)
                {
                    theSystem->addConstraint(newConstraints[i]);
                }
            }

            poweredChainMapper = hkpPoweredChainMapper::buildChainMapper (theConfig, theSystem->getConstraints(), chainEndPointsArray, HK_NULL);
        }
        else
        {
            HK_WARN_ALWAYS(0xabba6abc, "No chains can be created");
        }
    }

    // If we succeed
    if (poweredChainMapper)
    {
        Log_Info( "Powered chain mapper created" );

        // We add it as a new variant
        data.m_namedVariants.expandOne().set("PoweredChainMapper", poweredChainMapper );

        // We add the constraints in the powered chain mapper to the physics system
        for (int i=0; i<poweredChainMapper->m_chains.getSize(); i++)
        {
            theSystem->addConstraint(poweredChainMapper->m_chains[i]);
        }

    }
    else
    {
        HK_WARN_ALWAYS(0xabba38ce, "No powered chain mappers created");
    }
}

void hctCreateConstraintChainsFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctCreateConstraintChainsOptions* options = hctFilterUtils::getNativeOptions<hctCreateConstraintChainsOptions>(optVar))
    {
        m_options = *options;
        m_physicsSystemName = m_options.m_physicsSystemName;

        hctFilterUtils::createArrayFromDelimitedString( m_startNames, options->m_rbStartNames, ';' );
        hctFilterUtils::createArrayFromDelimitedString( m_endNames, options->m_rbEndNames, ';' );
        delete options;
    }
}

void hctCreateConstraintChainsFilter::getOptions(hkReflect::Any& buffer) const
{
    // make sure our char* are not to withing the buffer we are writting to (setOptions() uses it and we don't copy the strings then)
    m_options.m_physicsSystemName = const_cast<char*>( m_physicsSystemName.cString() );

    // Convert arrays to ";" delimited strings
    hkStringOld startNames;
    hctFilterUtils::createDelimitedStringFromArray (m_startNames, ";", startNames);
    m_options.m_rbStartNames = const_cast<char*> (startNames.cString());

    hkStringOld endNames;
    hctFilterUtils::createDelimitedStringFromArray (m_endNames, ";", endNames);
    m_options.m_rbEndNames = const_cast<char*> (endNames.cString());

    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
