// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterManager/hctFilterManager.h>
#include <ContentTools/Common/Filters/FilterManager/hctFilterManagerImpl.h>

#include <ContentTools/Common/Filters/Common/Error/hctFilterError.h>
#include <ContentTools/Common/Filters/FilterManager/DllManager/hctFilterDllManagerImpl.h>
#include <ContentTools/Common/Filters/Common/Options/hctFilterConfigurationSet.h>

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Config/hkConfigVersion.h>

#include <Common/Base/Serialize/Resource/hkResource.h>

#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>
#include <Common/Base/Config/hkOptionalComponent.h>

#include <time.h>

#include <Common/Base/Container/PointerMap/hkMap.hxx>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.filtermanager"
#include <Common/Base/System/Log/hkLog.hxx>


#pragma warning(disable: 4611) // setjmp usage with c++

#define HAVOK_FILTER_MANAGER_REG_KEY TEXT("Software\\Havok\\FilterManager")


extern HINSTANCE hInstance;

// Dialog processes
extern INT_PTR CALLBACK hkFilterManagerImplDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
extern INT_PTR CALLBACK hkFilterManagerProgressProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

// Helper struct to forward Log_Info to hctFilterError, where they can be displayed.
static class InfoLogForward
: public hkLog::Output
{
    public:
        InfoLogForward()
            : hkLog::Output(hkLog::Level::Info)
        {
        }

        virtual hkLog::Output::Action put(const hkLog::Message& m) HK_OVERRIDE
        {
            // We're only handling Level::Info now, as HK_WARNING/HK_ASSERT
            // still forward directly to the hkError instance.
            if(m.m_level == hkLog::Level::Info)
            {
                hkError::getInstance().message(hkError::MESSAGE_REPORT, m.m_id, m.m_txt, m.m_locFilename, m.m_locLine);
                return StopMessagePropagation; 
            }
            return PropagateToNextOutput;
        }
} s_infoLogForward;

//
// hkFilterManager interface
//

hctFilterManagerImpl::hctFilterManagerImpl(const char* havokPath,  const hkMemoryInitUtil::SyncInfo* memSyncInfo, hkBool interactive)
:   m_havokPath( havokPath ),
    m_errorHandler(HK_NULL),
    m_dllManager(HK_NULL),
    m_configurationSet(HK_NULL),
    m_originalContents(HK_NULL),
    m_mainWnd(HK_NULL),
    m_PresetConfigMenu(HK_NULL),
    m_result(-1)
{
    HK_OPTIONAL_COMPONENT_REQUEST(hkPatcher);
    m_processMode = PROCESS_NONE;

    // Setup error handler
    hkLog::accessGlobalOutput().addOutput(&s_infoLogForward);

    m_errorHandler = new hctFilterError(NULL, havokPath, interactive);
    
    
    hkError::replaceInstance( m_errorHandler );


    // Setup DLL manager
    m_dllManager = new hctFilterDllManagerImpl(this, m_havokPath + "\\filters", memSyncInfo );

    // Setup configuration manager
    m_configurationSet = new hctFilterConfigurationSet( this );

#ifdef HK_DEBUG
    hkMemoryInitUtil::refreshDebugSymbols();
#endif
}

hctFilterManagerImpl::~hctFilterManagerImpl()
{
    hkLog::accessGlobalOutput().removeOutput(&s_infoLogForward);

    if( m_configurationSet )
    {
        delete m_configurationSet;
        m_configurationSet = HK_NULL;
    }

    if( m_dllManager )
    {
        delete m_dllManager;
        m_dllManager = HK_NULL;
    }

    if( m_errorHandler )
    {
        // Reset error handler
        hkError::replaceInstance(NULL);
        m_errorHandler = NULL;
    }

    removeAllStoredObjects();
}

/*virtual*/ const hkRootLevelContainer* hctFilterManagerImpl::getOriginalContents() const
{
    return m_originalContents;
}

/*virtual*/ class hctSceneExportError* hctFilterManagerImpl::getErrorHandler()
{
    return m_errorHandler;
}

/*virtual*/ HWND hctFilterManagerImpl::getMainWindowHandle () const
{
    return m_mainWnd;
}

/*virtual*/ HWND hctFilterManagerImpl::getOwnerWindowHandle () const
{
    return m_ownerWnd;
}

/*virtual*/ hctFilterManagerInterface::ProcessMode hctFilterManagerImpl::getProcessMode() const
{
    return m_processMode;
}

/*virtual*/ int hctFilterManagerImpl::getResult() const
{
    return m_result;
}

const class hctFilterDllManager* hctFilterManagerImpl::getDllManager() const
{
    return m_dllManager;
}

const char* hctFilterManagerImpl::getHavokPath() const
{
    return m_havokPath.cString();
}

unsigned int hctFilterManagerImpl::getFilterManagerVersion() const
{
    return HCT_FILTER_MANAGER_VERSION;
}

int hctFilterManagerImpl::getConfigurationSet( void* optionData ) const
{
    return m_configurationSet->getData( optionData );
}

void hctFilterManagerImpl::setConfigurationSet( const void* optionData, int optionDataSize )
{
    m_configurationSet->setData( optionData, optionDataSize );
}


void hctFilterManagerImpl::openFilterManager ( HWND owner, const hkRootLevelContainer& data, hkBool& shouldSaveConfigOut  )
{
    m_originalContents = &data;
    {
        m_ownerWnd = owner;

        INT_PTR ret = DialogBoxParamW(hInstance, MAKEINTRESOURCEW(IDD_FILTER_MANAGER),
            m_ownerWnd, hkFilterManagerImplDlgProc, (LPARAM)this );

        m_dllManager->waitForModelessFilters();

        m_errorHandler->setWindow( 0 );

        m_ownerWnd = HK_NULL;

        switch (ret)
        {
        case IDC_CLOSE:
            shouldSaveConfigOut = true;
            break;
        case IDCANCEL:
        case IDC_CANCEL:
            shouldSaveConfigOut = false;
            break;
        default:
            HK_WARN_ALWAYS (0xabba121e, "Unexpected return code");
            break;
        }

    }
    m_originalContents = HK_NULL;
}

void hctFilterManagerImpl::processBatch ( HWND owner, const hkRootLevelContainer& data, hkBool& configModifiedOut, int configToRun, bool allowModlessFilters)
{
    m_originalContents = &data;
    {
        m_processMode = allowModlessFilters? PROCESS_BATCH_UI : PROCESS_BATCH;
        processNoWait( owner, configToRun, HK_NULL );
        m_dllManager->waitForModelessFilters();
        m_processMode = PROCESS_NONE;
    }
    m_originalContents = HK_NULL;
    configModifiedOut = m_configurationSet->getOverrideMode() == hctFilterConfigurationSet::MODE_UPGRADE;
}

hkRootLevelContainer* hctFilterManagerImpl::processBatchReturnData( HWND owner, const class hkRootLevelContainer& data, int configToRun, bool allowModlessFilters, hkResource*& sceneCopyStorage)
{
    hkRootLevelContainer* retValue = HK_NULL;
    m_originalContents = &data;
    {
        m_processMode = allowModlessFilters? PROCESS_BATCH_UI : PROCESS_BATCH;
        retValue = processNoWait( owner, configToRun, &sceneCopyStorage);
        m_dllManager->waitForModelessFilters();
        m_processMode = PROCESS_NONE;
    }
    m_originalContents = HK_NULL;
    return retValue;
}

static INT_PTR CALLBACK _hkFilterManagerMakeImageProc(HWND hwndDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
    case WM_INITDIALOG:
        if (lParam)
        {
            LONG numStages = (LONG)lParam;
            HWND progressBar = GetDlgItem(hwndDlg, IDC_PROGRESS_IMAGE);
            SendMessage( progressBar, PBM_SETSTEP, 1, 0);
            SendMessage( progressBar, PBM_SETRANGE, 0, MAKELPARAM(0,numStages) );
            SendMessage( progressBar, PBM_SETPOS, 0, 0);
            return TRUE;
        }
        break;
    }
    return FALSE;
}

bool hctFilterManagerImpl::getInputContentsToCurrentFilter(hkRootLevelContainer** outputData, hkResource*& sceneCopyData ) const
{
    const int& currentIndex = m_configurationSet->m_currentIndex;
    if( currentIndex < 0 || currentIndex >= m_configurationSet->m_configurations.getSize() )
    {
        HK_WARN_ALWAYS(0xabba3a81, "INTERNAL: Filter option image run needs a current working config.");
        return false;
    }

    const hctFilterConfigurationSet::Configuration& config = m_configurationSet->m_configurations[ currentIndex ];

    // For the moment, as we know we should only call this from filters doing option setup
    // they should be the current filter dialog open.
    if (m_processMode != PROCESS_NONE)
    {
        HK_WARN_ALWAYS(0xabba8201, "INTERNAL: Only the current filter can ask for images (and only for options). You can't (and there should be no need to) call this during normal processing.");
        return false;
    }

    m_processMode = OPTIONS_BATCH;

    int runToStage = m_currentFilterDialog.m_stageIndex;

    HWND imageProgressDlg = CreateDialogParamW(hInstance, MAKEINTRESOURCEW(IDD_MAKEIMAGE_DIALOG), m_mainWnd, _hkFilterManagerMakeImageProc, LONG(runToStage));
    HWND imageProgressDlgStatus = GetDlgItem(imageProgressDlg, IDC_STATIC_IMAGEPROGRESS);
    SetWindowText(imageProgressDlgStatus, "Caching scene data..");
    ShowWindow( imageProgressDlg , SW_SHOW );
    UpdateWindow( imageProgressDlg ); // force update


    *outputData = hctFilterUtils::deepCopyObject( m_originalContents, sceneCopyData );

    setEnvironmentVariables ( *outputData, config );

    SetWindowText(imageProgressDlgStatus, "Running filters up to this filter..");

    // run through the current config, up and to the current given filter
    const hctFilterDescriptor* finalDesc = m_dllManager->getFilterDescByIndex( config.m_filterStages[runToStage].m_index );
    {
        hkStringBuf reportString; reportString.printf("Gathering Filter Changes for %s", finalDesc->getShortName());
        HK_REPORT_SECTION_BEGIN(0x1a254999, reportString.cString());
    }

    // Disable reports during this sweep
    //hkError::getInstance().setEnabled(-1, false);
    for (int i=0; i < runToStage; ++i)
    {
        const hctFilterConfigurationSet::FilterStage& filterStage = config.m_filterStages[i];

        const hctFilterDescriptor* curDesc = m_dllManager->getFilterDescByIndex(filterStage.m_index);
        if (curDesc->getFilterBehaviour() == hctFilterDescriptor::HK_DATA_PASSTHRU)
        {
            continue; // does not alter data so can be ignored
        }
        else if (curDesc->getFilterBehaviour() == hctFilterDescriptor::HK_DATA_MUTATES_EXTERNAL)
        {
            continue; // just alters external data files, so should be able to be ignored (and probably preferably so)
        }

        hctFilterInterface* curFilter = curDesc->createFilter(this);

        // have a filter that alters the data stream
        if ( filterStage.m_options.var() )
        {
            curFilter->setOptions( filterStage.m_options.var() );
        }

        HK_REPORT_SECTION_BEGIN(0x54654364, curDesc->getShortName());

        // Disaster recovery...of sorts
        if (setjmp(*m_errorHandler->getJumpBuffer()) == 0)
        {
            // Process the filter, pretend it is batch mode too in case it does an special no UI modes
            curFilter->process( **outputData );
        }
        else
        {
            Log_Error( "** Fatal Error in filter. **" );
        }

        HK_REPORT_SECTION_END();

        // safe to delete filter (no modeless allowed)
        curFilter->removeReference();

        {
            HWND progressBar = GetDlgItem(imageProgressDlg, IDC_PROGRESS_IMAGE);
            SendMessage( progressBar, PBM_SETPOS, i, 0);
            UpdateWindow( imageProgressDlg ); // force update
        }
    }

    m_processMode = PROCESS_NONE;

    hkError::getInstance().setEnabled(-1, true);

    // cleanup
    DestroyWindow(imageProgressDlg);

    HK_REPORT_SECTION_END();


    return true;
}


void hctFilterManagerImpl::registerThreadCallback (const class hctFilterThreadCallback* cb)
{
    // Check whether the callback is already registered. Do not allow the same callback to be registered multiple times!
    const int idx = m_threadCallbacks.indexOf(cb);
    if ( idx < 0 )
    {
        m_threadCallbacks.pushBack(cb);
    }
}

void hctFilterManagerImpl::setThreadData (hkMemoryRouter* memRouter) const
{
    m_dllManager->initThread( memRouter );
    for (int i=0; i<m_threadCallbacks.getSize(); i++)
    {
        m_threadCallbacks[i]->newThreadCreated(memRouter);
    }
}

//
// Processing
//

hkRootLevelContainer* hctFilterManagerImpl::processNoWait( HWND owner, int configIndex, hkResource** sceneCopyStorage )
{
    // If there are no filters to process then don't do anything.
    if ( m_configurationSet->m_configurations.getSize() == 0 )
    {
        Log_Info( "** No filters present - no action taken **" );
        return HK_NULL;
    }

    // Get the range of configurations to process.
    // A configIndex of -1 means 'process all configurations'
    int curConfig;
    int lastConfigToProcess;
    if ( configIndex == -1 )
    {
        curConfig = 0;
        lastConfigToProcess = m_configurationSet->m_configurations.getSize() - 1;
    }
    else
    {
        curConfig = configIndex;
        lastConfigToProcess = curConfig;
    }

    // Check if we are overriding/upgrading
    const hctFilterConfigurationSet::OverrideMode mode = m_configurationSet->getOverrideMode();
    switch( mode )
    {
    case hctFilterConfigurationSet::MODE_OVERRIDE:
        Log_Info( "** Using override options file **" );
        break;
    case hctFilterConfigurationSet::MODE_UPGRADE:
        Log_Info( "** Using upgrade options file **" );
        break;
    default:
        if( configIndex == -1 )
        {
            HK_REPORT_SECTION_BEGIN( 0x492348a8, "Processing All Configurations..." );
        }
    }

    // Print the asset name and time of processing to the log file.
    hkStringBuf logMsg;
    {
        hkxScene* scene = m_originalContents->findObject<hkxScene>();
        if( scene )
        {
            logMsg.printf( "========== Asset: %s ==========", scene->m_asset.cString());
            m_errorHandler->printToFile( logMsg.cString() );
        }
    }

    // Clear storage
    removeAllStoredObjects();

    // Process the specified configurations
    hkRootLevelContainer* lastSceneCopy = HK_NULL;
    for( ; curConfig<=lastConfigToProcess; ++curConfig )
    {
        m_currentFilterBeingProcessed.m_configIndex = curConfig;
        m_currentFilterBeingProcessed.m_stageIndex = -1;

        const hctFilterConfigurationSet::Configuration& config = m_configurationSet->m_configurations[curConfig];

        // Make a copy of the data (since we may be processing multiple configurations).
        hkResource* defaultPtr;
        hkResource** packfileData = sceneCopyStorage? sceneCopyStorage : &defaultPtr;
        hkRootLevelContainer* sceneCopy = hctFilterUtils::deepCopyObject( m_originalContents, *packfileData );
        lastSceneCopy = sceneCopy;

        // Get the name of the configuration.
        hkStringOld curConfigName = config.m_configName;

        if( !sceneCopy )
        {
            hkStringBuf msg;
            msg.printf("Could not clone the current scene for the %s configuration", curConfigName.cString());
            MessageBox(owner, msg.cString(), "Havok Filters", MB_OK | MB_ICONERROR );
        }
        else
        {
            lastSceneCopy = sceneCopy;

            setEnvironmentVariables( sceneCopy, config );

            // Show configuration information in the Filter Preview dialog.
            hkStringBuf infoString;
            __time64_t lonm_time;
            {
                struct tm newtime;
                _time64( &lonm_time );                /* Get time as long integer. */
                _localtime64_s( &newtime, &lonm_time ); /* Convert to local time. */

                // 26 is minimum required by asctime_s: Www Mmm dd hh:mm:ss yyyy\n\0
                char buf[26];
                asctime_s(buf, HK_COUNT_OF(buf), &newtime);
                buf[24] = '\0';

                infoString.printf("\n========== Started: %s, Configuration: %s ==========\n", buf, curConfigName.cString() );

                // Dark gray print
                HK_REPORT2((int)RGB(80,80,80), infoString.cString());
            }

            // Reset counters
            hctFilterError::resetCounters();

            // Require some filters
            const hkArray<hctFilterConfigurationSet::FilterStage>& filterStages = config.m_filterStages;
            const int nf = filterStages.getSize();
            if( nf <= 0 )
            {
                Log_Info( "*** There is no filter in this configuration. ***" );
            }
            else
            {
                // Check the configuration for a PlatformWriter filter.
                {
                    hkBool noPlatformWriter = true;

                    for ( int i = 0; i < nf; i++ )
                    {
                        int filterIndex = filterStages[i].m_index;
                        const hctFilterDescriptor* desc = m_dllManager->getFilterDescByIndex( filterIndex );

                        if ( desc->getID() == 0xab787565 ) // Plaform writer ID
                        {
                            noPlatformWriter = false;
                            break;
                        }
                    }

                    if ( noPlatformWriter )
                    {
                        Log_Info( "*** No PlatformWriter filter found. No HKX data will be saved for this configuration. ***" );
                    }
                }

                // Initialize the seed of the UUID generator so we get deterministic UUIDs further-on.
                {
                    hkxScene* scene = sceneCopy->findObject<hkxScene>();

                    if ( scene && (scene->m_rootNode && scene->m_rootNode->m_uuid != hkUuid::getNil()) )
                    {
                        const hkUuid& uuid = scene->m_rootNode->m_uuid;
                        hkUint32 newSeed = uuid.getData<0>() ^ uuid.getData<1>() ^ uuid.getData<2>();
                        hkUuidPseudoRandomGenerator::getInstance().setSeed(newSeed);
                    }
                }

                // Process the filters.
                for (int i=0; i < nf; ++i)
                {
                    m_currentFilterBeingProcessed.m_stageIndex = i;

                    const hctFilterDescriptor* curDesc = m_dllManager->getFilterDescByIndex(filterStages[i].m_index);
                    infoString += curDesc->getShortName();
                    if (i < (nf-1))
                    {
                        infoString += ", ";
                    }

                    // Put infoString into hkxEnvironment
                    hkxEnvironment* environment =  (sceneCopy->findObject<hkxEnvironment>());
                    if (environment)
                    {
                        environment->setVariable("infoString", infoString.cString());

                        hkStringBuf slotStr; slotStr.printf("%d", i);
                        environment->setVariable("slot", slotStr.cString());
                    }

                    hctFilterInterface* curFilter = curDesc->createFilter(this);

                    if ( filterStages[i].m_options.var() )
                    {
                        curFilter->setOptions( filterStages[i].m_options.var() );
                    }

                    char statusBuf[255];
                    hkString::sprintf(statusBuf, "Processing : %s", curDesc->getShortName());
                    showStatus(statusBuf);

                    hkStringBuf sectionTitle;
                    sectionTitle.printf("%s Filter", curDesc->getShortName());
                    HK_REPORT_SECTION_BEGIN(0x54654364, sectionTitle.cString());

                    // Give a warning if current product doesn't support filter
                    hctFilterDescriptor::HavokComponentMask requiredComponents = curDesc->getRequiredHavokComponents();
                    if (!checkProductSupport(requiredComponents))
                    {
                        HK_WARN_ALWAYS (0xabba911c, "Filter not supported by current product selection");
                    }

                    // Disaster recovery. May leak memory, but will at least jump out and exit cleanly.
                    if (setjmp(*m_errorHandler->getJumpBuffer()) == 0)
                    {
                        // Process the filter
                        curFilter->process( *sceneCopy );
                    }
                    else
                    {
                        HK_REPORT_SECTION_END();
                        curFilter->removeReference();
                        break;
                    }
                    HK_REPORT_SECTION_END();

                    hkString::sprintf(statusBuf, "Finished : %s\n", curDesc->getShortName());
                    showStatus(statusBuf);
                    curFilter->removeReference();
                }
            }

            // Show timing info
            {
                struct tm totalTime;
                __time64_t finishTime;
                _time64( &finishTime );                /* Get time as long integer. */
                finishTime -= lonm_time;
                _gmtime64_s( &totalTime, &finishTime ); /* Convert to time struct. */

                hkStringBuf infoTimingString; infoTimingString.printf("\n========== Finished in %02d:%02d:%02d, %d Errors, %d Warnings ==========",totalTime.tm_hour, totalTime.tm_min, totalTime.tm_sec, hctFilterError::getErrorCount(), hctFilterError::getWarnCount() );

                if (hctFilterError::getWarnCount() == 0 && hctFilterError::getErrorCount() == 0)
                {
                    // Green for success
                    HK_REPORT2( ((int)RGB(0,140,0)), infoTimingString.cString() );
                }
                else if (hctFilterError::getErrorCount() == 0)
                {
                    // Orange for warnings
                    HK_REPORT2( ((int)RGB(255,127,0)), infoTimingString.cString() );
                }
                else
                {
                    // Red for errors
                    HK_REPORT2( ((int)RGB(255,0,0)), infoTimingString.cString() );
                }

                m_errorHandler->printToFile(""); // new line before next run

                infoString; infoString.printf("Finished in %02d:%02d:%02d, %d Errors, %d Warnings",totalTime.tm_hour, totalTime.tm_min, totalTime.tm_sec, hctFilterError::getErrorCount(), hctFilterError::getWarnCount() );
                showStatus(infoString.cString());
            }

            if (!sceneCopyStorage || ((*packfileData == *sceneCopyStorage) && (curConfig < lastConfigToProcess)) )
            {
                (*packfileData)->removeReference();
            }
            //HK_REPORT_SECTION_END();
        }
    }

    // Clear storage
    removeAllStoredObjects();

    if ( mode == hctFilterConfigurationSet::MODE_NORMAL && configIndex == -1 )
    {
        HK_REPORT_SECTION_END();
    }

    // Return Status
    m_result = 0; // Complete Success
    if (hctFilterError::getWarnCount() > 0)
    {
        m_result = 1; // Success with Warnings
    }
    if (hctFilterError::getErrorCount() > 0)
    {
        m_result = -1; // Failure
    }

    m_currentFilterBeingProcessed.m_configIndex = -1;
    m_currentFilterBeingProcessed.m_stageIndex = -1;

    // May still be modeless DLLs. They will only be waited on on close on the window (so
    // as to not disturb normal message handling in this dlg (if doing a test)

    return sceneCopyStorage? lastSceneCopy : HK_NULL;
}

void hctFilterManagerImpl::getFileDependencies(HWND owner, const hkRootLevelContainer& data, hkArray<hkStringPtr>& fileDependencies, int configIndex)
{
    hkResource** sceneCopyStorage = HK_NULL;
    m_originalContents = &data;

    m_processMode = PROCESS_BATCH;
    {
        // If there are no filters to process then don't do anything.
        if (m_configurationSet->m_configurations.getSize() == 0)
        {
            Log_Info("** No filters present - no action taken **");
            return;
        }

        // Get the range of configurations to process.
        // A configIndex of -1 means 'process all configurations'
        int curConfig;
        int lastConfigToProcess;
        if (configIndex == -1)
        {
            curConfig = 0;
            lastConfigToProcess = m_configurationSet->m_configurations.getSize() - 1;
        }
        else
        {
            curConfig = configIndex;
            lastConfigToProcess = curConfig;
        }

        // Clear storage
        removeAllStoredObjects();

        // Process the specified configurations
        hkRootLevelContainer* lastSceneCopy = HK_NULL;
        for (; curConfig <= lastConfigToProcess; ++curConfig)
        {
            m_currentFilterBeingProcessed.m_configIndex = curConfig;
            m_currentFilterBeingProcessed.m_stageIndex = -1;

            const hctFilterConfigurationSet::Configuration& config = m_configurationSet->m_configurations[curConfig];

            // Make a copy of the data (since we may be processing multiple configurations).
            hkResource* defaultPtr;
            hkResource** packfileData = sceneCopyStorage ? sceneCopyStorage : &defaultPtr;
            hkRootLevelContainer* sceneCopy = hctFilterUtils::deepCopyObject(m_originalContents, *packfileData);
            lastSceneCopy = sceneCopy;

            // Get the name of the configuration.
            hkStringOld curConfigName = config.m_configName;

            if (!sceneCopy)
            {
                Log_Info("*** Could not clone the current scene for the {} configuration. ***", curConfigName.cString());
            }
            else
            {
                lastSceneCopy = sceneCopy;

                setEnvironmentVariables(sceneCopy, config);

                hkStringBuf infoString;

                // Reset counters
                hctFilterError::resetCounters();

                // Require some filters
                const hkArray<hctFilterConfigurationSet::FilterStage>& filterStages = config.m_filterStages;
                const int nf = filterStages.getSize();
                if (nf <= 0)
                {
                    Log_Info("*** There is no filter in this configuration. ***");
                }
                else
                {
                    // Initialize the seed of the UUID generator so we get deterministic UUIDs further-on.
                    {
                        hkxScene* scene = sceneCopy->findObject<hkxScene>();

                        if (scene && (scene->m_rootNode && scene->m_rootNode->m_uuid != hkUuid::getNil()))
                        {
                            const hkUuid& uuid = scene->m_rootNode->m_uuid;
                            hkUint32 newSeed = uuid.getData<0>() ^ uuid.getData<1>() ^ uuid.getData<2>();
                            hkUuidPseudoRandomGenerator::getInstance().setSeed(newSeed);
                        }
                    }

                    // Process the filters.
                    for (int i = 0; i < nf; ++i)
                    {
                        m_currentFilterBeingProcessed.m_stageIndex = i;

                        const hctFilterDescriptor* curDesc = m_dllManager->getFilterDescByIndex(filterStages[i].m_index);
                        infoString += curDesc->getShortName();
                        if (i < (nf - 1))
                        {
                            infoString += ", ";
                        }

                        // Put infoString into hkxEnvironment
                        hkxEnvironment* environment = (sceneCopy->findObject<hkxEnvironment>());
                        if (environment)
                        {
                            environment->setVariable("infoString", infoString.cString());

                            hkStringBuf slotStr; slotStr.printf("%d", i);
                            environment->setVariable("slot", slotStr.cString());
                        }

                        hctFilterInterface* curFilter = curDesc->createFilter(this);

                        if (filterStages[i].m_options.var())
                        {
                            curFilter->setOptions(filterStages[i].m_options.var());
                        }

                        // Disaster recovery. May leak memory, but will at least jump out and exit cleanly.
                        if (setjmp(*m_errorHandler->getJumpBuffer()) == 0)
                        {
                            // Get the file dependencies
                            curFilter->getFileDependencies(*sceneCopy, fileDependencies);
                        }
                        else
                        {
                            curFilter->removeReference();
                            break;
                        }

                        curFilter->removeReference();
                    }
                }

                if (!sceneCopyStorage || ((*packfileData == *sceneCopyStorage) && (curConfig < lastConfigToProcess)))
                {
                    (*packfileData)->removeReference();
                }

            }
        }

        // Clear storage
        removeAllStoredObjects();

        // Return Status
        m_result = 0; // Complete Success
        if (hctFilterError::getWarnCount() > 0)
        {
            m_result = 1; // Success with Warnings
        }
        if (hctFilterError::getErrorCount() > 0)
        {
            m_result = -1; // Failure
        }

        m_currentFilterBeingProcessed.m_configIndex = -1;
        m_currentFilterBeingProcessed.m_stageIndex = -1;

        // May still be modeless DLLs. They will only be waited on on close on the window (so
        // as to not disturb normal message handling in this dlg (if doing a test)
    }

    m_dllManager->waitForModelessFilters();
    m_processMode = PROCESS_NONE;
    m_originalContents = HK_NULL;
}

void hctFilterManagerImpl::showStatus( const char* status ) const
{
    m_status = status;
    if( m_mainWnd )
    {
        SetWindowText( GetDlgItem( m_mainWnd, IDC_STATUS_TEXT ), status );
    }
}

void hctFilterManagerImpl::showHelp(HWND owner) const
{
    // EXP-754
    hkStringOld helpFile;
    hctSdkUtils::getHelpFileName(helpFile);

    const hkStringOld helpPath = m_havokPath + helpFile;

    ShellExecute( m_ownerWnd, "open", helpPath.cString(), NULL, m_havokPath.cString(), SW_SHOWNORMAL );
}


//
// Misc
//

hctFilterDescriptor::HavokComponentMask hctFilterManagerImpl::getAvailableHavokComponents() const
{
    hctOptionsRegistry registry ("filterManager");

    int availableComponents = registry.getIntWithDefault("availableComponents", hctFilterDescriptor::HK_COMPONENTS_ALL);

    // EXP-1356 : Make sure we only use valid products (No FX support any more)
    availableComponents &= hctFilterDescriptor::HK_COMPONENTS_ALL;

    return (hctFilterDescriptor::HavokComponentMask) availableComponents;

}

void hctFilterManagerImpl::setAvailableHavokComponents(hctFilterDescriptor::HavokComponentMask product) const
{
    hctOptionsRegistry registry ("filterManager");

    registry.setInt("availableComponents", product);
}

bool hctFilterManagerImpl::checkProductSupport (hctFilterDescriptor::HavokComponentMask requiredComponents) const
{
    hctFilterDescriptor::HavokComponentMask availableComponents = getAvailableHavokComponents();

    return (requiredComponents & availableComponents) == requiredComponents;
}

/*static*/ const char* hctFilterManagerImpl::getProductNameBasedOnComponents (hctFilterDescriptor::HavokComponentMask components)
{
    switch (components)
    {
        case hctFilterDescriptor::HK_COMPONENT_COMMON:
            return "(Havok Base)";

        case hctFilterDescriptor::HK_COMPONENT_ANIMATION:
            return "(Havok Animation)";

        case hctFilterDescriptor::HK_COMPONENT_PHYSICS:
            return "(Havok Physics)";

        case hctFilterDescriptor::HK_COMPONENTS_COMPLETE:
            return "(Havok Complete)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH:
            return "(Havok Cloth)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_ANIMATION:
            return "(Havok Animation + Cloth)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_PHYSICS:
            return "(Havok Physics + Cloth)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENTS_COMPLETE:
            return "(Havok Complete + Cloth)";

        case hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Animation + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Physics + Destruction)";

        case hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Complete + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Cloth + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Animation + Cloth + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Physics + Cloth + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION:
            return "(Havok Complete + Cloth + Destruction)";

        case hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok AI)";

        case hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Animation + AI)";

        case hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Physics + AI)";

        case hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Complete + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Cloth + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Animation + Cloth + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Physics + Cloth + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Complete + Cloth + AI)";

        case hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Animation + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Physics + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Complete + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Cloth + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_ANIMATION | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Animation + Cloth + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENT_PHYSICS | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Physics + Cloth + Destruction + AI)";

        case hctFilterDescriptor::HK_COMPONENT_CLOTH | hctFilterDescriptor::HK_COMPONENTS_COMPLETE | hctFilterDescriptor::HK_COMPONENT_DESTRUCTION | hctFilterDescriptor::HK_COMPONENT_AI:
            return "(Havok Complete + Cloth + Destruction + AI)";
    }

    return "Unknown Product";
}


void hctFilterManagerImpl::setEnvironmentVariables (hkRootLevelContainer* rootLevelcontainer, const hctFilterConfigurationSet::Configuration& config ) const
{
    // Look for the environment
    hkxEnvironment* environment = HK_NULL;
    hkxScene* scene = HK_NULL;
    {
        for (int i=0; i<rootLevelcontainer->m_namedVariants.getSize(); i++)
        {
            if (hkxEnvironment* e = rootLevelcontainer->m_namedVariants[i].getByTypeName<hkxEnvironment>())
            {
                environment = e;
            }
            if (hkxScene* s = rootLevelcontainer->m_namedVariants[i].getByTypeName<hkxScene>())
            {
                scene = s;
            }
        }
    }

    // Do not warn on resizing arrays
    const bool wasEnabled = hkError::getInstance().isEnabled(0xf3768206);
    hkError::getInstance().setEnabled(0xf3768206, false);

    if (!environment)
    {
        // Add one
        environment = new hkxEnvironment();

        // Put the new one first
        hkRootLevelContainer::NamedVariant newVariant("Environment Data", environment, hkReflect::getType<hkxEnvironment>());
        rootLevelcontainer->m_namedVariants.insertAt(0, newVariant);

        // If there was a scene, fill it with that
        if (scene)
        {
            hkxSceneUtils::fillEnvironmentFromScene(*scene, *environment);
        }
    }

    environment->setVariable( "configuration", config.m_configName.cString() );

    hkError::getInstance().setEnabled(0xf3768206, wasEnabled);
}

hkReferencedObject* hctFilterManagerImpl::getStoredObject( const char* name ) const
{
    hkStringMap< hkReferencedObject* >::Iterator it = m_storage.findKey( name );
    if ( !m_storage.isValid( it ) )
    {
        return HK_NULL;
    }

    return m_storage.getValue( it );
}

void hctFilterManagerImpl::storeObject( const char* name, hkReferencedObject* object )
{
    HK_ASSERT_NO_MSG( 0xbbb00021, object != HK_NULL );
    m_storage.insert( name, object );
}

void hctFilterManagerImpl::removeStoredObject( const char* name )
{
    hkStringMap< hkReferencedObject* >::Iterator it = m_storage.findKey( name );
    if ( !m_storage.isValid( it ) )
    {
        return;
    }

    #ifdef HK_DEBUG
    hkReferencedObject* oldObject = m_storage.getValue( it );
    HK_ASSERT_NO_MSG( 0xbbb00022, oldObject != HK_NULL );
//  oldObject->removeReference();
    #endif

    m_storage.remove( it );
}

void hctFilterManagerImpl::removeAllStoredObjects()
{
//  for ( hkStringMap< hkReferencedObject* >::Iterator it=m_storage.getIterator(); m_storage.isValid(it); it=m_storage.getNext(it) )
//  {
//      hkReferencedObject* oldObject = m_storage.getValue( it );
//      HK_ASSERT_NO_MSG( 0xbbb00023, oldObject != HK_NULL );
//      oldObject->removeReference();
//  }

    m_storage.clear();
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
