// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterManager/hctFilterManager.h>
#include <ContentTools/Common/Filters/FilterManager/DllManager/hctFilterDllManagerImpl.h>

#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>

hctFilterDllManagerImpl::hctFilterDllManagerImpl(hctFilterManagerInterface* filterInterface, const hkStringOld& filterPath, const hkMemoryInitUtil::SyncInfo* memSyncInfo)
{
    // The system info which will be applied to each DLL
    hkMemoryInitUtil::SyncInfo localSyncInfo;
    const hkMemoryInitUtil::SyncInfo* syncInfo = memSyncInfo;
    if (!syncInfo)
    {
        hkMemoryInitUtil::SyncInfo::getLocalInfo(localSyncInfo);
        syncInfo = &localSyncInfo;
    }

    hkArray<const hctFilterDescriptor*> filterDescs;

    // Search for any DLLs in the provided path
    WIN32_FIND_DATA FindFileData;
    HANDLE hFind = FindFirstFile( hkStringOld( filterPath + "\\*.dll" ).cString(), &FindFileData );
    while( hFind != INVALID_HANDLE_VALUE )
    {
        const hkStringOld dllName = filterPath + "\\" + FindFileData.cFileName;
        if (FindNextFile(hFind, &FindFileData) == 0)
        {
            hFind = INVALID_HANDLE_VALUE;
        }

        // Load the DLL
        HMODULE filterDllModule = LoadLibrary( dllName.cString() );
        if( !filterDllModule )
        {
            HK_WARN_ALWAYS( 0xabba9733, "Couldn't load DLL (" << dllName.cString() );
            continue;
        }

        // Require our exported function
        hkGetFilterDLLFunc getFilterDllFunc = (hkGetFilterDLLFunc) GetProcAddress( filterDllModule, "getFilterDll" );
        if( !getFilterDllFunc )
        {
            HK_WARN_ALWAYS( 0xabba9734, "Missing function(s) in DLL (" << dllName.cString() << "). Ignoring DLL." );
            FreeLibrary( filterDllModule );
            continue;
        }

        // Get our interface
        hctFilterDll* filterDll = getFilterDllFunc( filterDllModule );
        if( !filterDll )
        {
            HK_WARN_ALWAYS( 0xabba9735, "Couldn't get filter interface (" << dllName.cString() << "). Ignoring DLL." );
            FreeLibrary( filterDllModule );
            continue;
        }

        // Require correct version
        if( filterDll->getDllVersion() != hctBaseDll::getCurrentDllVersion() )
        {
            HK_WARN_ALWAYS( 0xabba30ff, "Incompatible filter DLL version (" << dllName.cString() << "). Ignoring DLL." );
            FreeLibrary( filterDllModule );
            continue;
        }

        // Require some filters
        if( filterDll->getNumberOfFilters() <= 0 )
        {
            HK_WARN_ALWAYS( 0xabba301f, "No filters in DLL (" << dllName.cString() << "). Ignoring DLL." );
            FreeLibrary( filterDllModule );
            continue;
        }

        // Init and keep the DLL
        filterDll->initFilterDll(filterInterface, *syncInfo, &hkError::getInstance() /*this is the local filter manager error log*/ );

        m_filterDlls.pushBack( filterDll );

        // Add the individual filters to the list
        for( int i=0; i<filterDll->getNumberOfFilters(); ++i )
        {
            const hctFilterDescriptor* desc = filterDll->getFilterDescriptor(i);
            if( desc )
            {
                filterDescs.pushBack( desc );
            }
        }
    }
    FindClose( hFind );

    // Verify that each filter has a unique ID
    for( int i=0; i<filterDescs.getSize(); ++i )
    {
        const hctFilterDescriptor* desc = filterDescs[i];
        for( int j=0; j<i; ++j )
        {
            const hctFilterDescriptor* otherDesc = filterDescs[j];
            if( otherDesc->getID() == desc->getID() )
            {
                hkStringOld str;
                str.printf( "Filters \"%s\" and \"%s\" have same ID [%d]. This will cause options to behave incorrectly.",
                            desc->getShortName(), otherDesc->getShortName(), desc->getID() );
                HK_WARN_ALWAYS( 0xabba924c, str.cString() );
            }
        }
    }

    // Add filters in alphabetic order
    hkArray<hkStringOld> names;
    hkStringMap<int> nameToIndex;

    for( int i=0; i<filterDescs.getSize(); ++i )
    {
        const hctFilterDescriptor* desc = filterDescs[i];
        hkStringOld name( desc->getShortName() );
        names.pushBack( name );
        nameToIndex.insert( desc->getShortName(), i );
    }

    if ( names.getSize() )  // allow for case where there are no filters at all (EXP-1203)
    {
        hkAlgorithm::quickSort( &names[0], names.getSize() );
        int indx;
        for( int i=0; i<names.getSize(); ++i )
        {
            HK_ON_DEBUG(hkResult found =) nameToIndex.get( names[i].cString(), &indx );
            HK_ASSERT_NO_MSG( 0x224321ea, found.isSuccess() );
            m_filterDescs.pushBack( filterDescs[indx] );
        }
    }
}

hctFilterDllManagerImpl::~hctFilterDllManagerImpl()
{
    // Quit and unload the DLLs
    for( int i=0; i<m_filterDlls.getSize(); ++i )
    {
        hctFilterDll* filterDll= m_filterDlls[i];
        if( filterDll )
        {
            filterDll->quitDll();
            // Never delete the filterdll ptrs, as they are usually statics.
            FreeLibrary( filterDll->getDllModule() );
        }
    }
    m_filterDlls.clear();
    m_filterDescs.clear();
}


const hctFilterDescriptor* hctFilterDllManagerImpl::getFilterDescByIndex( int index ) const
{
    if( index >= 0 && index < m_filterDescs.getSize() )
    {
        return m_filterDescs[ index ];
    }
    return HK_NULL;
}

static unsigned int _ImplicitFilterUpgrades[] =
{
    // Old preview to new. New preview will handle setting the old options (ignore them)
    0xa397b875, 0x700030cb
};

int hctFilterDllManagerImpl::getFilterIndexByID( unsigned int filterID ) const
{
    for( int i=0; i<m_filterDescs.getSize(); ++i )
    {
        if( m_filterDescs[i]->getID() == filterID )
        {
            return i;
        }
    }

    // not found, see if we have a replacement for it
    int numUpgrades = sizeof(_ImplicitFilterUpgrades) / sizeof (unsigned int);
    for (int u=0; u < numUpgrades; u += 2)
    {
        unsigned int f = _ImplicitFilterUpgrades[u];
        if (filterID == f)
        {
            return getFilterIndexByID( _ImplicitFilterUpgrades[u+1] );
        }
    }
    return -1;
}


void hctFilterDllManagerImpl::initThread( hkMemoryRouter* memRouter)
{
    if (memRouter)
    {
        hkBaseSystem::initThread( memRouter );
    }
    else
    {
        hkBaseSystem::quitThread();
    }

    for( int i=0; i < m_filterDlls.getSize(); ++i )
    {
        m_filterDlls[i]->initThread( memRouter );
    }
}


//
// Closing Down Dialog
//

#define MYTIMER 666


INT_PTR CALLBACK _waitForModelessDialogProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctFilterDllManagerImpl *manager = reinterpret_cast<hctFilterDllManagerImpl*>( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA) );

    bool shouldUpdateLabels = false;
    bool shouldClose = false;

    BOOL result = FALSE;

    switch(message)
    {
        case WM_INITDIALOG:
            {
                SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later
                manager = (hctFilterDllManagerImpl*)lParam;

                // Get a WM_TIMER event every .1s
                SetTimer(hWnd, MYTIMER, 100, NULL);

                shouldUpdateLabels = true;
                result = FALSE;
                break;
            }

        case WM_COMMAND:
            switch ( LOWORD(wParam) )
            {
              case IDCANCEL:
                  {
                      shouldClose = true;
                      result = TRUE;
                      break;
                  }
              }
            break;

        case WM_TIMER:
            {
                shouldUpdateLabels = true;
                result = TRUE;
                break;
            }
    }

    if (shouldUpdateLabels)
    {
        int numActive, numClosing;
        const int totalModeless = manager->countModelessFilters(numActive, numClosing);
        if (totalModeless == 0)
        {
            shouldClose = true;
        }

        char buffer[200];
        hkString::sprintf(buffer,"%d still active", numActive );
        SendMessage(GetDlgItem(hWnd, IDC_LAB_ACTIVE), WM_SETTEXT, 0, (LPARAM) buffer );

        hkString::sprintf(buffer,"%d closing down", numClosing );
        SendMessage(GetDlgItem(hWnd, IDC_LAB_CLOSING), WM_SETTEXT, 0, (LPARAM) buffer );

    }

    if (shouldClose)
    {
        KillTimer(hWnd, MYTIMER);
        EndDialog( hWnd, 0 );
    }

    return result;
}


extern HINSTANCE hInstance;

// Tries to close and then waits for all modeless filters. Returns false if the user cancels
bool hctFilterDllManagerImpl::closeAndWaitForModelessFilters(HWND parentWnd) const
{
    // See if there is any modeless filter
    int dummy1,dummy2;
    int numModeless = countModelessFilters(dummy1, dummy2);

    if (numModeless>0)
    {
        // Ask them to close
        askModelessFiltersToClose();
        // Open a dialog reporting the progress of the "closing down"
        DialogBoxParamW(hInstance, MAKEINTRESOURCEW(IDD_CLOSING_FILTERS), parentWnd, _waitForModelessDialogProc, (LPARAM) this);
    }

    // the user may have cancelled, so check again
    numModeless = countModelessFilters(dummy1, dummy2);

    return (numModeless==0);
}


// Ask all filters to close
void hctFilterDllManagerImpl::askModelessFiltersToClose () const
{
    for (int i=0; i < m_filterDlls.getSize(); ++i )
    {
        hctFilterDll* filterDll = m_filterDlls[i];
        for (int j=0; j <filterDll->getNumberOfFilters(); j++)
        {
            hctFilterDescriptor* filterDesc = filterDll->getFilterDescriptor(j);
            filterDesc->askModelessFiltersToClose();
        }
    }
}

// Returns how many modeless filters are present.
int hctFilterDllManagerImpl::countModelessFilters (int& numActiveOut, int& numClosingOut) const
{
    numActiveOut = numClosingOut = 0;

    for (int i=0; i < m_filterDlls.getSize(); ++i )
    {
        hctFilterDll* filterDll = m_filterDlls[i];
        for (int j=0; j <filterDll->getNumberOfFilters(); j++)
        {
            hctFilterDescriptor* filterDesc = filterDll->getFilterDescriptor(j);

            int filterActive=0, filterClosing=0;
            filterDesc->countModelessFilters(filterActive, filterClosing);

            numActiveOut += filterActive;
            numClosingOut += filterClosing;

        }
    }

    return numActiveOut+numClosingOut;
}


void hctFilterDllManagerImpl::waitForModelessFilters() const
{
    int dummy1, dummy2;
    while (countModelessFilters(dummy1, dummy2) > 0)
    {
        Sleep(100);
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
