// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterManager/hctFilterManager.h>
#include <ContentTools/Common/Filters/FilterManager/hctFilterManagerImpl.h>
#include <ContentTools/Common/Filters/Common/Error/hctFilterError.h>
#include <ContentTools/Common/Filters/FilterManager/DllManager/hctFilterDllManagerImpl.h>

#include <Common/Base/Container/String/hkUtf8.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.filtermanager"
#include <Common/Base/System/Log/hkLog.hxx>


extern HINSTANCE hInstance;

extern void _CloseCurrentFilterOptions(hctFilterManagerImpl* manager );
extern void _OpenFilterOptions( hctFilterManagerImpl* manager, int stageIndex );
extern void _RefreshFilterView( HWND listView, hctFilterManagerImpl* manager );


void _SetupConfigCombo( HWND combo, hctFilterManagerImpl* manager )
{
    // Clear the combo box
    SendMessage( combo, CB_RESETCONTENT, (WPARAM)0, 0 );

    // Load the current configurations
    int numConfigs = manager->m_configurationSet->m_configurations.getSize();
    for( int i=0; i<numConfigs; ++i )
    {
        const char* szBuf = manager->m_configurationSet->m_configurations[i].m_configName.cString();
        SendMessageW( combo, CB_ADDSTRING, 0, (LPARAM)hkUtf8::WideFromUtf8(szBuf).cString() );
    }

    // Select the active configuration
    SendMessageW( combo, CB_SETCURSEL, (WPARAM)manager->m_configurationSet->m_currentIndex, 0 );
}


INT_PTR CALLBACK hkFilterManagerFilterConfigurationsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctFilterManagerImpl *manager = reinterpret_cast<hctFilterManagerImpl*>( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA) );

    switch( message )
    {

    case WM_INITDIALOG:
        {
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later
            manager = (hctFilterManagerImpl*)lParam;

            // Setup the combo box
            _SetupConfigCombo( GetDlgItem( hWnd, IDC_CONFIG_COMBO ), manager );

            // Setup filter order list
            HWND orderWnd = GetDlgItem( hWnd, IDC_FILTERS_ORDER_LIST );
            {
                DWORD style = LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_INFOTIP;
                SendMessage( orderWnd, LVM_SETEXTENDEDLISTVIEWSTYLE, 0, style );

                HIMAGELIST hImageList = ImageList_LoadImage( hInstance, MAKEINTRESOURCE(IDB_BITMAP_CAT), 16, 1, RGB(255,255,255), IMAGE_BITMAP, LR_CREATEDIBSECTION );
                ListView_SetImageList( orderWnd, hImageList, LVSIL_SMALL );

                LVCOLUMN LvCol;
                LvCol.mask = LVCF_TEXT;
                LvCol.pszText = "Filter Order";
                ListView_InsertColumn( orderWnd, 0, &LvCol );
            }

            // Populate list
            const hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->getCurrentConfiguration();
            int numFilterStages = config.m_filterStages.getSize();
            for( int i=0; i < numFilterStages; ++i )
            {
                const int filterNum = config.m_filterStages[i].m_index;
                const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex( filterNum );
                if( desc )
                {
                    LVITEM LvItem;
                    LvItem.iItem = i;
                    LvItem.mask = LVIF_TEXT | LVIF_PARAM | LVIF_IMAGE;
                    LvItem.cchTextMax = 256;
                    LvItem.pszText = const_cast<LPSTR>( desc->getShortName() );
                    LvItem.lParam = filterNum;
                    LvItem.iImage = manager->categoryToIconID(desc->getCategory());
                    LvItem.iSubItem = 0;

                    ListView_InsertItem( orderWnd, &LvItem );
                }
            }

            // Select the first filter
            if( numFilterStages > 0 )
            {
                ListView_SetItemState( orderWnd, 0, LVIS_SELECTED | LVIS_FOCUSED, 0x0FF );
                ListView_SetSelectionMark( orderWnd, 0 );
                _OpenFilterOptions( manager, 0 );
            }
        }
        // FALL THROUGH

    case WM_SIZE:
        {
            RECT wndRect; GetClientRect( hWnd, (LPRECT)&wndRect );

            SetWindowPos( GetDlgItem( hWnd, IDC_HEADING ), NULL,
                5, 5, wndRect.right-10, 15, SWP_NOZORDER );

            SetWindowPos( GetDlgItem( hWnd, IDC_CONFIG_COMBO ), NULL,
                90, 25, wndRect.right-165, 20, SWP_NOZORDER );
            SetWindowPos( GetDlgItem( hWnd, IDC_NEW_CONFIGURATION ), NULL,
                wndRect.right-70, 25, 30, 20, SWP_NOZORDER );
            SetWindowPos( GetDlgItem( hWnd, IDC_DEL_CONFIGURATION ), NULL,
                wndRect.right-35, 25, 30, 20, SWP_NOZORDER );

            HWND orderWnd = GetDlgItem( hWnd, IDC_FILTERS_ORDER_LIST );
            SetWindowPos( orderWnd, NULL,
                90, 50, wndRect.right-95, wndRect.bottom-85, SWP_NOZORDER );

            LVCOLUMN LvCol;
            LvCol.mask = LVCF_WIDTH;
            RECT orderRect; GetClientRect( orderWnd, &orderRect );
            LvCol.cx = orderRect.right;
            ListView_SetColumn( orderWnd, 0, &LvCol );

            SetWindowPos( GetDlgItem( hWnd, IDC_RUN_CONFIGURATION ), NULL,
                90, wndRect.bottom-30, wndRect.right-95, 25, SWP_NOZORDER );

            InvalidateRect( hWnd, NULL, TRUE );
            UpdateWindow( hWnd );
        }
        break;

    case WM_NOTIFY:
        {
            LPNMHDR pnmh = (LPNMHDR)lParam;
            switch( wParam )
            {
            case IDC_FILTERS_ORDER_LIST:
                {
                    HWND orderWnd = GetDlgItem( hWnd, (int)wParam );
                    switch( pnmh->code )
                    {

                    case LVN_KEYDOWN:
                        {
                            LPNMLVKEYDOWN lpnmc = (LPNMLVKEYDOWN)lParam;
                            switch( lpnmc->wVKey )
                            {
                            case VK_DELETE:
                            case VK_BACK:
                                SendMessage( hWnd, WM_COMMAND, IDC_FILTER_REMOVE, 0 );
                                break;
                            case VK_SUBTRACT:
                                SendMessage( hWnd, WM_COMMAND, IDC_FILTER_MOVE_UP, 0 );
                                break;
                            case VK_ADD:
                                SendMessage( hWnd, WM_COMMAND, IDC_FILTER_MOVE_DOWN, 0 );
                                break;
                            case VK_LEFT:
                                SendMessage( hWnd, WM_COMMAND, IDC_FILTER_REMOVE, 0 );
                                break;
                            case VK_RIGHT:
                                SendMessage( hWnd, WM_COMMAND, IDC_FILTER_INSERT, 0 );
                                break;
                            }
                        }
                        break;

                    case NM_CLICK:
                    case NM_DBLCLK:
                        {
                            int stageIndex = ListView_GetSelectionMark( orderWnd );
                            if( stageIndex >= 0 )
                            {
                                _OpenFilterOptions( manager, stageIndex );
                            }

                            SetFocus( orderWnd ); // so keys work and the dlg doesn't get the initial focus
                        }
                        break;

                    case LVN_GETINFOTIP:
                        {
                            LPNMLVGETINFOTIP lplvgit = (LPNMLVGETINFOTIP)pnmh;
                            const hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->getCurrentConfiguration();
                            int filterIndex = config.m_filterStages[ lplvgit->iItem ].m_index;
                            const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex(filterIndex);
                            const hctFilterDescriptor::HavokComponentMask requiredComponents = desc->getRequiredHavokComponents();

                            if( manager->checkProductSupport(requiredComponents) )
                            {
                                lplvgit->pszText = const_cast<LPSTR>( desc->getLongName() );
                            }
                            else
                            {
                                lplvgit->pszText = "This filter is not supported by the current product selection";
                            }
                        }
                        break;

                    case NM_CUSTOMDRAW:
                        {
                            LPNMLVCUSTOMDRAW lplvcd = (LPNMLVCUSTOMDRAW)pnmh;
                            switch(lplvcd->nmcd.dwDrawStage)
                            {

                            case CDDS_PREPAINT:
                                SetWindowLong(hWnd, DWLP_MSGRESULT, CDRF_NOTIFYITEMDRAW);
                                return TRUE;

                            case CDDS_ITEMPREPAINT:
                                {
                                    int index = (int)( lplvcd->nmcd.dwItemSpec );
                                    const hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->getCurrentConfiguration();
                                    int filterIndex = config.m_filterStages[index].m_index;
                                    const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex( filterIndex );
                                    const hctFilterDescriptor::HavokComponentMask requiredComponents = desc->getRequiredHavokComponents();

                                    lplvcd->clrText = manager->checkProductSupport(requiredComponents)? RGB(0,0,0): RGB(200,20,20);
                                    SetWindowLong(hWnd, DWLP_MSGRESULT, CDRF_NEWFONT );
                                }
                                return TRUE;

                            }
                        }
                        break;
                    }
                }
                break;
            }
        }
        break;

    case WM_COMMAND:
        {
            HWND comboWnd = GetDlgItem( hWnd, IDC_CONFIG_COMBO );
            HWND orderWnd = GetDlgItem( hWnd, IDC_FILTERS_ORDER_LIST );

            switch ( LOWORD(wParam) )
            {

            case IDC_CONFIG_COMBO:
                {
                    switch( HIWORD(wParam) )
                    {

                    case CBN_SELCHANGE:
                        {
                            // Make sure the current configuration really has changed and that the current one hasn't just been reselected.
                            int configIndex = (int) SendMessage( comboWnd, CB_GETCURSEL, 0, 0 );
                            if( configIndex != manager->m_configurationSet->m_currentIndex )
                            {
                                // Clear the filter options dialog.
                                _CloseCurrentFilterOptions( manager );

                                // Select and display the new configuration.
                                manager->m_configurationSet->m_currentIndex = configIndex;
                                _RefreshFilterView( orderWnd, manager );
                            }
                        }
                        break;

                    case CBN_EDITUPDATE:
                        {
                            //[EXP-2832] Chinese character support in both input/output path
                            wchar_t newName[256];
                            GetWindowTextW( comboWnd, newName, 256 );
                            // Get the current text selection
                            LRESULT sel = SendMessage( comboWnd, CB_GETEDITSEL, NULL, NULL );

                            // Update the configuration name
                            const int configIndex = manager->m_configurationSet->m_currentIndex;
                            manager->m_configurationSet->m_configurations[ configIndex ].m_configName = hkUtf8::Utf8FromWide(newName);

                            //// Update the drop list
                            SendMessageW( comboWnd, CB_DELETESTRING, configIndex, 0 ); // remove old
                            SendMessageW( comboWnd, CB_INSERTSTRING, configIndex, (LPARAM)newName ); // add new in same place
                            SendMessageW( comboWnd, CB_SETCURSEL, (WPARAM)configIndex, 0 ); // select it
                            SendMessageW( comboWnd, CB_SETEDITSEL, 0, sel ); // apply the old selection
                        }
                        break;
                    }
                }
                break;

            case IDC_NEW_CONFIGURATION:
                {
                    // Clear the filter options dialog.
                    _CloseCurrentFilterOptions( manager );

                    // Add an entry for the new configuration to the drop down.
                    const char* configName = "NewConfig";
                    int configIndex = (int) SendMessage( comboWnd, CB_ADDSTRING, 0, (LPARAM)configName );

                    // Add it to the array of filter configurations (copy the current configuration).
                    hctFilterConfigurationSet::Configuration newConfig( manager->m_configurationSet->getCurrentConfiguration() );
                    newConfig.m_configName = configName;
                    manager->m_configurationSet->m_configurations.pushBack( newConfig );

                    // Select and display the new configuration.
                    manager->m_configurationSet->m_currentIndex = configIndex;
                    SendMessage( comboWnd, CB_SETCURSEL, (WPARAM)configIndex, 0 );
                    _RefreshFilterView( orderWnd, manager );

                    Log_Info( "Added new configuration" );
                    SetFocus( comboWnd );
                }
                break;

            case IDC_DEL_CONFIGURATION:
                {
                    // Must keep at least one configuration.
                    if( manager->m_configurationSet->m_configurations.getSize() == 1 )
                    {
                        hkStringBuf str;
                        str.printf( "Could not delete %s. There must always be at least one configuration.", manager->m_configurationSet->m_configurations[0].m_configName.cString() );
                        HK_WARN_ALWAYS( 0xabba8c66, str );
                        break;
                    }

                    // Clear the filter options dialog.
                    _CloseCurrentFilterOptions( manager );

                    // Get the configuration to delete.
                    const int configIndex = manager->m_configurationSet->m_currentIndex;
                    if ( configIndex > -1 )
                    {
                        hkStringPtr configName = manager->m_configurationSet->m_configurations[configIndex].m_configName;

                        // Remove the entry from the drop down.
                        SendMessage( comboWnd, CB_DELETESTRING, (WPARAM)configIndex, 0 );

                        // Remove the filter configuration itself.
                        manager->m_configurationSet->m_configurations.removeAtAndCopy( configIndex );

                        // Select and display the zero'th configuration.
                        {
                            manager->m_configurationSet->m_currentIndex = 0;
                            SendMessage( comboWnd, CB_SETCURSEL, 0, 0 );
                            _RefreshFilterView( orderWnd, manager );
                        }

                        Log_Info( "Deleted configuration: {}", configName );
                    }
                }
                break;

            case IDC_FILTER_INSERT:
                {
                    HWND fromWnd = GetDlgItem(manager->m_filtersWnd, IDC_FILTERS_LIST);

                    // As the manager stores the optiondlg<->stageindex mapping, when we remove stages
                    // we might change that, so we should just close the option dlg to be safe.
                    _CloseCurrentFilterOptions( manager );

                    int toIndex = ListView_GetItemCount( orderWnd );
                    int curItem = ListView_GetNextItem( fromWnd, -1, LVNI_SELECTED );

                    while (curItem >= 0)
                    {
                        LVITEM LvGetItem;
                        LvGetItem.iItem = curItem;
                        LvGetItem.mask = LVIF_PARAM;
                        ListView_GetItem( fromWnd, &LvGetItem );

                        int index = (int)LvGetItem.lParam;

                        // Get the description
                        const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex(index);

                        // Add to the list view
                        LVITEM LvItem;
                        LvItem.iItem = toIndex;
                        LvItem.mask = LVIF_TEXT | LVIF_PARAM | LVIF_IMAGE;
                        LvItem.cchTextMax = 256;
                        LvItem.pszText = const_cast<LPSTR>( desc->getShortName() );
                        LvItem.lParam = index;
                        LvItem.iImage = manager->categoryToIconID(desc->getCategory());
                        LvItem.iSubItem = 0;
                        ListView_InsertItem( orderWnd, &LvItem );

                        // Add it to the actual stages
                        hctFilterConfigurationSet::FilterStage newGuy;
                        newGuy.m_index = index;
                        newGuy.m_filterId = desc->getID();
                        newGuy.m_optionDataVersion = desc->getFilterVersion();
                        manager->m_configurationSet->getCurrentConfiguration().m_filterStages.insertAt( toIndex, newGuy );

                        _OpenFilterOptions( manager, toIndex );

                        // Next
                        ++toIndex;
                        curItem = ListView_GetNextItem( fromWnd, curItem, LVNI_SELECTED);
                    }

                    // Select the last inserted list item (and no others).
                    for (int i = 0; i < ListView_GetItemCount( orderWnd ); i++ )
                    {
                        ListView_SetItemState( orderWnd, i, 0x00, 0xFF );
                    }
                    ListView_SetItemState( orderWnd, toIndex-1, LVIS_SELECTED | LVIS_FOCUSED, 0x0FF );
                    ListView_SetSelectionMark( orderWnd, toIndex-1 );
                }
                break;

            case IDC_FILTER_REMOVE:
                {
                    // As the manager stores the optiondlg<->stageindex mapping, when we remove stages
                    // we might change that, so we should just close the option dlg to be safe.
                    _CloseCurrentFilterOptions( manager );

                    INT lastIndex = -1;
                    INT nextIndex = ListView_GetNextItem( orderWnd, -1, LVNI_SELECTED );
                    while( nextIndex >= 0 )
                    {
                        // remove from array
                        manager->m_configurationSet->getCurrentConfiguration().m_filterStages.removeAtAndCopy( nextIndex );

                        // remove from list view
                        ListView_DeleteItem( orderWnd, nextIndex );

                        // next
                        lastIndex = nextIndex;
                        nextIndex = ListView_GetNextItem( orderWnd, -1, LVNI_SELECTED );
                    }

                    // Select the proceeding item
                    lastIndex = hkMath::min2<INT>( lastIndex, ListView_GetItemCount( orderWnd )-1 );
                    if( lastIndex != -1 )
                    {
                        ListView_SetItemState( orderWnd, lastIndex, LVIS_SELECTED | LVIS_FOCUSED, 0x0FF );
                        ListView_SetSelectionMark( orderWnd, lastIndex );

                        _OpenFilterOptions( manager, lastIndex );
                        SetFocus( orderWnd );
                    }
                }
                break;

            case IDC_FILTER_MOVE_UP:
            case IDC_FILTER_MOVE_DOWN:
                {
                    INT fromIndex = ListView_GetSelectionMark( orderWnd );
                    INT numItems = ListView_GetItemCount( orderWnd );

                    // As the manager stores the optiondlg<->stageindex mapping, when we remove stages
                    // we might change that, so we should just close the option dlg to be safe.
                    _CloseCurrentFilterOptions( manager );

                    // just handle single selection for the moment. Multiple
                    // disperse selection harder to move consistently:

                    INT toIndex =  ( LOWORD(wParam) == IDC_FILTER_MOVE_UP )? fromIndex - 1: fromIndex + 1;
                    if ( (toIndex >= 0) && (toIndex < numItems) )
                    {
                        LVITEM LvGetItem;
                        memset(&LvGetItem,0,sizeof(LvGetItem)); // Reset Item Struct
                        LvGetItem.mask=LVIF_PARAM;  // just the lparam (filter index)
                        LvGetItem.iSubItem = 0;
                        LvGetItem.iItem = fromIndex;
                        ListView_GetItem(orderWnd, &LvGetItem);
                        ListView_DeleteItem(orderWnd, fromIndex);

                        int fi = (int)LvGetItem.lParam;
                        const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex( fi );

                        LVITEM LvItem;
                        LvItem.iItem = toIndex;
                        LvItem.mask = LVIF_TEXT | LVIF_PARAM | LVIF_IMAGE;
                        LvItem.cchTextMax = 256;
                        LvItem.pszText = const_cast<LPSTR>( desc->getShortName() );
                        LvItem.lParam = fi;
                        LvItem.iImage = manager->categoryToIconID(desc->getCategory());
                        LvItem.iSubItem = 0;
                        ListView_InsertItem( orderWnd, &LvItem );

                        // Set the selection to so that it is easier to repeat press:
                        ListView_SetSelectionMark( orderWnd, toIndex );
                        ListView_SetItemState( orderWnd, toIndex, LVIS_SELECTED, 0x0FF ); // low byte == normal state

                        hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->getCurrentConfiguration();
                        hctFilterConfigurationSet::FilterStage temp = config.m_filterStages[toIndex];
                        config.m_filterStages[toIndex] = config.m_filterStages[fromIndex];
                        config.m_filterStages[fromIndex] = temp;
                    }

                    // Re-open the selected filter panel
                    {
                        int stageIndex = ListView_GetSelectionMark( orderWnd );
                        if( stageIndex >= 0 )
                        {
                            _OpenFilterOptions( manager, stageIndex );
                        }

                        SetFocus( orderWnd ); // so keys work and the dlg doesn't get the initial focus
                    }
                }
                break;

            case IDC_RUN_CONFIGURATION:
                {
                    if( IsDlgButtonChecked(manager->m_processingWnd, IDC_CLEAR_LOG) == BST_CHECKED )
                    {
                        manager->m_errorHandler->clearLog();
                    }

                    _CloseCurrentFilterOptions( manager ); // in case it has options to be reflected

                    manager->m_processMode = hctFilterManagerInterface::PROCESS_INTERACTIVE;
                    manager->processNoWait( manager->m_mainWnd, manager->m_configurationSet->m_currentIndex, HK_NULL );
                    manager->m_processMode = hctFilterManagerInterface::PROCESS_NONE;

                    // Re-open the selected filter panel
                    {
                        HWND orderListWnd = GetDlgItem(hWnd, IDC_FILTERS_ORDER_LIST);
                        int stageIndex = ListView_GetSelectionMark(orderListWnd);
                        if( stageIndex >= 0 )
                        {
                            _OpenFilterOptions( manager, stageIndex );
                        }

                        SetFocus(orderListWnd); // so keys work and the dlg doesn't get the initial focus
                    }
                }
                break;

            }
        }
        break;

    }

    return FALSE;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
