// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterManager/hctFilterManager.h>
#include <ContentTools/Common/Filters/FilterManager/hctFilterManagerImpl.h>
#include <ContentTools/Common/Filters/Common/Error/hctFilterError.h>
#include <ContentTools/Common/Filters/FilterManager/DllManager/hctFilterDllManagerImpl.h>

#include <Common/Serialize/Util/hkSerializeUtil.h>

#include <Common/Base/Container/String/hkUtf8.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.filtermanager"
#include <Common/Base/System/Log/hkLog.hxx>


extern HINSTANCE hInstance;

extern INT_PTR CALLBACK hkFilterManagerAvailableFiltersProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
extern INT_PTR CALLBACK hkFilterManagerFilterConfigurationsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
extern INT_PTR CALLBACK hkFilterManagerFilterOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
extern INT_PTR CALLBACK hkFilterManagerProcessingProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

extern void _SetupConfigCombo( HWND combo, hctFilterManagerImpl* manager );
extern void _CloseCurrentFilterOptions( hctFilterManagerImpl* manager );
extern void _OpenFilterOptions( hctFilterManagerImpl* manager, int stageIndex );


/*static*/ int hctFilterManagerImpl::categoryToIconID (hctFilterDescriptor::FilterCategory category)
{
    switch (category)
    {
        case hctFilterDescriptor::HK_CATEGORY_CORE:                     return 1;
        case hctFilterDescriptor::HK_CATEGORY_COLLISION_DEPRECATED:     return 4;
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS_2012:             return 4;
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS:                  return 4;
        case hctFilterDescriptor::HK_CATEGORY_FX_PHYSICS_DEPRECATED:    return 0;
        case hctFilterDescriptor::HK_CATEGORY_ANIMATION:                return 0;
        case hctFilterDescriptor::HK_CATEGORY_CLOTH:                    return 6;
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION_2012:         return 7;
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION:              return 7;
        case hctFilterDescriptor::HK_CATEGORY_GRAPHICS:                 return 3;
        case hctFilterDescriptor::HK_CATEGORY_USER:                     return 5;
    }

    HK_ASSERT(0x17d29011, 0, "Internal error - invalid category");
    return 0;
}

//
// Utils
//

hctFilterDescriptor::FilterCategory _tabToCategory (int tabIdx)
{
    switch (tabIdx)
    {
        case 0: return hctFilterDescriptor::HK_CATEGORY_CORE;
        case 1: return hctFilterDescriptor::HK_CATEGORY_PHYSICS_2012;
        //case 2: return hctFilterDescriptor::HK_CATEGORY_PHYSICS;
        case 2: return hctFilterDescriptor::HK_CATEGORY_ANIMATION;
        case 3: return hctFilterDescriptor::HK_CATEGORY_CLOTH;
        case 4: return hctFilterDescriptor::HK_CATEGORY_DESTRUCTION_2012;
        case 5: return hctFilterDescriptor::HK_CATEGORY_GRAPHICS;
        case 6: return hctFilterDescriptor::HK_CATEGORY_USER;

        default:
            HK_ASSERT(0x622f8b9b, 0, "Internal error - invalid category tab");
            return hctFilterDescriptor::HK_CATEGORY_CORE;
    }
}

int _categoryToTab (hctFilterDescriptor::FilterCategory category)
{
    switch (category)
    {
        case hctFilterDescriptor::HK_CATEGORY_COLLISION_DEPRECATED: return 1;
        case hctFilterDescriptor::HK_CATEGORY_FX_PHYSICS_DEPRECATED:return 0;

        case hctFilterDescriptor::HK_CATEGORY_CORE:                 return 0;
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS_2012:         return 1;
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS:              return 1;   // Share with Physics 2012
        case hctFilterDescriptor::HK_CATEGORY_ANIMATION:            return 2;
        case hctFilterDescriptor::HK_CATEGORY_CLOTH:                return 3;
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION_2012:     return 4;
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION:          return 4;   // Share with Destruction 2012
        case hctFilterDescriptor::HK_CATEGORY_GRAPHICS:             return 5;
        case hctFilterDescriptor::HK_CATEGORY_USER:                 return 6;

        default:
            HK_ASSERT(0x3763bf77, 0, "Internal error - invalid category");
            return 0;
    }
}

void _FillCategory( hctFilterManagerImpl* manager, hctFilterDescriptor::FilterCategory cat, HWND listWnd )
{
    ListView_DeleteAllItems( listWnd );

    LVCOLUMN col;
    memset(&col, 0, sizeof(col));
    col.mask = LVCF_TEXT;
    hkStringOld str = hctFilterUtils::filterCategoryToString( cat );
    str += " Filters";
    col.pszText = (LPSTR) str.cString();
    ListView_SetColumn( listWnd, 0, &col );

    int catTab = _categoryToTab(cat);

    int curItem = 0;
    int filterNum = 0;
    while( true )
    {
        const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex(filterNum);
        if( !desc )
        {
            break;
        }

        int filterTab = _categoryToTab( desc->getCategory() );
        if( filterTab == catTab )
        {
            LVITEM LvItem;
            LvItem.iItem = curItem;

            // Filter name
            LvItem.iSubItem = 0;
            LvItem.mask = LVIF_TEXT | LVIF_PARAM;
            LvItem.cchTextMax = 256;
            LvItem.pszText = const_cast<LPSTR>( desc->getShortName() );
            LvItem.lParam = filterNum;
            ListView_InsertItem( listWnd, &LvItem );

            // Version
            LvItem.iSubItem = 1;
            LvItem.mask = LVIF_TEXT;
            LvItem.cchTextMax = 256;
            hkStringOld s;
            int version = desc->getFilterVersion();
            s.printf("%d.%d.%d", (version >> 16), ((version & 0xff00) >> 8), (version & 0xff) );
            LvItem.pszText = const_cast<LPSTR>( s.cString() );
            ListView_SetItem(listWnd, &LvItem);

            ++curItem;
        }

        ++filterNum;
    }
}

void _RefreshFilterView( HWND listView, hctFilterManagerImpl* manager )
{
    INT ni = ListView_GetItemCount( listView );
    while( ni != 0 )
    {
        ListView_DeleteItem(listView, 0);
        ni--;
    }

    for (int i=0; i < manager->m_configurationSet->getCurrentConfiguration().m_filterStages.getSize(); ++i)
    {
        int filterNum = manager->m_configurationSet->getCurrentConfiguration().m_filterStages[i].m_index;
        const hctFilterDescriptor* desc = manager->m_dllManager->getFilterDescByIndex( filterNum );

        LVITEM LvItem;
        LvItem.iItem = i;
        LvItem.mask = LVIF_TEXT | LVIF_PARAM | LVIF_IMAGE;
        LvItem.cchTextMax = 256;
        LvItem.pszText = const_cast<LPSTR>( desc->getShortName() );
        LvItem.lParam = filterNum;
        LvItem.iImage = manager->categoryToIconID(desc->getCategory());
        LvItem.iSubItem = 0;

        ListView_InsertItem( listView, &LvItem );
    }
}


//
// Menu items
//

static void _LoadConfigurationSet( void*& configurationSetData, int& configurationSetDataSize, HWND owner )
{
    configurationSetDataSize = 0;
    configurationSetData = HK_NULL;

    // Have to use Unicode version (in Max etc.)
    OPENFILENAMEW ofn;
    wchar_t szFile[1024];

    // Initialize OPENFILENAME
    ZeroMemory(&ofn, sizeof(OPENFILENAMEW));
    ZeroMemory(szFile, sizeof(szFile));
    ofn.lStructSize = sizeof(OPENFILENAMEW);
    ofn.hwndOwner = owner;
    ofn.lpstrFile = szFile;
    ofn.nMaxFile = HK_COUNT_OF(szFile);
    ofn.lpstrFilter = L"All Files\0*.*\0HKO Files\0*.hko\0";
    ofn.nFilterIndex = 2; // use .hko as default
    ofn.lpstrTitle = L"Load Configuration Set";
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrDefExt = L"hko";
    ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST; //  | OFN_ENABLESIZING;

    // Display the load dialog box.
    if ( GetOpenFileNameW(&ofn)==TRUE )
    {
        hkUtf8::Utf8FromWide szFileUtf8(szFile);

        HANDLE hFile = CreateFileW( szFile, GENERIC_READ, 0, NULL,
            OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

        if (hFile != INVALID_HANDLE_VALUE)
        {
            DWORD dwSize = GetFileSize(hFile, NULL);
            if (dwSize != INVALID_FILE_SIZE)
            {
                configurationSetData = hkAllocateChunk<char>( dwSize, HK_MEMORY_CLASS_EXPORT);
                if ( ReadFile(hFile, configurationSetData, dwSize, &dwSize, NULL) )
                {
                    configurationSetDataSize = dwSize;
                }
                else
                {
                    hkDeallocateChunk<char>((char*)configurationSetData, dwSize, HK_MEMORY_CLASS_EXPORT);
                    configurationSetData = HK_NULL;
                }
            }
            CloseHandle(hFile);

            Log_Info( "Loaded configuration set from '{}'.", szFileUtf8.cString() );
        }
        else
        {
            HK_WARN_ALWAYS( 0xabba6527, "Failed to load configuration set from file " << szFileUtf8.cString() << "." );
        }
    }
}

static void _SaveConfigurationSet( const void* configurationSetData, int configurationSetDataSize, HWND owner )
{
    // Have to use Unicode version (in Max etc.)
    OPENFILENAMEW ofn;
    wchar_t szFile[1024];

    // Initialize OPENFILENAME
    ZeroMemory(&ofn, sizeof(OPENFILENAMEW));
    ZeroMemory(szFile, sizeof(szFile));
    ofn.lStructSize = sizeof(OPENFILENAMEW);
    ofn.hwndOwner = owner;
    ofn.lpstrFile = szFile;
    ofn.nMaxFile = HK_COUNT_OF(szFile);
    ofn.lpstrFilter = L"All Files\0*.*\0HKO Files\0*.hko\0\0";
    ofn.nFilterIndex = 2; // use .hko as default
    ofn.lpstrTitle = L"Save Configuration Set";
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrDefExt = L"hko";
    ofn.Flags = OFN_EXPLORER | OFN_OVERWRITEPROMPT; //| OFN_ENABLESIZING;

    // Display the Save dialog box.
    if ( GetSaveFileNameW(&ofn)==TRUE )
    {
        hkUtf8::Utf8FromWide szFileUtf8(szFile);

        HANDLE hFile = CreateFileW( szFile, GENERIC_WRITE, 0, NULL,
            CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

        if (hFile != INVALID_HANDLE_VALUE)
        {
            DWORD numWrite = 0;
            WriteFile(hFile, configurationSetData, configurationSetDataSize, &numWrite, NULL);

            //  if (!wrote || (numWrite < src.m_memSize) )
            //  {
            //      OutputDebugString("Havok: Could not write filtered result to file.");
            //  }

            CloseHandle(hFile);

            Log_Info( "Saved the current configuration set to file: '{}'.", szFileUtf8.cString() );
        }
        else
        {
            HK_WARN_ALWAYS( 0xabba3115, "Failed to save the current configuration set to file " << szFileUtf8.cString() << "." );
        }
    }
}

static void _LoadPresetConfigurationSet( hkStringOld fileName, void*& configurationSetData, int& configurationSetDataSize, HWND owner )
{
    configurationSetDataSize = 0;
    configurationSetData = HK_NULL;

    HANDLE hFile = CreateFileW( hkUtf8::WideFromUtf8(fileName.cString()), GENERIC_READ, 0, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

    if (hFile != INVALID_HANDLE_VALUE)
    {
        DWORD dwSize = GetFileSize(hFile, NULL);
        if (dwSize != INVALID_FILE_SIZE)
        {
            configurationSetData = hkAllocateChunk<char>( dwSize, HK_MEMORY_CLASS_EXPORT);
            if ( ReadFile(hFile, configurationSetData, dwSize, &dwSize, NULL) )
            {
                configurationSetDataSize = dwSize;
            }
            else
            {
                hkDeallocateChunk<char>((char*)configurationSetData, dwSize, HK_MEMORY_CLASS_EXPORT);
                configurationSetData = HK_NULL;
            }
        }
        CloseHandle(hFile);

        Log_Info( "Loaded configuration set from '{}'.", fileName.cString() );
    }
    else
    {
        HK_WARN_ALWAYS( 0xabba6527, "Failed to load configuration set from file " << fileName.cString() << "." );
    }
}


static void _SaveSupportData( HWND owner, const hctFilterManagerImpl* manager )
{
    // Have to use Unicode version (in Max etc.)
    OPENFILENAMEW ofn;
    wchar_t szFile[1024];

    // Initialize OPENFILENAME
    ZeroMemory(&ofn, sizeof(OPENFILENAMEW));
    ZeroMemory(szFile, sizeof(szFile));
    ofn.lStructSize = sizeof(OPENFILENAMEW);
    ofn.hwndOwner = owner;
    ofn.lpstrFile = szFile;
    ofn.nMaxFile = HK_COUNT_OF(szFile);
    ofn.lpstrFilter = L"All Files\0*.*\0\0";
    ofn.nFilterIndex = 1; // use all files as default
    ofn.lpstrTitle = L"Save FilterSet";
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrDefExt = NULL;
    ofn.Flags = OFN_EXPLORER | OFN_OVERWRITEPROMPT; //| OFN_ENABLESIZING;

    // Display the Save dialog box.
    if ( GetSaveFileNameW(&ofn)==TRUE )
    {
        hkUtf8::Utf8FromWide szFileUtf8(szFile);

        // Save raw data
        {
            hkStringOld dataFile(szFileUtf8);
            dataFile += ".support.hkt";

            hkOstream dataF(dataFile.cString());
            if ( dataF.isOk() )
            {
                hkResult res = hkSerializeUtil::saveTagfile(manager->getOriginalContents(), dataF.getStreamWriter());

                if(res.isSuccess())
                {
                    Log_Info( "Saved support data as \"{}.support.hkt\".", szFileUtf8.cString() );
                }
                else
                {
                    HK_WARN_ALWAYS( 0xabba9184, "Could not save support data." );
                }
            }
            else
            {
                HK_WARN_ALWAYS( 0xabba9184, "Could not save support data." );
            }
        }

        // Save filter options
        {
            hkStringOld dataFile(szFileUtf8);
            dataFile += ".support.hko";

            hkOstream dataF(dataFile.cString());
            if ( dataF.isOk() )
            {
                int bufSize = manager->m_configurationSet->getData( HK_NULL );
                char* buf = hkAllocateChunk<char>( bufSize, HK_MEMORY_CLASS_EXPORT);
                manager->m_configurationSet->getData( buf );
                dataF.write( buf, bufSize );
                hkDeallocateChunk<char>( buf, bufSize, HK_MEMORY_CLASS_EXPORT);

                Log_Info( "Saved support options as \"{}.support.hko\".", szFileUtf8.cString() );
            }
            else
            {
                HK_WARN_ALWAYS( 0xabba2713, "Could not save support options." );
            }
        }
    }
}

INT_PTR CALLBACK hkAboutDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message)
    {
        case WM_INITDIALOG:
            {
                SetWindowText( GetDlgItem(hWnd, IDC_LAB_SDK_VERSION), HCT_CURRENT_VERSION_STRING);
            }
            break;
        case WM_COMMAND:
            {
                switch ( LOWORD(wParam) )
                {
                case IDOK:
                case IDCANCEL:
                    EndDialog(hWnd, wParam);
                    return TRUE;
                }
            }
            break;
    }
    return FALSE;
}


inline void _setMenuItem( HWND hWnd, UINT id, BOOL state )
{
    MENUITEMINFO mii; memset(&mii, 0, sizeof(MENUITEMINFO));
    mii.cbSize = sizeof(MENUITEMINFO);
    mii.fMask = MIIM_STATE;
    mii.fState = state? MFS_CHECKED : MFS_UNCHECKED;
    SetMenuItemInfo( GetMenu(hWnd), id, FALSE, &mii);
}

void _refreshProductMenu (HWND hWnd, hctFilterManagerImpl* manager)
{
    hctFilterDescriptor::HavokComponentMask supportedComponents = manager->getAvailableHavokComponents();

    _setMenuItem( hWnd, ID_PRODUCT_HAVOKANIMATION, supportedComponents & hctFilterDescriptor::HK_COMPONENT_ANIMATION );
    _setMenuItem( hWnd, ID_PRODUCT_HAVOKPHYSICS, supportedComponents &  hctFilterDescriptor::HK_COMPONENT_PHYSICS );
    _setMenuItem( hWnd, ID_PRODUCT_HAVOKCLOTH, supportedComponents & hctFilterDescriptor::HK_COMPONENT_CLOTH );
    _setMenuItem( hWnd, ID_PRODUCT_HAVOKDESTRUCTION, supportedComponents & hctFilterDescriptor::HK_COMPONENT_DESTRUCTION );
    _setMenuItem( hWnd, ID_PRODUCT_HAVOKAI, supportedComponents & hctFilterDescriptor::HK_COMPONENT_AI );

    const char* productName = hctFilterManagerImpl::getProductNameBasedOnComponents(supportedComponents);
    ModifyMenu (GetMenu(hWnd), ID_PRODUCT_TEXT, MF_BYCOMMAND | MF_STRING | MF_GRAYED, ID_PRODUCT_TEXT, productName );
}

void _searchDirForHKOFiles(HMENU parentMenu, hkStringOld parentDir, hkStringOld subDir, hkArray<hkStringOld> *filePaths)
{
    const hkStringOld fileType = ".HKO";

    WIN32_FIND_DATAW findFileData;

    hkStringOld findFileStr = hkStringOld(parentDir + subDir + "\\*.*");
    HANDLE hFind = FindFirstFileW(hkUtf8::WideFromUtf8(findFileStr.cString()), &findFileData );

    if(hFind == INVALID_HANDLE_VALUE)
    {
        return;
    }
    // go through the file names in the current directory
    do
    {
        const hkStringOld fileName(hkUtf8::Utf8FromWide(findFileData.cFileName));

        if (findFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
        {
            if(fileName != "." && fileName != "..")
            {
                hkStringOld newSubDir = hkStringOld(fileName);
                hkStringOld newParentDir = hkStringOld(parentDir + subDir + "\\");

                HMENU newDirMenu = CreatePopupMenu();
                AppendMenuW(parentMenu, MF_POPUP|MF_STRING|MF_ENABLED, (UINT_PTR)newDirMenu, findFileData.cFileName);
                _searchDirForHKOFiles(newDirMenu, newParentDir, newSubDir, filePaths);
            }
        }
        else
        {
            if(fileName.asUpperCase().endsWith(fileType.cString()))
            {
                UINT cmdId = (ID_PRESET_CONFIG_START + filePaths->getSize());
                AppendMenuW(parentMenu, MF_STRING | MF_ENABLED, cmdId, findFileData.cFileName);

                hkStringOld filePath(parentDir + subDir + "\\" + fileName);
                filePaths->pushBack(filePath);
            }
        }
    }
    while (FindNextFileW(hFind, &findFileData));
}

void _refreshPresetConfigurationMenuItem (HWND hWnd, hctFilterManagerImpl* manager)
{
    manager->m_HKOFilePaths.clear();

    HMENU hFileMenu = GetSubMenu(GetMenu(hWnd), 0);

    // Delete the menu (and it sub-menus) if it already exists
    if(manager->m_PresetConfigMenu != HK_NULL)
    {
        DeleteMenu(hFileMenu, (UINT) manager->m_PresetConfigMenu, MF_BYCOMMAND);
    }

    manager->m_PresetConfigMenu = CreatePopupMenu();
    AppendMenu(hFileMenu, MF_STRING | MF_POPUP | MF_ENABLED, (UINT_PTR) manager->m_PresetConfigMenu, "&Preset Configurations");

    // Recursively search for any .HKO files in the provided path and add a menu for each one
    _searchDirForHKOFiles(manager->m_PresetConfigMenu, hkStringOld( manager->m_havokPath + "\\"), hkStringOld("configurations"), &manager->m_HKOFilePaths);
}
//
// Main dialog
//

INT_PTR CALLBACK hkFilterManagerImplDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctFilterManagerImpl *manager = reinterpret_cast<hctFilterManagerImpl*>( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA) );

    // dialog sizes
    static RECT filtersRect;
    static RECT configurationRect;
    static RECT optionsRect;
    static RECT processingRect;

    // horizontal splitter
    static int splitterPos = 0;
    static bool overSplitter = false;
    static bool draggingSplitter = false;
    static const int splitterSize = 6;
    static HCURSOR arrowCursor = LoadCursor( NULL, IDC_SIZENS );

    switch(message)
    {
        case WM_INITDIALOG:
        {
            SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR)lParam); // so that it can be retrieved later
            manager = (hctFilterManagerImpl*)lParam;

            manager->m_mainWnd = hWnd;

            // Add child dialogs
            manager->m_filtersWnd = CreateDialogParamW( hInstance, MAKEINTRESOURCEW(IDD_AVAILABLE_FILTERS),
                                    hWnd, hkFilterManagerAvailableFiltersProc, (LPARAM)manager);
            manager->m_configurationWnd =   CreateDialogParamW( hInstance, MAKEINTRESOURCEW(IDD_FILTER_CONFIGURATION),
                                    hWnd, hkFilterManagerFilterConfigurationsProc, (LPARAM)manager );
            manager->m_optionsWnd = CreateDialogParamW( hInstance, MAKEINTRESOURCEW(IDD_FILTER_OPTIONS),
                                    hWnd, hkFilterManagerFilterOptionsProc, (LPARAM)manager );
            manager->m_processingWnd =  CreateDialogParamW( hInstance, MAKEINTRESOURCEW(IDD_PROCESSING),
                                    hWnd, hkFilterManagerProcessingProc, (LPARAM)manager );

            // Set window icon
            SendMessage( hWnd, WM_SETICON, ICON_SMALL, (LPARAM)LoadImage( hInstance, MAKEINTRESOURCE(IDI_HAVOKLOGO), IMAGE_ICON, 16, 16, 0 ) );
            SendMessage( hWnd, WM_SETICON, ICON_BIG, (LPARAM)LoadImage( hInstance, MAKEINTRESOURCE(IDI_HAVOKLOGO), IMAGE_ICON, 32, 32, 0 ) );

            // Set window size to fit child dialogs (adding extra for title, border, etc)
            GetClientRect( manager->m_filtersWnd, &filtersRect );
            GetClientRect( manager->m_configurationWnd, &configurationRect );
            GetClientRect( manager->m_optionsWnd, &optionsRect );
            GetClientRect( manager->m_processingWnd, &processingRect );
            const int totalWidth = filtersRect.right + configurationRect.right + optionsRect.right + 20;
            const int totalHeight = configurationRect.bottom + processingRect.bottom + splitterSize + 80;

            // Center the window - Assume it was centered with the original size
            RECT currentRect;
            GetWindowRect(hWnd, &currentRect);
            const int screenCenterX = (currentRect.left + currentRect.right) / 2;
            const int screenCenterY = (currentRect.top + currentRect.bottom) / 2;
            const int posX = screenCenterX - (totalWidth / 2 );
            const int posY = screenCenterY - (totalHeight / 2);
            SetWindowPos( hWnd, NULL, posX, posY,
                totalWidth,
                totalHeight,
                SWP_NOZORDER );

            // Start the splitter a bit below the configuration window
            splitterPos = configurationRect.bottom + 40;

            // Setup the log window
            manager->m_errorHandler->setWindow( GetDlgItem(manager->m_processingWnd, IDC_PROGRESS_LOG) );
            manager->m_errorHandler->refreshWindow();

            // Set the preset configuration files menu item
            _refreshPresetConfigurationMenuItem(hWnd, manager);

            // Set the product check menu item
            _refreshProductMenu(hWnd, manager);

            // Set initial status text
            SetWindowText( GetDlgItem( hWnd, IDC_STATUS_TEXT ), manager->m_status.cString() );

            Log_Info( "Filter Manager ready" );
        }
        // FALL THROUGH

        case WM_SIZE:
            {
                RECT dialogRect;
                GetClientRect( hWnd, (LPRECT)&dialogRect );
                int x,y,cx,cy;

                // Limit the window size
                dialogRect.right = hkMath::max2<int>( dialogRect.right, filtersRect.right + configurationRect.right + optionsRect.right );
                dialogRect.bottom = hkMath::max2<int>( dialogRect.bottom, configurationRect.bottom + processingRect.bottom + splitterSize + 20 );

                // Limit the splitter position
                splitterPos = hkMath::min2<int>( splitterPos, dialogRect.bottom - 120 );
                splitterPos = hkMath::max2<int>( splitterPos, configurationRect.bottom );

                // Child windows

                x = 0;
                y = 0;
                cx = filtersRect.right;
                cy = splitterPos;
                SetWindowPos( manager->m_filtersWnd, NULL, x, y, cx, cy, SWP_NOZORDER );

                x = filtersRect.right;
                y = 0;
                cx = ( dialogRect.right - optionsRect.right ) - filtersRect.right;
                cy = splitterPos;
                SetWindowPos( manager->m_configurationWnd, NULL, x, y, cx, cy, SWP_NOZORDER );

                x = dialogRect.right - optionsRect.right;
                y = 0;
                cx = optionsRect.right;
                cy = dialogRect.bottom - 20;
                SetWindowPos( manager->m_optionsWnd, NULL, x, y, cx, cy, SWP_NOZORDER );

                x = 0;
                y = splitterPos + splitterSize;
                cx = dialogRect.right - optionsRect.right;
                cy = ( ( dialogRect.bottom - splitterPos ) - splitterSize ) - 20;
                SetWindowPos( manager->m_processingWnd, NULL, x, y, cx, cy, SWP_NOZORDER );

                // Status bar

                x = 0;
                y = dialogRect.bottom-20;
                cx = dialogRect.right;
                cy = 20;
                SetWindowPos( GetDlgItem( hWnd, IDC_STATUS_TEXT ), NULL, x, y, cx, cy,  SWP_NOZORDER );
            }
            break;

        case WM_MOUSEMOVE:
            {
                // Check if we are over the splitter
                POINTS p = MAKEPOINTS( lParam );
                RECT dialogRect; GetClientRect( hWnd, (LPRECT)&dialogRect );
                overSplitter = ( ( p.x < dialogRect.right - optionsRect.right ) && p.y >= splitterPos && p.y <= splitterPos + splitterSize );

                if( overSplitter )
                {
                    SetCursor( arrowCursor );
                }
                if( draggingSplitter )
                {
                    splitterPos = p.y;
                    hkFilterManagerImplDlgProc( hWnd, WM_SIZE, NULL, NULL );
                }
            }
            break;

        case WM_LBUTTONDOWN:
            {
                draggingSplitter = overSplitter;
                if( draggingSplitter )
                {
                    SetCapture( hWnd );
                }
            }
            break;

        case WM_LBUTTONUP:
            {
                if( draggingSplitter )
                {
                    ReleaseCapture();
                }
                draggingSplitter = false;
            }
            break;

        case WM_KEYDOWN:
            {
                switch (wParam)
                {
                case VK_ESCAPE:
                    SendMessage( manager->m_processingWnd, WM_COMMAND, IDCANCEL, 0 );
                    return TRUE;
                }
            }
            break;

        case WM_COMMAND:
            switch ( LOWORD(wParam) )
            {
                // Handle all close/cancel messages here
                case IDC_CLOSE:
                case IDCANCEL:
                    {
                        // This stores current changes in the filter
                        _CloseCurrentFilterOptions( manager );

                        // Try to close all modeless filters
                        if( !manager->m_dllManager->closeAndWaitForModelessFilters( manager->m_mainWnd ) )
                        {
                            return TRUE;
                        }

                        // If there have been any changes to the filter options, ask the user what to do
                        int buttonPressed = LOWORD(wParam);
                        {
                            // Now, compare with previous filter options
                            if( manager->m_configurationSet->hasDataChanged() )
                            {
                                // Ask for confirmation and possibly save changes
                                int confButton = MessageBox(hWnd, "Do you want to save changes to the configuration(s)?", "Save Changes", MB_YESNOCANCEL|MB_DEFBUTTON3);
                                switch (confButton)
                                {
                                    case IDNO:
                                        buttonPressed = IDC_CANCEL;
                                        break;
                                    case IDYES:
                                        buttonPressed = IDC_CLOSE;
                                        break;
                                    case IDCANCEL:
                                        return TRUE;
                                }
                            }
                        }

                        manager->m_errorHandler->clearLog();
                        EndDialog( hWnd, buttonPressed );
                    }
                    break;

                case ID_LOAD_CONFIGURATIONSET:  // Load a configuration set from file
                    {
                        // This stores current changes in the filter
                        _CloseCurrentFilterOptions( manager );

                        bool shouldLoad = true;

                        // If loading a configuration set and there has been any changes, ask the user what to do
                        // Now, compare with previous filter options
                        if( manager->m_configurationSet->hasDataChanged() )
                        {
                            // Ask for confirmation and possibly save changes
                            int confButton = MessageBox(hWnd, "Changes have been made to the configuration(s) without saving. Are you sure you want to continue?", "Warning", MB_YESNO|MB_DEFBUTTON2);
                            switch (confButton)
                            {
                            case IDNO:
                                shouldLoad = false;
                                break;
                            case IDYES:
                                shouldLoad = true;
                                break;
                            }
                        }

                        if (shouldLoad)
                        {
                            void* dataPtr = HK_NULL;
                            int dataSize=0;
                            _LoadConfigurationSet( dataPtr, dataSize, hWnd );

                            if( dataPtr )
                            {
                                manager->setConfigurationSet( dataPtr, dataSize );
                                hkDeallocateChunk<char>( (char*)dataPtr, dataSize, HK_MEMORY_CLASS_EXPORT );

                                // Populate the combo box.
                                _SetupConfigCombo( GetDlgItem( manager->m_configurationWnd, IDC_CONFIG_COMBO ), manager );

                                // Show the active configuration.
                                _RefreshFilterView( GetDlgItem( manager->m_configurationWnd, IDC_FILTERS_ORDER_LIST ), manager );
                            }
                        }
                        else
                        {
                            return TRUE;
                        }
                    }
                    break;

                case ID_SAVE_CONFIGURATIONSET: // Save the configuration set to a file
                    {
                        // Ensure any changes to the current options are saved.
                        _CloseCurrentFilterOptions( manager );

                        int bufSize = manager->m_configurationSet->getData( HK_NULL );
                        char* buf = hkAllocateChunk<char>( bufSize, HK_MEMORY_CLASS_EXPORT);
                        manager->m_configurationSet->getData( buf );

                        _SaveConfigurationSet( buf, bufSize, hWnd );
                        hkDeallocateChunk<char>( buf, bufSize, HK_MEMORY_CLASS_EXPORT );

                        // refresh the preset configuration menu
                        _refreshPresetConfigurationMenuItem(hWnd, manager);
                    }
                    break;

                case ID_PRODUCT_HAVOKANIMATION:
                case ID_PRODUCT_HAVOKPHYSICS:
                case ID_PRODUCT_HAVOKCLOTH:
                case ID_PRODUCT_HAVOKDESTRUCTION:
                case ID_PRODUCT_HAVOKAI:
                    {
                        hctFilterDescriptor::HavokComponent component = hctFilterDescriptor::HK_COMPONENT_COMMON;
                        switch ( LOWORD(wParam) )
                        {
                            case ID_PRODUCT_HAVOKPHYSICS:
                                component = hctFilterDescriptor::HK_COMPONENT_PHYSICS;
                                break;
                            case ID_PRODUCT_HAVOKANIMATION:
                                component = hctFilterDescriptor::HK_COMPONENT_ANIMATION;
                                break;
                            case ID_PRODUCT_HAVOKCLOTH:
                                component = hctFilterDescriptor::HK_COMPONENT_CLOTH;
                                break;
                            case ID_PRODUCT_HAVOKDESTRUCTION:
                                component = hctFilterDescriptor::HK_COMPONENT_DESTRUCTION;
                                break;
                            case ID_PRODUCT_HAVOKAI:
                                component = hctFilterDescriptor::HK_COMPONENT_AI;
                                break;
                        }

                        hctFilterDescriptor::HavokComponentMask availableComponents = manager->getAvailableHavokComponents();

                        availableComponents ^= component;

                        manager->setAvailableHavokComponents(availableComponents);

                        _refreshProductMenu(hWnd, manager);

                        // Refresh all list views
                        HWND tabWnd = GetDlgItem( manager->m_filtersWnd, IDC_CAT_TAB );
                        HWND listWnd = GetDlgItem( manager->m_filtersWnd, IDC_FILTERS_LIST );
                        HWND orderWnd = GetDlgItem( manager->m_configurationWnd, IDC_FILTERS_ORDER_LIST );
                        _FillCategory( manager, _tabToCategory( TabCtrl_GetCurSel(tabWnd) ), listWnd );
                        _RefreshFilterView( orderWnd, manager );
                    }
                    break;

                case ID_HELP_HELP:
                    {
                        manager->showHelp(hWnd);
                    }
                    break;

                case ID_HELP_ABOUT:
                    {
                        DialogBoxParamW(hInstance, MAKEINTRESOURCEW(IDD_ABOUT_DLG),hWnd, hkAboutDlgProc,(LPARAM)manager);
                    }
                    break;

                case ID_SUPPORTDATA_SAVE:
                    {
                        // Save the temp filter set to file.
                        _SaveSupportData(hWnd, manager ); // fs will be cleaned up by manager later
                    }
                    break;

                default:
                    if(LOWORD(wParam) >= ID_PRESET_CONFIG_START && LOWORD(wParam) <= ID_PRESET_CONFIG_END)
                    {
                        int filePathID = LOWORD(wParam) - ID_PRESET_CONFIG_START;

                        if( filePathID >= 0 && filePathID < manager->m_HKOFilePaths.getSize())
                        {
                            // Ensure any changes to the current options are saved.
                            _CloseCurrentFilterOptions( manager );

                            bool shouldLoad = true;

                            // If loading a configuration set and there has been any changes, ask the user what to do
                            // Now, compare with previous filter options
                            if( manager->m_configurationSet->hasDataChanged() )
                            {
                                // Ask for confirmation and possibly save changes
                                int confButton = MessageBox(hWnd, "Changes have been made to the configuration(s) without saving. Are you sure you want to continue?", "Warning", MB_YESNO|MB_DEFBUTTON2);
                                switch (confButton)
                                {
                                case IDNO:
                                    shouldLoad = false;
                                    break;
                                case IDYES:
                                    shouldLoad = true;
                                    break;
                                }
                            }

                            if (shouldLoad)
                            {
                                void* dataPtr = HK_NULL;
                                int dataSize=0;
                                _LoadPresetConfigurationSet(manager->m_HKOFilePaths[filePathID], dataPtr, dataSize, hWnd );

                                if( dataPtr )
                                {
                                    manager->setConfigurationSet( dataPtr, dataSize );
                                    hkDeallocateChunk<char>( (char*)dataPtr, dataSize, HK_MEMORY_CLASS_EXPORT );

                                    // Populate the combo box.
                                    _SetupConfigCombo( GetDlgItem( manager->m_configurationWnd, IDC_CONFIG_COMBO ), manager );

                                    // Show the active configuration.
                                    _RefreshFilterView( GetDlgItem( manager->m_configurationWnd, IDC_FILTERS_ORDER_LIST ), manager );
                                }
                            }
                            else
                            {
                                return TRUE;
                            }
                        }
                    }
                    break;
            }
            break;

        case WM_DESTROY:
            {
                manager->m_dllManager->waitForModelessFilters();
            }
            break;
    }
    return FALSE;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
