// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterAsset/hctFilterAsset.h>
#include <ContentTools/Common/Filters/FilterAsset/PruneTypes/hctPruneTypesFilter.h>

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Camera/hkxCamera.h>
#include <Common/SceneData/Light/hkxLight.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>
#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>
#include <Common/SceneData/Selection/hkxNodeSelectionSet.h>

#if defined(HK_FEATURE_PRODUCT_ANIMATION)
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Animation/hkaAnimation.h>
#include <Animation/Animation/Animation/hkaAnnotationTrack.h>
#include <Animation/Animation/Rig/hkaBoneAttachment.h>
#include <Animation/Animation/Animation/hkaAnimationBinding.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapper.h>
#endif

#if defined(HK_FEATURE_PRODUCT_ANIMATION) && defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
#include <Animation/Physics2012Bridge/Instance/hkaRagdollInstance.h>
#endif

#if defined(HK_FEATURE_PRODUCT_CLOTH)
#include <Cloth/Cloth/Container/hclClothContainer.h>
#endif

#include <Common/Base/Container/PointerMap/hkPointerMap.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#include <Common/SceneData/Environment/hkxEnvironment.h>

#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.asset.prune"
#include <Common/Base/System/Log/hkLog.hxx>


hctPruneTypesFilterDesc g_pruneTypesDesc;

hctPruneTypesFilter::hctPruneTypesFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface(owner),
    m_optionsDialog(NULL), m_fillingControls (false)
{
    m_customClasses = "";
    m_selectionSets = "";
}

hctPruneTypesFilter::~hctPruneTypesFilter()
{
}

static void _NullAllMeshReferences(hkxNode* node)
{
    if ( node->hasA<hkxMesh>() || node->hasA<hkxSkinBinding>() )
    {
        node->m_object = HK_NULL;
    }

    for (int c=0; c < node->m_children.getSize(); ++c)
    {
        _NullAllMeshReferences(node->m_children[c]);
    }
}

static void _ZeroMeshChannels(hkxMesh* mesh)
{
    for (int s=0; s<mesh->m_sections.getSize(); s++)
    {
        mesh->m_sections[s]->m_userChannels.setSize(0);
    }
    mesh->m_userChannelInfos.setSize(0);
}

static int _CountNodes(hkxNode* node)
{
    int cc = 1; // node count for reports
    for (int c=0; c < node->m_children.getSize(); ++c)
    {
        cc += _CountNodes(node->m_children[c]);
    }
    return cc;
}

static void _removeAttributesRecursive (hkxNode* theNode)
{
    if (!theNode) return;

    theNode->m_attributeGroups.setSize(0);

    for (int i=0; i<theNode->m_children.getSize(); ++i)
    {
        _removeAttributesRecursive(theNode->m_children[i]);
    }
}


/****************************************************************************/
/*                       Node selection set pruning                         */
/****************************************************************************/

void _makeSelectionSetNamesUnique( hkArray<hkStringOld>& namesIn )
{
    hkArray<hkStringOld> uniqueNames;
    for (int i=0; i<namesIn.getSize(); ++i)
    {
        hkStringOld& name = namesIn[i];
        hkBool unique = true;
        for (int j=0; j<uniqueNames.getSize(); ++j)
        {
            if ( uniqueNames[j] == name )
            {
                unique = false;
                break;
            }
        }
        if (unique) uniqueNames.pushBack(name);
    }
    namesIn = uniqueNames;
}

/*
    We assume that some of the selection-sets may have identical names, e.g. (order is irrelevant):

    sets = {cat_0, dog_0, rabbit_0, cat_1, horse_0, dog_1}

    where name_n indicates the set with the nth occurence of that name. In this case, if the set of unique picked names
    was pickedNames = {cat, rabbit, iguana}, we partition the sets into:

    pickedSets = {cat_0, rabbit_0},
    unpickedSets = {dog_0, cat_1, horse_0, dog_1}

    i.e. we only operate on the first set we find whose name matches the picked name. If we find two or more sets with the
    same name, e.g here {cat_0, cat_1} and {dog_0, dog_1}, we warn that we found the duplicated names {cat, dog}. We also
    warn that no set with the name "iguana" was found.

    (In practice the modelers seem to enforce uniqueness of selection set names anyway..)
*/
static void _mapNamesToSets( hkArray< hkRefPtr<hkxNodeSelectionSet> >& pickedSets, hkArray< hkRefPtr<hkxNodeSelectionSet> >& unpickedSets,
                            hkArray<hkStringOld>& pickedNames, const hkxScene* scenePtr )
{
    // Make sure there are no duplicated names in pickedNames
    _makeSelectionSetNamesUnique( pickedNames );

    hkLocalArray<hkBool> pickedThisName( pickedNames.getSize() );
    pickedThisName.setSize( pickedNames.getSize(), false );
    hkLocalArray<hkBool> pickedThisSet( scenePtr->m_selectionSets.getSize() );
    pickedThisSet.setSize( scenePtr->m_selectionSets.getSize(), false );

    // Work through all (unique) picked set names
    for (int s=0; s<pickedNames.getSize(); ++s)
    {
        const hkStringOld& nameOfSetToRemove = pickedNames[s];
        hkBool seenThisNameAlready = false;

        // Find the first occurrence of this (unique) name in the list of sets, and flag the corresponding set to be removed
        for (int i=0; i<scenePtr->m_selectionSets.getSize(); ++i)
        {
            hkxNodeSelectionSet* set = scenePtr->m_selectionSets[i];
            if (pickedNames[s].compareTo(set->m_name)==0)
            {
                // If multiple sets are found having the same name, remove only the first.
                if (seenThisNameAlready)
                {
                    HK_WARN_ALWAYS( 0xabba2cdd, "Found duplicated selection set name \"" <<  nameOfSetToRemove.cString()  \
                        << "\". Ignoring all but the first set found with that name" );
                    break;
                }
                else
                {
                    pickedSets.pushBack(set);
                    pickedThisName[s] = true;
                    pickedThisSet[i] = true;
                    seenThisNameAlready = true;
                }
            }
        }
    }

    // Unpicked sets
    for (int i=0; i<scenePtr->m_selectionSets.getSize(); ++i)
    {
        hkxNodeSelectionSet* set = scenePtr->m_selectionSets[i];
        if (!pickedThisSet[i]) unpickedSets.pushBack(set);
    }

    // Warn if some of the picked selection set names were not found in the scene
    for (int s=0; s<pickedNames.getSize(); ++s)
    {
        if ( !pickedThisName[s] )
        {
            const hkStringOld& nameOfPickedSet = pickedNames[s];
            HK_WARN_ALWAYS( 0xabba2cdd, "Could not find selection set with name \"" <<  nameOfPickedSet.cString() << "\"" );
        }
    }
}

static hkResult _removeNodeFromScene(const hkxNode* node, hkxScene* scenePtr)
{
    // Find this node's parent
    hkArray<const hkxNode*> path;
    hkResult result = scenePtr->getFullPathToNode(node, path);
    if (result.isFailure() || path.getSize()<1)
    {
        return HK_FAILURE;
    }

    if (path.getSize() == 1)
    {
        // Node is scene graph root
        scenePtr->m_rootNode = HK_NULL;
    }
    else
    {
        // Remove the pointer to the selected child node in the parent.
        // We're modifying the scene graph here, so need a const cast.
        hkxNode* parent = const_cast<hkxNode*>( path[ path.getSize()-2 ] );

        for (int m=0; m<parent->m_children.getSize(); ++m)
        {
            hkxNode* child = parent->m_children[m];
            if (child == node)
            {
                // Delete pointer from array by shunting subsequent ones down a slot
                parent->m_children.removeAtAndCopy(m);
                break;
            }
        }
    }

    return HK_SUCCESS;
}

// Remove the nodes referred to by the picked selection sets from the scene. Remove the selection sets themselves also.
static void _removeSelectionSets( hkArray<hkStringOld>& namesOfSelectionSetsToRemove, hkxScene* scenePtr )
{
    // Find which sets to remove given the picked names
    hkArray<hkRefPtr<hkxNodeSelectionSet> > selectionSetsToRemove;
    hkArray<hkRefPtr<hkxNodeSelectionSet> > selectionSetsToRetain;
    _mapNamesToSets( selectionSetsToRemove, selectionSetsToRetain, namesOfSelectionSetsToRemove, scenePtr );

    // Remove the pointers to the selected nodes in the scene graph
    int numDeletedNodes = 0;

    for (int i=0; i<selectionSetsToRemove.getSize(); ++i)
    {
        hkxNodeSelectionSet* selectionSet = selectionSetsToRemove[i];
        const char* name = selectionSet->m_name;
        int numNodes = 0;

        for (int n=0; n<selectionSet->m_selectedNodes.getSize(); ++n)
        {
            // If root node has been nulled already, we're done
            if (!scenePtr->m_rootNode) break;

            const hkxNode* node = selectionSet->m_selectedNodes[n];
            if (node)
            {
                hkResult result = _removeNodeFromScene(node, scenePtr);
                if (result.isSuccess()) numNodes++;
            }
        }

        if (numNodes>0)
        {
            Log_Info( "Removing {} nodes associated with selection set \"{}\".", numNodes, name );
        }
        numDeletedNodes += numNodes;
    }

    if (numDeletedNodes == 0) return;
    scenePtr->m_selectionSets.clear();
    scenePtr->m_selectionSets.append(selectionSetsToRetain.begin(), selectionSetsToRetain.getSize());
}

enum NodeRemovalFlag
{
    REMOVE_ME = 0,
    KEEP_ME
};

static void _initNodeRemovalMap(hkPointerMap<const hkxNode*,int>& nodeRemovalMap, hkxNode* node)
{
    nodeRemovalMap.insert(node, REMOVE_ME);

    for (int n=0; n<node->m_children.getSize(); ++n)
    {
        hkxNode* childNode = node->m_children[n];
        _initNodeRemovalMap(nodeRemovalMap, childNode);
    }
}

// Recursively descend scene graph, snipping out nodes flagged to be removed as we go.
static void _removeFlaggedNodes(hkPointerMap<const hkxNode*,int>& nodeRemovalMap, hkxNode* node, hkxScene* scenePtr, int& numRemovedNodes)
{
    if (!node) return;
    {
        int flag;
        nodeRemovalMap.get(node, &flag);
        if (flag == REMOVE_ME)
        {
            hkResult result = _removeNodeFromScene(node, scenePtr);
            if (result.isSuccess())
            {
                numRemovedNodes++;
            }
            else
            {
                HK_WARN_ALWAYS( 0xabba2cdd, "Could not remove node " << node->m_name );
            }

            // We can return now, since this node's children have also been disconnected from the scene root
            return;
        }
    }

    // Need to store array of child nodes prior to removing them, since child array is modified during each removal step
    hkLocalArray<hkxNode*> childNodes( node->m_children.getSize() );
    childNodes.setSize( node->m_children.getSize() );
    for (int n=0; n<node->m_children.getSize(); ++n)
    {
        childNodes[n] = node->m_children[n];
    }

    for (int n=0; n<childNodes.getSize(); ++n)
    {
        hkxNode* childNode = childNodes[n];
        _removeFlaggedNodes(nodeRemovalMap, childNode, scenePtr, numRemovedNodes);
    }
}

// Keep the nodes referred to by the picked selection sets, and all of their ancestor nodes. Delete everything else from the scene.
static void _removeAllExceptSelectionSets( hkArray<hkStringOld>& namesOfSelectionSetsToRetain, hkxScene* scenePtr )
{
    // Find which sets to remove given the picked names
    hkArray< hkRefPtr<hkxNodeSelectionSet> > selectionSetsToRetain;
    hkArray< hkRefPtr<hkxNodeSelectionSet> > unpickedSets;
    _mapNamesToSets( selectionSetsToRetain, unpickedSets, namesOfSelectionSetsToRetain, scenePtr );

    // Make a pointer map between node pointers and an integer flag of whether the node is to be removed or retained.
    // Initialize map with all nodes in the scene, flagged to be removed.
    hkPointerMap<const hkxNode*,int> nodeRemovalMap;
    _initNodeRemovalMap(nodeRemovalMap, scenePtr->m_rootNode);

    // Flag all ancestor nodes of the picked selection set nodes to be retained, as well as the nodes themselves.
    for (int i=0; i<selectionSetsToRetain.getSize(); ++i)
    {
        hkxNodeSelectionSet* selectionSet = selectionSetsToRetain[i];
        for (int n=0; n<selectionSet->m_selectedNodes.getSize(); ++n)
        {
            const hkxNode* node = selectionSet->m_selectedNodes[n];
            hkArray<const hkxNode*> path;
            scenePtr->getFullPathToNode(node, path);

            for (int p=0; p<path.getSize(); ++p)
            {
                nodeRemovalMap.insert(path[p], KEEP_ME);
            }
        }
    }

    int numRemovedNodes = 0;
    _removeFlaggedNodes(nodeRemovalMap, scenePtr->m_rootNode, scenePtr, numRemovedNodes);

    if (numRemovedNodes>0)
    {
        Log_Info( "Removing {} nodes from scene.", numRemovedNodes );
    }
}

/****************************************************************************/


static void _getListFromSemicolonDelimitedString( const char* stringIn, hkArray<hkStringOld>& listOut )
{
    if(stringIn)
    {
        int i=0, j=0;
        while (stringIn[j])
        {
            i = j;
            while (stringIn[i] && (stringIn[i] != ';'))
            {
                ++i;
            }
            if ((i - j) > 0)
            {
                hkStringOld type(&stringIn[j], i-j);
                listOut.pushBack(type);
            }
            j = stringIn[i]? i + 1 : i;
        }
    }
}

// Helper class to make the scene consistent after pruning:
// create an instance of this class before pruning, and call
// its run member function after pruning.
// This will remove all deleted objects from the top-level arrays,
// while keeping those that were not part of the scene tree before
// pruning.
class ConsistencyCheck
{
    hkxScene* m_scene;
    hkArray< hkRefPtr<class hkxCamera> > m_cameras;
    hkArray< hkRefPtr<class hkxLight> > m_lights;
    hkArray< hkRefPtr<class hkxMesh> > m_meshes;
    hkArray< hkRefPtr<class hkxMaterial> > m_materials;
    hkArray< hkRefPtr<class hkxTextureInplace> > m_inplaceTextures;
    hkArray< hkRefPtr<class hkxTextureFile> > m_externalTextures;
    hkArray< hkRefPtr<class hkxSkinBinding> > m_skinBindings;

    // Add an object to the objectsToRetain array if:
    // - it is not already there
    // - it is contained in the existingObjects array.
    // No runtime type checking on the object is performed.
    // Only call this function with objects of the correct type.
    // A RefPtr to the casted object is returned.
    template <typename T>
    static hkRefPtr<T> markObject(hkArray< hkRefPtr<T> >& objectsToRetain,
        const hkArray< hkRefPtr<T> >& existingObjects,
        const hkRefVariant& object)
    {
        hkRefPtr<T> x(static_cast<T*>(object.val()));
        if (objectsToRetain.indexOf(x) == -1 && existingObjects.indexOf(x) != -1)
        {
            objectsToRetain.pushBack(x);
        }
        return x;
    }

    // overwrite array with elements from other
    template <typename T>
    static void rebuild(hkArray<T>& array, hkArray<T>& other)
    {
        array.clear();
        array.append(other.begin(), other.getSize());
        other.clear();
    }

    template <typename T>
    static void arrayDifference(const hkArray<T>& array1,
        hkArray<T>& array2)
    {
        hkArray<T> difference;
        for (int i = 0; i < array1.getSize(); i++)
        {
            if (array2.indexOf(array1[i]) == -1)
                difference.pushBack(array1[i]);
        }

        array2.clear();
        array2.append(difference.begin(), difference.getSize());
    }

    // Add an object to the top-level arrays of a scene, and recursively add all
    // contained objects.
    void addObject(const hkRefVariant& object);

    void addMesh(const hkRefVariant& object);

    void addMaterial(const hkRefVariant& object);

    void addSkinBinding(const hkRefVariant& object);

    // Scan through all the selection sets defined in the scene, and check whether
    // the nodes referred to in each exist in the tree (since they may have been deleted).
    // If not, modify the selection set to no longer refer to those nodes.
    void cleanupSelectionSets();

    void rebuildArrays();

    void removeReferencedObjects();

    void harvestObjects();

public:

    ConsistencyCheck(hkxScene* scene);

    void run();
};


bool _hasSuperClass(const hkReflect::Type* klass, const char* superClassName)
{
    while(klass)
    {
        if( const char* name = klass->getName() )
        {
            if(hkString::strCmp(name, superClassName) == 0)
            {
                return true;
            }
        }
        if( const hkReflect::RecordType* r = klass->asRecord() )
        {
            klass = r->getParentRecord();
        }
    }
    return false;
}


void hctPruneTypesFilter::process( hkRootLevelContainer& data  )
{
    hkArray<hkStringOld> customTypesToRemove;
    _getListFromSemicolonDelimitedString( m_options.m_customClasses, customTypesToRemove );

    hkArray<hkStringOld> pickedSelectionSetNames;
    _getListFromSemicolonDelimitedString( m_options.m_selectionSets, pickedSelectionSetNames );

    hkArray<hkRootLevelContainer::NamedVariant> remainingVariants;

    int numSkeletonMappersRemoved = 0;
    int numRagdollInstancesRemoved = 0;

    for (int nv=0; nv<data.m_namedVariants.getSize(); nv++)
    {
        hkRootLevelContainer::NamedVariant& namedVariant = data.m_namedVariants[nv];

        if (namedVariant.getByTypeName<hkxEnvironment>())
        {
            // Environment: Prune it?
            if (m_options.m_pruneEnvironmentData)
            {
                Log_Info( "Removing hkxEnvironment (\"{}\")", namedVariant.getName() );
                continue;
            }
        }

        if (hkResourceContainer* resource_Data = namedVariant.isA<hkResourceContainer>())
        {
            if (m_options.m_pruneResourceData)
            {
                Log_Info( "Removing Resource Data completely." );
                continue;
            }

            if (m_options.m_pruneDestructionData)
            {
                Log_Info( "Removing Destruction data." );

                hkArray<hkResourceContainer*> containers;
                resource_Data->findAllContainersRecursively(containers);

                for (int i = 0; i < containers.getSize(); i++)
                {
                    hkResourceContainer* container = containers[i];

                    hkArray<hkResourceHandle*> handlesToDelete;
                    for (hkResourceHandle* handle = container->findResourceByName(HK_NULL, HK_NULL, HK_NULL); handle;  handle = container->findResourceByName(HK_NULL, HK_NULL, handle))
                    {
                        const hkReflect::Type* klass = handle->getObjectType();

                        if (_hasSuperClass(klass, "hkdAction"                ) ||
                            _hasSuperClass(klass, "hkdBody"                  ) ||
                            _hasSuperClass(klass, "hkdController"            ) ||
                            _hasSuperClass(klass, "hkdDestructionDemoConfig" ) ||
                            _hasSuperClass(klass, "hkdFracture"              ) ||
                            _hasSuperClass(klass, "hkdInfo"                  ) ||
                            _hasSuperClass(klass, "hkdRaycastGun"            ) ||
                            _hasSuperClass(klass, "hkdShape"                 ))
                        {
                            handlesToDelete.pushBack(handle);
                        }
                    }

                    for (int handleIndex = 0; handleIndex < handlesToDelete.getSize(); handleIndex++)
                    {
                        container->destroyResource(handlesToDelete[handleIndex]);
                    }

                    int numHandlesLeft = container->getNumResources();
                    if ((numHandlesLeft == 0) && (container->getParent() != HK_NULL))
                    {
                        container->getParent()->destroyContainer(container);
                    }
                }
            }
        }

        if (hkxScene* scenePtr = namedVariant.getByTypeName<hkxScene>())
        {
            // Scene
            // Prune it all?
            if (m_options.m_pruneAllSceneData)
            {
                Log_Info( "Removing hkxScene (\"{}\")", namedVariant.getName() );
                continue;
            }

            // else
            ConsistencyCheck check(scenePtr);

            // SCENE DATA
            if (scenePtr->m_rootNode && m_options.m_pruneSceneData)
            {
                int numNodes = _CountNodes(scenePtr->m_rootNode);

                Log_Info( "Removing {} nodes.", numNodes );
                Log_Info( "Removing {} lights.", scenePtr->m_lights.getSize() );
                Log_Info( "Removing {} cameras.", scenePtr->m_cameras.getSize() );
                Log_Info( "Removing {} materials.", scenePtr->m_materials.getSize() );
                Log_Info( "Removing {} inplaceTextures.", scenePtr->m_inplaceTextures.getSize() );
                Log_Info( "Removing {} externalTextures.", scenePtr->m_externalTextures.getSize() );

                // remove scene tree
                scenePtr->m_rootNode = HK_NULL;

                // remove selection sets
                scenePtr->m_selectionSets.clear();

                // all other objects will be removed by the consistency check
            }

            // ATTRIBUTES
            if (scenePtr->m_rootNode && m_options.m_pruneAttributes)
            {
                _removeAttributesRecursive (scenePtr->m_rootNode);
            }

            // SELECTION SETS
            if (scenePtr->m_rootNode)
            {
                if (m_options.m_pruneSelectionSets)
                {
                    // remove the selection sets themselves
                    scenePtr->m_selectionSets.setSize(0);
                }
                else
                {
                    // remove the nodes referred to by the selection sets defined in the filter (or retain them and remove the others)
                    hkBool deleteNotRetain = (m_options.m_selectionDeletionMode != hctPruneTypesOptions::HK_SELECTION_DELETE_NOT_SELECTED);
                    if (deleteNotRetain)
                    {
                        _removeSelectionSets( pickedSelectionSetNames, scenePtr );
                    }
                    else
                    {
                        _removeAllExceptSelectionSets( pickedSelectionSetNames, scenePtr );
                    }
                }
            }

            // MESH DATA
            if (m_options.m_pruneMeshData)
            {
                Log_Info( "Removing {} meshes.", scenePtr->m_meshes.getSize() );
                Log_Info( "Removing {} skin bindings.", scenePtr->m_skinBindings.getSize() );

                // mesh
                scenePtr->m_meshes.setSize(0);

                // skin bindings
                scenePtr->m_skinBindings.setSize(0);

                if (scenePtr->m_rootNode && !m_options.m_pruneSceneData)
                {
                    _NullAllMeshReferences(scenePtr->m_rootNode);
                }
            }

            // MESH USER CHANNELS
            else if (m_options.m_pruneMeshUserChannels)
            {
                Log_Info( "Removing user channels from {} meshes.", scenePtr->m_meshes.getSize() );

#if defined(HK_FEATURE_PRODUCT_CLOTH)
                if (hclClothContainer* clothContainer = data.findObject<hclClothContainer>())
                {
                    for (int m = 0; m < scenePtr->m_meshes.getSize(); m++)
                    {
                        hkxMesh* mesh = scenePtr->m_meshes[m];

                        if (!mesh->m_userChannelInfos.isEmpty())
                        {
                            HK_WARN_ALWAYS(0xabbae8a9, "Removing user channels from meshes may cause cloth execution to fail.");
                            break;
                        }
                    }
                }
#endif

                for (int m=0; m<scenePtr->m_meshes.getSize(); m++)
                {
                    hkxMesh* mesh = scenePtr->m_meshes[m];
                    _ZeroMeshChannels(mesh);
                }
            }

            // make sure all deleted objects are removed from the top-level arrays
            check.run();
        }

#if defined(HK_FEATURE_PRODUCT_ANIMATION)
        if (hkaAnimationContainer* animContainer = namedVariant.getByTypeName<hkaAnimationContainer>())
        {
            // Prune it all?
            if (m_options.m_pruneAllAnimationData)
            {
                Log_Info( "Removing hkaAnimationContainer (\"{}\")", namedVariant.getName() );
                continue;
            }

            if (m_options.m_pruneSkeletonData)
            {
                Log_Info( "Removing {} skeletons.", animContainer->m_skeletons.getSize() );

                //Skels ref the bones, so just null the skels
                animContainer->m_skeletons.clear();
            }

            if (m_options.m_pruneAnimationData)
            {
                Log_Info( "Removing {} animations.", animContainer->m_animations.getSize() );

                animContainer->m_animations.clear();
                animContainer->m_bindings.clear();
            }

            if (m_options.m_pruneAttachments)
            {
                Log_Info( "Removing {} mesh attachments.", animContainer->m_attachments.getSize() );
                // attachments
                animContainer->m_attachments.clear();
            }

            if ( !m_options.m_pruneAnimationData )
            {
                if ( m_options.m_pruneAnimationTracks && m_options.m_pruneAnnotations )
                {
                    int totalTracksRemoved = 0;
                    for (int an=0; an<animContainer->m_animations.getSize(); an++)
                    {
                        hkaAnimation* skelAnim = animContainer->m_animations[an];

                        totalTracksRemoved += skelAnim->m_annotationTracks.getSize();

                        skelAnim->m_annotationTracks.clear();
                    }

                    Log_Info( "Removing {} tracks from {} animation(s).", totalTracksRemoved , animContainer->m_animations.getSize() );
                }
                else if ( m_options.m_pruneAnimationTracks )
                {
                    int totalTracksRemoved = 0;
                    for (int an=0; an<animContainer->m_animations.getSize(); an++)
                    {
                        hkaAnimation* skelAnim = animContainer->m_animations[an];

                        totalTracksRemoved += skelAnim->m_annotationTracks.getSize();

                        for ( int track = 0; track < skelAnim->m_annotationTracks.getSize(); track++ )
                        {
                            skelAnim->m_annotationTracks[ track ].m_trackName = "";
                        }
                    }

                    Log_Info( "Removing {} track names from {} animation(s).", totalTracksRemoved , animContainer->m_animations.getSize() );
                }
                else if ( m_options.m_pruneAnnotations )
                {
                    int totalAnnotationsRemoved = 0;
                    int totalTracks = 0;
                    for (int an=0; an<animContainer->m_animations.getSize(); an++)
                    {
                        hkaAnimation* skelAnim = animContainer->m_animations[an];
                        for (int t=0; t<skelAnim->m_annotationTracks.getSize(); t++)
                        {
                            hkaAnnotationTrack& track = skelAnim->m_annotationTracks[t];

                            totalAnnotationsRemoved += track.m_annotations.getSize();

                            track.m_annotations.clear();
                        }
                        totalTracks += skelAnim->m_annotationTracks.getSize();
                    }

                    Log_Info( "Removing {} annotations from {} tracks ({} animation(s)).", totalAnnotationsRemoved , totalTracks, animContainer->m_animations.getSize() );
                }

                if( m_options.m_pruneIdentityBindingIndices )
                {
                    int totalTransformIdentityBindingsRemoved = 0;
                    int totalFloatIdentityBindingsRemoved = 0;
                    for (int b=0; b<animContainer->m_bindings.getSize(); b++)
                    {
                        hkaAnimationBinding* binding = animContainer->m_bindings[b];
                        int idx = 0;
                        for (idx=0; idx<binding->m_transformTrackToBoneIndices.getSize(); idx++)
                        {
                            if(binding->m_transformTrackToBoneIndices[idx] != idx)
                            {
                                idx = -1;
                                break;
                            }
                        }
                        if( (binding->m_transformTrackToBoneIndices.getSize() != 0) && (idx != -1))
                        {
                            binding->m_transformTrackToBoneIndices.clear();
                            totalTransformIdentityBindingsRemoved++;
                        }

                        for (idx=0; idx<binding->m_floatTrackToFloatSlotIndices.getSize(); idx++)
                        {
                            if(binding->m_floatTrackToFloatSlotIndices[idx] != idx)
                            {
                                idx = -1;
                                break;
                            }
                        }
                        if( (binding->m_floatTrackToFloatSlotIndices.getSize() != 0) && (idx != -1))
                        {
                            binding->m_floatTrackToFloatSlotIndices.clear();
                            totalFloatIdentityBindingsRemoved++;
                        }
                    }

                    Log_Info( "Removing {} Transform and {} Float identity binding index arrays from {} bindings(s).", totalTransformIdentityBindingsRemoved, totalFloatIdentityBindingsRemoved, animContainer->m_bindings.getSize() );
                }

                if( m_options.m_pruneQuantizedBindings )
                {
                    int totalBindingsRemoved = 0;

                    // For each binding
                    for (int b=0; b<animContainer->m_bindings.getSize(); b++)
                    {
                        hkaAnimationBinding* binding = animContainer->m_bindings[b];

                        // Check if the type is quantized
                        if (    ( binding->m_animation->getType() == hkaAnimation::HK_QUANTIZED_COMPRESSED_ANIMATION ) ||
                                ( binding->m_animation->getType() == hkaAnimation::HK_PREDICTIVE_COMPRESSED_ANIMATION ) )
                        {
                            // Set the binding to null
                            animContainer->m_bindings[b] = HK_NULL;
                            ++totalBindingsRemoved;
                        }
                    }

                    Log_Info( "Removing {} hkaAnimtionBinding instances associated with hkaQuantizedAnimation instances", totalBindingsRemoved );
                }
            }

            if(m_options.m_pruneMeshBindingData)
            {
                Log_Info( "Removing {} skins.", animContainer->m_skins.getSize() );
                animContainer->m_skins.clear();
            }
        }

        if (namedVariant.getByTypeName<hkaSkeletonMapper>())
        {
            if (m_options.m_pruneRagdollAndMapperData || m_options.m_pruneAllAnimationData)
            {
                // Skip skeleton mappers
                numSkeletonMappersRemoved++;
                continue;
            }
        }
#endif

#if defined(HK_FEATURE_PRODUCT_ANIMATION) && defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
        if (namedVariant.getByTypeName<hkaRagdollInstance>())
        {
            if (m_options.m_pruneRagdollAndMapperData || m_options.m_pruneAllAnimationData)
            {
                // Skip ragdoll instance
                numRagdollInstancesRemoved++;
                continue;
            }

        }
#endif

        // CUSTOM CLASSES, separated by ';'
        {
            bool removeIt = false;
            const char* typeName = namedVariant.getTypeName();
            for (int c=0; c<customTypesToRemove.getSize(); c++)
            {
                if (customTypesToRemove[c] == typeName)
                {
                    removeIt = true;
                    Log_Info( "Removing {} of type {}.", namedVariant.getName(), customTypesToRemove[c].cString() );
                    continue;
                }
            }

            if (removeIt) continue;
        }

        // We keep this variant
        remainingVariants.pushBack(namedVariant);
    }

    if(numRagdollInstancesRemoved > 0)
    {
        Log_Info( "Removing {} ragdoll instances.", numRagdollInstancesRemoved );
    }
    if(numSkeletonMappersRemoved > 0)
    {
        Log_Info( "Removing {} skeleton mappers.", numSkeletonMappersRemoved );
    }

    if (remainingVariants.getSize() != data.m_namedVariants.getSize()) // some variants removed
    {
        data.m_namedVariants.clearAndDeallocate();
        remainingVariants.swap(data.m_namedVariants);
    }
}

void hctPruneTypesFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctPruneTypesOptions* options = hctFilterUtils::getNativeOptions<hctPruneTypesOptions>(optVar))
    {
        m_customClasses = options->m_customClasses;
        m_selectionSets = options->m_selectionSets;
        m_options = *options;
        delete options;
    }
}

void hctPruneTypesFilter::getOptions(hkReflect::Any& buffer) const
{
    hctPruneTypesOptions options(m_options);
    buffer.setFromObj( options );
}

// implementation of ConsistencyCheck

void ConsistencyCheck::addObject( const hkRefVariant& object )
{
    if (!object)
    {
        return;
    }

    if (hkReflect::exactMatchDynCast<hkxMesh>(object))
    {
        addMesh(object);
    }
    else if (hkReflect::exactMatchDynCast<hkxCamera>(object))
    {
        markObject(m_cameras, m_scene->m_cameras, object);
    }
    else if (hkReflect::exactMatchDynCast<hkxLight>(object))
    {
        markObject(m_lights, m_scene->m_lights, object);
    }
    else if (hkReflect::exactMatchDynCast<hkxMaterial>(object))
    {
        addMaterial(object);
    }
    else if (hkReflect::exactMatchDynCast<hkxTextureInplace>(object))
    {
        markObject(m_inplaceTextures, m_scene->m_inplaceTextures, object);
    }
    else if (hkReflect::exactMatchDynCast<hkxTextureFile>(object))
    {
        markObject(m_externalTextures, m_scene->m_externalTextures, object);
    }
    else if (hkReflect::exactMatchDynCast<hkxSkinBinding>(object))
    {
        addSkinBinding(object);
    }
}

void ConsistencyCheck::addMesh(const hkRefVariant& object)
{
    hkRefPtr<hkxMesh> mesh = markObject(m_meshes, m_scene->m_meshes, object);
    // add materials
    for (int i = 0; i < mesh->m_sections.getSize(); i++)
    {
        if (mesh->m_sections[i]->m_material!=HK_NULL)
        {
            addMaterial(hkRefVariant(mesh->m_sections[i]->m_material));
        }
    }
}

void ConsistencyCheck::addMaterial(const hkRefVariant& object)
{
    hkRefPtr<hkxMaterial> material = markObject(m_materials, m_scene->m_materials, object);
    // add sub-materials
    for (int i = 0; i < material->m_subMaterials.getSize(); i++)
    {
        addMaterial(hkRefVariant(material->m_subMaterials[i]));
    }
    // add textures
    for (int i = 0; i < material->m_stages.getSize(); i++)
    {
        addObject(material->m_stages[i].m_texture);
    }
}

void ConsistencyCheck::addSkinBinding(const hkRefVariant& object)
{
    hkRefPtr<hkxSkinBinding> skinBinding = markObject(m_skinBindings, m_scene->m_skinBindings, object);
    // add the mesh contained in this skin binding
    addMesh(hkRefVariant(skinBinding->m_mesh));
}

void ConsistencyCheck::cleanupSelectionSets()
{
    hkArray< hkRefPtr<hkxNode> > existingNodes;

    if ( (!m_scene->m_rootNode) && (m_scene->m_selectionSets.getSize() > 0))
    {
        // If scene root is null, selection sets cannot refer to any nodes
        m_scene->m_selectionSets.setSize(0);
        return;
    }

    for (int i=0; i < m_scene->m_selectionSets.getSize(); ++i)
    {
        hkxNodeSelectionSet* selectionSet = m_scene->m_selectionSets[i];
        existingNodes.clear();

        for (int n=0; n<selectionSet->m_selectedNodes.getSize(); ++n)
        {
            hkxNode* node = selectionSet->m_selectedNodes[n];
            if (node)
            {
                // Find this node in the scene tree
                hkArray<const hkxNode*> path;
                hkResult result = m_scene->getFullPathToNode(node, path);
                if (result.isSuccess())
                {
                    existingNodes.pushBack(node);
                }
            }
        }

        selectionSet->m_selectedNodes.setSize(0);
        selectionSet->m_selectedNodes.append(existingNodes.begin(), existingNodes.getSize());
    }

    // If a selection set no longer refers to any nodes, the set itself is deleted from the scene.
    hkArray< hkRefPtr<hkxNodeSelectionSet> > selectionSetsToRetain;
    for (int i=0; i < m_scene->m_selectionSets.getSize(); ++i)
    {
        hkxNodeSelectionSet* selectionSet = m_scene->m_selectionSets[i];
        if (selectionSet->m_selectedNodes.getSize() != 0)
        {
            selectionSetsToRetain.pushBack(selectionSet);
        }
    }
    m_scene->m_selectionSets.clear();
    m_scene->m_selectionSets.append(selectionSetsToRetain.begin(), selectionSetsToRetain.getSize() );
}


void ConsistencyCheck::rebuildArrays()
{
    rebuild(m_scene->m_cameras, m_cameras);
    rebuild(m_scene->m_lights, m_lights);
    rebuild(m_scene->m_meshes, m_meshes);
    rebuild(m_scene->m_materials, m_materials);
    rebuild(m_scene->m_inplaceTextures, m_inplaceTextures);
    rebuild(m_scene->m_externalTextures, m_externalTextures);
    rebuild(m_scene->m_skinBindings, m_skinBindings);
}

void ConsistencyCheck::removeReferencedObjects()
{
    arrayDifference(m_scene->m_cameras, m_cameras);
    arrayDifference(m_scene->m_lights, m_lights);
    arrayDifference(m_scene->m_meshes, m_meshes);
    arrayDifference(m_scene->m_materials, m_materials);
    arrayDifference(m_scene->m_inplaceTextures, m_inplaceTextures);
    arrayDifference(m_scene->m_externalTextures, m_externalTextures);
    arrayDifference(m_scene->m_skinBindings, m_skinBindings);
}

void ConsistencyCheck::harvestObjects()
{
    hkArray< hkRefPtr<hkxNode> > nodes;
    hkxSceneUtils::findAllNodes(m_scene->m_rootNode, nodes);

    // discover all objects in the scene tree
    for (int i = 0; i < nodes.getSize(); i++)
    {
        addObject(nodes[i]->m_object);
    }
}

ConsistencyCheck::ConsistencyCheck(hkxScene* scene)
: m_scene(scene)
{
    // initialize arrays with objects NOT in the scene tree
    harvestObjects();
    removeReferencedObjects();
}

void ConsistencyCheck::run()
{
    harvestObjects();
    rebuildArrays();
    cleanupSelectionSets();
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
