// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/Filters/Common/Utils/hctFilterUtils.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Container/String/hkUtf8.h>

#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlWriteFormat.h>
#include <Common/Base/Serialize/Util/Xml/hkXmlParser.h>

#include <Common/Base/System/Io/FileSystem/hkNativeFileSystem.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>

#include <Common/Serialize/Util/hkSerializeUtil.h>


#include <shlwapi.h>

const wchar_t hctFilterUtils::s_DialogFileExtensions[] = L"Havok Files (*.hkx;*.hkt;*.xml)\0*.hkx;*.hkt;*.xml;\0Binary Packfiles (*.hkx)\0*.hkx\0Tagfiles (*.hkt)\0*.hkt\0XML Packfiles (*.xml)\0*.xml\0All Files\0*.*\0\0";

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.common"
#include <Common/Base/System/Log/hkLog.hxx>

#if defined (HK_ARCH_X64)
    static const CHAR* _HAVOK_FILTERS_REG_KEY = "Software\\Havok\\hkFilters_x64";
    static const CHAR* _HAVOK_MANAGEDTOOLS_REG_KEY = "Software\\Havok\\ManagedTools_x64";
    static const CHAR* _HAVOK_TOOLS_ENV_VAR = "HAVOK_TOOLS_ROOT_X64";
#else
    static const CHAR* _HAVOK_FILTERS_REG_KEY = "Software\\Havok\\hkFilters";
    static const CHAR* _HAVOK_MANAGEDTOOLS_REG_KEY = "Software\\Havok\\ManagedTools";
    static const CHAR* _HAVOK_TOOLS_ENV_VAR = "HAVOK_TOOLS_ROOT";
#endif

namespace
{
    template<typename T>
    void createArrayFromDelimitedStringInternal( hkArray<T>& buf, const char* delimitedString, const char delimiter )
    {
        buf.setSize( 0 );

        int curCh = 0;
        int prevCh = 0;

        while ( delimitedString && delimitedString[curCh] )
        {
            prevCh = curCh;
            while ( delimitedString[curCh] && ( delimitedString[curCh] != delimiter ) )
            {
                curCh++;
            }
            if ( curCh > prevCh )
            {
                T* str = buf.expandBy(1);
                *str = T( &( delimitedString[prevCh] ), curCh - prevCh );
            }
            if ( delimitedString[curCh] )
            {
                curCh++;
            }
        }
    }

    template<typename T>
    void createArrayOfArrayOfStringsFromDelimitedStringInternal(
        hkArray< hkArray< T > >& out, const char* delimitedString, char inner, char outer )
    {
        // While there are characters
        while ( *delimitedString )
        {
            char c = *delimitedString++;

            if ( c == outer )
            {
                out.pushBack( hkArray< T >() );
            }
            else if ( c == inner )
            {
                out.back().pushBack( T() );
            }
            else
            {
                hkStringBuf buf(out.back().back().cString());
                buf.append(&c, 1);
                out.back().back() = buf.cString();
            }
        }
    }

    template<typename T>
    void createDelimitedStringFromArrayInternal( const hkArray<T>& array, const char* delimiter, hkStringOld& delimitedStringOut)
    {
        delimitedStringOut = "";

        for (int i=0; i<array.getSize(); i++)
        {
            delimitedStringOut += array[i];
            delimitedStringOut += delimiter;
        }
    }

    template<typename T>
    void createDelimitedStringFromArrayOfArrayOfStringsInternal(
        const hkArray< hkArray< T > >& in, hkStringOld& out, char inner, char outer )
    {
        char ostr[ 2 ] = { outer, 0 };
        char istr[ 2 ] = { inner, 0 };

        // For each outer array
        for ( int i = 0; i < in.getSize(); i++ )
        {
            out += ostr;

            // For each inner array
            for ( int j = 0; j < in[ i ].getSize(); j++ )
            {
                out += istr;
                out += in[ i ][ j ];
            }
        }
    }
}

void* hctFilterUtils::deepCopyObject(const void* obj, const hkReflect::Type* klass, hkResource*& pd )
{
    
    

    // EXP-698 : Temporarily disable warnings regarding non-registered structs
    const bool wasEnabled = hkError::getInstance().isEnabled(0x4e34ea5f);
    hkError::getInstance().setEnabled(0x4e34ea5f, false);

    hkArray<char> buf;
    hkSerializeUtil::save( obj, klass, hkOstream(buf).getStreamWriter() );
    pd = hkSerializeUtil::load( buf.begin(), buf.getSize() );

    hkError::getInstance().setEnabled(0x4e34ea5f, wasEnabled);

    return pd->getContentsPointer( (char*)HK_NULL );
}

void hctFilterUtils::createArrayFromDelimitedString( hkArray<hkStringOld>& buf, const char* delimitedString, const char delimiter )
{
    createArrayFromDelimitedStringInternal(buf, delimitedString, delimiter);
}

void hctFilterUtils::createArrayFromDelimitedString( hkArray<hkStringPtr>& buf, const char* delimitedString, const char delimiter )
{
    createArrayFromDelimitedStringInternal(buf, delimitedString, delimiter);
}

void hctFilterUtils::createArrayFromDelimitedString( hkArray<hkStringOld>& buf, const char* delimitedString, const hkStringOld& delimiter )
{
    // First replace all those instances with a fixed single character delimiter, then call the single char version
    // TODO? - what if user alreay has this delimiter in their names???
    hkStringOld s(delimitedString);
    hkStringOld newDelim = ";"; // TODO? Generalize this????
    hkStringOld stringModified = s.replace( delimiter, newDelim, hkString::REPLACE_ALL);
    hctFilterUtils::createArrayFromDelimitedString( buf, stringModified.cString(), ';');
}

void hctFilterUtils::createDelimitedStringFromArray(
    const hkArray<hkStringOld>& array, const char* delimiter, hkStringOld& delimitedStringOut)
{
    createDelimitedStringFromArrayInternal(array, delimiter, delimitedStringOut);
}

void hctFilterUtils::createDelimitedStringFromArray(
    const hkArray<hkStringPtr>& array, const char* delimiter, hkStringOld& delimitedStringOut)
{
    createDelimitedStringFromArrayInternal(array, delimiter, delimitedStringOut);
}

void hctFilterUtils::createArrayOfArrayOfStringsFromDelimitedString(
    hkArray< hkArray< hkStringOld > >& out, const char* delimitedString, char inner, char outer )
{
    createArrayOfArrayOfStringsFromDelimitedStringInternal(out, delimitedString, inner, outer);
}

void hctFilterUtils::createArrayOfArrayOfStringsFromDelimitedString(
    hkArray< hkArray< hkStringPtr > >& out, const char* delimitedString, char inner, char outer )
{
    createArrayOfArrayOfStringsFromDelimitedStringInternal(out, delimitedString, inner, outer);
}

void hctFilterUtils::createDelimitedStringFromArrayOfArrayOfStrings(
    const hkArray< hkArray< hkStringOld > >& in, hkStringOld& out, char inner, char outer )
{
    createDelimitedStringFromArrayOfArrayOfStringsInternal(in, out, inner, outer);
}

void hctFilterUtils::createDelimitedStringFromArrayOfArrayOfStrings(
    const hkArray< hkArray< hkStringPtr > >& in, hkStringOld& out, char inner, char outer )
{
    createDelimitedStringFromArrayOfArrayOfStringsInternal(in, out, inner, outer);
}


void hctFilterUtils::getAssetFolder (const hkRootLevelContainer& rootLevelContainer, hkStringOld& assetFolderOut )
{
    // Look for the asset folder environment variable if we have it
    const char* assetFolder = getEnvironmentVariable(rootLevelContainer, "assetFolder");

    if (assetFolder)
    {
        assetFolderOut = assetFolder;
    }
    else
    {
        assetFolderOut = "";
    }
}

const char* hctFilterUtils::getEnvironmentVariable (const hkRootLevelContainer& contents, const char* variable)
{
    // Find the environment in the root level container
    hkxEnvironment tempEnvironment;
    hkxEnvironment* environmentPtr = HK_NULL;
    {
        environmentPtr = contents.findObject<hkxEnvironment>();

        // If no environment is found, use a temporary one
        if (!environmentPtr)
        {
            environmentPtr = &tempEnvironment;

            // If we have a scene, fill the environment with it
            hkxScene* scenePtr = contents.findObject<hkxScene>();
            if (scenePtr)
            {
                hkxSceneUtils::fillEnvironmentFromScene(*scenePtr, tempEnvironment);
            }
        }
    }

    return environmentPtr->getVariableValue(variable);
}


const char* hctFilterUtils::getOSEnvironmentVariable (const char* variable)
{
    return ::getenv( variable );
}


void hctFilterUtils::getFullPath ( const hkRootLevelContainer& rootLevelContainer, const char* userPath, hkStringOld& fullPathOut)
{
    hkStringOld userPathStr (userPath);

    hkStringOld assetFolder;
    getAssetFolder( rootLevelContainer, assetFolder);

    // No asset folder? return the user path as given
    if (assetFolder.getLength()==0)
    {
        fullPathOut = userPathStr;
        return;
    }

    // purely relative? just add asset path
    if (userPathStr.beginsWith("."))
    {
        fullPathOut = assetFolder + userPathStr;
        return;
    }

    // a root offset? (and not a server path) then add the start (drive) of asset path
    if (assetFolder.getLength()>=2 && (userPathStr.beginsWith("\\") && !userPathStr.beginsWith("\\\\") || userPathStr.beginsWith("/")) )
    {
        hkStringOld driveLetter = assetFolder.substr(0,2);
        fullPathOut = driveLetter + userPathStr;
        return;
    }

    // might either be a absolute path (starts with drive and not a server path) or a normal filename (then add asset path)
    if ( (userPathStr.getLength() > 4) && !userPathStr.beginsWith("\\\\") && !hkString::strChr(userPathStr.substr(0,2).cString(), ':'))
    {
        // normal filename (has not got a : in the first 2 entries (or rather [1]), add asset path
        fullPathOut = assetFolder + userPathStr;
        return;
    }

    // must be absolute or something we don't handle wrt to paths
    fullPathOut = userPathStr;

    return;
}


void hctFilterUtils::getReducedFilename( const wchar_t* fname, const hkStringOld& assetPath, hkStringOld& reducedName )
{
    // Take m_assetPath out of fname
    wchar_t rel[MAX_PATH];

    hkStringOld assetPathLessLast( assetPath.cString(), assetPath.getLength() > 0? assetPath.getLength() - 1 : 0 );

    hkUtf8::WideFromUtf8 assetPathLessLastWide(assetPathLessLast.cString());

    if ( PathFileExistsW( assetPathLessLastWide ) &&
         PathRelativePathToW( rel, assetPathLessLastWide, FILE_ATTRIBUTE_DIRECTORY, fname, FILE_ATTRIBUTE_NORMAL) )
    {
        reducedName = hkUtf8::Utf8FromWide(rel);
    }
    else
    {
        reducedName = hkUtf8::Utf8FromWide(fname); // ah well...
    }
}

const char* hctFilterUtils::filterCategoryToString( hctFilterDescriptor::FilterCategory c )
{
    switch (c)
    {
        case hctFilterDescriptor::HK_CATEGORY_CORE: return "Core";
        case hctFilterDescriptor::HK_CATEGORY_COLLISION_DEPRECATED: return "Collision";
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS_2012:
        case hctFilterDescriptor::HK_CATEGORY_PHYSICS: return "Physics";
        case hctFilterDescriptor::HK_CATEGORY_FX_PHYSICS_DEPRECATED: return "FX";
        case hctFilterDescriptor::HK_CATEGORY_ANIMATION: return "Animation";
        case hctFilterDescriptor::HK_CATEGORY_GRAPHICS: return "Graphics";
        case hctFilterDescriptor::HK_CATEGORY_USER: return "User";
        case hctFilterDescriptor::HK_CATEGORY_CLOTH: return "Cloth";
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION_2012:
        case hctFilterDescriptor::HK_CATEGORY_DESTRUCTION: return "Destruction";
        default: return "Unknown";
    }
}


/*static*/ bool hctFilterUtils::replaceVariables (
    const hkxEnvironment& environment, const char* inputString, hkStringOld& outputString, hkBool useOSEnvironment)
{
    outputString = inputString;

    bool everythingOk = true;

    // Check any $ symbols in the string.
    int dollarIndex = outputString.indexOf( '$' );
    while( dollarIndex != -1 )
    {
        const int bracketIndex1 = outputString.indexOf( '(', dollarIndex );
        const int colonIndex = outputString.indexOf( ':', bracketIndex1);
        const int bracketIndex2 = outputString.indexOf( ')', bracketIndex1 );


        if( ( bracketIndex1 == dollarIndex+1 ) && ( bracketIndex2 > bracketIndex1 ) )
        {
            const bool hasDefault = (colonIndex>bracketIndex1) && (colonIndex<bracketIndex2);

            const int variableNameLength = hasDefault ? (colonIndex - bracketIndex1 - 1) :  (bracketIndex2 - bracketIndex1 -1);

            hkStringOld variableString; // i.e. "$(xxxx[:yyy])"
            variableString = outputString.substr( dollarIndex, bracketIndex2+1 - dollarIndex );

            hkStringOld varName; // i.e. "xxxx"
            varName = variableString.substr(2, variableNameLength);

            const char* varValue = environment.getVariableValue(varName.cString());

            hkStringOld replacementString;
            bool doReplace = false;

            if (varValue)
            {
                replacementString = varValue;
                doReplace = true;
            }
            else if (useOSEnvironment)
            {
                // Try OS environment variables too
                const char* envVarValue = getenv( varName.cString() );

                if (envVarValue)
                {
                    replacementString = envVarValue;
                    doReplace = true;
                }
            }

            if (!doReplace && hasDefault)
            {
                hkStringBuf defaultStr = variableString.cString();
                replacementString = variableString.substr(colonIndex-dollarIndex+1, bracketIndex2-colonIndex-1);
                doReplace = true;
            }

            if (doReplace)
            {
                outputString = outputString.substr( 0, dollarIndex )
                        + replacementString
                        + outputString.substr( dollarIndex + variableString.getLength() );

            }
            else
            {
                HK_WARN_ALWAYS(0xabba8dc2, "Couldn't replace variable "<<variableString.cString());
                everythingOk = false;
            }
        }

        dollarIndex = outputString.indexOf( '$', dollarIndex + 1 );

    }

    return everythingOk;
}


/* static */ bool hctFilterUtils::isExportable( const hkRootLevelContainer::NamedVariant& a )
{
    // Null-terminated array of names of non-exportable classes
    // Add any new non-exportable class names here
    static char* nonExportables[] = { "hkaFootstepAnalysisInfoContainer", "hkaAnimationPreviewColorContainer", "hkAlignSceneToNodeOptions", 0 };

    hkStringBuf buf;
    const char* name = a.getTypeName();
    // For each (non-null) entry of the array...
    for ( int i = 0; nonExportables[ i ] != 0; i++ )
    {
        if ( hkString::strCmp( name, nonExportables[ i ] ) == 0 )
        {
            // A match was found.  This class is not exportable.
            return false;
        }
    }

    // No matches were found.  This class is exportable.
    return true;
}

/* static */ bool hctFilterUtils::compareExportableLess( const hkRootLevelContainer::NamedVariant& a, const hkRootLevelContainer::NamedVariant& b )
{
    // Enforce that non-exportables are greater than exportables
    if ( isExportable( a ) && !isExportable( b ) )
    {
        return true;
    }

    return false;
}

/* static */ int hctFilterUtils::moveNonExportablesToEnd( hkRootLevelContainer& data )
{
    // Oh the joys of bubblesort!  Want a *stable* sort here.
    // Moves all non exportable classes to the end.
    hkAlgorithm::bubbleSort( data.m_namedVariants.begin(), data.m_namedVariants.getSize(), compareExportableLess );

    // Find the number of non-exportables
    for ( int i = 1; i <= data.m_namedVariants.getSize(); i++ )
    {
        if ( isExportable( data.m_namedVariants[ data.m_namedVariants.getSize() - i ] ) )
        {
            // An exportable class was found.  Return the count.
            return i - 1;
        }
    }

    // All classes were non-exportable
    return data.m_namedVariants.getSize();
}

//
//  Attempts to resolve the absolute path to the given file, based on the path of the current asset

hkResult hctFilterUtils::resolvePath(const char* fileName, const char* assetPath, hkStringBuf& filePathOut)
{
    hkStringBuf havokFileName = fileName;
    havokFileName.pathNormalize();

    // Try to open the file
    {
        hkIstream iStream(havokFileName.cString());
        if ( iStream.isOk() )
        {
            // We've managed to open the file!
            filePathOut = havokFileName;
            return HK_SUCCESS;
        }
    }

    // Failed to open the file. Get the asset path and try to create a file path based on that
    if ( assetPath )
    {
        hkStringBuf havokAssetPath = assetPath;
        havokAssetPath.pathAppend("..", fileName);
        havokAssetPath.pathNormalize();

        // Append script file
        hkIstream iStream(havokAssetPath.cString());
        if ( iStream.isOk() )
        {
            // We've managed to open the file!
            filePathOut = havokAssetPath;
            return HK_SUCCESS;
        }
    }

    // Could not open the file or infer its path, abandon!
    return HK_FAILURE;
}

hkReflect::Var hctFilterUtils::getNativeOptions(const hkReflect::Var& loadedOptions, const hkReflect::Type* nativeType)
{
    
    if (loadedOptions)
    {
        if (!nativeType->equals(loadedOptions.getType()))
        {
            HK_WARN_ALWAYS(0xabba5fab, "Cannot load options of type '" << nativeType->getName()
                << "' (found type '" << loadedOptions.getType()->getName() << "'");
            return hkReflect::Var();
        }
        hkReflect::Var res = nativeType->newInstance();
        HK_ON_DEBUG(hkResult r = )res.assign(loadedOptions);
        HK_ASSERT_NO_MSG(0x6f364a79, r.isSuccess());
        return res;
    }
    else
    {
        return hkReflect::Var();
    }
}

int hctFilterUtils::getValueOfName(const Enum& en, const char* name, int defValue)
{
    for (Enum::const_iterator it = en.begin(); it != en.end(); ++it)
    {
        if (hkString::strCmp(name, it->m_name) == 0)
        {
            return it->m_value;
        }
    }
    return defValue;
}


hkResult hctFilterUtils::writeToPlatform(class hkRootLevelContainer& data, hkStringOld& fileName, const hctPlatformWriterOptions& options)
{
    hkResult res = HK_FAILURE;

    // Pre-process to move non-exportable root-level classes to the end
    int numNonExportable = hctFilterUtils::moveNonExportablesToEnd(data);
    HK_ASSERT_NO_MSG(0x70e9be70, (data.m_namedVariants.getSize() - numNonExportable) >= 0);
    hkArray<hkRootLevelContainer::NamedVariant> nonExportable;

    if (numNonExportable > 0)
    {
        int indexOfFirstNonExportable = data.m_namedVariants.getSize() - numNonExportable;
        nonExportable.insertAt(0, data.m_namedVariants.begin() + indexOfFirstNonExportable, numNonExportable);
        data.m_namedVariants.setSize(indexOfFirstNonExportable);
    }

    // Find the environment in the root level container
    hkxEnvironment tempEnvironment;
    hkxEnvironment* environmentPtr = HK_NULL;
    bool prunedEnvironment = false;
    {
        environmentPtr = data.findObject<hkxEnvironment>();

        // If no environemnt is found, use a temporary one
        if (!environmentPtr)
        {
            prunedEnvironment = true;
            environmentPtr = &tempEnvironment;

            // If we have a scene, fill the environment with ti
            hkxScene* scenePtr = data.findObject<hkxScene>();
            if (scenePtr)
            {
                hkxSceneUtils::fillEnvironmentFromScene(*scenePtr, tempEnvironment);
            }
        }
    }

    hkStringOld err;
    bool isPortable = (options.m_format != hctPlatformWriterOptions::FORMAT_NATIVE);
    int curPresetIndex = hctPlatformWriterOptions::findTargetIndex(options.m_target);
    if (isPortable || curPresetIndex >= 0)
    {
        // If the filename hasn't been set, use the default.
        if (fileName.getLength() == 0)
        {
            fileName = ".\\$(asset)_$(configuration).hkx";
        }

        hkStringOld userFilename;
        const bool substitutionOk = hctFilterUtils::replaceVariables(*environmentPtr, fileName.cString(), userFilename, true);

        if (!substitutionOk && prunedEnvironment)
        {
            HK_WARN_ALWAYS(0xabba9a9f, "No environment data (required for variable substitutions) found - possibly pruned.");
        }

        // Construct the full filename if the path seems to be relative
        hkStringOld fullFilename;
        hctFilterUtils::getFullPath(data, userFilename.cString(), fullFilename);

        hkOstream f(fullFilename.cString());
        if (f.isOk())
        {
            // EXP-462
            // If we don't want to save the environment, go through the variants
            hkRootLevelContainer* container = &data;
            hkRootLevelContainer localContainer;
            if (!options.m_saveEnvironmentData)
            {
                container = &localContainer;
                localContainer.m_namedVariants.reserve(data.m_namedVariants.getSize());
                for (int i = 0; i < data.m_namedVariants.getSize(); i++)
                {
                    HK_ASSERT_NO_MSG(0x74fd5f45, data.m_namedVariants[i].getType());
                    if (!data.m_namedVariants[i].isA<hkxEnvironment>()) // not an environment
                    {
                        localContainer.m_namedVariants.expandOne() = data.m_namedVariants[i];
                    }
                }
            }

            // save file
            const char* target = isPortable ? "Portable" : options.m_target;

            switch (options.m_format)
            {
            case hctPlatformWriterOptions::FORMAT_PORTABLE:
            {
                res = hkSerialize::Save().contentsPtr(container, f.getStreamWriter());
                break;
            }
            case hctPlatformWriterOptions::FORMAT_XML:
            {
                res = hkSerialize::Save().withFormat<hkSerialize::XmlWriteFormat>().contentsPtr(container, f.getStreamWriter());
                break;
            }
            case hctPlatformWriterOptions::FORMAT_NATIVE:
            {
                if (hkString::strChr(HAVOK_VERSION_RELEASE, '$'))
                {
                    HK_WARN_ALWAYS(
                        0xabbab0ba, "Serializing Pack File using $$ version [" << HAVOK_REFLECTION_VERSION_STRING << "]. Not a good idea.");
                }
                res = hkSerialize::Save().withTarget(options.m_target).contentsPtr(container, f.getStreamWriter());
                break;
            }
            }

            if (res.isSuccess())
            {
                Log_Info("Wrote layout for [{}] to file [{}].", target, fullFilename.cString());
            }
            else
            {
                err.printf("Could not write layout for [%s] to file [%s]. File may now be corrupt.", target, fullFilename.cString());
                HK_WARN_ALWAYS(0xabba9832, err.cString());
            }
        }
        else
        {
            err.printf("Could not create file [%s] to write to. Aborting.", fullFilename.cString());
            HK_WARN_ALWAYS(0xabba9833, err.cString());
        }
    }
    else
    {
        HK_WARN_ALWAYS(0xabba4323, "Preset layout not valid. Aborting.");
    }

    if (numNonExportable > 0)
    {
        data.m_namedVariants.insertAt(data.m_namedVariants.getSize(), nonExportable.begin(), numNonExportable);
    }

    return res;
}

static bool _makeFileNameAndCheck(const hkStringBuf& toolPath, const hkStringBuf& toolBaseName, hkStringBuf& fullExeName)
{
    bool hasSlash = toolPath.endsWith("\\") || toolPath.endsWith("/");
    fullExeName = toolPath;
    fullExeName.append(!hasSlash ? "\\" : "");
    fullExeName.append(toolBaseName);

    hkIstream icheck(fullExeName.cString());
    return icheck.isOk();
}

static bool _findToolExe(const hkStringBuf& toolBaseName, hkStringBuf& fullExeName)
{
    // Check ENV VAR, then REG
    const char* filterToolPathEnv = ::getenv(_HAVOK_TOOLS_ENV_VAR);
    hkStringBuf filterToolPath;
    bool found = false;
    if (filterToolPathEnv)
    {
        found = _makeFileNameAndCheck(filterToolPathEnv, toolBaseName, fullExeName);
    }

    if (!found)
    {
        HKEY filterToolReg;
        DWORD dispos;
        BYTE filtersPathBuf[1024];
        DWORD filtersPathSize = 1024;

        // try the managed tool path root. This is only the case in dev when we set the path to the bin dir
        RegCreateKeyExA(
            HKEY_CURRENT_USER, _HAVOK_MANAGEDTOOLS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
            &filterToolReg, &dispos);

        if (RegQueryValueExA(filterToolReg, "ToolsDir", NULL, NULL, filtersPathBuf, &filtersPathSize) == ERROR_SUCCESS)
        {
            filterToolPath = (const char*)filtersPathBuf;
            found = _makeFileNameAndCheck(filterToolPath, toolBaseName, fullExeName);
        }
        RegCloseKey(filterToolReg);

        // try the filter manager path. This is the normal install location for users.
        if (!found)
        {
            RegCreateKeyExA(
                HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &filterToolReg, &dispos);

            if (RegQueryValueExA(filterToolReg, "FilterPath", NULL, NULL, filtersPathBuf, &filtersPathSize) == ERROR_SUCCESS)
            {
                filterToolPath = (const char*)filtersPathBuf;
                found = _makeFileNameAndCheck(filterToolPath, toolBaseName, fullExeName);
            }
            RegCloseKey(filterToolReg);
        }
    }
    return found;
}

hkResult hctFilterUtils::remoteUpdateTool(
    hkRootLevelContainer& data, const hctFilterToolStubRemoteOptions& options, const hkStringBuf& autoLaunchToolName)
{
    // Find a running instance of the Havok Tool (or anything else that is able to accept the data)
    // XX user selected through the options?
    // XX user selected IP address
    // XX user selected option to enable auto launch

    hkResult connectionResult = HK_FAILURE;

    hkSocket* socket = hkSocket::create();
    hkStringOld errorString;
    if (socket)
    {
        //XX User Options
        const char* remote = options.m_connectAddr;
        int port = options.m_connectPort; // VDB uses 25001 by default
        int sleepTime = options.m_sleepTimeInMs; //ms
        bool autoLaunchOnFail = options.m_allowAutoLaunch;
        int maxWaits = options.m_reconnectAttempts;

        int retryCount;

    RECONNECT:

        retryCount = 0;
        while (retryCount < maxWaits)
        {
            Log_Info("Trying to connect to [{}:{}]...", remote, port);
            connectionResult = socket->connect(remote, port);
            if (connectionResult.isFailure())
            {
                Sleep(sleepTime);
                retryCount++;
            }
            else
            {
                break;
            }
        }

        if (autoLaunchOnFail && (connectionResult.isFailure()))
        {
            Log_Info("Connect to [{}:{}] failed, trying to launch the standalone app locally instead.", remote, port);

            PROCESS_INFORMATION ProcessInfo;
            STARTUPINFOW StartupInfo = { 0 };
            StartupInfo.cb = sizeof(STARTUPINFO);

            hkStringBuf toolBaseName = !autoLaunchToolName.isEmpty() ? autoLaunchToolName : HCT_PREVIEW_TOOL_FILENAME;

            hkStringBuf appFriendlyName = (toolBaseName.compareTo(HCT_PREVIEW_TOOL_FILENAME) == 0) ?
                "Standalone Preview Tool" : autoLaunchToolName;

            hkStringBuf fullExeName;
            bool found = _findToolExe(toolBaseName, fullExeName);
            if (found)
            {
                wchar_t cmdLine[4096];
                int len = fullExeName.getLength() < 4095 ? fullExeName.getLength() + 1 : 4095;
                mbstowcs(cmdLine, fullExeName.cString(), len);
                cmdLine[4095] = TEXT('0');

                if (CreateProcessW(NULL, cmdLine, NULL, NULL, FALSE,
                    0, NULL, NULL, &StartupInfo, &ProcessInfo))
                {
                    //WaitForSingleObject(ProcessInfo.hProcess, INFINITE);

                    Log_Info(
                        "Started up new instance of the Standalone Preview Tool as requested, due to inactivity on communication socket.");

                    maxWaits *= 10;
                    if (maxWaits == 0)
                    {
                        maxWaits = 5;
                    }
                    else if (maxWaits > 20)
                    {
                        maxWaits = 20;
                    }

                    if (sleepTime < 0)
                    {
                        sleepTime = 1;
                    }

                    autoLaunchOnFail = false; // don't try again if this fails again.
                    remote = "localhost";
                    port = 25010;

                    goto RECONNECT;
                }
                else
                {
                    errorString.printf(
                        "Failed to start up new instance of the %s as requested (due to inactivity on communication socket). "
                        "Will not continue to try and remote preview. Was looking in : [%s]",
                        appFriendlyName.cString(), fullExeName.cString());

                    HK_WARN_ALWAYS(0xabba9933, errorString.cString());//..Won't see this as in a modeless filter dll
                }
            }
            else
            {
                errorString.printf("Failed to find the %s to startup. Can not remote preview.", appFriendlyName.cString());
                HK_WARN_ALWAYS(0xabba9933, errorString.cString());//..Won't see this as in a modeless filter dll
            }
        }

        if ((connectionResult.isSuccess()) && (socket->isOk()))
        {
            hkArray<char> buff;
            hkArrayStreamWriter asw(&buff, hkArrayStreamWriter::ARRAY_BORROW);

            hkResult res = hkSerializeUtil::saveTagfile(&data, &asw); //, false); //to dump ascii file

            if (res.isSuccess())
            {
                hkOArchive a(&(socket->getWriter()), hkBool(false) /*hkBool(lr.m_littleEndian != HK_ENDIAN_LITTLE)*/);
                res = HK_FAILURE;
                a.write32(buff.getSize());
                if (a.isOk())
                {
                    a.writeRaw(buff.begin(), buff.getSize());
                    if (a.isOk())
                    {
                        res = HK_SUCCESS;
                    }
                }
            }

            if (res.isSuccess())
            {
                Log_Info("Wrote scene data to socket [{}:{}].", remote, port);
            }
            else
            {
                errorString.printf("Could not write data to socket [%s:%d].", remote, port);
                HK_WARN_ALWAYS(0xabba9832, errorString.cString());//..Won't see this as in a modeless filter dll
            }
        }
        else
        {
            errorString.printf("Could not open connection to socket [%s:%d] after %d attempts.", remote, port, retryCount);
            HK_WARN_ALWAYS(0xabba9833, errorString.cString());//..Won't see this as in a modeless filter dll

        }
        socket->removeReference();
    }

    return connectionResult;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
