// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/Filters/Common/Error/hctFilterError.h>
#include <ContentTools/Common/Filters/Common/FilterManager/hctFilterDllManager.h>
#include <ContentTools/Common/Filters/Common/Options/hctFilterConfigurationSet.h>

#include <Common/Base/Serialize/Format/Compat/hkCompatFormats.h>

#include <Common/Base/Container/Tree/hkTree.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlWriteFormat.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.utils"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    void createDefaultConfig(hkArray<hctFilterConfigurationSet::Configuration>& configurations)
    {
        // Create a completely empty configuration, in case there's no data with the asset or in the registry.
        configurations.setSize(1);
        configurations[0].m_configName = "Default";
        configurations[0].m_filterStages.setSize(0);
    }
}


//
// Copy constructors
//

hctFilterConfigurationSet::Configuration::Configuration( const hctFilterConfigurationSet::Configuration& other )
:   m_configName(other.m_configName)
{
    m_filterStages.setSize( other.m_filterStages.getSize() );
    for( int i = 0; i < m_filterStages.getSize(); i++ )
    {
        m_filterStages[i] = other.m_filterStages[i];
    }
}



//
// Configuration set methods
//

hctFilterConfigurationSet::hctFilterConfigurationSet( const class hctFilterManagerInterface* filterManager )
:   m_filterManager( filterManager )
{
    createDefaultConfig(m_configurations);

    m_currentIndex = 0;
    m_overrideMode = MODE_NORMAL;
}

hctFilterConfigurationSet::~hctFilterConfigurationSet()
{
}

int hctFilterConfigurationSet::getData( void* optionData ) const
{
    // May want the original data we passed in
    if( m_overrideMode == MODE_OVERRIDE )
    {
        if (optionData)
        {
            hkString::memCpy( optionData, m_originalData.begin(), m_originalData.getSize() );
        }
        return m_originalData.getSize();
    }

    // Save out the configurations.
    hkArray<char>::Temp tempBuffer;
    hkArrayStreamWriter sw(&tempBuffer, hkMemoryRouter::getInstance().temp(), hkArrayStreamWriter::ARRAY_BORROW);
    hkSerialize::Save().withFormat<hkSerialize::XmlWriteFormat>().contentsPtr(this, &sw);

    if (optionData)
    {
        hkString::memCpy( optionData, tempBuffer.begin(), tempBuffer.getSize() );
    }
    return tempBuffer.getSize();
}

void hctFilterConfigurationSet::setData( const void* optionData, int optionDataSize )
{
    // Store original data
    m_originalData.setSize( optionDataSize );
    hkString::memCpy( m_originalData.begin(), optionData, optionDataSize );

    // Check for file-based data (update optionDataSize if found)
    m_overrideMode = MODE_NORMAL;
    hkArray<char> loadedData;
    {
        const hkStringOld havokPath( m_filterManager->getHavokPath() );

        // Check for an override file
        {
            hkStringOld overrideFilename( havokPath + "\\hkOverride.hko" );
            if( hkLoadUtil( overrideFilename.cString() ).toArray(loadedData) )
            {
                optionData = loadedData.begin();
                optionDataSize = loadedData.getSize();

                m_overrideMode = MODE_OVERRIDE;

                Log_Info( "Using settings from override options file: '{}'. Changes won't be saved.", overrideFilename.cString() );
            }
        }

        // Check for an upgrade file
        {
            hkStringOld upgradeFilename( havokPath + "\\hkUpgrade.hko" );
            if( hkLoadUtil( upgradeFilename.cString() ).toArray(loadedData) )
            {
                optionData = loadedData.begin();
                optionDataSize = loadedData.getSize();

                m_overrideMode = MODE_UPGRADE;

                Log_Info( "Using settings from upgrade options file: '{}'", upgradeFilename.cString() );
            }
        }
    }

    if ( optionDataSize <= 0 )
    {
        return;
    }

    hctFilterConfigurationSet* loaded = HK_NULL;

    {
        // We disable the long jump here because we are not calling setjmp anywhere in this code which
        // means if there is an error in any of the code below it will try to do a long jump to invalid
        // memory space (potentially). Another approach here is to call setjmp in numerous places to handle
        // the error more correctly, but it'd require changing a lot of code. Instead, we assume that
        // even if we assert the code following is safe. More details and history in EXP-2989.
        hctFilterErrorDisableExceptionHandler disableExceptionHandler;

        // Load options file.
        
        loaded = hkSerialize::Load()
            .withIgnoreInvalid().toObject<hctFilterConfigurationSet>(optionData, optionDataSize);

        if (!loaded && hkSerialize::Detail::detectFormat(optionData, optionDataSize) == HK_NULL)
        {
            // No known format, try compatibility loader.
            if (hkCompatFormats::getInstance().isLoaded())
            {
                // Check if this is a pre-2015.1 file.
                hkArray<char>::Temp convertedOptions;
                hkArrayStreamWriter sw(&convertedOptions, hkMemoryRouter::getInstance().temp(), hkArrayStreamWriter::ARRAY_BORROW);
                hkCompatFormats::getInstance().convertConfigurationSet(optionData, optionDataSize, &sw);

                // Load the upgraded options.
                loaded = hkSerialize::Load()
                    .withIgnoreInvalid().toObject<hctFilterConfigurationSet>(convertedOptions.begin(), convertedOptions.getSize());
            }
            else
            {
                HK_WARN_ALWAYS(0xabba1457, "Cannot try compatibility loader for filter options, a valid hkCompatFormats.dll was not found");
            }
        }
    }

    if (loaded)
    {
        m_configurations.swap(loaded->m_configurations);
        m_currentIndex = loaded->m_currentIndex;
        delete loaded;

        // Set the filter index on all configuration stages.
        for (int ci = 0; ci < m_configurations.getSize(); ++ci)
        {
            Configuration& config = m_configurations[ci];
            for (int fi = 0; fi < config.m_filterStages.getSize(); ++fi)
            {
                FilterStage& stage = config.m_filterStages[fi];
                stage.m_index = getFilterManager()->getDllManager()->getFilterIndexByID(stage.m_filterId);
                if (stage.m_index < 0)
                {
                    // Warn the user
                    hkStringOld str;
                    str.printf("Unknown filter found, ID = 0x%x. Could not load. Check for missing filter DLLs.", stage.m_filterId);
                    HK_WARN_ALWAYS(0xabba8111, str.cString());

                    // The filter has not been loaded, remove the stage
                    config.m_filterStages.removeAtAndCopy(fi, 1);
                    --fi;

                }
            }
        }
    }
    else
    {
        HK_WARN_ALWAYS(0xabba4f67, "Could not load the options file");
    }

    if (m_configurations.getSize() == 0)
    {
        createDefaultConfig(m_configurations);
    }

    // HKA-629
    if ( (m_currentIndex<0) || (m_currentIndex > (m_configurations.getSize()-1)) )
    {
        HK_WARN_ALWAYS( 0xabba5edc, "Current configuration index out of bounds" );
        m_currentIndex = 0;
    }
}

bool hctFilterConfigurationSet::hasDataChanged()
{
    // Get current data
    const int size = getData( HK_NULL );
    hkArray<char> currentData(size);
    getData( currentData.begin() );

    // Check if it differs from the original
    return (
        ( currentData.getSize() != m_originalData.getSize() ) ||
        ( hkString::memCmp( currentData.begin(), m_originalData.begin(), size ) != 0 )
    );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
