// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/Filters/Common/Error/hctFilterError.h>

#include <Common/Base/System/Io/OStream/hkOStream.h>
#include <Common/Base/System/Io/Platform/Stdio/hkStdioStreamWriter.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <richedit.h>
#include <Common/Base/Fwd/hkcstdio.h>

static jmp_buf s_hkJumpBuffer;

int hctFilterError::s_warnCount = 0;
int hctFilterError::s_errorCount = 0;

hctFilterError::hctFilterError(HWND dialog, const char* logFilePath, hkBool interactive)
    : m_interactive(interactive)
    , m_dialog(dialog)
    , m_logFile(HK_NULL)
    , m_logFileName("")
    , m_logVerboseLevel(0)
    , m_logFilePath(logFilePath)
    , m_stdOutput(false)
    , m_enableWarningFilter(true)
    , m_useJumpBuffer(true)
{
    m_indent = 0;
    m_ownerThreadId = hkThread::getMyThreadId();
}

hctFilterError::~hctFilterError()
{
    if (m_logFile)
    {
        delete m_logFile;
    }
}


int hctFilterError::message(Message type, int id, const char* description, const char* file, int line)
{
    const bool mainThread = m_ownerThreadId == hkThread::getMyThreadId();
    if (mainThread && (m_skippedLogEntries.getSize() > 0))
    {
        // Good time to flush queue messages before dealing with any new ones
        for (int si = 0; si < m_skippedLogEntries.getSize(); ++si)
        {
            addEntryToWindow(m_log[m_skippedLogEntries[si]]);
        }

        m_skippedLogEntries.clear();
        UpdateWindow(m_dialog);
    }

    // Only report tools warnings
    // This avoids dll memory issues when running in debug mode
    if (type == MESSAGE_WARNING)
    {
        if (id == 0xf0f0f0f0)
        {
            m_enableWarningFilter = false;
            return 0;
        }

        if (id == 0xf0f0f0f1)
        {
            m_enableWarningFilter = true;
            return 0;
        }

        if (m_enableWarningFilter)
        {
            if ((id & 0xffff0000) != 0xabba0000)
            {
                return 0;
            }
        }
    }

    // Verify the log file, first time only
    static bool logFileVerified = false;
    if (!logFileVerified)
    {
        if (m_logFile && !m_logFile->isOk())
        {
            hkStringOld str = "Error opening '";
            str += m_logFileName;
            str += "' for writing. No log file will be written.";

            LogEntry entry(MESSAGE_WARNING, 0, str.cString(), HK_NULL, 0);
            addEntryToWindow(entry);
        }
        logFileVerified = true;
    }

    // Skip disabled ids
    if (!isEnabled(id))
    {
        return 0;
    }

    // EXP-1331 count warnings
    switch (type)
    {
    case hkError::MESSAGE_WARNING:
    {
        s_warnCount++;
    }
    break;
    case hkError::MESSAGE_ERROR:
    {
        s_errorCount++;
    }
    break;
    }

    // Create the new log entry
    LogEntry entry(type, id, description, file, line);

    // If a custom color was provided through the id
    if (type == hkError::MESSAGE_REPORT && id != -1 && (id <= 0x00ffffff) && (id >= 0))
    {
        entry.m_customColor = true;
        entry.m_color = (COLORREF)id;
    }

    // Add any warnings/errors or coloured text (must be important) to the log file (before indentation)
    if (entry.m_text.getLength() > 0 && m_logVerboseLevel > 0)
    {
        if (m_logVerboseLevel == 1)
        {
            if (type != MESSAGE_REPORT || entry.m_customColor)
            {
                printToFile(entry.m_text);
            }
        }
        else // 2 or higher
        {
            printToFile(entry.m_text);
        }
    }

    // Prefix the current indent
    for (int i = 0; i < m_indent; ++i)
    {
        entry.m_text = hkStringOld("    ") + entry.m_text;
    }

    // Store the entry
    m_criticalAccess.enter();
    m_log.pushBack(entry);
    const int lastAddedIdx = m_log.getSize() - 1;
    m_criticalAccess.leave();

    if (m_interactive)
    {
        if (!mainThread)
        {
            m_criticalAccess.enter();
            m_skippedLogEntries.pushBack(lastAddedIdx);
            m_criticalAccess.leave();
        }
        else
        {
            // Add it to the window
            addEntryToWindow(entry);
            UpdateWindow(m_dialog);
        }
    }
    else if (m_stdOutput)
    {
        hkStringOld text(entry.m_text);
        text += "\n";
        fputs(text.cString(), type != MESSAGE_ERROR ? stdout : stderr);
    }

    // In release 'jump' on errors - this allows the filter manager to cleanly exit
    // (with a possible memory leak)
    // In debug, force a breakpoint so that user can debug the code properly.
    if (type == MESSAGE_ERROR || type == MESSAGE_ASSERT)
    {
        m_enableWarningFilter = true;
#ifdef HK_DEBUG
        HK_BREAKPOINT(0);
#endif
        if (mainThread && m_useJumpBuffer)
        {
            longjmp(s_hkJumpBuffer, 1);
        }
        // else hope for the best.. but will prob crash.. :(
    }

    return false;
}


jmp_buf* hctFilterError::getJumpBuffer()
{
    return &s_hkJumpBuffer;
}

void hctFilterError::disableJumpBuffer()
{
    m_useJumpBuffer = false;
}

void hctFilterError::enableJumpBuffer()
{
    m_useJumpBuffer = true;
}

void hctFilterError::setWindow(HWND dialog)
{
    m_dialog = dialog;
}

void hctFilterError::refreshWindow()
{
    if (m_dialog)
    {
        SetWindowText(m_dialog, "");

        for (int i = 0; i < m_log.getSize(); ++i)
        {
            addEntryToWindow(m_log[i]);
        }
        m_skippedLogEntries.clear();

        UpdateWindow(m_dialog);
    }
}

void hctFilterError::clearLog()
{
    if (m_dialog)
    {
        SetWindowText(m_dialog, "");
    }
    m_skippedLogEntries.clear();
    m_log.clear();
}

/*virtual*/ void hctFilterError::setLogFile(const char* logFile)
{
    if (logFile == HK_NULL)
        return;

    m_logFileName = logFile;

    hkStringBuf logLocation(m_logFilePath);
    logLocation += "\\";
    logLocation += logFile;

    hkRefPtr<hkStreamWriter> appendWriter = hkFileSystem::getInstance().openWriter(logLocation.cString(), hkFileSystem::OpenFlags(hkFileSystem::OPEN_DEFAULT_WRITE & (~hkFileSystem::OPEN_TRUNCATE)));
    if (appendWriter)
    {
        appendWriter->seek(0, hkStreamWriter::STREAM_END);
    }

    m_logFile = new hkOstream(appendWriter);
}

/*virtual*/ void hctFilterError::setLogVerboseLevel(unsigned int logVerboseLevel)
{
    m_logVerboseLevel = logVerboseLevel;
}

/*virtual*/ void hctFilterError::setStdOutput(bool output)
{
    m_stdOutput = output;
}

void hctFilterError::addEntryToWindow(const hctFilterError::LogEntry& entry)
{
    hctPrintLogEntryToRichEditControl(entry, m_dialog);
}

void hctFilterError::printToFile(hkStringOld errMsg)
{
    if (m_logFile && m_logFile->isOk() && m_logVerboseLevel > 0)
    {
        m_logFile->printf("%s\r\n", errMsg.cString());
    }
}

hctFilterErrorDisableExceptionHandler::hctFilterErrorDisableExceptionHandler()
{
    hctFilterError *filterManagerError = dynamic_cast<hctFilterError*>(&hkError::getInstance());
    HK_ASSERT(0x10733def, filterManagerError != HK_NULL, "You should not be using hctFilterErrorDisableExceptionHandler unless you are in filter code");
    if (filterManagerError)
        filterManagerError->disableJumpBuffer();
}

hctFilterErrorDisableExceptionHandler::~hctFilterErrorDisableExceptionHandler()
{
    hctFilterError *filterManagerError = dynamic_cast<hctFilterError*>(&hkError::getInstance());
    if (filterManagerError)
        filterManagerError->enableJumpBuffer();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
