// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : CONVEX_DECOMPOSITION+PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/ConvexDecompUtils/hctConvexDecompUtils.h>
#include <ContentTools/Common/ConvexDecompUtils/hctConvexDecompUtilsImpl.h>
#include <ContentTools/Common/Filters/Common/hctBaseDll.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>
#include <Common/Internal/ConvexHull/hkGeometryUtility.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <Common/Base/KeyCode.h>

extern HINSTANCE hInstance;

//
// DLL loading
//

hctBaseDll* HK_CALL getBaseDllInterface()
{
    return new hctBaseDll(hInstance); // no need to sub class as it has everything we need already
}

hctConvexDecompUtils* HK_CALL getConvexDecompUtilInterface()
{
    return new hctConvexDecompUtilsImpl();
}

hctConvexDecompUtils::DllError HK_CALL getConvexDecompDllError()
{
    return hctConvexDecompUtils::DLL_NO_ERROR;
}

//
// Progress bar events
//

bool hctConvexDecompProgress::startJob( int index, int totalJobs ) const
{
    if ( m_areSeparateDecompositions )
    {
        if ( 1 == totalJobs )
        {
            m_areSeparateDecompositions = false;
            return true;
        }

        // Start progress bar on first job
        if ( 0 == m_progressJobHandler->m_nJobs )
        {
            m_progressJobHandler->m_nFinished = 0;
            m_progressJobHandler->m_nJobs = totalJobs;
            return true;
        }

        // If not first job, allow user cancel
        return !didUserCancel();
    }
    return true;
}

void hctConvexDecompProgress::endJob() const
{
    if ( m_areSeparateDecompositions && !m_progressJobHandler->m_isCancelled_mainThread )
    {
        ++m_progressJobHandler->m_nFinished;

        float percent = ( m_progressJobHandler->m_nFinished / (float)max( m_progressJobHandler->m_nJobs, 1 ) ) * 100;
        percent = max( 0.0f, min( 99.0f, percent ) );
        hkStringBuf buf;
        buf.printf( "Decomposing shape %d", m_progressJobHandler->m_nFinished );

        m_progressJobHandler->setJob( buf.cString(), static_cast<int>(percent) );
    }
}

bool hctConvexDecompProgress::didUserCancel() const
{
    bool b = false;
    m_progressJobHandler->m_interfaceLock->enter();
    b = m_progressJobHandler->m_isCancelled;
    if ( b )
    {
        m_progressJobHandler->m_wasCancelPressed = true;
    }
    m_progressJobHandler->m_interfaceLock->leave();
    return b;
}

bool hctConvexDecompProgress::onDecomposePlanes( int currentPlane, int totalPlanes ) const
{
    if ( !m_areSeparateDecompositions )
    {
        float percent = (currentPlane / (float)max(totalPlanes, 1)) * 100;
        percent = max( 0.0f, min( 99.0f, percent ) );
        hkStringBuf buf;
        buf.printf( "Decomposing plane %d", currentPlane );

        m_progressJobHandler->setJob( buf.cString(), static_cast<int>(percent) );
        return !didUserCancel();
    }
    return true;
}

bool hctConvexDecompProgress::onReduce( int currentHullCount,int totalHulls ) const
{
    if ( !m_areSeparateDecompositions )
    {
        float percent = ( currentHullCount / (float)max( totalHulls, 1 ) ) * 100.0f;
        percent = max( 0.0f, min( 99.0f, percent ) );
        hkStringBuf buf;
        buf.printf( "Reducing (n hulls %d)", currentHullCount );

        m_progressJobHandler->setJob( buf.cString(), static_cast<int>(percent) );
        return !didUserCancel();
    }
    return true;
}

bool hctConvexDecompProgress::onDecimate( int currentHullCount, int objectiveHullsCount ) const
{
    if ( !m_areSeparateDecompositions )
    {
        float percent;
        if ( 0 == m_nOrigHulls )
        {
            percent = ( objectiveHullsCount / (float)max( currentHullCount, 1 ) ) * 100.0f;
        }
        else
        {
            percent = ( currentHullCount / (float)max( m_nOrigHulls, 1 ) ) * 100;
        }
        percent = max( 0.0f, min( 99.0f, percent ) );
        hkStringBuf buf;
        buf.printf( "Decimating (n hulls %d)", currentHullCount );

        m_progressJobHandler->setJob( buf.cString(), static_cast<int>(percent) );
        return !didUserCancel();
    }
    return true;
}

//
// Convex Decomposition
//

hctConvexDecompUtilsImpl::hctConvexDecompUtilsImpl()
{
}

hctConvexDecompUtilsImpl::~hctConvexDecompUtilsImpl()
{
}

struct hctConvexDecompThreadParams
{
    hkgpConvexDecomposition::Config*    m_config;
    bool                                m_parts;
    const hkGeometry*                   m_geometry;
    hkArray< hkArray< hkVector4 > >*    m_shapesOut;
    hctConvexDecompProgressHandler*     m_progressJobHandler;
    hkArray< hkArray< hkVector4 > >**   m_shapesToCombine;
    hkSceneExportMemThreadCallback*     m_threadCallback;

    hctConvexDecompThreadParams() {}
};

void* _decompose( void* arg )
{
    hkMemoryRouter router;
    hkMemorySystem::getInstance().threadInit( router, "decompose" );
    hkBaseSystem::initThread( &router );
    {
        hctConvexDecompThreadParams* params = (hctConvexDecompThreadParams*)arg;

        // Callback to set this memory router back in the modeler dll
        if( params->m_threadCallback )
        {
            params->m_threadCallback->initThread( &router );
        }

        hkArray< hkArray< hkVector4 > >* shapesToCombine = HK_NULL;
        if ( params->m_shapesToCombine )
        {
            shapesToCombine = *params->m_shapesToCombine;
        }

        // Call convex decomposition
        //hkgpConvexDecomposition::Config   CD_config = *params->m_config;
        //CD_config.m_mergeAllParts =   params->m_parts?false:true;
        params->m_config->m_mergeAllParts   =   (hkBool) (params->m_parts?false:true);
        if(shapesToCombine)
        {
            params->m_shapesOut->clear();
            *params->m_shapesOut = *shapesToCombine;
        }
        hkgpConvexDecomposition::decompose( *params->m_config, *params->m_geometry, *params->m_shapesOut/*, shapesToCombine*/ );
        if ( params->m_progressJobHandler )
        {
            // Tell progress bar job-handler to stop waiting for jobs
            params->m_progressJobHandler->setJob( "quit", 0 );
        }

        // Release semaphore so that main thread will continue executing
        params->m_progressJobHandler->m_endThreadEvent->release();
    }
    hkBaseSystem::quitThread();
    hkMemorySystem::getInstance().threadQuit( router );

    return 0;
}

bool hctConvexDecompUtilsImpl::decompose( const hctConvexDecompUtils::Input& input, const hkGeometry& geometry, hkArray<hkArray<hkVector4> >& shapesOut )
{
    hctConvexDecompProgress progressUpdater;
    {
        if ( input.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler = input.m_progressJobHandler;
            progressUpdater.m_progressJobHandler->m_isCombine = false;
            progressUpdater.m_areSeparateDecompositions = input.m_parts;
        }
    }

    hkgpConvexDecomposition::Config config;
    {
        config.m_accuracy = input.m_accuracy;
        config.m_sphereGuards.reserve(0);
        config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_GLOBAL;
        config.m_reduceOverlapsPasses = input.m_reduceOverlapsPasses;
        config.m_decimateOverlapsPasses = input.m_reduceOverlapsPasses;
        config.m_primaryMethod = hkgpConvexDecomposition::Config::SURFACE;
        config.m_useMaterialBoundaries = input.m_useMaterialBoundaries;
        config.m_guardsGenerator.m_method = (hkgpConvexDecomposition::GuardGenConfig::eMethod)input.m_guardsGenerator.m_method;
        config.m_guardsGenerator.m_offset = input.m_guardsGenerator.m_offset;
        config.m_guardsGenerator.m_edgeSamplingSpacing = input.m_guardsGenerator.m_edgeSamplingSpacing;
        config.m_guardsGenerator.m_maxGlobalGuardsOctreeDepth = input.m_guardsGenerator.m_maxGlobalGuardsOctreeDepth;
        config.m_maxVertices = input.m_maxVertices;

        switch(input.m_method)
        {
            case    0:  config.m_primaryMethod = hkgpConvexDecomposition::Config::SURFACE;break;
            case    1:  config.m_primaryMethod = hkgpConvexDecomposition::Config::SOLID;break;
            case    2:  config.m_primaryMethod = hkgpConvexDecomposition::Config::WRAP;break;
        }

        switch(input.m_reduceMethod)
        {
            case    0:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_DISABLE;break;
            case    1:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_LOCAL;break;
            case    2:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_GLOBAL;break;
        }

        config.m_maxWrapIterations  =   input.m_wrapIterations;
        config.m_maxWrapConcacity   =   input.m_wrapConcavity;

        if ( input.m_progressJobHandler )
        {
            config.m_iprogress = &progressUpdater;
        }
        else
        {
            config.m_iprogress = NULL;
        }
    }

    hctConvexDecompThreadParams threadParams;
    {
        threadParams.m_config = &config;
        threadParams.m_parts = input.m_parts;
        threadParams.m_geometry = &geometry;
        threadParams.m_shapesOut = &shapesOut;
        threadParams.m_shapesToCombine = HK_NULL;
        threadParams.m_progressJobHandler = input.m_progressJobHandler;
        threadParams.m_threadCallback = m_threadCallback;
    }

    bool result = true;
    hkThread* decomposeThread = new hkThread();
    {
        if ( progressUpdater.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler->setupBar();
        }

        // Call utility
        decomposeThread->startThread( &_decompose, &threadParams, "decomposing" );

        if ( progressUpdater.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler->processJobs();
            progressUpdater.m_progressJobHandler->m_endThreadEvent->acquire();
            result = !progressUpdater.m_progressJobHandler->m_wasCancelPressed;
            progressUpdater.m_progressJobHandler->cleanupBar( result );
        }

        delete decomposeThread;
    }

    return result;
}

bool hctConvexDecompUtilsImpl::combine( const hctConvexDecompUtils::Input& input, const hkGeometry& geometry, hkArray< hkArray< hkVector4 > >& shapesOut, hkArray< hkArray< hkVector4 > >* shapesToCombine )
{
    hctConvexDecompProgress progressUpdater;
    {
        if ( input.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler = input.m_progressJobHandler;
            progressUpdater.m_progressJobHandler->m_isCombine = true;
            progressUpdater.m_areSeparateDecompositions = false;

            if ( GuardGenConfig::NONE != input.m_guardsGenerator.m_method )
            {
                progressUpdater.m_nOrigHulls = shapesToCombine->getSize();
            }
            else
            {
                progressUpdater.m_nOrigHulls = 0;
            }
        }
    }

    hkgpConvexDecomposition::Config config;
    {
        config.m_accuracy = input.m_accuracy;
        config.m_sphereGuards.reserve(0);
        config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_GLOBAL;
        config.m_reduceOverlapsPasses = input.m_reduceOverlapsPasses;
        config.m_decimateOverlapsPasses = input.m_reduceOverlapsPasses;
        config.m_useMaterialBoundaries = input.m_useMaterialBoundaries;
        config.m_guardsGenerator.m_method = (hkgpConvexDecomposition::GuardGenConfig::eMethod)input.m_guardsGenerator.m_method;
        config.m_guardsGenerator.m_offset = input.m_guardsGenerator.m_offset;
        config.m_guardsGenerator.m_edgeSamplingSpacing = input.m_guardsGenerator.m_edgeSamplingSpacing;
        config.m_guardsGenerator.m_maxGlobalGuardsOctreeDepth = input.m_guardsGenerator.m_maxGlobalGuardsOctreeDepth;

        switch(input.m_reduceMethod)
        {
            case    0:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_DISABLE;break;
            case    1:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_LOCAL;break;
            case    2:  config.m_reduceMethod = hkgpConvexDecomposition::Config::REDUCE_GLOBAL;break;
        }

        if ( input.m_progressJobHandler )
        {
            config.m_iprogress = &progressUpdater;
        }
        else
        {
            config.m_iprogress = NULL;
        }
    }

    hctConvexDecompThreadParams threadParams;
    {
        threadParams.m_config = &config;
        threadParams.m_parts = input.m_parts;
        threadParams.m_geometry = &geometry;
        threadParams.m_shapesOut = &shapesOut;
        threadParams.m_shapesToCombine = &shapesToCombine;
        threadParams.m_progressJobHandler = input.m_progressJobHandler;
        threadParams.m_threadCallback = m_threadCallback;
    }

    bool result = true;
    hkThread* decomposeThread = new hkThread();
    {
        if ( progressUpdater.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler->setupBar();
        }

        // Call utility
        decomposeThread->startThread( &_decompose, &threadParams, "combining" );

        if ( progressUpdater.m_progressJobHandler )
        {
            progressUpdater.m_progressJobHandler->processJobs();
            progressUpdater.m_progressJobHandler->m_endThreadEvent->acquire();
            result = !progressUpdater.m_progressJobHandler->m_wasCancelPressed;
            progressUpdater.m_progressJobHandler->cleanupBar( result );
        }

        delete decomposeThread;
    }

    return result;
}

//
// Geometry
//

void hctConvexDecompUtilsImpl::createConvexGeometry( hkArray< hkVector4 >& shape, hkGeometry& shapeGeometry )
{
    hkStridedVertices stridedVertices;
    stridedVertices.m_numVertices = shape.getSize();
    stridedVertices.m_vertices = &((shape)[0](0));
    stridedVertices.m_striding = sizeof(hkVector4);

    hkGeometryUtility::createConvexGeometry( stridedVertices, shapeGeometry );
}

int hctConvexDecompUtilsImpl::getConvexHullPlanes( const hkGeometry& geometry )
{
    hkStridedVertices stridedVertices;
    {
        stridedVertices.m_numVertices = geometry.m_vertices.getSize();
        stridedVertices.m_striding = sizeof( hkVector4 );
        stridedVertices.m_vertices = &( geometry.m_vertices[0](0) );
    }

    hkGeometry geometryOut;
    hkArray<hkVector4> planes;

    hkGeometryUtility::createConvexGeometry( stridedVertices, geometryOut, planes );

    return planes.getSize();
}

void hctConvexDecompUtilsImpl::mergeGeometries(const hkGeometry& input,hkGeometry& output)
{
    const int ibase = output.m_vertices.getSize();
    hkVector4* vertices = output.m_vertices.expandBy( input.m_vertices.getSize() );
    hkGeometry::Triangle* triangles = output.m_triangles.expandBy( input.m_triangles.getSize() );

    for ( int i = 0; i < input.m_vertices.getSize(); ++i )
    {
        vertices[i] = input.m_vertices[i];
    }
    for ( int i = 0; i < input.m_triangles.getSize(); ++i )
    {
        triangles[i].m_a = input.m_triangles[i].m_a + ibase;
        triangles[i].m_b = input.m_triangles[i].m_b + ibase;
        triangles[i].m_c = input.m_triangles[i].m_c + ibase;
    }
}

hkReal hctConvexDecompUtilsImpl::normalizeGeometry( hkGeometry& geom, hkReal size, hkTransform* normalizedToWorld, hkVector4* normalizeCenter )
{
    if ( geom.m_vertices.getSize() == 0 )
    {
        return 0;
    }

    hkAabb box;
    box.m_min=geom.m_vertices[0];
    box.m_max=geom.m_vertices[0];
    for( int i = 1; i < geom.m_vertices.getSize(); ++i )
    {
        box.m_max.setMax4( box.m_max, geom.m_vertices[i] );
        box.m_min.setMin4( box.m_min, geom.m_vertices[i] );
    }

    hkVector4 ext;
    ext.setSub4( box.m_max, box.m_min );
    hkVector4 ctr;
    ctr.setAdd4( box.m_max, box.m_min );
    ctr.mul4( 0.5f );
    hkReal scl = 1 / ext( ext.getMajorAxis3() ) * size;

    *normalizeCenter = ctr;

    for(int i = 0; i < geom.m_vertices.getSize(); ++i)
    {
        geom.m_vertices[i].sub4( ctr );
        geom.m_vertices[i].mul4( scl );
    }

    if ( normalizedToWorld )
    {
        *normalizedToWorld = hkTransform::getIdentity();
        hkReal invScl = 1.0f / scl;
        hkMatrix3Util::_setDiagonal( invScl, invScl, invScl, normalizedToWorld->getRotation() );
        normalizedToWorld->getTranslation() = ctr;
    }

    return scl;
}

void hctConvexDecompUtilsImpl::normalizeShapes( hkArray< hkArray< hkVector4 > >& shapes, hkReal scale, hkVector4& normalizeCenter )
{
    for( int ai = 0; ai < shapes.getSize(); ++ai )
    {
        hkArray< hkVector4 >& shape = shapes[ai];

        for ( int vi = 0; vi < shape.getSize(); ++vi )
        {
            shape[vi].sub4( normalizeCenter );
            shape[vi].mul4( scale );
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
