// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkVisualDebuggerReporting.h>
#include <Common/Visualize/hkServerTextHandler.h>

#include <Common/Base/System/Error/hkErrorRouter.h>
#include <Common/Base/System/StackTracer/hkStackTracer.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

//
// DeferredOutputFwd
//

#define MAX_STACK_TRACE_LEVEL 128

class DeferredOutputFwd : public hkVisualDebuggerReporter::Output
{
public:
    HK_DECLARE_CLASS( DeferredOutputFwd, New );
    DeferredOutputFwd( hkVisualDebuggerReporter::Output& output ) :
        m_criticalSection( 1000 ),
        m_callTree( hkMemHeapAllocator() ),
        m_output( output ),
        m_maxStackTraceLevel( hkLog::Level::Error )
    {}
    virtual ~DeferredOutputFwd() {}

    void setMaximumStackTraceLevel( hkLog::Level::Enum lvl )
    {
        hkCriticalSectionLock lock( &m_criticalSection );
        m_maxStackTraceLevel = lvl;
    }

    hkLog::Level::Enum getMaximumStackTraceLevel()
    {
        hkCriticalSectionLock lock( &m_criticalSection );
        return m_maxStackTraceLevel;
    }

    virtual hkResult outputText( hkUint64 id, hkLog::Level::Enum level, const char* text, int tag ) HK_OVERRIDE
    {
        hkCriticalSectionLock lock( &m_criticalSection );
        if ( level <= m_maxStackTraceLevel )
        {
            hkStringPtr& textStorage = m_outputQueueStorage.emplaceBack( text );
            hkUlong trace[MAX_STACK_TRACE_LEVEL];
            int ntrace = m_tracer.getStackTrace( trace, MAX_STACK_TRACE_LEVEL );
            hkStackTracer::CallTree::TraceId traceId =
                ( ntrace > 0 ) ?
                m_callTree.insertCallStack( trace, ntrace ) :
                -1;
            m_outputQueue.emplaceBack( id, level, textStorage.cString(), tag, traceId );
        }
        else
        {
            hkStringPtr& textStorage = m_outputQueueStorage.emplaceBack( text );
            m_outputQueue.emplaceBack( id, level, textStorage.cString(), tag, -1 );
        }
        return HK_SUCCESS;
    }

    virtual hkResult flush() HK_OVERRIDE
    {
        bool success = true;
        {
            hkCriticalSectionLock lock( &m_criticalSection );
            hkStringBuf buf;
            for ( auto& args : m_outputQueue )
            {
                if ( args.m_4 >= 0 )
                {
                    buf.clear();
                    buf.set( args.m_2 );
                    buf.append( "Stack trace is:\n" );
                    {
                        hkUlong trace[MAX_STACK_TRACE_LEVEL];
                        const int ntrace = m_callTree.getCallStack( args.m_4, trace, MAX_STACK_TRACE_LEVEL );
                        m_tracer.dumpStackTrace(
                            
                            // It'd be nice to skip the low-level framework of hkError and hkLog...
                            // Doing it in a performant way will likely mean a custom append.
                            // Alternatively, could store off known hkUlong addrs for entry points
                            // as determined in a ctor?
                            // For now we just add 3 to accommodate the reporter
                            trace + 3,
                            ntrace - 3,
                            []( const char* s, void* handle )
                            {
                                hkStringBuf* bufPtr = reinterpret_cast< hkStringBuf* >( handle );
                                bufPtr->append( s );
                            },
                            &buf );
                    }
                    buf.append( "Stack trace end\n" );
                    success &= ( m_output.outputText( args.m_0, args.m_1, buf.cString(), args.m_3 ).isSuccess() );
                }
                else
                {
                    success &= ( m_output.outputText( args.m_0, args.m_1, args.m_2, args.m_3 ).isSuccess() );
                }
            }
            m_outputQueue.clear();
            m_outputQueueStorage.clear();
            m_callTree.quit(); m_callTree.init( hkMemHeapAllocator() );
        }
        success &= m_output.flush().isSuccess();
        return success ? HK_SUCCESS : HK_FAILURE;
    }

    hkCriticalSection m_criticalSection;
    hkStackTracer m_tracer;
    hkStackTracer::CallTree m_callTree;
    hkVisualDebuggerReporter::Output& m_output;
    hkLog::Level::Enum m_maxStackTraceLevel;
    hkArray<hkTuple<hkUint64, hkLog::Level::Enum, const char*, int, hkStackTracer::CallTree::TraceId>> m_outputQueue;
    hkArray<hkStringPtr> m_outputQueueStorage;
};

//
// ErrorOutput
//

hkVisualDebuggerReporter::ErrorReporter::ErrorReporter( hkVisualDebuggerReporter::Output& output ) :
    m_output( output ),
    m_reportDisabledMessages( true )
{
    // Just to make sure if we change some defaults to hkDefaultError in the future, etc.
    // we will always get everything coming through.
    hkDefaultError::enableAll();
    hkDefaultError::setMinimumMessageLevel( hkError::MESSAGE_ALL );
}

hkBool hkVisualDebuggerReporter::ErrorReporter::isEnabled( int id )
{
    hkCriticalSectionLock lock( &m_lock );

    return
        // Check for someone explicitly disabled it *on* the Error reporter
        hkDefaultError::isEnabled( id )
        // If no one's explicitly disabled it, see if we are reporting
        // framework-disabled errors; and if not, fall back on installed hkError.
        && ( m_reportDisabledMessages || hkError::getInstance().isEnabled( id ) );
}

hkError::Message hkVisualDebuggerReporter::ErrorReporter::getMinimumMessageLevel()
{
    hkCriticalSectionLock lock( &m_lock );

    // Start with the min message level set *on* the Error reporter.
    int minMessageLevel = hkDefaultError::getMinimumMessageLevel();

    // Allow the installed error handler to reduce this if we aren't reporting all errors.
    if ( !m_reportDisabledMessages )
    {
        minMessageLevel = hkMath::max2( minMessageLevel, hkError::getInstance().getMinimumMessageLevel() );
    }

    return hkError::Message( minMessageLevel );
}

void hkVisualDebuggerReporter::ErrorReporter::reportDisabledMessages( hkBool32 on )
{
    hkCriticalSectionLock lock( &m_lock );
    m_reportDisabledMessages = on;
}

hkBool32 hkVisualDebuggerReporter::ErrorReporter::isReportingDisabledMessages() const
{
    hkCriticalSectionLock lock( &m_lock );
    return m_reportDisabledMessages;
}

void hkVisualDebuggerReporter::ErrorReporter::showMessage(
    hkError::Message msg,
    int id,
    const char* desc,
    const char* file,
    int line ) const
{
    hkLog::Level::Enum level = getMessageLogLevel( msg );
    if ( level == hkLog::Level::Disabled )
    {
        return;
    }

    hkStringBuf buf;
    hkDefaultError::formMessage( msg, id, desc, file, line, buf );
    if ( buf.getLength() )
    {
        // Output our text
        // We use -1 for tag to indicate that this was a disabled message.
        m_output.outputText( id, level, buf.cString(), hkError::getInstance().isEnabled( id ) ? 0 : -1 );

        
        
        
        
        
    }
}

hkLog::Level::Enum hkVisualDebuggerReporter::ErrorReporter::getMessageLogLevel( hkError::Message msg )
{
    // Always err on the side of being *more likely* to report
    // by choosing the lowest value level that makes sense.
    switch ( msg )
    {
        case MESSAGE_REPORT:
            return hkLog::Level::Info;
        case MESSAGE_WARNING:
            return hkLog::Level::Warning;
        case MESSAGE_ASSERT:
        case MESSAGE_ERROR:
        case MESSAGE_FATAL_ERROR:
            return hkLog::Level::Error;
        case MESSAGE_NONE:
        default:
            return hkLog::Level::Disabled;
    }
}

//
// LogOutput
//

hkVisualDebuggerReporter::LogReporter::LogReporter( hkVisualDebuggerReporter::Output& output ) :
    m_output( output )
{}

hkLog::Output::Action hkVisualDebuggerReporter::LogReporter::put( const hkLog::Message& msg )
{
    
    if ( ( msg.m_level > m_level ) || ( msg.m_level == hkLog::Level::Disabled ) )
    {
        return hkLog::Output::PropagateToNextOutput;
    }

    hkStringBuf buf;
    if ( getMessageText( msg, buf ) )
    {
        // Output our text
        m_output.outputText( msg.m_id, msg.m_level, buf.cString() );

        
        
        
        
        
    }

    return hkLog::Output::PropagateToNextOutput;
}

//
// hkVisualDebuggerReporter
//

hkVisualDebuggerReporter::hkVisualDebuggerReporter( hkServerTextHandler& handler ) :
    m_deferredOutputFwd( new DeferredOutputFwd( handler ) ),
    m_reporterError( new ErrorReporter( *m_deferredOutputFwd ) ),
    m_reporterLogOutput( new LogReporter( *m_deferredOutputFwd ) )
{
    hkErrorRouter::getInstance().addHandler( m_reporterError );
}

hkVisualDebuggerReporter::~hkVisualDebuggerReporter()
{
    hkErrorRouter::getInstance().removeHandler( m_reporterError );
    delete m_reporterError;
    delete m_reporterLogOutput;
    delete m_deferredOutputFwd;
    m_deferredOutputFwd = HK_NULL;
}

void hkVisualDebuggerReporter::setMaximumStackTraceLevel( hkLog::Level::Enum lvl )
{
    m_deferredOutputFwd->setMaximumStackTraceLevel( lvl );
}

hkLog::Level::Enum hkVisualDebuggerReporter::getMaximumStackTraceLevel()
{
    return m_deferredOutputFwd->getMaximumStackTraceLevel();
}

void hkVisualDebuggerReporter::flush()
{
    m_deferredOutputFwd->flush();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
