// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/hkBase.h>
#include <Common/Base/Math/Util/hkPack.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#ifndef HK_VDB_MAX_CMD_SIZE



/// The max size for a vdb cmd.  Cmds that are larger than this value are considered errors.
/// This should not be set to more than an hkArray of bytes can hold.
#define HK_VDB_MAX_CMD_SIZE ( hkArray<hkInt8>::CAPACITY_MASK )
#endif

#ifndef HK_VDB_RESERVED_PERSISTENT_ID_BITS
/// Reserved bits for use by the system to persist the cmd into a replay buffer.
/// This is based on the very high bar of accommodating a 1hr long recording at 60fps.
/// 1hr movie -> 3600sec movie -> 216000 movie frames (< 2^18-1)
#define HK_VDB_RESERVED_PERSISTENT_ID_BITS ( 18 )
#endif

/// Utils for use with cmds.
namespace hkVdbCmdUtils
{
    /// size used to send an (u)int8
    const int s8 = sizeof( hkInt8 );
    /// size used to send an (u)int16
    const int s16 = sizeof( hkInt16 );
    /// size used to send an (u)int32
    const int s32 = sizeof( hkInt32 );
    /// size used to send an (u)int64
    const int s64 = sizeof( hkInt64 );
    /// size used to send an float32
    const int sF = sizeof( hkFloat32 );
    /// size used to send an hkTransform
    const int sT = ( sF * 7 );
    /// size used to send an hkVector4
    const int sV = ( sF * 3 );
    /// size used to send an hkQuaternion
    const int sQ = ( sF * 4 );
    /// size used to send an hkMatrix4
    const int sM = ( sF * 16 );
    /// size used to send an hkAabb
    const int sA = ( sV * 2 );
}

//////////////////////////////////////////////////////////////////////////////////////////////
/// This namespace is used for constructing and verifying ids used in commands.
/// It's goal is to generate unique ids for various contexts.
//////////////////////////////////////////////////////////////////////////////////////////////
namespace hkVdbCmdId
{
    using namespace hkPack;

    //////////////////////////////////////////////////////////////////////////////////////////////
    /// Returns true if the id is okay to persist a cmd in the circular buffer.
    //////////////////////////////////////////////////////////////////////////////////////////////
    static HK_INLINE hkBool32 canIdPersist( hkUint64 id )
    {
        // id 0 is a special id which cannot persist
        return /*id && */( ( id >> ( ( sizeof( id ) * 8 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS ) ) == 0 );
    }

    //////////////////////////////////////////////////////////////////////////////////////////////
    /// Returns true if the ids are all okay to persist a cmd in the circular buffer.
    //////////////////////////////////////////////////////////////////////////////////////////////
    static HK_INLINE hkBool32 canIdsPersist( hkUint64* ids, int numIds )
    {
        for ( int i = 0; i < numIds; i++ ) { if ( !canIdPersist( ids[i] ) ) return false; }
        return true;
    }

    //////////////////////////////////////////////////////////////////////////////////////////////
    /// This function creates an id to be streamed over to the VDB.
    /// It's useful for creating ids that are unique within there given context (Eg. object handler).
    /// The id is packed as followed into 64 bits:
    /// | HK_VDB_RESERVED_PERSISTENT_ID_BITS | sizeof(Shared) | ->(variable)<- | sizeof(Variadic) |
    /// | 0                                  | head           | 0              | tail...          |
    //////////////////////////////////////////////////////////////////////////////////////////////
#ifdef HK_VARIADIC_TEMPLATES
    template<typename Shared, typename... Variadic>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head, const Variadic&... tail )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ON_DEBUG( hkUint8 bitsForTail = BitSizeOf<hkUint8>( tail... ); )
        HK_ASSERT(
            0x22441174,
            ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead + bitsForTail ) <= hkBitSizeOf( hkUint64 ),
            "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = Pack<hkUint64>( tail... );
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
#else
    template<typename Shared>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ASSERT(
            0x22441174,
            ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead ) <= hkBitSizeOf( hkUint64 ),
            "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = 0;
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
    template<typename Shared, typename T0>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head, const T0& tail0 )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ON_DEBUG( hkUint8 bitsForTail = BitSizeOf<hkUint8>( tail0 ); )
            HK_ASSERT(
                0x22441174,
                ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead + bitsForTail ) <= hkBitSizeOf( hkUint64 ),
                "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = Pack<hkUint64>( tail0 );
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
    template<typename Shared, typename T0, typename T1>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head, const T0& tail0, const T1& tail1 )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ON_DEBUG( hkUint8 bitsForTail = BitSizeOf<hkUint8>( tail0, tail1 ); )
            HK_ASSERT(
                0x22441174,
                ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead + bitsForTail ) <= hkBitSizeOf( hkUint64 ),
                "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = Pack<hkUint64>( tail0, tail1 );
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
    template<typename Shared, typename T0, typename T1, typename T2>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head, const T0& tail0, const T1& tail1, const T2& tail2 )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ON_DEBUG( hkUint8 bitsForTail = BitSizeOf<hkUint8>( tail0, tail1, tail2 ); )
            HK_ASSERT(
                0x22441174,
                ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead + bitsForTail ) <= hkBitSizeOf( hkUint64 ),
                "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = Pack<hkUint64>( tail0, tail1, tail2 );
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
    template<typename Shared, typename T0, typename T1, typename T2, typename T3>
    static HK_INLINE hkUint64 computeStreamId( const Shared& head, const T0& tail0, const T1& tail1, const T2& tail2, const T3& tail3 )
    {
        hkUint8 bitsForHead = BitSizeOf<hkUint8>( head );
        HK_ON_DEBUG( hkUint8 bitsForTail = BitSizeOf<hkUint8>( tail0, tail1, tail2, tail3 ); )
            HK_ASSERT(
                0x22441174,
                ( hkUint8( HK_VDB_RESERVED_PERSISTENT_ID_BITS ) + bitsForHead + bitsForTail ) <= hkBitSizeOf( hkUint64 ),
                "Not enough room in streamId" );
        hkUint64 header = ( hkUint64( head ) << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - bitsForHead ) );
        hkUint64 footer = Pack<hkUint64>( tail0, tail1, tail2, tail3 );
        hkUint64 streamId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441175, ( header ^ footer ) == streamId );
        HK_ASSERT( 0x22441176, hkVdbCmdId::canIdPersist( streamId ), "Id cannot persist" );
        return streamId;
    }
#endif

    //////////////////////////////////////////////////////////////////////////////////////////////
    /// This function creates a persistent id given the streamId and the frame.
    /// The id is packed as followed into 64 bits:
    /// | HK_VDB_RESERVED_PERSISTENT_ID_BITS | 64 - HK_VDB_RESERVED_PERSISTENT_ID_BITS |
    /// | frame (wrapped)                    | streamId                                |
    //////////////////////////////////////////////////////////////////////////////////////////////
    static HK_INLINE hkUint64 computePersistentId( hkUint32 frame, hkUint64 streamId )
    {
        HK_ASSERT( 0x22441144, hkVdbCmdId::canIdPersist( streamId ), "Stream id uses reserved bits" );
        hkUint64 maxReserved = hkBitmask64( HK_VDB_RESERVED_PERSISTENT_ID_BITS );
        hkUint64 uniqueFrame = ( hkUint64( frame ) % maxReserved ); // assumes circ buffer is large enough
        hkUint64 header = ( uniqueFrame << ( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS ) );
        hkUint64 footer = streamId;
        hkUint64 persistentId = ( header + footer );
        HK_ASSERT_NO_MSG( 0x22441177, ( header ^ footer ) == persistentId );
        // id 0 is a special id which cannot persist
        return ( streamId != 0 ) * persistentId;
    }

    //////////////////////////////////////////////////////////////////////////////////////////////
    /// This function masks out the persistent bits to return the streamId
    //////////////////////////////////////////////////////////////////////////////////////////////
    static HK_INLINE hkUint64 extractStreamId( hkUint64 persistentId )
    {
        hkUint64 mask = hkBitmask64( hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS );
        hkUint64 streamId = ( persistentId & mask );
        return streamId;
    }
}

//////////////////////////////////////////////////////////////////////////////////////////////
/// This struct describes a vdb cmd that has been read from a vdb server.
/// The commands are stored in a backing buffer and have the following layout:
/// | HEADER     | BODY                                                    | META              |
/// | m_bodySize | m_serverType, m_data[m_bodySize - sizeof(m_serverType)] | patchedType, etc. |
//////////////////////////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_COMMON hkVdbCmd
{
    /// Returns the cmd index in the frame in which it occurred.
    HK_INLINE hkUint32 getIndex() const
    {
        // By convention this is the second item in the meta data section of the backing store
        // which is located just passed the data section.
        HK_ASSERT_DEV( 0x22440988, getMetaDataSize() >= ( sizeof( hkUint8 ) + sizeof( hkUint32 ) ), "Index information was not stored" );
        hkUint32 idx = *reinterpret_cast< const hkUint32* >( reinterpret_cast< const hkInt8* >( this ) + getCmdSize() + sizeof( hkUint8 ) );
        return idx;
    }

    /// Sets the cmd index in the frame in which it occurred (advanced use only).
    HK_INLINE void setIndex( hkUint32 idx )
    {
        // By convention this is the second item in the meta data section of the backing store
        // which is located just passed the data section.
        HK_ASSERT_DEV( 0x22441130, getMetaDataSize() >= ( sizeof( hkUint8 ) + sizeof( hkUint32 ) ), "Index storage was not reserved" );
        *reinterpret_cast< hkUint32* >( reinterpret_cast< hkInt8* >( this ) + getCmdSize() + sizeof( hkUint8 ) ) = idx;
    }

    /// Returns the cmd's patched/resolved type.
    HK_INLINE hkVdbCmdType::Enum getType() const
    {
        // By convention this is the first item in the meta data section of the backing store
        // which is located just passed the data section.
        HK_ASSERT_DEV( 0x22440989, getMetaDataSize() >= sizeof( hkUint8 ), "Type information was not stored" );
        hkUint8 patchedType = *( reinterpret_cast< const hkInt8* >( this ) + getCmdSize() );
        return hkVdbCmdType::Enum( patchedType );
    }

    /// Sets the cmd's patched/resolved type (advanced use only).
    HK_INLINE void setType( hkVdbCmdType::Enum type )
    {
        // By convention this is the first item in the meta data section of the backing store
        // which is located just passed the data section.
        HK_ASSERT_DEV( 0x22441131, getMetaDataSize() >= sizeof( hkUint8 ), "Type storage was not reserved" );
        *( reinterpret_cast< hkInt8* >( this ) + getCmdSize() ) = hkInt8( type );
    }

    /// Returns the cmd's server type (directly from server stream).
    HK_INLINE hkUint8 getServerType() const
    {
        return m_serverType;
    }

    /// Returns the size of the header for the command.
    static HK_INLINE hkUint32 getHeaderSize()
    {
        return sizeof( BodySizeType );
    }

    /// Returns the size of the body of the command.
    HK_INLINE hkUint32 getBodySize() const
    {
        return m_bodySize;
    }

    /// Returns the size of the cmd's data.
    HK_INLINE hkUint32 getDataSize() const
    {
        // By convention m_size includes everything but the m_size member.
        // Subtract off the m_command storage
        HK_ASSERT_NO_MSG( 0x22441126, getBodySize() >= sizeof( m_serverType ) );
        return ( getBodySize() - sizeof( m_serverType ) );
    }

    /// Returns the size of this cmd, including header information.
    /// Note: this does not include the size of meta-data which shouldn't be included in copies for saving anyway.
    HK_INLINE hkUint32 getCmdSize() const
    {
        // By convention m_size includes everything but the m_size member.
        // Add in the m_size storage
        return ( getHeaderSize() + getBodySize() );
    }

    /// Returns the size of the meta data added to the command by the reader.
    /// If you are writing the cmd to be read back during a later session, then you shouldn't
    /// include the metadata, just getCmdSize(), as the meta data is specific to the client
    /// implementation/version that is reading the cmd.
    static HK_INLINE hkUint32 getMetaDataSize()
    {
        return META_DATA_SIZE;
    }

    /// Raw access to the cmd data.
    HK_INLINE const void* getData( int offset = 0 ) const
    {
        return ( m_data + offset );
    }

    /// Returns a complete copy of this cmd (including metadata)
    HK_INLINE hkVdbCmd* copy() const
    {
        const int sizeInBackingStore = ( getCmdSize() + getMetaDataSize() );
        hkInt8* dst = hkMemHeapBlockAlloc<hkInt8>( sizeInBackingStore );
        hkString::memCpy( dst, reinterpret_cast< const hkInt8* >( this ), sizeInBackingStore );
        return reinterpret_cast< hkVdbCmd* >( dst );
    }

    /// Destroys a previous copied cmd, not safe to call on original back-end data
    HK_INLINE void destroyCopy()
    {
        const int sizeInBackingStore = ( getCmdSize() + getMetaDataSize() );
        hkMemHeapBlockFree(
            reinterpret_cast< hkInt8* >( this ),
            sizeInBackingStore );
    }

    /// Verifies that the vdb cmd is okay (behavior is unspecified if the header hasn't been read)
    HK_INLINE hkBool32 isOk( bool metaDataIsValid = true ) const
    {
        const int type = metaDataIsValid ? (int)getType() : m_serverType;
        return
            ( type >= 0 ) && ( type <= getMaxTypeValue() ) &&
            ( ( getCmdSize() + getMetaDataSize() ) < HK_VDB_MAX_CMD_SIZE );
    }

    /// Return the maximum value that the cmd type can be set to
    static HK_INLINE int getMaxTypeValue()
    {
        hkVdbCmd cmd;
        return hkBitmask( hkBitSizeOf( cmd.m_serverType ) );
    }

    /// Size of extra data stored with the cmd during read: patched type + cmd index
    static const int META_DATA_SIZE = sizeof( hkUint8 ) + sizeof( hkUint32 );

private:

    hkVdbCmd() {}
    ~hkVdbCmd() {}

    typedef hkUint32 BodySizeType;
    BodySizeType m_bodySize;
    hkUint8 m_serverType;
    hkInt8 m_data[1];
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
