// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Container/Hash/hkHashSet.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/hkProcessContext.h>

class hkStreamReader;
class hkStreamWriter;
class hkProcess;
class hkNetLobby;
class hkSocket;
class hkServerProcessHandler;
class hkVisualDebuggerReporter;

/// Default port used to listen for vdb clients.
#if defined(HK_PLATFORM_DURANGO)
static const hkUint32 HK_VISUAL_DEBUGGER_DEFAULT_PORT = 65001;
#else
static const hkUint32 HK_VISUAL_DEBUGGER_DEFAULT_PORT = 25001;
#endif

/// Default port used to advertise the server to vdb clients.
/// This port is selected because it's available on major consoles (which may restrict port range).
/// Note: The port must still be opened on Xbox One using the app manifest (see Reserved Ports and Firewall Exceptions in the XDK for more info)
static const hkUint32 HK_VDB_ADVERTISE_SERVER_DEFAULT_PORT = 15001;

/// Id used with hkNetLobby to distinguish vdb server discovery requests from other hkNetLobby traffic on the same port.
static const int HK_VDB_ADVERTISE_SERVER_DEFAULT_GROUP_ID = ( ( ( 'H' ) << 24 ) | ( ( 'V' ) << 16 ) | ( ( 'D' ) << 8 ) | ( ( 'B' ) << 0 ) );

/// Information about connected Vdb clients.
struct HK_EXPORT_COMMON hkVisualDebuggerClient
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_UTILITIES, hkVisualDebuggerClient );

        /// The socket connection to the Vdb client
    hkSocket* m_socket;

        /// A handler for registering, creating, deleting, etc. the hkProcesses for this Vdb client
    hkServerProcessHandler* m_processHandler;

        /// A reporter used to output hkError and hkLog text to the client.
    hkVisualDebuggerReporter* m_vdbReporter;

        /// Expected authentication acknowledgment.
        /// (see hkVisualDebugger::pollForNewClients for more information).
    hkUint64 m_authenticationHashAck;

        /// The tick count at which the authentication ack will timeout.
        /// (see hkVisualDebugger::pollForNewClients for more information).
    hkUint64 m_authenticationHashAckTimeout;

        /// Enum describing stages of authentication.
        /// (see hkVisualDebugger::pollForNewClients for more information).
    enum class AuthenticationStatus : int
    {
        Timedout = -2,
        Failed = -1,
        Awaiting = 0,
        Passed = 1,
    };

        /// Indicates authentication status.
        /// (see hkVisualDebugger::pollForNewClients for more information).
    AuthenticationStatus m_authenticationStatus;
};

/// This class facilitates communication with visual debugger (vdb) clients.
class HK_EXPORT_COMMON hkVisualDebugger : public hkReferencedObject
{
    public:
        HK_DECLARE_CLASS( hkVisualDebugger, New );

            /// Creates a visual debugger for a given set of contexts.
            /// A context is a container for interesting game data such
            /// as hknpWorlds or user / animation data.
            /// This ctor will set the context owner as this VDB (assumes only one VDB).
            /// It will also register common hkProcesses and enable some default processes
            /// (see addDefaultProcess).
        hkVisualDebugger( const hkArray<hkProcessContext*>& contexts, const void* classReg = HK_NULL );

        
            /// Sets up the visual debugger for listening to clients and advertising itself using the provided
            /// TCP/IP port numbers (IPv4 only). For the server to actually work hkVisualDebugger::step must be called.
            /// Returns HK_FAILURE if there was an error creating sockets.
            /// After failure, it is safe to call serve() again with different port numbers.
        hkResult serve(
            int listenPort = HK_VISUAL_DEBUGGER_DEFAULT_PORT,
            int advertisePort = HK_VDB_ADVERTISE_SERVER_DEFAULT_PORT,
            const char* advertiseName = HK_NULL,
            int advertiseGroupId = HK_VDB_ADVERTISE_SERVER_DEFAULT_GROUP_ID );
        

        
            /// Instructs the visual debugger to start capturing the output of all
            /// of the default viewers to the specified filename. Note you do not
            /// have to call hkVisualDebugger::serve this is completely independent
            /// and can be used without any network serving.
        void capture(const char* captureFilename);

            /// Instructs the visual debugger to create a connection to a memory stream. After
            /// connected the debugger will communicate over these memory streams as if they
            /// were a network connection.
        void capture(hkMemoryTrack* outgoingMemory, hkMemoryTrack* incommingMemory);

            /// End (all) captures to file/memory.
        void endCapture();
        

            /// Shutdown the VDB. Can call shutdown and then serve again later.
            /// It removes all clients and shuts down the server.
            /// Shutdown is called automatically by the dtor, but is safe to call more than once.
        void shutdown();

            /// Clear out the default and required process lists
        void clearDefaultAndRequiredProcesses();

        
            /// Add a process to the list of default processes. All processes in this
            /// list will be selected by a client when a new connection is established.
        void addDefaultProcess(const char* processName);

            /// Removes a viewer from the list of default viewers. If a viewer is not
            /// in this list it will not be selected by a client when a new connection
            /// is established.
        void removeDefaultProcess(const char* processName);
        

            /// Immediately create this process on the server when a client connects,
            /// don't wait for a client to request/validate it.
        void addRequiredProcess(const char* processName);

            /// Creates any required or default processes.
        void createRequiredAndDefaultProcessList( hkVisualDebuggerClient& vdbClient, hkStringBuf& viewerNames );

        
            /// Actions which step can perform.
            struct HK_EXPORT_COMMON StepActions : public hkFlagsEx<hkUint8>
            {
                enum Bits
                {
                    // Polls for new clients (see pollForNewClients() for more details).
                    PollForNewClients = ( 1 << 0 ),
                    // New clients will have basic authentication performed (see pollForNewClients() for more details).
                    AuthenticateNewClients = ( 1 << 1 ),
                    // Responds to server broadcasts if requested (see respondToServerDiscoveryBroadcasts()).
                    RespondToServerDiscoveryBroadcasts = ( 1 << 2 ),
                    // Default step actions.
                    Default = ( PollForNewClients | AuthenticateNewClients | RespondToServerDiscoveryBroadcasts )
                };
                HK_DECLARE_FLAGS_EX_CLASS( StepActions, StepActions::Bits );
            };
            /// Advances current clients by frameTimeInMs and performs other requested StepActions.
            /// If no value is passed in for frameTimeInMs, then it will automatically be computed
            /// as the difference from the last step. This call is essential for all hkVisualDebugger
            /// operations such as client connection/creation, sending data, receiving data, etc.
        virtual void step(
            hkReal frameTimeInMs = 0,
            StepActions actions = StepActions::Default );
        

            /// Called internally by step() to poll for new clients.
            /// New clients are authenticated upon connection if requested. They do so by responding to a
            /// client-specific hash computed on the server with an acknowledgment hash the server
            /// also knows how to compute.
            /// It is a simple authentication system to prevent the server from crashing in cases
            /// where non-vdb-clients connect to a served port. It is a not a robust security measure.
        virtual void pollForNewClients( bool authenticateNewClients = true );

            /// Called internally by step() to respond to any pending server discovery requests.
            /// Server discovery requests come from clients which have the capability and have
            /// enabled this feature.
        virtual void respondToServerDiscoveryBroadcasts();

            // dtor.
        virtual ~hkVisualDebugger();

            /// Get the contexts that the Processes can run under
            /// such as a context that lists hknpWorlds, or one that has animation data for instance.
        const hkArray<hkProcessContext*>& getContexts() { return m_contexts; }

            /// Gets a list of process used by clients of the server.
            /// Returns number of appended processes for convenience.
            /// Optionally provide a name or tag to prune the results.
        int getCurrentProcesses( hkArray<hkProcess*>& processes, const char* name = HK_NULL, int tag = -1 ) const;

        
            /// Add/remove a process context to provide to created processes.
        void addContext( hkProcessContext* context );
        void removeContext( hkProcessContext* context );
        

            /// By default ::step will use the wall clock time if frameTimeInMs is equal
            /// to 0.  Call disableOverrideFrameTimeIfZero to disable this behavior.
        void disableOverrideFrameTimeIfZero() { m_overrideFrameTimeIfZero = false; }

            /// Return true if there's a connected client.
        bool hasClients() const { return m_clients.isEmpty() == false; }

            /// Returns the currently connected clients.
            /// For advanced use only.
        hkArrayView<hkVisualDebuggerClient> getClients() { return m_clients; }

            /// Wait for connecting clients.
            /// Returns immediately if "server" has not been called and returns false in that case.
        bool waitForClients();

        
            /// Clears all log patterns being sent to clients.
        void clearLogPatterns();

            /// Add/Remove a log pattern to a stored list.
            /// Any newly created clients will register to have these log patterns sent to them.
        void addLogPattern( const char* logPattern );
        void removeLogPattern( const char* logPattern );

            /// Adds default log patterns to the stored list.
            /// (This is called during ctor).
        void createDefaultLogPatterns();

            /// Get the current set of log patterns to be registered with newly created clients.
        const hkHashSet<hkStringPtr>& getLogPatterns() const;

            /// Sets the level for reporting the registered patterns. Does not set the origin
            /// level as that may overwrite the level registered by other origins and also
            /// will prune data for other log listeners.
        void setLogReportingLevel( hkLog::Level::Enum level );
        

    protected:

            /// Derive from hkVisualDebugger and override these two functions for book keeping
            /// You can then synchronize across existing and new clients within your derived class
        virtual void deleteClient( int i );
        virtual void createClient(
            hkSocket* socket,
            hkStreamReader* reader,
            hkStreamWriter* writer,
            bool authenticateClient );

            /// Update authentication status for the client.
        void updateAuthenticationStatus( hkVisualDebuggerClient& client );

            /// write the step cmd to given client index.
        void writeStep( int i, float t );

            /// Server socket (we listen for new Clients on it)
        hkSocket* m_server;
            /// Net lobby for server discovery requests
        hkNetLobby* m_netLobby;

            /// All clients get a separate process handler.
        hkArray<hkVisualDebuggerClient> m_clients;
        hkArray<hkProcessContext*> m_contexts;

            /// Rand generator for authentication hash.
        hkPseudoRandomGenerator m_rand;

            /// Processes that will always service a Client (auto created)
        hkArray<hkStringPtr> m_defaultProcesses;

            /// Processes that will be created without any handshaking required (create instead of select)
        hkArray<hkStringPtr> m_requiredProcesses;

            /// Timing variables.
        hkBool m_amTimingFrame;
        hkStopwatch m_frameTimer;
        hkBool m_overrideFrameTimeIfZero;

            /// List of log patterns that will be registered with any newly created clients.
        hkHashSet<hkStringPtr> m_registeredLogPatterns;
        hkLog::Level::Enum m_logLevel;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
