// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkVisualDebugger.h>
#include <Common/Base/System/Platform/hkPlatformInfo.h>
#include <Common/Base/System/Stopwatch/hkSystemClock.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Io/Socket/hkNetLobby.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Visualize/hkProcessRegisterUtil.h>
#include <Common/Visualize/hkServerProcessHandler.h>
#include <Common/Visualize/hkVersionReporter.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>
#include <Common/Visualize/hkVisualDebuggerReporting.h>
#include <Common/Visualize/Process/hkDebugDisplayProcess.h>
#include <Common/Visualize/Process/hkRemoteObjectProcess.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "common.visualize.vdb"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    // Note: This doesn't have to be unique, we also aren't relying on any distribution properties, etc.
    // It's just intended to be a bit of magic numbering as a base for a computed hash and hash ack.
    HK_INLINE hkUint64 computeAuthenticationHash( hkPseudoRandomGenerator& rand, int clientIdx )
    {
        // Note: zero hash is disable, so we avoid this.
        return
            ( hkUint64( rand.getRand32() ) << 30 ) + rand.getRand32() +
            ( ( clientIdx + 1 ) * ( rand.getRandChar( 256 ) + 1 ) );
    }

    
    HK_INLINE void connectOutput( const char* pattern, hkLog::Output* output, hkLog::Level::Enum lvl )
    {
        // Set the output level *first* as it can *also* cause multi-output level to change!
        output->setLevel( lvl );
        // Pass in "Disabled" so the level is never affected from this call (::connectToOutput only sets the level if it's
        // higher than what's currently there)
        hkLog::connectToOutput( pattern, output, hkLog::Level::Disabled );
    }
}

hkVisualDebugger::hkVisualDebugger( const hkArray<hkProcessContext*>& contexts, const void* )
    : m_server( HK_NULL ),
    m_netLobby( HK_NULL ),
    m_rand( 0xffffff00 ),
    m_amTimingFrame( false ),
    m_frameTimer( "Frame Timer" ),
    m_overrideFrameTimeIfZero( true )
{
    for ( int c = 0; c < contexts.getSize(); ++c )
    {
        addContext( contexts[c] );
    }

    HK_REPORT_SECTION_BEGIN( 0x1293adef, "Visual Debugger" );

    //Log_Info( "VDB Server instance has been created" );
    hkProcessRegisterUtil::registerAllCommonProcesses();
    addDefaultProcess( hkDebugDisplayProcess::getName() );
    addDefaultProcess( hkRemoteObjectProcess::getName() );

    setLogReportingLevel( hkLog::Level::Warning );
    createDefaultLogPatterns();

    HK_REPORT_SECTION_END();
}

hkVisualDebugger::~hkVisualDebugger()
{
    shutdown();
    for ( int c = 0; c < m_contexts.getSize(); ++c )
    {
        m_contexts[c]->m_contextDestroyedSignal.unsubscribeAll(this);
    }
    m_defaultProcesses.clear();
    m_requiredProcesses.clear();
}

void hkVisualDebugger::addContext( hkProcessContext* context )
{
    HK_WARN_ON_DEBUG_IF( m_clients.getSize() > 0, 0x47cf27af, "Trying to add a context after a Visual Debugger client is connected. This probably won't work." );
    context->m_contextDestroyedSignal.subscribe( this, &hkVisualDebugger::removeContext, "hkVisualDebugger" );
    m_contexts.pushBack( context );
}

void hkVisualDebugger::shutdown()
{
    // cleanup existing clients
    HK_REPORT_SECTION_BEGIN( 0x1293adef, "Shutting down Visual Debugger.." );
    for ( int i = ( m_clients.getSize() - 1 ); i >= 0; i-- )
    {
        // Send a step command to make sure that this frame gets processed
        writeStep( i, 0 );

        // delete it
        deleteClient( i );
        Log_Info( "Client deleted." );
    }

    if ( m_server )
    {
        delete m_server;
        m_server = HK_NULL;
        Log_Info( "Server deleted." );
    }

    if ( m_netLobby )
    {
        delete m_netLobby;
        m_netLobby = HK_NULL;
        Log_Info( "Net lobby deleted." );
    }

    HK_REPORT_SECTION_END();
}

hkResult hkVisualDebugger::serve( int listenPort, int advertisePort, const char* advertiseName, int advertiseGroupId )
{
    HK_REPORT_SECTION_BEGIN( 0x1293ade8, "Serving" );

    hkResult serverResult = HK_FAILURE;
    if ( !m_server )
    {
        m_server = hkSocket::create();
        if ( m_server )
        {
            serverResult = m_server->listen( listenPort );
            if ( serverResult.isSuccess() )
            {
                Log_Info( "Server created and will poll for new client(s) on port {} every frame", listenPort );
            }
            else
            {
                m_server->removeReference();
                m_server = HK_NULL;
            }
        }

        if ( serverResult.isFailure() )
        {
            Log_Error( "Server could not create connection, usually this means something is already listening on port {}, or a permissions or hardware error is preventing a socket from being created.", listenPort );
        }
    }
    else
    {
        serverResult = HK_SUCCESS;
        Log_Warning( "Server has already been created, only one server allowed per visual debugger instance" );
    }

    hkResult advertiseResult = HK_SUCCESS;
    if ( m_server && ( advertisePort > 0 ) )
    {
        if ( !m_netLobby )
        {
            hkStringPtr defaultNameStorage;
            if ( !advertiseName )
            {
                hkPlatformInfo::getDeviceName( defaultNameStorage );
                advertiseName = defaultNameStorage;
            }

            hkNetLobby::SessionInfo info;
            m_server->getInetAddr( info.m_serverAddr );
            info.m_serverID = advertiseGroupId;
            int strWithNullSize = hkMath::min2( hkString::strLen( advertiseName ), hkNetLobby::SessionInfo::MAX_INFO_SIZE - 1 ) + 1;
            hkString::strNcpy(
                info.m_sessionInfo,
                hkNetLobby::SessionInfo::MAX_INFO_SIZE,
                advertiseName,
                strWithNullSize);
            info.m_infoSize = strWithNullSize;

            m_netLobby = new hkNetLobby();
            advertiseResult = m_netLobby->advertiseSession( info, advertisePort );
            if ( advertiseResult.isSuccess() )
            {
                Log_Info( "Server discovery initialized; will respond to discovery broadcasts on port {} every frame", advertisePort );
            }
            else
            {
                delete m_netLobby;
                m_netLobby = HK_NULL;
            }
        }
        else
        {
            Log_Warning( "Server discovery has already been initialized, only allowed once per visual debugger instance" );
        }
    }

    HK_REPORT_SECTION_END();

    return ( serverResult.isFailure() ) ? serverResult : advertiseResult;
}

void hkVisualDebugger::createRequiredAndDefaultProcessList( hkVisualDebuggerClient& vdbClient, hkStringBuf& viewerNames )
{
    viewerNames = "Turning on the following viewers: [";

    // Create required processes first
    for( int j = 0; j < m_requiredProcesses.getSize(); j++ )
    {
        int tag = vdbClient.m_processHandler->getProcessTag( m_requiredProcesses[j] );
        if ( tag >= 0 )
        {
            viewerNames.appendJoin(" ", m_requiredProcesses[j]);
            vdbClient.m_processHandler->createProcess(tag);
        }
    }

    // Create default processes next
    for(int i = 0; i < m_defaultProcesses.getSize(); i++)
    {
        int tag = vdbClient.m_processHandler->getProcessTag( m_defaultProcesses[i] );

        // Only add processes that were not created already
        if (tag >= 0 && m_requiredProcesses.indexOf( m_defaultProcesses[i] ) == -1)
        {
            viewerNames.appendJoin(" ", m_defaultProcesses[i]);
            vdbClient.m_processHandler->createProcess( tag );
        }
    }

    viewerNames += "]";
}

void hkVisualDebugger::capture( const char* captureFilename )
{
    HK_REPORT_SECTION_BEGIN( 0x1293ade7, "Capturing" );

    hkRefPtr<hkStreamWriter> writer = hkFileSystem::getInstance().openWriter( captureFilename );
    if ( writer && writer->isOk() )
    {
        Log_Info( "Capturing simulation state to '{}'", captureFilename );

        createClient( HK_NULL, HK_NULL, writer, false );

        // At this point, the ids have been placed in the stream to be selected
        // by default by the client, but as it is an inert file,
        // we will go ahead and confirm them all
        hkVisualDebuggerClient& vdbC = m_clients.back();
        hkStringBuf viewerNames;
        createRequiredAndDefaultProcessList( vdbC, viewerNames );

        Log_Info( viewerNames );
    }
    else
    {
        Log_Error( "Capture file '{}' could not be opened for writing", captureFilename );
    }

    HK_REPORT_SECTION_END();
}

void hkVisualDebugger::capture( hkMemoryTrack* outgoingMemory, hkMemoryTrack* incommingMemory )
{
    HK_REPORT_SECTION_BEGIN( 0xbebefec4, "Capturing to memory stream" );

    hkStreamWriter* writer = new hkMemoryTrackStreamWriter( outgoingMemory, hkMemoryTrackStreamWriter::TRACK_BORROW );
    hkStreamReader* reader = new hkMemoryTrackStreamReader( incommingMemory, hkMemoryTrackStreamReader::MEMORY_INPLACE, true );
    if ( writer && writer->isOk() && reader && reader->isOk() )
    {
        Log_Info( "Capturing simulation state to memory stream." );

        createClient( HK_NULL, reader, writer, false );
        writer->removeReference();
        reader->removeReference();

        // At this point, the ids have been placed in the stream to be selected
        // by default by the client, but as it is an inert file,
        // we will go ahead and confirm them all
        hkVisualDebuggerClient& vdbC = m_clients.back();
        hkStringBuf viewerNames;
        createRequiredAndDefaultProcessList( vdbC, viewerNames );

        Log_Info( viewerNames );

        // Flush the client's output stream so all data is sent after connection
        vdbC.m_processHandler->m_outStream->getStreamWriter()->flush();
    }
    else
    {
        Log_Error( "Could not create memory stream for writing." );
    }

    HK_REPORT_SECTION_END();
}

void hkVisualDebugger::endCapture()
{
    int nc = m_clients.getSize();
    for ( int i = ( nc - 1 ); i >= 0; --i ) // backwards as they may be deleted.
    {
        hkVisualDebuggerClient& client = m_clients[i];
        if ( !client.m_socket ) // then must be a file based capture
        {
            deleteClient( i );
        }
    }
}

void hkVisualDebugger::respondToServerDiscoveryBroadcasts()
{
    if ( m_netLobby != HK_NULL)
    {
        m_netLobby->step();
    }
}

void hkVisualDebugger::deleteClient( int i )
{
    hkVisualDebuggerClient& client = m_clients[i];

    // Disconnect logs
    hkLog::TextOutput* textOutput = client.m_vdbReporter->getLogReporter();
    for( hkHashSet<hkStringPtr>::Iterator iter = m_registeredLogPatterns.getIterator();
        m_registeredLogPatterns.isValid( iter );
        iter = m_registeredLogPatterns.getNext( iter ) )
    {
        const hkStringPtr& pattern = m_registeredLogPatterns.get( iter );
        hkLog::disconnect( pattern, textOutput );
    }

    // Clean up
    if ( client.m_vdbReporter ) client.m_vdbReporter->removeReference();
    if ( client.m_processHandler ) client.m_processHandler->removeReference();
    if ( client.m_socket ) client.m_socket->removeReference();

    m_clients.removeAt( i );
}

void hkVisualDebugger::createClient(
    hkSocket* socket,
    hkStreamReader* reader,
    hkStreamWriter* writer,
    bool authenticateClient )
{
    if ( authenticateClient && !reader )
    {
        HK_WARN_ALWAYS( 0x22441441, "Must have a reader for vdb clients in order to authenticate them" );
        authenticateClient = false;
    }

    // Send version information (the first thing in the stream back to Client / file).
    hkUint64 authenticationHashAck = 0;
    if ( writer )
    {
        hkVersionReporter::writeServerInformationCmd(
            writer,
            authenticateClient ? computeAuthenticationHash( m_rand, m_clients.getSize() ) : 0,
            authenticationHashAck );
    }

    // Create/init our new client data.
    hkVisualDebuggerClient* newClient = m_clients.expandBy( 1 );
    newClient->m_socket = socket;
    newClient->m_processHandler = new hkServerProcessHandler( m_contexts, reader, writer );
    newClient->m_vdbReporter = new hkVisualDebuggerReporter( *newClient->m_processHandler->m_textHandler );
    newClient->m_authenticationHashAck = authenticationHashAck;
    // 10s timeout should be very large and accommodate any slow network traffic for initial handshake.
    newClient->m_authenticationHashAckTimeout = hkUint64( hkSystemClock::getTickCounter() + 10.0f * hkSystemClock::getTicksPerSecond() );
    newClient->m_authenticationStatus =
        ( authenticationHashAck ) ?
        hkVisualDebuggerClient::AuthenticationStatus::Awaiting :
        hkVisualDebuggerClient::AuthenticationStatus::Passed;

    // Register available processes to the client.
    newClient->m_processHandler->registerAllAvailableProcesss();

    // Register log patterns.
    for ( hkHashSet<hkStringPtr>::Iterator iter = m_registeredLogPatterns.getIterator();
        m_registeredLogPatterns.isValid( iter );
        iter = m_registeredLogPatterns.getNext( iter ) )
    {
        const hkStringPtr& pattern = m_registeredLogPatterns.get( iter );
        connectOutput( pattern, newClient->m_vdbReporter->getLogReporter(), m_logLevel );
    }

    // Create the processes we require.
    for(int i = 0; i < m_requiredProcesses.getSize(); i++)
    {
        int tag = newClient->m_processHandler->getProcessTag( m_requiredProcesses[i].cString() );
        if (tag >= 0)
        {
            newClient->m_processHandler->createProcess(tag);
            newClient->m_processHandler->selectProcess(tag);
        }
    }
    for ( int i = 0; i < m_defaultProcesses.getSize(); i++ )
    {
        int tag = newClient->m_processHandler->getProcessTag( m_defaultProcesses[i].cString() );
        if (tag >= 0  && m_requiredProcesses.indexOf( m_defaultProcesses[i] ) == -1 )
        {
            newClient->m_processHandler->selectProcess( tag );
        }
    }

    // Step one frame.
    writeStep( m_clients.getSize() - 1, 0 );
}

void hkVisualDebugger::pollForNewClients( bool authenticateNewClients )
{
    if ( m_amTimingFrame && m_frameTimer.isRunning() )
        m_frameTimer.stop();

    // see if there is a new client trying to connect
    if ( m_server )
    {
        hkSocket* socket = m_server->pollForNewClient();
        if ( socket )
        {
            Log_Info( "A new network client has been received (host name not available at present)" );
            hkStreamWriter* writer = &socket->getWriter();
            hkStreamReader* reader = &socket->getReader();
            createClient( socket, reader, writer, authenticateNewClients );
        }
    }

    if ( m_amTimingFrame )
        m_frameTimer.start();
}

void hkVisualDebugger::updateAuthenticationStatus( hkVisualDebuggerClient& client )
{
    if ( client.m_authenticationStatus == hkVisualDebuggerClient::AuthenticationStatus::Awaiting )
    {
        if ( client.m_processHandler->m_inStream->getStreamReader() )
        {
            // We assume these are the same readers.
            HK_ASSERT(
                0x22441440,
                !client.m_socket || client.m_processHandler->m_inStream->getStreamReader() == &client.m_socket->getReader(),
                "ProcessHandler and socket have different stream readers" );

            // If we are using a socket, wait until it's ready.
            if ( client.m_socket && ( !client.m_socket->isOk() || !client.m_socket->canRead() ) )
            {
                // Check for time-out.
                if ( hkSystemClock::getTickCounter() > client.m_authenticationHashAckTimeout )
                {
                    client.m_authenticationStatus = hkVisualDebuggerClient::AuthenticationStatus::Timedout;
                }
                return;
            }

            // Read our ack.
            hkUint64 ack = client.m_processHandler->m_inStream->read64u();
            if ( client.m_authenticationHashAck == ack )
            {
                client.m_authenticationStatus = hkVisualDebuggerClient::AuthenticationStatus::Passed;
            }
            else
            {
                client.m_authenticationStatus = hkVisualDebuggerClient::AuthenticationStatus::Failed;
            }
        }
        else
        {
            client.m_authenticationStatus = hkVisualDebuggerClient::AuthenticationStatus::Passed;
        }
    }
}

void hkVisualDebugger::writeStep( int i, float t )
{
    hkVisualDebuggerClient& client = m_clients[i];
    if ( client.m_processHandler->isOk() )
    {
        hkVdbOStream* os = client.m_processHandler->m_outStream;
        os->write32u( sizeof( hkUint8 ) + sizeof( float ) ); // cmd size
        os->write8u( hkVdbCmdType::STEP );// cmd
        os->writeFloat32( t ); // cmd data
        os->getStreamWriter()->flush(); // end of frame, so flush
    }
}

void hkVisualDebugger::step( hkReal frameTimeInMs, StepActions actions )
{
    hkReal frameTime = 16.0f; // (this is an approximation for the first frame)
    if ( m_amTimingFrame )
    {
        frameTime = m_frameTimer.getElapsedSeconds() * 1000.0f;
        m_frameTimer.stop();
    }
    m_amTimingFrame = true;
    if ( hkMath::fabs( frameTimeInMs ) > HK_REAL_EPSILON || !m_overrideFrameTimeIfZero )
    {
        frameTime = frameTimeInMs;
    }

    // Check for server discovery broadcasts
    if ( actions & StepActions::RespondToServerDiscoveryBroadcasts )
    {
        respondToServerDiscoveryBroadcasts();
    }

    // Check connection status.
    if ( actions & StepActions::PollForNewClients )
    {
        pollForNewClients( bool( actions & StepActions::AuthenticateNewClients ) );
    }

    // step all clients
    int nc = m_clients.getSize();
    for ( int i = ( nc - 1 ); i >= 0; --i ) // backwards as they may be deleted.
    {
        hkVisualDebuggerClient& client = m_clients[i];
        updateAuthenticationStatus( client );

        if ( client.m_processHandler->isOk()
            && ( client.m_authenticationStatus == hkVisualDebuggerClient::AuthenticationStatus::Passed ) )
        {
            client.m_processHandler->step( frameTime );
            client.m_vdbReporter->flush();
            writeStep( i, hkFloat32( frameTime ) ); // a STEP command followed by the float32 for time
        }

        if ( !client.m_processHandler->isOk()
            || ( int( client.m_authenticationStatus ) < 0 ) )
        {
            HK_REPORT_SECTION_BEGIN( 0x76e3a642, "Client Dies" );
            if ( client.m_authenticationStatus == hkVisualDebuggerClient::AuthenticationStatus::Timedout )
            {
                Log_Warning( "Connection authentication timed out, cleaning up..." );
            }
            else if ( client.m_authenticationStatus == hkVisualDebuggerClient::AuthenticationStatus::Failed )
            {
                Log_Warning( "Connection did not pass authentication (it is not a VDB client), cleaning up..." );
            }
            else
            {
                Log_Info( "VDB client has died, cleaning up..." );
            }
            HK_REPORT_SECTION_END();
            deleteClient( i ); // not much use
        }
    }

    if ( m_amTimingFrame )
    {
        m_frameTimer.reset();
        m_frameTimer.start();
    }
}

void hkVisualDebugger::clearDefaultAndRequiredProcesses()
{
    m_defaultProcesses.clear();
    m_requiredProcesses.clear();
}

void hkVisualDebugger::addDefaultProcess( const char* processName )
{
    if ( m_defaultProcesses.indexOf( processName ) == -1 )
    {
        m_defaultProcesses.expandOne() = processName;
    }
}

void hkVisualDebugger::removeDefaultProcess( const char* processName )
{
    for ( int i = 0; i < m_defaultProcesses.getSize(); i++ )
    {
        if ( hkString::strCmp( m_defaultProcesses[i].cString(), processName ) == 0 )
        {
            // Keep the processes in the same order in case they have dependencies
            // (eg, hkRemoteObjectProcess must be created before hkbBehaviorServer).
            m_defaultProcesses.removeAtAndCopy( i );
            return;
        }
    }

    HK_REPORT_SECTION_BEGIN( 0x76565454, "removeDefaultProcess" );
    Log_Error( "The default process '{}' cannot not be removed from the default process list as it cannot be found.", processName );
    HK_REPORT_SECTION_END();
}

void hkVisualDebugger::addRequiredProcess(const char* processName)
{
    if ( m_requiredProcesses.indexOf( processName ) == -1 )
    {
        m_requiredProcesses.expandOne() = processName;
    }
}

int hkVisualDebugger::getCurrentProcesses( hkArray<hkProcess*>& processes, const char* name, int tag ) const
{
    tag =
        ( tag != -1 ) ? tag :
        ( name != HK_NULL ) ? hkProcessFactory::getInstance().getProcessTag( name ) :
        -1;
    int nc = m_clients.getSize();
    int prevProcessesCount = processes.getSize();
    for ( int ic = 0; ic < nc; ++ic )
    {
        const hkVisualDebuggerClient& client = m_clients[ic];
        if ( client.m_processHandler )
        {
            const hkArray<hkProcess*>& proc = client.m_processHandler->getProcessList();
            if ( tag == -1 )
            {
                processes.insertAt( processes.getSize(), proc.begin(), proc.getSize() );
            }
            else
            {
                for ( int ip = 0; ip < proc.getSize(); ++ip )
                {
                    if ( proc[ip]->getProcessTag() == tag )
                    {
                        processes.pushBack( proc[ip] );
                    }
                }
            }
        }
    }
    return ( processes.getSize() - prevProcessesCount );
}

void hkVisualDebugger::removeContext( hkProcessContext* context )
{
    int index = m_contexts.indexOf( context );
    if ( index >= 0 )
    {
        m_contexts.removeAtAndCopy( index );
    }
}

void hkVisualDebugger::addLogPattern( const char* logPattern )
{
    hkHashSet<hkStringPtr>::Iterator iter = m_registeredLogPatterns.find( logPattern );
    if( !m_registeredLogPatterns.isValid( iter ) )
    {
        // Add new value
        m_registeredLogPatterns.insert( logPattern );
    }

    // Connect for the first time or after a disconnect with different warning level
    for( int i = 0; i < m_clients.getSize(); i++ )
    {
        hkLog::TextOutput* textOutput = m_clients[i].m_vdbReporter->getLogReporter();
        connectToOutput( logPattern, textOutput, m_logLevel );
    }
}

bool hkVisualDebugger::waitForClients()
{
    while ( m_server && !hasClients() )
    {
        pollForNewClients();
    }
    return hasClients();
}

void hkVisualDebugger::clearLogPatterns()
{
    for( hkHashSet<hkStringPtr>::Iterator iter = m_registeredLogPatterns.getIterator();
        m_registeredLogPatterns.isValid( iter );
        iter = m_registeredLogPatterns.getNext( iter ) )
    {
        const hkStringPtr& pattern = m_registeredLogPatterns.get( iter );
        removeLogPattern( pattern.cString() );
    }
}

void hkVisualDebugger::removeLogPattern( const char* logPattern )
{
    hkResult result = m_registeredLogPatterns.remove( logPattern );

    if( result.isSuccess() )
    {
        for( int i = 0; i < m_clients.getSize(); i++ )
        {
            hkLog::TextOutput* textOutput = m_clients[i].m_vdbReporter->getLogReporter();
            hkLog::disconnect( logPattern, textOutput );
        }
    }
}

void hkVisualDebugger::createDefaultLogPatterns()
{
    addLogPattern( "ReadFormat" );
    addLogPattern( "s11n" );
    addLogPattern( "reflect" );
    addLogPattern( "ver" );
    addLogPattern( "io" );
    addLogPattern( "Filesystem" );
}

const hkHashSet<hkStringPtr>& hkVisualDebugger::getLogPatterns() const
{
    return m_registeredLogPatterns;
}

void hkVisualDebugger::setLogReportingLevel( hkLog::Level::Enum level )
{
    
    
    for ( int i = 0; i < m_clients.getSize(); i++ )
    {
        hkLog::TextOutput* textOutput = m_clients[i].m_vdbReporter->getLogReporter();
        textOutput->setLevel( level );
    }
    m_logLevel = level;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
