// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Visualize/hkVisualize.h>
#include <Common/Base/System/Platform/hkPlatformInfo.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Visualize/hkVersionReporter.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#define READ_STRING( _STRBUF, _STREAM ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( infoOut.m_protocol >= hkProtocolVersion::SUPPORTS_CONNECTION_AUTHENTICATION ) { _STREAM.readString(_STRBUF); } \
        else \
        { \
            unsigned int length = _STREAM.read16u(); \
            hkInplaceArray<char, 128> buffer; \
            buffer.setSize( length + 1 ); \
            _STREAM.readRaw( buffer.begin(), length ); \
            buffer[length] = '\0'; \
            _STRBUF = &buffer[0]; \
        } \
    HK_MULTILINE_MACRO_END

namespace
{
    // Copied from hkPseudoRandomGenerator, to avoid changes in that class impacting
    // our ability to authenticate against older servers.
    HK_INLINE char getRandChar( hkUint32& seed )
    {
        seed = 1664525L * seed + 1013904223L;
        hkUint32 temp = seed;
        temp >>= 13;
        temp = temp % 256;
        return char( temp );
    }

    // Computes an ack for the authentication hash.
    HK_INLINE hkUint64 computeAuthenticationAck( hkUint64 authenticationHash )
    {
        // Zero is considered disabled.
        if ( authenticationHash == 0 ) return 0;

        // Add some randomness to our hash for flare.
        hkUint32 seed = hkUint32( authenticationHash >> 32 ) + hkUint32( authenticationHash );
        char buf[100];
        for ( int i = 0; i < HK_COUNT_OF( buf ) - 1; i++ )
        {
            buf[i] = getRandChar( seed );
        }
        buf[HK_COUNT_OF( buf ) - 1] = '\0';

        // Return our computed ack.
        return hkHash::appendCrc64( authenticationHash, buf );
    }
}

hkResult hkVersionReporter::writeServerInformationCmd(
    hkStreamWriter* connection,
    hkUint64 authenticationHash,
    hkUint64& authenticationHashAckOut )
{
    Info serverInfo;
    computeInformation( serverInfo );

    // send the data chunk describing the version information
    hkArray<char> rawData;
    hkVdbOStream commandStream( rawData );
    commandStream.write8u( hkVdbCmdType::SEND_SERVER_INFO );

    // version and minimum compatible version
    commandStream.write32( serverInfo.m_protocol );
    commandStream.write32( serverInfo.m_minimumProtocol );
    commandStream.write32( serverInfo.m_sdkVersion );

    // send magic platform string
    const char* platformInfo = serverInfo.m_platformString;
    commandStream.writeString( platformInfo );

    // send compiler string
    const char* compilerInfo = serverInfo.m_compilerString;
    commandStream.writeString( compilerInfo );

    // Write the host layout
    commandStream.write8u( serverInfo.m_sizeOfPointer );
    commandStream.write8u( serverInfo.m_isLittleEndian );

    // Write auth hash base.
    commandStream.write64u( authenticationHash );

    // actually write the packet.
    hkVdbOStream connectionStream( connection );
    connectionStream.write32( rawData.getSize() );
    connectionStream.writeRaw( rawData.begin(), rawData.getSize() );

    // Determine the hash ack.
    authenticationHashAckOut = computeAuthenticationAck( authenticationHash );

    return HK_SUCCESS;
}

hkResult HK_CALL hkVersionReporter::readServerInformationCmdData(
    hkStreamReader* connection,
    Info& infoOut,
    hkUint64& authenticationHashAckOut )
{
    hkVdbIStream chunkStream( connection );
    return readServerInformationCmdData( chunkStream, infoOut, authenticationHashAckOut );
}

hkResult HK_CALL hkVersionReporter::readServerInformationCmdData(
    hkVdbIStream& chunkStream,
    Info& infoOut,
    hkUint64& authenticationHashAckOut )
{
    infoOut.m_protocol = chunkStream.read32();
    infoOut.m_minimumProtocol = chunkStream.read32();
    if ( infoOut.m_protocol >= hkProtocolVersion::VDB_2017_1 )
    {
        infoOut.m_sdkVersion = chunkStream.read32();
    }
    else
    {
        infoOut.m_sdkVersion = -1;
    }

    // read magic platform string
    {
        hkStringBuf buff;
        READ_STRING( buff, chunkStream );
        infoOut.m_platformString = buff;
    }

    // read compiler string if available
    if ( infoOut.m_protocol >= hkProtocolVersion::SUPPORTS_COMPILER_STRING )
    {
        hkStringBuf buff;
        READ_STRING( buff, chunkStream );
        infoOut.m_compilerString = buff;
    }
    else
    {
        infoOut.m_compilerString = "UNAVAILABLE";
    }

    if ( infoOut.m_protocol >= hkProtocolVersion::CONTAINS_HOST_LAYOUT_RULES )
    {
        infoOut.m_sizeOfPointer = chunkStream.read8u();
        infoOut.m_isLittleEndian = chunkStream.read8u();
        if ( infoOut.m_protocol < hkProtocolVersion::SUPPORTS_CONNECTION_AUTHENTICATION )
        {
            // Unused, read for backwards compatibility.
            /*hkUint8 reusePaddingOptimization = */chunkStream.read8u();
            /*hkUint8 emptyBaseClassOptimization = */chunkStream.read8u();
        }
    }
    else
    {
        // Previous protocol versions attempted to determine the host layout from the version string.
        // This is incomplete, but we need to keep this in order to ensure compatibility when
        // connecting to older servers.
        infoOut.m_isLittleEndian =
            ( ( hkString::strStr( infoOut.m_platformString.cString(), "XOTHER" ) == HK_NULL ) &&
            ( hkString::strStr( infoOut.m_platformString.cString(), "XBOX360" ) == HK_NULL ) &&
                ( hkString::strStr( infoOut.m_platformString.cString(), "PS3" ) == HK_NULL ) &&
                ( hkString::strStr( infoOut.m_platformString.cString(), "WII" ) == HK_NULL ) &&
                ( hkString::strStr( infoOut.m_platformString.cString(), "MAC" ) == HK_NULL ) );

        // Most of our 64 bit platforms report their version string with an _64; this is
        // as complete as we can get without the explicit info added in 14200.
        infoOut.m_sizeOfPointer =
            ( hkString::strStr( infoOut.m_platformString.cString(), "_64" ) != HK_NULL ) ? 8 : 4;
    }

    if ( infoOut.m_protocol >= hkProtocolVersion::SUPPORTS_CONNECTION_AUTHENTICATION )
    {
        hkUint64 authenticationHash = chunkStream.read64u();
        authenticationHashAckOut = computeAuthenticationAck( authenticationHash );
    }
    else
    {
        authenticationHashAckOut = 0;
    }

    return HK_SUCCESS;
}

void HK_CALL hkVersionReporter::computeInformation( Info& info )
{
    info.m_protocol = m_protocolVersion;
    info.m_minimumProtocol = m_protocolMinimumCompatible;
    info.m_sdkVersion = HAVOK_SDK_VERSION;
    info.m_platformString = hkPlatformInfo::getPlatformName();
    info.m_compilerString = hkPlatformInfo::getCompilerName();
    info.m_sizeOfPointer = sizeof( void* );
    info.m_isLittleEndian = HK_ENDIAN_LITTLE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
