// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkServerProcessHandler.h>

#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

#include <Common/Visualize/hkVisualDebuggerReporting.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>
#include <Common/Visualize/hkServerObjectHandler.h>
#include <Common/Visualize/hkServerFileHandler.h>
#include <Common/Visualize/hkServerTextHandler.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

int hkServerProcessHandler::s_tag = 0;

hkServerProcessHandler::hkServerProcessHandler(
    hkArray<hkProcessContext*>& contexts,
    hkStreamReader* inStream,
    hkStreamWriter* outStream )
{
    m_outStreamLock = new hkCriticalSection( 1000 ); // usually no contention

    m_commandRouter.registerProcess( this );
    m_contexts = contexts;

    // The hkProcess base class. The streams will  be the master streams for all sub-spawned processes
    m_inStream.setAndDontIncrementRefCount( inStream ? new hkVdbIStream( inStream ) : HK_NULL );
    m_outStream.setAndDontIncrementRefCount( outStream ? new hkVdbOStream( outStream, true ) : HK_NULL );
    hkServerDebugDisplayHandler *const serverDisplayHandler = new hkServerDebugDisplayHandler( m_outStream, m_outStreamLock, m_inStream );
    m_commandRouter.registerProcess( serverDisplayHandler );
    m_displayHandler = hkRefNew<hkServerDebugDisplayHandler>( serverDisplayHandler );
    m_objectHandler = HK_NULL;
    m_serializer = hkRefNew<hkServerObjectSerializer>( new hkServerObjectSerializer( m_outStream, m_outStreamLock ) );
    m_fileHandler = hkRefNew<hkServerFileHandler>( new hkServerFileHandler( m_outStream, m_outStreamLock ) );
    m_textHandler = hkRefNew<hkServerTextHandler>( new hkServerTextHandler( m_outStream, m_outStreamLock ) );
    m_processHandler = this;
}

hkServerProcessHandler::~hkServerProcessHandler()
{
    for ( int i = m_processList.getSize() - 1; i >= 0; --i )
    {
        hkProcess* p = m_processList[i];
        m_processList.removeAt( i );
        delete p;
    }
    delete m_outStreamLock;
}

static hkUint8 _cmds[] = { hkVdbCmdType::CREATE_PROCESS, hkVdbCmdType::DELETE_PROCESS };

void hkServerProcessHandler::getConsumableCommands( hkUint8*& commands, int& numCommands )
{
    commands = _cmds;
    numCommands = 2;
}

void hkServerProcessHandler::consumeCommand( hkUint8 command )
{
    switch ( command )
    {
        case hkVdbCmdType::CREATE_PROCESS:
        {
            int tag = m_inStream->read32();
            if ( m_inStream->isOk() )
                createProcess( tag );
        }
        break;

        case hkVdbCmdType::DELETE_PROCESS:
        {
            int tag = m_inStream->read32();
            if ( m_inStream->isOk() )
                deleteProcess( tag );
        }
        break;
    }
}

// This function returns a new instance of a process specified by name
// NB: The caller must look after the cleanup of the returned Process instance
hkResult hkServerProcessHandler::createProcess( int tag )
{
    int curIndex = findProcessByTag( tag );
    if ( curIndex >= 0 )
        return HK_SUCCESS; // already created and running for this client

    hkProcess* p = hkProcessFactory::getInstance().createProcess( tag, m_contexts );
    if ( p )
    {
        p->m_inStream = m_inStream;
        p->m_outStream = m_outStream;
        p->m_displayHandler = m_displayHandler;
        p->m_objectHandler = hkRefNew<hkServerObjectHandler>( new hkServerObjectHandler( *m_serializer ) );
        p->m_fileHandler = m_fileHandler;
        p->m_textHandler = m_textHandler;
        p->m_processHandler = this;
        m_processList.pushBack( p );
        m_commandRouter.registerProcess( p ); // may or may not have any commands to register
        p->init();

        // notify the client that the process was created
        
        
        
        
        
        selectProcess( tag );

        return HK_SUCCESS;
    }
    else
    {
        return HK_FAILURE;
    }
}

int hkServerProcessHandler::findProcessByTag( int tag )
{
    int np = m_processList.getSize();
    for ( int i = 0; i < np; ++i )
    {
        if ( m_processList[i]->getProcessTag() == tag )
            return i;
    }
    return -1;
}

hkResult hkServerProcessHandler::deleteProcess( int tag )
{
    int index = findProcessByTag( tag );
    if ( index >= 0 )
    {
        hkProcess* p = m_processList[index];
        m_processList.removeAt( index );
        m_commandRouter.unregisterProcess( p );
        delete p;
    }
    else
    {
        HK_WARN( 0xfe452761, "VDB: Process scheduler could not find process to delete." );
    }
    return HK_SUCCESS;
}

hkProcess* hkServerProcessHandler::getProcessByName( const char* name )
{
    int tag = getProcessTag( name );
    if ( tag != -1 )
    {
        int index = findProcessByTag( tag );
        if ( index != -1 )
        {
            return m_processList[index];
        }
    }

    return HK_NULL;
}

void hkServerProcessHandler::getProcessesByType( int type, hkArray<hkProcess*>& processes )
{
    processes.clear();

    for ( int i = 0; i < m_processList.getSize(); ++i )
    {
        if ( m_processList[i]->getType() == type )
        {
            processes.pushBack( m_processList[i] );
        }
    }
}

hkResult hkServerProcessHandler::registerProcess( const char* name, int tag )
{
    using namespace hkVdbCmdUtils;

    const int numCharacters = ( hkString::strLen( name ) + 1 ); // include null terminator
    const int packetSize = s8 + s32 + ( s8 * numCharacters );

    m_outStream->write32( packetSize );
    m_outStream->write8u( hkVdbCmdType::REGISTER_PROCESS );

    m_outStream->write32( tag );
    m_outStream->writeString( name );

    return HK_SUCCESS;
}

hkResult hkServerProcessHandler::registerAllAvailableProcesss()
{
    // send a list of all the viewers over to the client
    const hkArray<hkProcessFactory::ProcessTagPair>& factoryProcessList =
        hkProcessFactory::getInstance().m_name2creationFunction;

    for ( int i = 0; i < factoryProcessList.getSize(); i++ )
    {
        registerProcess( factoryProcessList[i].m_name.cString(), factoryProcessList[i].m_tag );
    }

    return HK_SUCCESS;
}

hkResult hkServerProcessHandler::selectProcess( int tag )
{
    // packet size
    m_outStream->write32( 1 + 4 );

    // send command header
    m_outStream->write8u( hkVdbCmdType::SELECT_PROCESS );
    // send tag
    m_outStream->write32( tag );

    return HK_SUCCESS;
}

void hkServerProcessHandler::step( hkReal frameTimeInMs )
{
    // Step the Processes
    // As a process can delete itself, we can just
    // iter in reverse to prevent overrun
    for ( int i = m_processList.getSize() - 1; i >= 0; --i )
    {
        m_processList[i]->step( frameTimeInMs );
    }

    // For debugging purposes
    m_serializer->step( frameTimeInMs );

    // Interpret commands (after the process step so that Processes like the plugin
    // handlers have time to create processes, that in turn may be required for a new command
    if ( m_inStream )
    {
        m_commandRouter.consumeCommands( m_inStream );
    }

    // The debug display handler is not in the process list, so we step it separately.
    static_cast< hkServerDebugDisplayHandler* >( m_displayHandler.val() )->step( frameTimeInMs );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
