// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Common/Visualize/hkProcess.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/Serialize/hkVdbSerialize.h>

/// This serializes hkReflect::Vars from the server to the vdb client.
class HK_EXPORT_COMMON hkServerObjectSerializer :
    public hkReferencedObject,
    public hkProcess,
    public hkObjectTypeRegistry<hkInt32>,
    public hkConnectedObjectRegistry<hkUint64, hkUint64>
{
public:
    typedef hkUint8 TypeStreamId;
    HK_DECLARE_CLASS( hkServerObjectSerializer, New );

    /// Create a handler that writes to the provided outstream.
    hkServerObjectSerializer( hkVdbOStream* outStream, hkCriticalSection* outStreamLock );
    ~hkServerObjectSerializer();

#ifdef HK_VARIADIC_TEMPLATES
    /// Compute a unique id for the given type and args.
    /// No ids computed with different types will ever be equal.
    /// Ids for the same type are only equal if the args are equal.
    template<typename... Args>
    hkUint64 computeId( const hkReflect::Type* type, const Args&... args )
    {
        TypeStreamId typeId = getOrCreateTypeStreamId( type );
        hkUint64 id = hkVdbCmdId::computeStreamId( typeId, args... );
        HK_ASSERT( 0x22441193, id, "Id zero is reserved" );
        return id;
    }

    /// Decompose an id computed earlier from computeId (type is not returned).
    template<typename... Args>
    void decomposeId( hkUint64 id, Args&... args ) const
    {
        using namespace hkPack;
        HK_ASSERT( 0x22441270, id, "Id zero is reserved" );
        
        hkUint64 streamId = hkVdbCmdId::extractStreamId( id );
        TypeStreamId typeIdOut;
        Unpack( streamId, typeIdOut, args... );
    }
#else
    template<typename T0>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0 )
    {
        TypeStreamId typeId = getOrCreateTypeStreamId( type );
        hkUint64 id = hkVdbCmdId::computeStreamId( typeId, tail0 );
        HK_ASSERT( 0x22441193, id, "Id zero is reserved" );
        return id;
    }
    template<typename T0, typename T1>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1 )
    {
        TypeStreamId typeId = getOrCreateTypeStreamId( type );
        hkUint64 id = hkVdbCmdId::computeStreamId( typeId, tail0, tail1 );
        HK_ASSERT( 0x22441193, id, "Id zero is reserved" );
        return id;
    }
    template<typename T0, typename T1, typename T2>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1, const T2& tail2 )
    {
        TypeStreamId typeId = getOrCreateTypeStreamId( type );
        hkUint64 id = hkVdbCmdId::computeStreamId( typeId, tail0, tail1, tail2 );
        HK_ASSERT( 0x22441193, id, "Id zero is reserved" );
        return id;
    }
    template<typename T0, typename T1, typename T2, typename T3>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1, const T2& tail2, const T3& tail3 )
    {
        TypeStreamId typeId = getOrCreateTypeStreamId( type );
        hkUint64 id = hkVdbCmdId::computeStreamId( typeId, tail0, tail1, tail2, tail3 );
        HK_ASSERT( 0x22441193, id, "Id zero is reserved" );
        return id;
    }

    template<typename T0>
    void decomposeId( hkUint64 id, T0& tail0 ) const
    {
        using namespace hkPack;
        HK_ASSERT( 0x22441270, id, "Id zero is reserved" );
        hkUint64 streamId = hkVdbCmdId::extractStreamId( id );
        TypeStreamId typeIdOut;
        Unpack( streamId, typeIdOut, tail0 );
    }
    template<typename T0, typename T1>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1 ) const
    {
        using namespace hkPack;
        HK_ASSERT( 0x22441270, id, "Id zero is reserved" );
        hkUint64 streamId = hkVdbCmdId::extractStreamId( id );
        TypeStreamId typeIdOut;
        Unpack( streamId, typeIdOut, tail0, tail1 );
    }
    template<typename T0, typename T1, typename T2>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1, T2& tail2 ) const
    {
        using namespace hkPack;
        HK_ASSERT( 0x22441270, id, "Id zero is reserved" );
        hkUint64 streamId = hkVdbCmdId::extractStreamId( id );
        TypeStreamId typeIdOut;
        Unpack( streamId, typeIdOut, tail0, tail1, tail2 );
    }
    template<typename T0, typename T1, typename T2, typename T3>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1, T2& tail2, T3& tail3 ) const
    {
        using namespace hkPack;
        HK_ASSERT( 0x22441270, id, "Id zero is reserved" );
        hkUint64 streamId = hkVdbCmdId::extractStreamId( id );
        TypeStreamId typeIdOut;
        Unpack( streamId, typeIdOut, tail0, tail1, tail2, tail3 );
    }
#endif

    /// Returns the streamId used for the given type.
    /// This is used both for the hkObjectTypeRegistry and the uid computation (computeId).
    hkResult getTypeStreamId( const hkReflect::Type* type, TypeStreamId& idOut ) const;
    TypeStreamId getOrCreateTypeStreamId( const hkReflect::Type* type );

    /// Extract the streamId for the type from the given packed streamId.
    static TypeStreamId extractTypeStreamId( hkUint64 streamId );

    //
    // hkObjectTypeRegistry interface
    //

    HK_INLINE_VIRTUAL virtual bool isObjectTypeRegistered( TypeId typeId ) const HK_OVERRIDE
    { hkCriticalSectionLock lock( m_accessLock ); return ( typeId >= 0 ) && ( typeId < m_registeredObjectTypeInfos.getSize() ); }
    HK_INLINE bool isObjectTypeRegistered( const hkReflect::Type* type ) const
    { hkCriticalSectionLock lock( m_accessLock ); return m_registeredObjectTypeInfosMap.contains( getRuntimeTypeUid( type ) );  }
    HK_INLINE_VIRTUAL virtual bool registerObjectType( TypeId typeId, const hkReflect::Type* type ) HK_OVERRIDE
    { return ( registerObjectTypeInternal( typeId, type ) != HK_NULL ); }

    //
    // hkConnectedObjectRegistry interface
    //

    HK_INLINE_VIRTUAL virtual bool hasObject( ObjectId id ) const HK_OVERRIDE
    {
        HK_ASSERT( 0x22441380, false, "hkServerObjectSerializer doesn't track added objects for performance reasons, use hkServerObjectHandler" );
        return false;
    }
    HK_INLINE_VIRTUAL virtual int addObjects(
        const hkArrayView<ObjectId>& ids,
        const hkArrayView<hkReflect::Var>& objects,
        hkObjectFlags flags = hkObjectFlags::DEFAULT,
        int tag = 0 ) HK_OVERRIDE;
    virtual int updateObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects ) HK_OVERRIDE;
    virtual int removeObjects( const hkArrayView<ObjectId>& ids ) HK_OVERRIDE;
    HK_INLINE_VIRTUAL virtual bool isConnected( ConnectionId id1, ConnectionId id2 ) HK_OVERRIDE
    {
        HK_ASSERT( 0x22441379, false, "hkServerObjectSerializer doesn't track connections for performance reasons, use hkServerObjectHandler" );
        return false;
    }
    
    
    virtual int connect( ConnectionId id, const hkArrayView<ConnectionId>& ids, int tag = 0 ) HK_OVERRIDE;
    virtual int disconnect( ConnectionId id, const hkArrayView<ConnectionId>& ids ) HK_OVERRIDE;
    virtual int disconnect( ConnectionId id, hkConnectivityFlags flags = hkConnectivityFlags::ALL ) HK_OVERRIDE;
    using hkConnectedObjectRegistry<hkUint64, hkUint64>::disconnect;

    //
    // hkProcess interface
    //

    virtual int getProcessTag() const HK_OVERRIDE { return 0; }
    virtual void getConsumableCommands( hkUint8*& commands, int& numCommands ) HK_OVERRIDE;
    virtual void consumeCommand( hkUint8 command ) HK_OVERRIDE;
    virtual void step( hkReal frameTimeInMs ) HK_OVERRIDE;

protected:

    struct ServerTypeInfo : public hkVdbReflect::TypeInfo
    {
        HK_DECLARE_CLASS(ServerTypeInfo, New);
        TypeId m_id;
        TypeStreamId m_streamId;
    };

    void writeAddOrUpdateObjectsCmdHeader( bool addOrUpdate, int numObjects, int totalObjectDataSize, hkObjectFlags* flags = HK_NULL );
    void writeAddOrUpdateObjectsCmdObjectStart( bool addOrUpdate, ObjectId id, hkVdbReflect::Var var, TypeId typeId );
    void writeAddOrUpdateObjectsCmdFooter( bool addOrUpdate, int* tag = HK_NULL );

    ServerTypeInfo* registerObjectTypeInternal( hkUlong typeUid, const hkReflect::Type* type );

    int connectOrDisconnectIds(
        bool connectOrDisconnect,
        ConnectionId id,
        const hkArrayView<ConnectionId>* ids,
        hkConnectivityFlags flags,
        int tag );

    HK_INLINE_VIRTUAL hkUlong getRuntimeTypeUid( const hkReflect::Type* type ) const
    {
        if ( const hkReflect::Detail::InheritanceInfo* info = hkReflect::TypeDetail::getInheritance( type ) )
        {
            return info->asUlong();
        }
        else
        {
            HK_ASSERT( 0x22441384, false, "hkServerObjectSerializer only supports native types" );
            return hkUlong( -1 );
        }
    }

    HK_INLINE ServerTypeInfo* getTypeInfo( const hkReflect::Type* type )
    {
        hkCriticalSectionLock lock( m_accessLock );
        int idx = m_registeredObjectTypeInfosMap.getWithDefault( getRuntimeTypeUid( type ), -1 );
        if ( ( idx >= 0 ) && ( idx < m_registeredObjectTypeInfos.getSize() ) )
        {
            return &m_registeredObjectTypeInfos[idx];
        }
        else
        {
            return HK_NULL;
        }
    }

    HK_INLINE const ServerTypeInfo* getTypeInfo( const hkReflect::Type* type ) const
    {
        return const_cast< hkServerObjectSerializer* >( this )->getTypeInfo( type );
    }

    /// Lock for out stream shared with other handlers that can be called MT.
    mutable hkCriticalSection* m_outStreamLock;

    /// Lock for objectHandler shared access.
    mutable hkCriticalSection* m_accessLock;

    hkArray<ServerTypeInfo> m_registeredObjectTypeInfos;
    hkVdbIdMap<hkUlong, int> m_registeredObjectTypeInfosMap;
#ifdef HK_VDB_USE_HK_SERIALIZE
    hkSerialize::Save* m_objectSaver;
#else
    hkVdbSerialize::TypeSave* m_typeSaver;
#endif
    hkReflect::MutableTypeReg::Subscription m_typeRegSubscription;
};

/// This class facilitates sending hkReflect::Vars from the server to the vdb client.
/// It uses the hkServerObjectSerializer, but adds book-keeping to ease the proper
/// use of the serializer.  Unlike other "handlers" which are shared across processes
/// this is intended to be created for each process that will use object inspection.
class HK_EXPORT_COMMON hkServerObjectHandler :
    public hkReferencedObject,
    public hkObjectTypeRegistry<hkInt32>,
    public hkConnectedObjectRegistry<hkUint64, hkUint64>
{
public:
    HK_DECLARE_CLASS( hkServerObjectHandler, New );

    /// Create a handler which uses the provided serializer to send hkReflect::Vars.
    hkServerObjectHandler( hkServerObjectSerializer& serializer );
    ~hkServerObjectHandler();

#ifdef HK_VARIADIC_TEMPLATES
    /// Forwarder to underlying hkServerObjectSerializer.
    template<typename... Args>
    hkUint64 computeId( const hkReflect::Type* type, const Args&... args ) { return m_serializer->computeId( type, args... ); }
    /// Forwarder to underlying hkServerObjectSerializer.
    template<typename ObjectType, typename... Args>
    hkUint64 computeId( const Args&... args ) { return computeId( hkReflect::getType<ObjectType>(), args... ); }
    /// Forwarder to underlying hkServerObjectSerializer.
    template<typename... Args>
    void decomposeId( hkUint64 id, Args&... args ) { return m_serializer->decomposeId( id, args... ); }
#else
    template<typename T0>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0 ) { return m_serializer->computeId<>( type, tail0 ); }
    template<typename T0, typename T1>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1 ) { return m_serializer->computeId<>( type, tail0, tail1 ); }
    template<typename T0, typename T1, typename T2>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1, const T2& tail2 ) { return m_serializer->computeId<>( type, tail0, tail1, tail2 ); }
    template<typename T0, typename T1, typename T2, typename T3>
    hkUint64 computeId( const hkReflect::Type* type, const T0& tail0, const T1& tail1, const T2& tail2, const T3& tail3 ) { return m_serializer->computeId<>( type, tail0, tail1, tail2, tail3 ); }

    template<typename ObjectType, typename T0>
    hkUint64 computeId( const T0& tail0 ) { return computeId<>( hkReflect::getType<ObjectType>(), tail0 ); }
    template<typename ObjectType, typename T0, typename T1>
    hkUint64 computeId( const T0& tail0, const T1& tail1 ) { return computeId<>( hkReflect::getType<ObjectType>(), tail0, tail1 ); }
    template<typename ObjectType, typename T0, typename T1, typename T2>
    hkUint64 computeId( const T0& tail0, const T1& tail1, const T2& tail2 ) { return computeId<>( hkReflect::getType<ObjectType>(), tail0, tail1, tail2 ); }
    template<typename ObjectType, typename T0, typename T1, typename T2, typename T3>
    hkUint64 computeId( const T0& tail0, const T1& tail1, const T2& tail2, const T3& tail3 ) { return computeId<>( hkReflect::getType<ObjectType>(), tail0, tail1, tail2, tail3 ); }

    template<typename T0>
    void decomposeId( hkUint64 id, T0& tail0 ) { return m_serializer->decomposeId<>( id, tail0 ); }
    template<typename T0, typename T1>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1 ) { return m_serializer->decomposeId<>( id, tail0, tail1 ); }
    template<typename T0, typename T1, typename T2>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1, T2& tail2 ) { return m_serializer->decomposeId<>( id, tail0, tail1, tail2 ); }
    template<typename T0, typename T1, typename T2, typename T3>
    void decomposeId( hkUint64 id, T0& tail0, T1& tail1, T2& tail2, T3& tail3 ) { return m_serializer->decomposeId<>( id, tail0, tail1, tail2, tail3 ); }
#endif

    //
    // hkObjectTypeRegistry interface
    //

    HK_INLINE_VIRTUAL virtual bool isObjectTypeRegistered( TypeId typeId ) const HK_FINAL_OVERRIDE
    { return m_serializer->isObjectTypeRegistered( typeId ); }
    HK_INLINE bool isObjectTypeRegistered( const hkReflect::Type* type ) const
    { return m_serializer->isObjectTypeRegistered( type ); }
    HK_INLINE_VIRTUAL virtual bool registerObjectType( TypeId typeId, const hkReflect::Type* type ) HK_FINAL_OVERRIDE
    { return m_serializer->registerObjectType( typeId, type ); }

    //
    // hkConnectedObjectRegistry interface
    //

    virtual bool hasObject( ObjectId id ) const HK_FINAL_OVERRIDE;
    virtual int addObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects, hkObjectFlags flags = hkObjectFlags::DEFAULT, int tag = 0 ) HK_FINAL_OVERRIDE;
    virtual int updateObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects ) HK_FINAL_OVERRIDE;
    virtual int removeObjects( const hkArrayView<ObjectId>& ids ) HK_FINAL_OVERRIDE;
    virtual bool isConnected( ConnectionId id1, ConnectionId id2 ) HK_FINAL_OVERRIDE;
    virtual int connect( ConnectionId id, const hkArrayView<ConnectionId>& ids, int tag = 0 ) HK_FINAL_OVERRIDE;
    virtual int disconnect( ConnectionId id, const hkArrayView<ConnectionId>& ids ) HK_FINAL_OVERRIDE;
    virtual int disconnect( ConnectionId id, hkConnectivityFlags flags = hkConnectivityFlags::ALL ) HK_FINAL_OVERRIDE;
    using hkConnectedObjectRegistry<hkUint64, hkUint64>::disconnect;

    /// Get the underlying serializer.
    hkServerObjectSerializer* getSerializer() { return m_serializer; }

protected:
    void cleanupAddedObjects();
    void cleanupConnectedObjects();

    mutable hkCriticalSection* m_accessLock;
    hkRefPtr<hkServerObjectSerializer> m_serializer;

    hkVdbIdSet<ObjectId> m_addedObjects;
    hkVdbIdMap<ConnectionId, hkArray<ConnectionId>*> m_connectedObjects;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
