// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkServerObjectHandler.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#include <Common/Visualize/Serialize/hkVdbOStream.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkServerObjectHandler::ConnectionId, hkArray<hkServerObjectHandler::ConnectionId>*>;
template class HK_EXPORT_COMMON hkMap<hkServerObjectHandler::ConnectionId, hkArray<hkServerObjectHandler::ConnectionId>*>;
#endif

using namespace hkVdbCmdUtils;

namespace
{
    void onBuiltinTypeRegChanged( const hkReflect::MutableTypeReg::TypesChangedFuncArgs& args, void* cbData )
    {
        hkVdbIdMap<hkUlong, int>* registeredObjectTypeInfosMap = reinterpret_cast< hkVdbIdMap<hkUlong, int>* >( cbData );
        registeredObjectTypeInfosMap->clear();
    }

#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING

    hkUint64 s_objectCmdWriteTickAcc = 0;
    hkUint64 s_objectCmdWriteTick = 0;
    hkUint64 s_objectCmdReportTickAcc = 0;
    hkUint64 s_objectCmdReportTick = 0;
    void checkAndReportObjectCmdTime()
    {
        hkUint64 tick = hkSystemClock::getTickCounter();
        s_objectCmdReportTickAcc += ( tick - s_objectCmdReportTickAcc );
        if ( hkSystemClock::secondsFromTicks( s_objectCmdReportTickAcc ) > 1.0f )
        {
            HK_REPORT2(0x248edeb3,
                "Object Cmd Write: " << hkSystemClock::secondsFromTicks( s_objectCmdWriteTickAcc ) << "s" );
            s_objectCmdWriteTickAcc = 0;
            s_objectCmdReportTickAcc = 0;
        }
    }
}

#define START_OBJECT_CMD_WRITE() s_objectCmdWriteTick = hkSystemClock::getTickCounter()
#define END_OBJECT_CMD_WRITE() s_objectCmdWriteTickAcc += ( hkSystemClock::getTickCounter() - s_objectCmdWriteTick )

#else
#define START_OBJECT_CMD_WRITE()
#define END_OBJECT_CMD_WRITE()
#endif
}

hkServerObjectSerializer::hkServerObjectSerializer( hkVdbOStream* outStream, hkCriticalSection* outStreamLock )
{
    m_outStream = outStream;
    m_outStreamLock = outStreamLock;
    m_accessLock = new hkCriticalSection( 1000 );

    // We use null platform type to avoid cross-platform type.
    // That way, we always read the raw data as-if on the target platform.
    // The impl takes care of translating reads from target to windows platform.
#ifdef HK_VDB_USE_HK_SERIALIZE
    m_objectSaver = new hkSerialize::Save();
    m_objectSaver->withMultiBundle();
#else
    m_typeSaver = new hkVdbSerialize::TypeSave();
#endif

    // Register for type registry changes so we can invalidate our registered object types
    m_typeRegSubscription = hkReflect::getTypeReg()->subscribeForChange( onBuiltinTypeRegChanged, &m_registeredObjectTypeInfosMap );
}
hkServerObjectSerializer::~hkServerObjectSerializer()
{
    m_typeRegSubscription.unsubscribe();
    delete m_accessLock;
#ifdef HK_VDB_USE_HK_SERIALIZE
    delete m_objectSaver;
#else
    delete m_typeSaver;
#endif
}

hkServerObjectSerializer::ServerTypeInfo* hkServerObjectSerializer::registerObjectTypeInternal( hkUlong typeUid, const hkReflect::Type* type )
{
    using namespace hkReflect;
    using namespace hkSerialize;

    hkCriticalSectionLock lock( m_accessLock );
    ServerTypeInfo* typeInfo = HK_NULL;
    {
        int potentialIdx = m_registeredObjectTypeInfos.getSize();
        hkVdbIdMap<hkUint32, int>::Iterator iter = m_registeredObjectTypeInfosMap.findOrInsertKey( typeUid, potentialIdx );
        int idx = m_registeredObjectTypeInfosMap.getValue( iter );
        if ( idx == potentialIdx )
        {
            // Note: local viewers don't send data, but will call this function to compute ids
            typeInfo = &m_registeredObjectTypeInfos.expandOne();
            typeInfo->m_id = idx;
            HK_ASSERT( 0x22441386, m_registeredObjectTypeInfos.getSize() <= hkInt32( hkBitmask( hkBitSizeOf( TypeStreamId ) ) ), "TypeStreamId overflow" );
            typeInfo->m_streamId = TypeStreamId( m_registeredObjectTypeInfos.getSize() ); // will never == 0, which is reserved

#ifndef HK_VDB_DISABLE_OBJECT_CMD_WRITING
            if ( m_outStream )
            {
                hkArrayStreamWriter typeBufferWriter;
#ifdef HK_VDB_USE_HK_SERIALIZE
                if ( m_objectSaver->contentsVar( hkReflect::Var( type ), &typeBufferWriter ) == HK_SUCCESS )
#else
                if ( hkVdbSerialize::serializeType(
                    typeBufferWriter,
                    *m_typeSaver,
                    type,
                    *typeInfo,
                    m_outStreamLock ).isSuccess() )
#endif
                {
                    START_OBJECT_CMD_WRITE();
                    m_outStreamLock->enter();

                    m_outStream->write32( s8 + s32 + typeBufferWriter.getDataSize() );
                    m_outStream->write8u( hkVdbCmdType::REGISTER_OBJECT_TYPE );
                    m_outStream->write32( idx );
                    m_outStream->writeRaw( typeBufferWriter.getData(), typeBufferWriter.getDataSize() );

                    m_outStreamLock->leave();
                    END_OBJECT_CMD_WRITE();
                }
                else
                {
                    m_registeredObjectTypeInfosMap.remove( iter );
                    m_registeredObjectTypeInfos.setSize( potentialIdx );
                    typeInfo = HK_NULL;
                }
            }
#endif
        }
        else
        {
            typeInfo = &m_registeredObjectTypeInfos[idx];
        }
    }
    return typeInfo;
}

int hkServerObjectSerializer::addObjects(
    const hkArrayView<ObjectId>& ids,
    const hkArrayView<hkReflect::Var>& objects,
    hkObjectFlags flags,
    int tag )
{
    if ( !m_outStream || !m_outStream->isOk() || ( ids.getSize() != objects.getSize() ) )
    {
        
        
        HK_ASSERT( 0x22441383, ids.getSize() == objects.getSize(), "Arrays must be the same size" );
        return 0;
    }

#ifndef HK_VDB_DISABLE_OBJECT_CMD_WRITING

    using namespace hkReflect;

    
    
    
    
    
    
    
    
    

    int objectDataSize = 0;
    hkLocalArray<ObjectId> addedIds( ids.getSize() );
    hkLocalArray<Var> addedObjects( ids.getSize() );
#ifdef HK_VDB_USE_HK_SERIALIZE
    hkLocalArray<hkArray<char>> addedObjectSerializeBuffers( ids.getSize() );
#else
    hkLocalArray<ServerTypeInfo*> addedObjectTypeInfos( ids.getSize() );
    hkLocalArray<hkArray<hkUint16>> addedObjectLocalOffets( ids.getSize() );
#endif
    if ( ids.getSize() )
    {
        hkCriticalSectionLock lock( m_accessLock );

        for ( int i = 0; i < ids.getSize(); i++ )
        {
            ObjectId id = ids[i];

            const Type* type = objects[i].getType();
            ServerTypeInfo* typeInfo = registerObjectTypeInternal( getRuntimeTypeUid( type ), type );
            if ( !typeInfo )
            {
                continue;
            }

            Var vdbObject = objects[i];
            addedIds.pushBack( id );
            addedObjects.pushBack( vdbObject );

#ifdef HK_VDB_USE_HK_SERIALIZE
            hkArray<char>& addedObjectSerializeBuffer = addedObjectSerializeBuffers.expandOne();
            m_objectSaver->contentsVar( vdbObject, &addedObjectSerializeBuffer );
            objectDataSize += addedObjectSerializeBuffer.getSize();
#else
            addedObjectTypeInfos.pushBack( typeInfo );
            objectDataSize += hkVdbSerialize::computeObjectSize( vdbObject, *typeInfo, addedObjectLocalOffets.expandOne() );
#endif
        }
    }

    const int numObjectsToSend = addedIds.getSize();
    if ( numObjectsToSend > 0 )
    {
        START_OBJECT_CMD_WRITE();
        {
            hkCriticalSectionLock lock( m_outStreamLock );

            writeAddOrUpdateObjectsCmdHeader(
                true,
                numObjectsToSend,
                objectDataSize,
                &flags );

            for ( int i = 0; i < numObjectsToSend; i++ )
            {
                ObjectId id = addedIds[i];
                hkVdbReflect::Var obj = addedObjects[i];

#ifdef HK_VDB_USE_HK_SERIALIZE
                writeAddOrUpdateObjectsCmdObjectStart( true, id, obj );
                hkArray<char>& addedObjectSerializeBuffer = addedObjectSerializeBuffers[i];
                m_outStream->writeRaw( addedObjectSerializeBuffer.begin(), addedObjectSerializeBuffer.getSize() );
#else
                ServerTypeInfo* typeInfo = addedObjectTypeInfos[i];
                hkArray<hkUint16>& localOffets = addedObjectLocalOffets[i];
                writeAddOrUpdateObjectsCmdObjectStart( true, id, obj, typeInfo->m_id );
                hkVdbSerialize::serializeObject(
                    *m_outStream->getStreamWriter(),
                    obj,
                    *typeInfo,
                    localOffets,
                    m_outStreamLock );
#endif
            }

            writeAddOrUpdateObjectsCmdFooter( true, &tag );
        }
        END_OBJECT_CMD_WRITE();
    }

    return addedIds.getSize();
#else
    return 0;
#endif
}

int hkServerObjectSerializer::updateObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects )
{
    if ( !m_outStream || !m_outStream->isOk() || ( ids.getSize() != objects.getSize() ) )
    {
        
        
        HK_ASSERT( 0x22441382, ids.getSize() == objects.getSize(), "Arrays must be the same size" );
        return 0;
    }

#ifndef HK_VDB_DISABLE_OBJECT_CMD_WRITING

    using namespace hkReflect;

    int objectDataSize = 0;
    hkLocalArray<hkUint64> updatedObjectIds( ids.getSize() );
    hkLocalArray<Var> updatedObjects( ids.getSize() );
#ifdef HK_VDB_USE_HK_SERIALIZE
    hkLocalArray<hkArray<char>> updatedObjectSerializeBuffers( ids.getSize() );
#else
    hkLocalArray<ServerTypeInfo*> updatedObjectTypeInfos( ids.getSize() );
    hkLocalArray<hkArray<hkUint16>> updatedObjectLocalOffets( ids.getSize() );
#endif
    for ( int i = 0; i < ids.getSize(); i++ )
    {
        Var vdbObject = objects[i];
        const Type* type = vdbObject.getType();
        ServerTypeInfo* typeInfo = registerObjectTypeInternal( getRuntimeTypeUid( type ), type );
        if ( !typeInfo )
        {
            continue;
        }

        ObjectId id = ids[i];
        updatedObjectIds.pushBack( id );
        updatedObjects.pushBack( vdbObject );

#ifdef HK_VDB_USE_HK_SERIALIZE
        hkArray<char>& updatedObjectSerializeBuffer = updatedObjectSerializeBuffers.expandOne();
        m_objectSaver->contentsVar( vdbObject, &updatedObjectSerializeBuffer );
        objectDataSize += updatedObjectSerializeBuffer.getSize();
#else
        updatedObjectTypeInfos.pushBack( typeInfo );
        objectDataSize += hkVdbSerialize::computeObjectSize( vdbObject, *typeInfo, updatedObjectLocalOffets.expandOne() );
#endif
    }

    if ( objectDataSize > 0 )
    {
        START_OBJECT_CMD_WRITE();
        {
            hkCriticalSectionLock lock( m_outStreamLock );

            writeAddOrUpdateObjectsCmdHeader(
                false,
                updatedObjectIds.getSize(),
                objectDataSize );

            for ( int i = 0; i < updatedObjectIds.getSize(); i++ )
            {
                ObjectId id = updatedObjectIds[i];
#ifndef HK_VDB_DISABLE_OBJECT_CMD_CONSISTENCY_CHECKS
                HK_ASSERT( 0x22441268, hasObject( id ), "You must add an object before it can be updated" );
#endif
                hkVdbReflect::Var obj = updatedObjects[i];

#ifdef HK_VDB_USE_HK_SERIALIZE
                writeAddOrUpdateObjectsCmdObjectStart( false, id, obj );
                hkArray<char>& updatedObjectSerializeBuffer = updatedObjectSerializeBuffers[i];
                m_outStream->writeRaw( updatedObjectSerializeBuffer.begin(), updatedObjectSerializeBuffer.getSize() );
#else
                ServerTypeInfo* typeInfo = updatedObjectTypeInfos[i];
                hkArray<hkUint16>& localOffets = updatedObjectLocalOffets[i];
                writeAddOrUpdateObjectsCmdObjectStart( false, id, obj, typeInfo->m_id );
                hkVdbSerialize::serializeObject(
                    *m_outStream->getStreamWriter(),
                    obj,
                    *typeInfo,
                    localOffets,
                    m_outStreamLock );
#endif
            }

            writeAddOrUpdateObjectsCmdFooter( false );
        }
        END_OBJECT_CMD_WRITE();
    }

    return updatedObjectIds.getSize();
#else
    return 0;
#endif
}

void hkServerObjectSerializer::writeAddOrUpdateObjectsCmdHeader(
    bool addOrUpdate,
    int numObjects,
    int totalObjectDataSize,
    hkObjectFlags* flags )
{
    const int headerAndFooter = s8 + s32 + ( addOrUpdate * ( s8 + s32 ) );
    const int objectHeader =
#ifdef HK_VDB_USE_HK_SERIALIZE
        ( sizeof( ObjectId ) );
#else
        ( sizeof( ObjectId ) + s32 );
#endif
    const int packetSize = headerAndFooter + ( objectHeader * numObjects ) + totalObjectDataSize;

    m_outStream->write32( packetSize );
    m_outStream->write8u( hkUint8( addOrUpdate ? hkVdbCmdType::ADD_OBJECTS : hkVdbCmdType::UPDATE_OBJECTS ) );
    if ( addOrUpdate )
    {
        HK_ASSERT( 0x22441348, flags, "Must provide flags for add cmd" );
        m_outStream->write8u( *flags );
    }
    m_outStream->write32( numObjects );
}
void hkServerObjectSerializer::writeAddOrUpdateObjectsCmdObjectStart( bool addOrUpdate, ObjectId id, hkVdbReflect::Var obj, int vdbTypeId = -1 )
{
    m_outStream->writePersistentId( id );
#ifndef HK_VDB_USE_HK_SERIALIZE
    m_outStream->write32( vdbTypeId );
#endif
}
void hkServerObjectSerializer::writeAddOrUpdateObjectsCmdFooter( bool addOrUpdate, int* tag )
{
    if ( addOrUpdate )
    {
        HK_ASSERT( 0x22441349, tag, "Must provide tag for add cmd" );
        m_outStream->write32( *tag );
    }
}

int hkServerObjectSerializer::removeObjects( const hkArrayView<ObjectId>& ids )
{
#ifndef HK_VDB_DISABLE_OBJECT_CMD_WRITING

    if ( !m_outStream || !m_outStream->isOk() )
    {
        
        
        return 0;
    }

    using namespace hkReflect;

    if ( ids.getSize() )
    {
        hkCriticalSectionLock alock( m_accessLock );
        START_OBJECT_CMD_WRITE();
        {
            hkCriticalSectionLock olock( m_outStreamLock );
            const int packetSize = s8 + s32 + ( sizeof(ObjectId) * ids.getSize() );
            m_outStream->write32( packetSize );
            m_outStream->write8u( hkVdbCmdType::REMOVE_OBJECTS );
            m_outStream->write32( ids.getSize() );

            for ( int i = 0; i < ids.getSize(); i++ )
            {
                ObjectId id = ids[i];
                m_outStream->write64u( id );
            }
        }
        END_OBJECT_CMD_WRITE();
    }

    return ids.getSize();
#else
    return 0;
#endif
}

int hkServerObjectSerializer::connect( ConnectionId id, const hkArrayView<ConnectionId>& ids, int tag )
{
    return connectOrDisconnectIds( true, id, &ids, hkConnectivityFlags::OUTGOING_CONNECTIONS, tag );
}

int hkServerObjectSerializer::disconnect( ConnectionId id, const hkArrayView<ConnectionId>& ids )
{
    return connectOrDisconnectIds( false, id, &ids, hkConnectivityFlags::OUTGOING_CONNECTIONS, 0 );
}

int hkServerObjectSerializer::disconnect( ConnectionId id, hkConnectivityFlags flags )
{
    return connectOrDisconnectIds( false, id, HK_NULL, flags, 0 );
}

int hkServerObjectSerializer::connectOrDisconnectIds(
    bool connectOrDisconnect,
    ConnectionId id,
    const hkArrayView<ConnectionId>* ids,
    hkConnectivityFlags flags,
    int tag )
{
#ifndef HK_VDB_DISABLE_OBJECT_CMD_WRITING

#ifdef HK_DEBUG
    if ( ids )
    {
        for ( int i = 0; i < ids->getSize(); i++ )
        {
            ConnectionId toId = ( *ids )[i];
            HK_ASSERT( 0x22441285, id != toId, "You should not be creating a circular connection" );
        }
    }
#endif

    if ( !m_outStream || !m_outStream->isOk() )
    {
        
        
        return 0;
    }

    if (
        // If we are doing a disconnect all
        ( ( ids == HK_NULL ) && ( flags != hkConnectivityFlags::NONE ) ) ||
        // If we are disconnecting or connecting specific ids
        ( ( ids != HK_NULL ) && ids->getSize() ) )
    {
        hkVdbCmdType::Enum type =
            ( connectOrDisconnect ) ?
            hkVdbCmdType::CONNECT_OBJECTS :
            hkVdbCmdType::DISCONNECT_OBJECTS;
        hkConnectivityFlags disconnectAllFlags =
            ( !connectOrDisconnect && !ids ) ?
            flags :
            hkConnectivityFlags::NONE;

        HK_ASSERT( 0x22441143, !ids || hkVdbCmdId::canIdsPersist( ids->begin(), ids->getSize() ), "Id uses reserved bits" );

        const int packetSize =
            // Cmd type
            s8 +
            // The fromId
            sizeof(ConnectionId) +
            // The number of toIds
            s32 +
            // The toIds
            ( ( ids ? ids->getSize() : 0 ) * sizeof(ConnectionId) ) +
            // The tag or flags
            s32;

        START_OBJECT_CMD_WRITE();
        {
            hkCriticalSectionLock lock( m_outStreamLock );
            m_outStream->write32( packetSize );
            m_outStream->write8u( hkUint8( type ) );
            m_outStream->writePersistentId( id );
            m_outStream->write32( ids ? ids->getSize() : 0 );
            if ( ids ) m_outStream->writeArray64u( ids->begin(), ids->getSize() );
            m_outStream->write32( connectOrDisconnect ? tag : int( disconnectAllFlags ) ); // # tag or disconnect flags
        }
        END_OBJECT_CMD_WRITE();
        return ( ids ) ? ids->getSize() : -1;
    }
#endif

    return 0;
}

hkResult hkServerObjectSerializer::getTypeStreamId( const hkReflect::Type* type, TypeStreamId& idOut ) const
{
    if ( const ServerTypeInfo* typeInfo = getTypeInfo( type ) )
    {
        idOut = typeInfo->m_streamId;
        return HK_SUCCESS;
    }
    else
    {
        idOut = TypeStreamId( 0 );
        return HK_FAILURE;
    }
}

hkServerObjectSerializer::TypeStreamId hkServerObjectSerializer::getOrCreateTypeStreamId( const hkReflect::Type* type )
{
    if ( const ServerTypeInfo* typeInfo = getTypeInfo( type ) )
    {
        return typeInfo->m_streamId;
    }
    else if ( const ServerTypeInfo* newTypeInfo = registerObjectTypeInternal( getRuntimeTypeUid( type ), type ) )
    {
        return newTypeInfo->m_streamId;
    }
    HK_ASSERT( 0x22441148, false, "Didn't register object type" );
    return 0;
}

hkServerObjectSerializer::TypeStreamId hkServerObjectSerializer::extractTypeStreamId( hkUint64 streamId )
{
    using namespace hkPack;
    TypeStreamId typeStreamId;
    hkPackInt<hkBitSizeOf( hkUint64 ) - HK_VDB_RESERVED_PERSISTENT_ID_BITS - hkBitSizeOf( TypeStreamId )> rest;
    Unpack( streamId, typeStreamId, rest );
    return typeStreamId;
}

hkServerObjectHandler::hkServerObjectHandler( hkServerObjectSerializer& serializer )
    : m_serializer( &serializer )
{
    m_accessLock = new hkCriticalSection( 1000 );
}

hkServerObjectHandler::~hkServerObjectHandler()
{
    cleanupAddedObjects();
    cleanupConnectedObjects();
    delete m_accessLock;
}

bool hkServerObjectHandler::hasObject( ObjectId id ) const
{
    hkCriticalSectionLock lock( m_accessLock );
    return ( m_addedObjects.contains( id ) != 0 );
}

int hkServerObjectHandler::addObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects, hkObjectFlags flags, int tag )
{
    HK_ASSERT( 0x22441383, ids.getSize() == objects.getSize(), "Arrays must be the same size" );
    hkLocalArray<ObjectId> addedIds( ids.getSize() );
    hkLocalArray<hkReflect::Var> addedObjects( ids.getSize() );
    m_accessLock->enter();
    for( int i = 0; ( i < ids.getSize() ) && ( i < objects.getSize() ); i++ )
    {
        if ( m_addedObjects.insert( ids[i] ) )
        {
            addedIds.pushBack( ids[i] );
            addedObjects.pushBack( objects[i] );
        }
    }
    m_accessLock->leave();
    return m_serializer->addObjects( addedIds, addedObjects, flags, tag );
}

int hkServerObjectHandler::updateObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects )
{
    HK_ASSERT( 0x22441385, ids.getSize() == objects.getSize(), "Arrays must be the same size" );
    hkLocalArray<ObjectId> updatedIds( ids.getSize() );
    hkLocalArray<hkReflect::Var> updatedObjects( ids.getSize() );
    m_accessLock->enter();
    for ( int i = 0; ( i < ids.getSize() ) && ( i < objects.getSize() ); i++ )
    {
        if ( m_addedObjects.contains( ids[i] ) )
        {
            updatedIds.pushBack( ids[i] );
            updatedObjects.pushBack( objects[i] );
        }
    }
    m_accessLock->leave();
    return m_serializer->updateObjects( updatedIds, updatedObjects );
}

int hkServerObjectHandler::removeObjects( const hkArrayView<ObjectId>& ids )
{
    hkLocalArray<ObjectId> removedIds( ids.getSize() );
    m_accessLock->enter();
    for ( int i = 0; i < ids.getSize(); i++ )
    {
        if ( m_addedObjects.remove( ids[i] ).isSuccess() )
        {
            removedIds.pushBack( ids[i] );
        }
    }
    m_accessLock->leave();
    return m_serializer->removeObjects( removedIds );
}

bool hkServerObjectHandler::isConnected( ConnectionId id1, ConnectionId id2 )
{
    bool connected = false;
    m_accessLock->enter();
    if ( hkArray<ConnectionId>* connections = m_connectedObjects.getWithDefault( id1, HK_NULL ) )
    {
        connected = ( connections->indexOf( id2 ) != -1 );
    }
    m_accessLock->leave();
    return connected;
}

int hkServerObjectHandler::connect( ConnectionId id, const hkArrayView<ConnectionId>& ids, int tag )
{
    m_accessLock->enter();
    {
        hkVdbIdMap<ConnectionId, hkArray<ConnectionId>*>::Iterator iter = m_connectedObjects.findOrInsertKey( id, HK_NULL );
        hkArray<ConnectionId>* connections = m_connectedObjects.getValue( iter );
        if( !connections )
        {
            connections = new hkArray<ConnectionId>();
            m_connectedObjects.setValue( iter, connections );
        }
        hkArray<ConnectionId> combinedConnections;
        combinedConnections.reserve( connections->getSize() + ids.getSize() );
        
        hkAlgorithm::insertionSort( const_cast<ConnectionId*>( ids.begin() ), ids.getSize() );
        int combinedSize = hkAlgorithm::unionOfSortedLists(
            connections->begin(), connections->getSize(),
            ids.begin(), ids.getSize(),
            combinedConnections.begin() );
        combinedConnections.setSize( combinedSize );
        connections->swap( combinedConnections );
    }
    m_accessLock->leave();
    return m_serializer->connect( id, ids, tag );
}

int hkServerObjectHandler::disconnect( ConnectionId id, const hkArrayView<ConnectionId>& ids )
{
    m_accessLock->enter();
    if( hkArray<ConnectionId>* connections = m_connectedObjects.getWithDefault( id, HK_NULL ) )
    {
        
        hkAlgorithm::insertionSort( const_cast<ConnectionId*>( ids.begin() ), ids.getSize() );
        int newSize = hkAlgorithm::inplaceDifferenceOfSortedLists(
            connections->begin(), connections->getSize(),
            ids.begin(), ids.getSize() );
        connections->setSize( newSize );
    }
    m_accessLock->leave();
    return m_serializer->disconnect( id, ids );
}

int hkServerObjectHandler::disconnect( ConnectionId id, hkConnectivityFlags flags )
{
    int disconnectedCount = 0;
    if( flags.anyIsSet( hkConnectivityFlags::OUTGOING_CONNECTIONS ) )
    {
        m_accessLock->enter();
        if( hkArray<ConnectionId>* connections = m_connectedObjects.getWithDefault( id, HK_NULL ) )
        {
            disconnectedCount += m_serializer->disconnect( id, *connections );
            m_connectedObjects.remove( id );
            delete connections;
        }
        m_accessLock->leave();
    }
    if( flags.anyIsSet( hkConnectivityFlags::INCOMING_CONNECTIONS ) )
    {
        
        
        
        
        
        disconnectedCount += m_serializer->disconnect( id, hkConnectivityFlags::INCOMING_CONNECTIONS );
    }
    return disconnectedCount;
}

void hkServerObjectHandler::cleanupAddedObjects()
{
    hkCriticalSectionLock lock( m_accessLock );
    hkLocalArray<ObjectId> addedIds( m_addedObjects.getSize() );
    for( hkVdbIdSet<ObjectId>::Iterator iter = m_addedObjects.getIterator();
        m_addedObjects.isValid( iter );
        iter = m_addedObjects.getNext( iter ) )
    {
        auto val = m_addedObjects.getElement( iter );
        addedIds.pushBack( val );
    }
    m_serializer->removeObjects( addedIds );
    m_addedObjects.clear();
}

void hkServerObjectHandler::cleanupConnectedObjects()
{
    hkCriticalSectionLock lock( m_accessLock );
    for( hkVdbIdMap<ConnectionId, hkArray<ConnectionId>*>::Iterator iter = m_connectedObjects.getIterator();
        m_connectedObjects.isValid( iter );
        iter = m_connectedObjects.getNext( iter ) )
    {
        if ( hkArray<ConnectionId>* connections = m_connectedObjects.getValue( iter ) )
        {
            ConnectionId id = m_connectedObjects.getKey( iter );
            m_serializer->disconnect( id, *connections );
            delete connections;
        }
    }
    m_connectedObjects.clear();
}



void hkServerObjectSerializer::getConsumableCommands( hkUint8*& commands, int& numCommands )
{
    
    
    commands = HK_NULL;
    numCommands = 0;
}

void hkServerObjectSerializer::consumeCommand( hkUint8 command )
{
#if 0
    switch ( command )
    {
        case hkVisualDebuggerProtocol::HK_TWEAK_DATA:
        {
        }
        break;
    }
#endif
}

void hkServerObjectSerializer::step( hkReal frameTimeInMs )
{
#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING
    checkAndReportObjectCmdTime();
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
