// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkServerDebugDisplayHandler.h>

#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Common/Visualize/Serialize/hkVdbOStream.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Common/Visualize/Shape/hkDisplayMesh.h>

#define DEBUG_LOG_IDENTIFIER "common.visualize.debugdisplay"
#include <Common/Base/System/Log/hkLog.hxx>

// These are a misuse of the apis and can produce incorrect results; but aren't catestophic.
//#define ASSERT_ON_MISORDERED_GEOMETRY_COMMANDS
#ifdef ASSERT_ON_MISORDERED_GEOMETRY_COMMANDS
#define ASSERT_ON_MISORDERED_CMD HK_ASSERT
#else
#define ASSERT_ON_MISORDERED_CMD(...)
#endif

using namespace hkVdbCmdUtils;

namespace
{
    static void _sendGeometryData( hkVdbOStream* outStream, const hkArrayBase<hkDisplayGeometry*>& geometries )
    {
        outStream->write32( geometries.getSize() );
        for ( int i = 0; i < geometries.getSize(); i++ )
        {
            outStream->writeDisplayGeometry( geometries[i] );
        }
    }

    static hkUint32 _getGeometryByteSize( const hkArrayBase<hkDisplayGeometry*>& geometries )
    {
        // We send the size, so start with s32
        hkUint32 bytes = s32;
        for ( int i = 0; i < geometries.getSize(); i++ )
        {
            bytes += hkVdbOStream::computeBytesRequired( geometries[i] );
        }
        return bytes;
    }
}

class DeprecatedDebugDisplayHandlerImpl : public hkReferencedObject, public hkDeprecatedDebugDisplayHandlerInterface
{
public:

    HK_DECLARE_CLASS( DeprecatedDebugDisplayHandlerImpl, New );

    DeprecatedDebugDisplayHandlerImpl( hkVdbOStream* outStream, hkCriticalSection* outStreamLock ) :
        m_outStreamLock( outStreamLock ),
        m_outStream ( outStream )
    {}

    using hkDeprecatedDebugDisplayHandlerInterface::addGeometryHash;
    virtual hkResult addGeometryHash(
        const hkReferencedObject* source,
        hkUint64 hash,
        const hkAabb& aabb,
        hkColor::Argb color,
        const hkTransform& transform,
        hkUint64 id,
        int tag ) HK_OVERRIDE
    {
        if ( const hkMeshBody* sourceMesh = hkDynCast<hkMeshBody>( source ) )
        {
            bool streamOK;

            m_outStreamLock->enter();
            {
                if ( m_outStream )
                {
                    const int packetSize = 1 + sizeof( hkUint64 ) + 6 * 4 + 4 + 7 * 4 + 8 + 4;
                    const hkUint64 longId = ( hkUint64 ) id;

                    m_outStream->write32( packetSize );
                    // send a serialized version of the displayObject
                    m_outStream->write8u( hkVdbCmdType::DEPRECATED_ADD_GEOMETRY_HASH );
                    m_outStream->write64u( longId );
                    m_outStream->write64u( hash );
                    m_outStream->writeAabb( aabb );
                    m_outStream->write32u( color );
                    m_outStream->writeTransform( transform );
                    m_outStream->write32( tag );
                }
                streamOK = ( m_outStream && m_outStream->isOk() );
            }
            m_outStreamLock->leave();

            // Store information about the hash
            {
                // Do we have a source and builder already for this hash?
                hkHashMap<hkUint64, int>::Iterator iterator = m_hashCountMap.findOrInsertKey( hash, 0 );
                const int count = m_hashCountMap.getValue( iterator );
                if ( count == 0 )
                {
                    // No.
                    // Add the info to the geometries awaiting requests array.
                    hkTuple<hkUint64, hkRefPtr<const hkReferencedObject>>& newEntry = m_geometriesAwaitingRequests.expandOne();
                    newEntry.m_0 = hash;
                    newEntry.m_1 = sourceMesh;
                }
                m_hashCountMap.setValue( iterator, count + 1 );
            }
            return  streamOK ? HK_SUCCESS : HK_FAILURE;
        }
        else
        {
            HK_ASSERT( 0x22441319, false, "Only suitable for hkMeshBody use" );
            return HK_FAILURE;
        }
    }

    hkResult addGeometryHash(
        const hkReferencedObject* source,
        hkUint64 hash,
        hkUint64 id,
        int tag )
    {
        hkAabb dummy;
        dummy.setEmpty();
        return addGeometryHash( source, hash, dummy, 0, hkTransform::getIdentity(), id, tag );
    }

    hkResult useGeometryForHash( const hkArray<hkDisplayGeometry*>& geometries, hkUint64 hash )
    {
        bool streamOK;

        m_outStreamLock->enter();
        {
            if ( m_outStream )
            {
                const int packetSize = 1 + sizeof( hkUint64 ) + _getGeometryByteSize( geometries );

                m_outStream->write32( packetSize );
                // send a serialized version of the displayObject
                m_outStream->write8u( ( hkUint8 ) hkVdbCmdType::DEPRECATED_ADD_GEOMETRY_HASH_FINAL );
                m_outStream->write64u( hash );
                _sendGeometryData( m_outStream, geometries );
            }
            streamOK = ( m_outStream && m_outStream->isOk() );
        }
        m_outStreamLock->leave();

        return  streamOK ? HK_SUCCESS : HK_FAILURE;
    }

    int findIndexForGeometryAwaitingRequest( hkUint64 hash ) const
    {
        const int numPending = m_geometriesAwaitingRequests.getSize();
        for ( int i = 0; i < numPending; ++i )
        {
            if ( hash == m_geometriesAwaitingRequests[i].m_0 )
            {
                return i;
            }
        }
        return -1;
    }

    void processGeometryWithHashRequested( hkUint64 hash )
    {
        const int index = findIndexForGeometryAwaitingRequest( hash );

        HK_ASSERT( 0xa8b7d6de, index != -1, "Can't find source object for geometry requested command" );

        // Move the info from one array to the other.
        m_geometriesAwaitingDeparture.expandOne() = m_geometriesAwaitingRequests[index];
        m_geometriesAwaitingRequests.removeAt( index );
    }

    void processGeometryWithHashNotRequested( hkUint64 hash )
    {
        hkHashMap<hkUint64, int>::Iterator iterator = m_hashCountMap.findKey( hash );
        HK_ASSERT( 0xea8b291e, m_hashCountMap.isValid( iterator ), "Count for source object invalid." );
        const int newCount = m_hashCountMap.getValue( iterator ) - 1;
        HK_ASSERT( 0xea8b291f, newCount >= 0, "Count for source object invalid." );
        m_hashCountMap.setValue( iterator, newCount );
        if ( newCount == 0 )
        {
            // The info must be in the awaiting requests list.
            const int index = findIndexForGeometryAwaitingRequest( hash );
            HK_ASSERT( 0xa8b7d6de, index != -1, "Can't find source object for geometry not requested command" );
            m_geometriesAwaitingRequests.removeAt( index );
            // Remove the map entry.
            m_hashCountMap.remove( iterator );
        }
    }

    void buildAndSendGeometries()
    {
        for ( int geometriesIt = 0; geometriesIt < m_geometriesAwaitingDeparture.getSize(); geometriesIt++ )
        {
            hkTuple<hkUint64, hkRefPtr<const hkReferencedObject>>& info = m_geometriesAwaitingDeparture[geometriesIt];

            hkInplaceArray<hkDisplayGeometry*, 8> displayGeometries;

            if ( const hkMeshBody* mesh = hkDynCast<hkMeshBody>( info.m_1 ) )
            {
                displayGeometries.pushBack( new hkDisplayMesh( const_cast< hkMeshBody* >( mesh ) ) );
            }
            else
            {
                continue;
            }

            for ( int i = ( displayGeometries.getSize() - 1 ); i >= 0; i-- )
            {
                if ( ( displayGeometries[i]->getType() == HK_DISPLAY_ANY ) &&
                    ( displayGeometries[i]->getGeometry() == HK_NULL ) )
                {
                    HK_WARN_ONCE( 0x22441320, "Unable to build some display geometry data" );
                    displayGeometries.removeAt( i );
                }
            }

            useGeometryForHash( displayGeometries, info.m_0 );

            {
                hkHashMap<hkUint64, int>::Iterator iterator = m_hashCountMap.findKey( info.m_0 );
                HK_ASSERT( 0xea8b291e, m_hashCountMap.isValid( iterator ), "Count for source object invalid." );
                const int newCount = m_hashCountMap.getValue( iterator ) - 1;
                HK_ASSERT( 0xea8b291f, newCount >= 0, "Count for source object invalid." );
                m_hashCountMap.setValue( iterator, newCount );
                if ( newCount == 0 )
                {
                    // Remove the map entry.
                    m_hashCountMap.remove( iterator );
                }
                else
                {
                    // Move it back into the awaiting requests list.
                    m_geometriesAwaitingRequests.expandOne() = info;
                }
            }

            for ( int i = 0; i < displayGeometries.getSize(); ++i )
            {
                delete displayGeometries[i];
            }
        }

        // Remove those we've successfully built and sent.
        m_geometriesAwaitingDeparture.clear();
    }

    mutable hkCriticalSection* m_outStreamLock;
    hkVdbOStream* m_outStream;

    /// Information about geometries we've sent to the client (in addGeometryHash
    /// messages), but haven't yet received a command telling us whether it needs them or
    /// not.
    hkArray<hkTuple<hkUint64, hkRefPtr<const hkReferencedObject>>> m_geometriesAwaitingRequests;

    /// Geometries that the client does want us to build and send.
    /// The first element of this list may have been partial built and send in the
    /// previous frame. This will be the case if m_continueData is non-null.
    hkArray<hkTuple<hkUint64, hkRefPtr<const hkReferencedObject>>> m_geometriesAwaitingDeparture;

    /// References to one source and builder are kept for each hash, controlled
    /// by the usage count in this map. The algorithms in this class assume that
    /// only one do-request for a geometry corresponding to a particular hash can
    /// exist at one time (but many do-not-requests can co-exist with a do-request).
    hkHashMap<hkUint64, int> m_hashCountMap;
};

hkServerDebugDisplayHandler::hkServerDebugDisplayHandler(
    hkVdbOStream* outStream,
    hkCriticalSection* outStreamLock,
    hkVdbIStream* inStream ) :
    m_outStreamLock( outStreamLock ),
    m_deprecatedHandler( hkRefNew<DeprecatedDebugDisplayHandlerImpl>( new DeprecatedDebugDisplayHandlerImpl( outStream, outStreamLock ) ) )
{
    m_outStream = outStream;
    m_inStream = inStream;
}
hkServerDebugDisplayHandler::~hkServerDebugDisplayHandler()
{
    ASSERT_ON_MISORDERED_CMD( 0x22441280, m_addedGeometries.getSize() == 0, "Some geometry was never removed" );
}

hkResult hkServerDebugDisplayHandler::setDisplayOptions( Options& options )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + sV + sF + sA;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::SET_DISPLAY_OPTIONS );
            m_outStream->writeQuadVector4( options.m_up );
            m_outStream->writeFloat32( hkFloat32( options.m_unitSize ) );
            m_outStream->writeAabb( options.m_displayBounds );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return  streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::addGeometry(
    hkUint64 id,
    const hkArrayBase<hkDisplayGeometry*>& geometries,
    const hkTransform& transform,
    int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        ASSERT_ON_MISORDERED_CMD( 0x22441276, m_addedGeometries.insert( id ), "Id is already in use" );
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + _getGeometryByteSize( geometries ) + sT + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::ADD_GEOMETRY );
            m_outStream->writePersistentId( id );
            _sendGeometryData( m_outStream, geometries );
            m_outStream->writeTransform( transform );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return  streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::addGeometryEx(
    hkUint64 id,
    const hkArrayBase<hkDisplayGeometry*>& geometries,
    const hkTransform& transform,
    const hkColor::Argb color,
    hkDisplayGeometryFlags flags,
    int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        ASSERT_ON_MISORDERED_CMD( 0x22441277, m_addedGeometries.insert( id ), "Id is already in use" );
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + _getGeometryByteSize( geometries ) + sT + s32 + s8 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::ADD_GEOMETRY_EX );
            m_outStream->writePersistentId( id );
            _sendGeometryData( m_outStream, geometries );
            m_outStream->writeTransform( transform );
            m_outStream->write32u( color );
            m_outStream->write8u( flags.get() );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return HK_SUCCESS;
}

hkResult hkServerDebugDisplayHandler::addGeometryInstance(
    hkUint64 id,
    hkUint64 originalGeomId,
    const hkTransform& transform,
    int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        ASSERT_ON_MISORDERED_CMD( 0x22441278, m_addedGeometries.insert( id ), "Id is already in use" );
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + s64 + sT + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::INSTANCE_GEOMETRY );
            m_outStream->writePersistentId( id );
            m_outStream->writePersistentId( originalGeomId );
            m_outStream->writeTransform( transform );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::setGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + s8;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::SET_GEOMETRY_FLAG_BITS );
            m_outStream->write64u( id );
            m_outStream->write8u( flagBits );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::clearGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + s8;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::CLEAR_GEOMETRY_FLAG_BITS );
            m_outStream->write64u( id );
            m_outStream->write8u( flagBits );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::setGeometryColor( hkUint64 id, hkColor::Argb color )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::SET_GEOMETRY_COLOR );
            m_outStream->write64u( id );
            m_outStream->write32u( color );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::setGeometryAlpha( hkUint64 id, hkReal alpha )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + sF;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::SET_GEOMETRY_ALPHA );
            m_outStream->write64u( id );
            m_outStream->writeFloat32( hkFloat32( alpha ) );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::updateGeometryTransform( hkUint64 id, const hkTransform& transform )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    if ( m_outStream )
    {
        const int packetSize = s8 + s64 + sT;
        m_outStream->write32( packetSize );

        m_outStream->write8u( hkVdbCmdType::UPDATE_GEOMETRY_TRANSFORM );
        m_outStream->write64u( id );
        m_outStream->writeTransform( transform );
    }
    streamOK = ( m_outStream && m_outStream->isOk() );
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::updateGeometryTransformEx( hkUint64 id, const hkMatrix4& transform )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + sM;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::UPDATE_GEOMETRY_TRANSFORM_EX );
            m_outStream->write64u( id );
            m_outStream->writeMatrix( transform );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::updateParticleTransforms( hkUint64 id, int updateId, const hkQuaternion* orientations, const hkVector4* translations, hkUint32 numTransforms )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            hkUint8 useOrientations = orientations ? 1 : 0;
            const int packetSize = s8 + s64 + s32 + s8 + s32 + numTransforms * ( sV + s32 + sQ * useOrientations );
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::UPDATE_PARTICLE_TRANSFORMS );
            m_outStream->write64u( id );
            m_outStream->write32( updateId );
            m_outStream->write8u( useOrientations );
            m_outStream->write32u( numTransforms );
            for ( hkUint32 i = 0; i < numTransforms; i++ )
            {
                if ( orientations )
                {
                    m_outStream->writeArrayFloat32( &orientations[i].m_vec( 0 ), 4 );
                }
                m_outStream->writeQuadVector4( translations[i] );
                m_outStream->write32u( *(hkColor::Argb*)&translations[i]( 3 ) );
            }
        }
        streamOK = (m_outStream && m_outStream->isOk());
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::updateGeometryVerts( hkUint64 id, const hkArray<hkVector4>& newVerts )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();

    const int packetSize = s8 + s64 + s32 + ( sV * newVerts.getSize() );
    m_outStream->write32( packetSize );

    m_outStream->write8u( hkVdbCmdType::UPDATE_GEOMETRY_VERTICES );
    m_outStream->write64u( id );
    m_outStream->write32u( newVerts.getSize() );
    for ( int i = 0; i < newVerts.getSize(); i++ )
    {
        m_outStream->writeQuadVector4( newVerts[i] );
    }
    streamOK = ( m_outStream && m_outStream->isOk() );
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::removeGeometry( hkUint64 id )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        ASSERT_ON_MISORDERED_CMD( 0x22441279, m_addedGeometries.remove( id ).isSuccess(), "Id was never added" );
        if ( m_outStream )
        {
            const int packetSize = s8 + s64;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::REMOVE_GEOMETRY );
            m_outStream->write64u( id );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::updateCamera(
    const char* name,
    hkVector4Parameter from,
    hkVector4Parameter to,
    hkVector4Parameter up,
    hkReal nearPlane,
    hkReal farPlane,
    hkReal fov )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int numCharacters = ( hkString::strLen( name ) + 1 ); // include null terminator
            const int packetSize = s8 + ( sV * 3 ) + ( sF * 3 ) + ( s8 * numCharacters );
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::UPDATE_CAMERA );
            m_outStream->writeQuadVector4( from );
            m_outStream->writeQuadVector4( to );
            m_outStream->writeQuadVector4( up );
            m_outStream->writeFloat32( hkFloat32( nearPlane ) );
            m_outStream->writeFloat32( hkFloat32( farPlane ) );
            m_outStream->writeFloat32( hkFloat32( fov ) );
            m_outStream->writeString( name );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::removeCamera( const char* name )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int numCharacters = ( hkString::strLen( name ) + 1 ); // include null terminator
            const int packetSize = s8 + ( s8 * numCharacters );
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::REMOVE_CAMERA );
            m_outStream->writeString( name );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::displayPoint( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + sV + s32 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_POINT );
            m_outStream->write64u( id );
            m_outStream->writeQuadVector4( a );
            m_outStream->write32u( color );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::displayPointEx( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, hk1PointDisplayStyle::Enum style, hkReal scale, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + sV + s32 + s8 + sF + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_POINT_EX );
            m_outStream->write64u( id );
            m_outStream->writeQuadVector4( a );
            m_outStream->write32u( color );
            m_outStream->write8u( hkUint8( style ) );
            m_outStream->writeFloat32( hkFloat32( scale ) );
            m_outStream->write32( tag );
        }

        streamOK = m_outStream && m_outStream->isOk();
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::display2Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + ( sV * 2 ) + s32 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_2_POINTS );
            m_outStream->write64u( id );
            m_outStream->writeQuadVector4( a );
            m_outStream->writeQuadVector4( b );
            m_outStream->write32u( color );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::display2PointsEx( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, hk2PointDisplayStyle::Enum style, hkReal thickness, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + ( sV * 2 ) + s32 + s8 + sF + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_2_POINTS_EX );
            m_outStream->write64u( id );
            m_outStream->writeQuadVector4( a );
            m_outStream->writeQuadVector4( b );
            m_outStream->write32u( color );
            m_outStream->write8u( hkUint8( style ) );
            m_outStream->writeFloat32( hkFloat32( thickness ) );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::display3Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + ( sV * 3 ) + s32 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_3_POINTS );
            m_outStream->write64u( id );
            m_outStream->writeQuadVector4( a );
            m_outStream->writeQuadVector4( b );
            m_outStream->writeQuadVector4( c );
            m_outStream->write32u( color );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::displayText( hkUint64 id, const char* text, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int numCharacters = ( hkString::strLen( text ) + 1 );
            const int packetSize = s8 + s64 + ( s8 * numCharacters ) + s32 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_TEXT );
            m_outStream->write64u( id );
            m_outStream->writeString( text );
            m_outStream->write32u( color );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::display3dText( hkUint64 id, const char* text, hkVector4Parameter pos, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int numCharacters = ( hkString::strLen( text ) + 1 );
            const int packetSize = s8 + s64 + ( s8 * numCharacters ) + s32 + sV + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_3D_TEXT );
            m_outStream->write64u( id );
            m_outStream->writeString( text );
            m_outStream->write32( color );
            m_outStream->writeQuadVector4( pos );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::displayGeometry( hkUint64 id, const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int tag )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s64 + _getGeometryByteSize( geometries ) + sT + s32 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::DISPLAY_GEOMETRY );
            m_outStream->write64u( id );
            _sendGeometryData( m_outStream, geometries );
            m_outStream->writeTransform( transform );
            m_outStream->write32u( color );
            m_outStream->write32( tag );
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkServerDebugDisplayHandler::flushDisplay( bool clearDisplayGeometry )
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if ( m_outStream )
        {
            const int packetSize = s8 + s32;
            m_outStream->write32( packetSize );

            m_outStream->write8u( hkVdbCmdType::FLUSH_DISPLAY );
            // Note: writing as a 32 bit integer so that we can expand to "flags" later if needed
            m_outStream->write32( clearDisplayGeometry );

            
            m_outStream->getStreamWriter()->flush();
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return streamOK ? HK_SUCCESS : HK_FAILURE;
}

hkDeprecatedDebugDisplayHandlerInterface* hkServerDebugDisplayHandler::getDeprecatedInterface()
{
    return m_deprecatedHandler;
}

hkResult hkServerDebugDisplayHandler::flushStream()
{
    hkBool32 streamOK;
    m_outStreamLock->enter();
    {
        if( m_outStream )
        {
            m_outStream->getStreamWriter()->flush();
        }
        streamOK = ( m_outStream && m_outStream->isOk() );
    }
    m_outStreamLock->leave();
    return  streamOK ? HK_SUCCESS : HK_FAILURE;
}

//////////////////////////////////////////////////////////////////////////
// Deprecated hashing display support code
//////////////////////////////////////////////////////////////////////////

static hkUint8 _serverDebugDisplayHandler_cmds[] = {
    hkVdbCmdType::DEPRECATED_REQUEST_GEOMETRY_WITH_HASH,
    hkVdbCmdType::DEPRECATED_DONT_REQUEST_GEOMETRY_WITH_HASH
};

void hkServerDebugDisplayHandler::getConsumableCommands( hkUint8*& commands, int& numCommands )
{
    commands = _serverDebugDisplayHandler_cmds;
    numCommands = sizeof( _serverDebugDisplayHandler_cmds );
}

void hkServerDebugDisplayHandler::consumeCommand( hkUint8 command )
{
    HK_ASSERT( 0x22441376, m_inStream, "You must construct hkServerDebugDisplayHandler with an instream to consume incoming commands" );
    switch ( command )
    {
        case hkVdbCmdType::DEPRECATED_REQUEST_GEOMETRY_WITH_HASH:
        {
            hkUint64 hash = m_inStream->read64u();
            m_deprecatedHandler->processGeometryWithHashRequested( hash );
            break;
        }
        case hkVdbCmdType::DEPRECATED_DONT_REQUEST_GEOMETRY_WITH_HASH:
        {
            hkUint64 hash = m_inStream->read64u();
            m_deprecatedHandler->processGeometryWithHashNotRequested( hash );
            break;
        }
        default:
            break;
    }
}

void hkServerDebugDisplayHandler::step( hkReal frameTimeInMs )
{
    m_deprecatedHandler->buildAndSendGeometries();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
