// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Visualize/Shape/hkDisplayGeometryFlags.h>

class hkMeshBody;
class hkDisplayGeometry;
class hkAabb;
struct hkFrustum;

/// Marker class for display id association via hkServerObjectHandler/Serializer.
struct HK_EXPORT_COMMON hkDebugDisplayMarker
{
    HK_DECLARE_CLASS( hkDebugDisplayMarker, New, Reflect, EmptyCtor );
    enum Type
    {
        /// The parent node is represented by a geometry.
        /// Note: Because geometry persists, a GEOMETRY marker must be added the same frame
        /// as the geometry you wish to connect into the object inspection tree.
        GEOMETRY,

        /// The parent node is represented by one or more immediate-mode displays.
        IMMEDIATE,

        NUM_TYPES
    };
    hkDebugDisplayMarker( Type type ) : m_type( type ) {}
    Type m_type;
};
HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkDebugDisplayMarker::Type );

/// Different styles for displaying single points
struct HK_EXPORT_COMMON hk1PointDisplayStyle
{
    enum Enum
    {
        DOT,
        STAR
        //FILLED_DIAMOND
    };
};

/// Different styles for displaying two points
struct HK_EXPORT_COMMON hk2PointDisplayStyle
{
    enum Enum
    {
        LINE_SEGMENT,
        ARROW
        //PLANE
    };
};

/// Different styles for displaying three points
struct HK_EXPORT_COMMON hk3PointDisplayStyle
{
    enum Enum
    {
        TRIANGLE
        
        
        
    };
};

/// Interface for deprecated debug display apis
class HK_EXPORT_COMMON hkDeprecatedDebugDisplayHandlerInterface
{
    public:
        virtual hkResult skinGeometry(
            const hkUint64* ids,
            int numIds,
            const hkMatrix4* poseModel,
            int numPoseModel,
            const hkMatrix4& worldFromModel) { return HK_FAILURE; }
        virtual hkResult skinGeometry(
            const hkUint64* ids,
            int numIds,
            const hkQsTransform* poseModel,
            int numPoseModel,
            const hkQsTransform& worldFromModel );
        virtual hkResult addTextureSearchPath( const char* path ) { return HK_FAILURE; }
        virtual hkResult clearTextureSearchPaths() { return HK_FAILURE; }
        virtual hkResult addGeometryHash(
            const hkReferencedObject* source,
            hkUint64 hash,
            const hkAabb& aabb,
            hkColor::Argb color,
            const hkTransform& transform,
            hkUint64 id,
            int tag) { return HK_FAILURE; }
        HK_INLINE hkResult addGeometryHash(
            const hkReferencedObject* source,
            hkUint64 hash,
            hkUint64 id,
            int tag )
        {
            hkAabb dummy;
            dummy.setEmpty();
            return addGeometryHash( source, hash, dummy, 0, hkTransform::getIdentity(), id, tag );
        }
};

/// This class exposes visual debugger debug display capabilities.
/// It is also used locally by the havok demo framework.
///
/// IDs: IDs are used to manage persistent objects which are added/updated
/// and removed. It is the responsibility of the user to supply a unique
/// ID for every persistent object added and to use this ID consistently when
/// updating and removing objects.
/// Additionally, if an hkDebugDisplayMarker is being used to associate
/// debug display data with an object, the same ID must be used for the
/// hkDebugDisplayMarker in Object Inspection as is used here for drawing.
///
/// TAGs: Unique tags (process IDs) are allocated to processes/viewers by the hkProcessFactory.
/// These are used for separating display information into different
/// categories so that they can be configured and filtered independently.
/// For example, each Viewer has one or more unique tags.
///
/// It is expected that an implementation of displayText() and display3dText() manages
/// to show the text in some manner in any case.
class HK_EXPORT_COMMON hkDebugDisplayHandler : public hkReferencedObject
{
    //+hk.MemoryTracker(ignore=True)
public:
    virtual ~hkDebugDisplayHandler() {}

    
    /// Options for setting up the debug display
    struct HK_EXPORT_COMMON Options
    {
        Options() :
            m_unitSize( 1.0f )
        {
            m_up.set( 0.0f, 1.0f, 0.0f, 0.0f );
            m_displayBounds.setFull();
        }

        /// The global up for the display.
        /// Some implementations may only accept/snap-to cardinal axes.
        hkVector4 m_up;

        /// The user-defined size of a single unit.
        /// Can be used for things like grids, textures, even camera movement speeds.
        ///<todo.jg.3vdb flush this out a bit more.
        hkReal m_unitSize;

        /// The scope of the display world.
        /// Can be used for optimizations or other effects.
        hkAabb m_displayBounds;
    };

    /// Set display options.
    virtual hkResult setDisplayOptions( Options& options ) { return HK_FAILURE; }
    

    //
    // Geometry functions
    //

    
    /// Adds a set of geometries to the display world managed by this display handler.
    /// There's an overload which takes a single hkDisplayGeometry instead of the array for convenience.
    HK_INLINE_VIRTUAL virtual hkResult addGeometry(
        hkUint64 id,
        const hkArrayBase<hkDisplayGeometry*>& geometries,
        const hkTransform& transform,
        int tag = 0 )
    { return addGeometryEx( id, geometries, transform, hkColor::NONE ); }

    /// Adds a set of geometries to the display world managed by this display handler.
    /// This call allows for specifying extended information for the geometry which can help the
    /// implementation optimize the graphics representation.
    /// Additionally, it avoids handler defaults for the provided data.
    /// There's an overload which takes a single hkDisplayGeometry instead of the array for convenience.
    /// Note: hkColor::NONE has a value of 0x00ffffff and is interpretted as no color. If you wish to set
    /// your geometry to transparent white use hkColor::WHITE here followed by setGeometryAlpha.
    
    virtual hkResult addGeometryEx(
        hkUint64 id,
        const hkArrayBase<hkDisplayGeometry*>& geometries,
        const hkTransform& transform,
        const hkColor::Argb color,
        hkDisplayGeometryFlags flags = hkDisplayGeometryFlags::DEFAULT,
        int tag = 0 )
    { return HK_FAILURE; }

    /// Adds an instance of a geometry that has already been added to the display world managed by this display handler.
    
    
    virtual hkResult addGeometryInstance(
        hkUint64 id,
        hkUint64 originalGeomId,
        const hkTransform& transform,
        int tag = 0 )
    { return HK_FAILURE; }
    

    /// Set geometry flag bits. The flagBits are ORed with the current flags.
    /// Some implementation may not allow certain flag bits to be set after the geometry was added (see addGeometryEx).
    virtual hkResult setGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits ) { return HK_FAILURE; }

    /// Clear geometry flag bits. The ~flagBits are ANDed with the current flags.
    /// Some implementation may not allow certain flag bits to be cleared after the geometry was added (see addGeometryEx).
    virtual hkResult clearGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits ) { return HK_FAILURE; }

    /// Sets the color of an object's geometries, previously added to the display world.
    /// Note: hkColor::NONE has a value of 0x00ffffff and is interpretted as no color. If you wish to set
    /// your geometry to transparent white use hkColor::WHITE here followed by setGeometryAlpha.
    virtual hkResult setGeometryColor( hkUint64 id, hkColor::Argb color ) { return HK_FAILURE; }

    /// Sets the transparency of an object's geometries, previously added to the display world.
    virtual hkResult setGeometryAlpha( hkUint64 id, hkReal alpha ) { return HK_FAILURE; }

    /// Updates the transform of an object in the display world.
    virtual hkResult updateGeometryTransform( hkUint64 id, const hkTransform& transform ) { return HK_FAILURE; }

    /// Updates the transform of an object in the display world. The matrix can include
    /// scale. Some implementations may only support scale for some object types.
    /// HKG, for instance, only supports scale for geometries of type hkDisplayMesh.
    virtual hkResult updateGeometryTransformEx( hkUint64 id, const hkMatrix4& transform ) { return HK_FAILURE; }

    /// Updates the root transform of a display object in the display world. The hkQsTransform
    /// can include scale. Some implementations may only support scale for some object types.
    /// HKG, for instance, only supports scale for geometries of type hkDisplayMesh.
    hkResult updateGeometryTransformEx( hkUint64 id, const hkQsTransform& transform );

    /// Determines the number of instances of geometry created with hkDisplayGeometryFlags::PARTICLE_TEMPLATE and their transforms.
    /// The instance color is stored in the low 32 bits of translation.W. Orientations are optional, if none are provided then identity will be used.
    /// If an instance is picked in the VDB, then the updateId from the call that set its transform will be passed as the first GRAB_GEOMETRY argument.
    /// This method is much faster than updating transforms individually with updateGeometryTransform.
    virtual hkResult updateParticleTransforms( hkUint64 id, int updateId, const hkQuaternion* orientations, const hkVector4* translations, hkUint32 numTransforms ) { return HK_FAILURE; }

    /// Update the vertices of a geometry created with hkDisplayGeometryFlags::MUTABLE_VERTICES.
    virtual hkResult updateGeometryVerts( hkUint64 id, const hkArray<hkVector4>& newVerts ) { return HK_FAILURE; }

    /// Removes an object from the display world managed by this display handler.
    virtual hkResult removeGeometry( hkUint64 id ) { return HK_FAILURE; }

    //
    // Camera Functionality
    //

    /// Adds or updates a camera.
    virtual hkResult updateCamera(
        const char* name,
        hkVector4Parameter from,
        hkVector4Parameter to,
        hkVector4Parameter up,
        hkReal nearPlane,
        hkReal farPlane,
        hkReal fov )
    { return HK_FAILURE; }

    /// Removes a previously added camera
    virtual hkResult removeCamera( const char* name ) { return HK_FAILURE; }

    //
    // Immediate Mode Functions
    //

    
    /// Puts a display dot into the display buffer for display in the next frame.
    HK_INLINE_VIRTUAL virtual hkResult displayPoint( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, int tag = 0 )
    { return displayPointEx( id, a, color, hk1PointDisplayStyle::DOT, 0.0f, tag ); }

    /// Puts a display dot/star/etc. into the display buffer for display in the next frame.
    /// A radius can be explicitly specified for the size of the point.
    virtual hkResult displayPointEx( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, hk1PointDisplayStyle::Enum style = hk1PointDisplayStyle::DOT, hkReal scale = 0, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display line into the display buffer for display in the next frame.
    HK_INLINE_VIRTUAL virtual hkResult display2Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag = 0 )
    { return display2PointsEx( id, a, b, color, hk2PointDisplayStyle::LINE_SEGMENT, 0.0f, tag ); }

    /// Puts a display line/arrow/etc. into the display buffer for display in the next frame.
    virtual hkResult display2PointsEx( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, hk2PointDisplayStyle::Enum style = hk2PointDisplayStyle::LINE_SEGMENT, hkReal thickness = 0, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display triangle into the display buffer for display in the next frame.
    HK_INLINE_VIRTUAL virtual hkResult display3Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 )
    { return display3PointsEx( id, a, b, c, color, hk3PointDisplayStyle::TRIANGLE, tag ); }

    /// Puts a display triangle/filled triangle/etc. into the display buffer for display in the next frame.
    virtual hkResult display3PointsEx( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, hk3PointDisplayStyle::Enum style = hk3PointDisplayStyle::TRIANGLE, int tag = 0 ) { return HK_FAILURE; }
    

    /// Outputs user text to the display. The manner in which the text
    /// is displayed depends on the implementation of the display handler.
    virtual hkResult displayText( hkUint64 id, const char* text, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Outputs text at provided point in 3d space.
    virtual hkResult display3dText( hkUint64 id, const char* text, hkVector4Parameter pos, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display point in 2D [-1..1] normalized windows coordinates into the display buffer for display in the next frame.
    virtual hkResult displayPoint2d( hkUint64 id, hkVector4Parameter position, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display line in 2D [-1..1] normalized windows coordinates into the display buffer for display in the next frame.
    virtual hkResult displayLine2d( hkUint64 id, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display triangle in 2D [-1..1] normalized windows coordinates into the display buffer for display in the next frame.
    virtual hkResult displayTriangle2d( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Outputs text in 2D [-1..1] normalized windows coordinates. Also scales the text wrt the font size using sizeScale.
    virtual hkResult displayText2d( hkUint64 id, const char* text, hkVector4Parameter pos, hkReal sizeScale, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Puts a display geometry into the display buffer for the next frame.
    virtual hkResult displayGeometry( hkUint64 id, const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int tag = 0 ) { return HK_FAILURE; }

    /// Flush the display pipeline.  Optionally clear previous display geometry for the current frame.
    virtual hkResult flushDisplay( bool clearDisplayGeometry = true ) { return HK_FAILURE; }

    /// Notify debug display handler that everything in the display buffer has been drawn and handlers may flush their cached points/lines/triangles etc.
    virtual hkResult endFrame() { return HK_FAILURE; }

    //
    // Deprecated interface access (if supported).
    //

    virtual hkDeprecatedDebugDisplayHandlerInterface* getDeprecatedInterface() { return HK_NULL; }

    //
    // Utility functions (just call other display functions above)
    // Functions which make multiple calls to underlying functions are made virtual so that impls can optimize those functions if needed.
    //

    hkResult displayTransformedLine( hkUint64 id, const hkTransform& transform, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag = 0 );
    hkResult displayTransformedLine( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayLitTriangle( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 );
    HK_INLINE hkResult displayFrame( hkUint64 id, const hkTransform& transform, hkReal size, int tag = 0 ) { return displayFrame( id, transform, size, HK_NULL, tag ); }
    HK_INLINE hkResult displayFrame( hkUint64 id, const hkQsTransform& transform, hkReal size, int tag = 0 ) { return displayFrame( id, transform, size, HK_NULL, tag ); }
    hkResult displayFrame( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb* color, int tag = 0 );
    virtual hkResult displayFrame( hkUint64 id, const hkQsTransform& transform, hkReal size, const hkColor::Argb* color, int tag = 0 );
    virtual hkResult displayTetra( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb& color, int tag = 0 );
    HK_INLINE hkResult displayArrow( hkUint64 id, hkVector4Parameter from, hkVector4Parameter dir, hkColor::Argb color, int tag = 0 ) { return display2PointsEx( id, from, dir, color, hk2PointDisplayStyle::ARROW, 0, tag ); }
    HK_INLINE hkResult displayStar( hkUint64 id, hkVector4Parameter position, hkColor::Argb color, hkReal scale, int tag = 0 ) { return displayPointEx( id, position, color, hk1PointDisplayStyle::STAR, scale, tag ); }
    hkResult displayTransformedStar( hkUint64 id, const hkTransform& transform, hkVector4Parameter position, hkReal scale, hkColor::Argb color, int tag = 0 );
    hkResult displayTransformedStar( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter position, hkReal scale, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayPlane( hkUint64 id, hkVector4Parameter plane, hkVector4Parameter offset, hkReal scale, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayModelSpacePose( hkUint64 id, int numTransforms, const hkInt16* parentIndices, const hkQsTransform* modelSpacePose, const hkQsTransform& worldFromModel, hkColor::Argb color, int tag = 0 );
    hkResult displayLocalSpacePose( hkUint64 id, int numTransforms, const hkInt16* parentIndices, const hkQsTransform* localSpacePose, const hkQsTransform& worldFromModel, hkColor::Argb color, int tag = 0 );
    hkResult displayRay( hkUint64 id, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag = 0 );
    hkResult displayTransformedRay( hkUint64 id, const hkTransform& transform, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag = 0 );
    hkResult displayTransformedRay( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayAabb( hkUint64 id, const hkAabb& aabb, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayTransformedAabb( hkUint64 id, const hkTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag = 0 );
    hkResult displayTransformedAabb( hkUint64 id, const hkQsTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag = 0 );
    virtual hkResult displayFrustum( hkUint64 id, const hkFrustum& frustum, hkColor::Argb color, int tag = 0 );
    hkResult displaySphere( hkUint64 id, hkVector4Parameter center, hkReal radius, hkColor::Argb color, int tag = 0 );
    HK_INLINE hkResult displayLine( hkUint64 id, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag = 0 ) { return display2Points( id, start, end, color, tag ); }
    virtual hkResult displayLines( hkUint64 id, const hkStridedVertices &starts, const hkStridedVertices &ends, int numLines, hkColor::Argb color, int tag = 0 );
    HK_INLINE hkResult displayTriangle( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 ) { return display3Points( id, a, b, c, color, tag ); }
    //void displayGeometry( hkUint64 id, const hkGeometry& geometry, const hkTransform& transform, hkColor::Argb color, hkGeometryDisplayMode mode, int tag = 0 );
    virtual hkResult displaySpline( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 );

    hkResult addGeometry( hkUint64 id, hkDisplayGeometry* geometry, const hkTransform& transform, int tag = 0 );
    hkResult addGeometry( hkUint64 id, hkGeometry* geometry, const hkTransform& transform, int tag = 0 );
    hkResult addGeometryEx(
        hkUint64 id,
        hkDisplayGeometry* geometry,
        const hkTransform& transform,
        const hkColor::Argb color,
        hkDisplayGeometryFlags flags = hkDisplayGeometryFlags::DEFAULT,
        int tag = 0 );
    hkResult addGeometryEx(
        hkUint64 id,
        hkGeometry* geometry,
        const hkTransform& transform,
        const hkColor::Argb color,
        hkDisplayGeometryFlags flags = hkDisplayGeometryFlags::DEFAULT,
        int tag = 0 );
    hkResult displayGeometry( hkUint64 id, hkDisplayGeometry* geometry, const hkTransform& transform, hkColor::Argb color, int tag = 0 );
    hkResult displayGeometry( hkUint64 id, hkGeometry* geometry, const hkTransform& transform, hkColor::Argb color, int tag = 0 );

    /// Batch update functions for potential speed increases from underlying implementations
    virtual void lockForUpdate() {}
    virtual void unlockForUpdate() {}
};

/// Utils for use with the debug display handler class
struct HK_EXPORT_COMMON hkDebugDisplayHandlerUtils
{
    /// Call display2PointsFunc so as to draw a star
    /// with the provided properties.
    template<typename FUNC_2_POINT>
    static void HK_CALL displayStarUsing2PointBatchesFunc(
        FUNC_2_POINT display2PointsFunc,
        hkUint64 id,
        hkVector4Parameter position,
        hkColor::Argb color,
        hkReal scale,
        int tag = 0 )
    {
        for ( int k = 0; k < 3; k++ )
        {
            hkVector4 star, pt1, pt2;
            star.setZero();

            star( k ) = scale;
            pt1.setAdd( position, star );
            pt2.setSub( position, star );
            display2PointsFunc( id, pt1, pt2, color, tag );
        }
    }

    /// Call handler's 2 point display function so as to draw a star
    /// with the provided properties.
    static hkResult HK_CALL displayStarUsing2PointBatches(
        hkDebugDisplayHandler& handler,
        hkUint64 id,
        hkVector4Parameter position,
        hkColor::Argb color,
        hkReal scale,
        int tag = 0 );

    /// Call display2PointsFunc so as to draw an arrow
    /// with the provided properties.
    template<typename FUNC_2_POINT>
    static void HK_CALL displayArrowUsing2PointBatchesFunc(
        FUNC_2_POINT display2PointsFunc,
        hkUint64 id,
        hkVector4Parameter from,
        hkVector4Parameter dir,
        hkColor::Argb color,
        int tag = 0 )
    {
        const hkSimdReal length = dir.length<3>();
        if ( length < hkSimdReal_Eps )
        {
            return;
        }

        hkVector4 to; to.setAdd( from, dir );
        hkVector4 ort; hkVector4Util::calculatePerpendicularVector( dir, ort );
        ort.normalize<3>();
        hkVector4 ort2; ort2.setCross( dir, ort );
        ort.mul( length );

        const hkReal maxArrowHeadRadius = 0.05f;
        const hkReal arrowHeadPercentage = hkMath::min2( 0.15f, maxArrowHeadRadius / length.getReal() );

        hkSimdReal c0; c0.setFromFloat( 1.0f - arrowHeadPercentage );
        hkSimdReal c; c.setFromFloat( arrowHeadPercentage * 0.5f );

        hkVector4 p; p.setInterpolate( from, to, c0 );
        hkVector4 p0; p0.setAddMul( p, ort, c );
        hkVector4 p1; p1.setAddMul( p, ort, -c );
        hkVector4 p2; p2.setAddMul( p, ort2, c );
        hkVector4 p3; p3.setAddMul( p, ort2, -c );

        display2PointsFunc( id, from, to, color, tag );
        display2PointsFunc( id, to, p0, color, tag );
        display2PointsFunc( id, to, p1, color, tag );
        display2PointsFunc( id, to, p2, color, tag );
        display2PointsFunc( id, to, p3, color, tag );
    }

    /// Call handler's 2 point display function so as to draw an arrow
    /// with the provided properties.
    static hkResult HK_CALL displayArrowUsing2PointBatches(
        hkDebugDisplayHandler& handler,
        hkUint64 id,
        hkVector4Parameter from,
        hkVector4Parameter dir,
        hkColor::Argb color,
        int tag = 0 );
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
