// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Types/Geometry/Frustum/hkFrustum.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Common/Visualize/Shape/hkDisplaySphere.h>

#define DEBUG_LOG_IDENTIFIER "common.visualize.displayhandler"
#include <Common/Base/System/Log/hkLog.hxx>

hkResult hkDeprecatedDebugDisplayHandlerInterface::skinGeometry(
    const hkUint64* ids,
    int numIds,
    const hkQsTransform* poseModel,
    int numPoseModel,
    const hkQsTransform& worldFromModel )
{
    hkLocalBuffer<hkMatrix4> poseWorldAsMatrices( numPoseModel );
    for ( hkInt32 i = 0; i < numPoseModel; ++i )
    {
        poseWorldAsMatrices[i].set( poseModel[i] );
    }
    hkMatrix4 worldFromModelAsMatrix;
    worldFromModelAsMatrix.set( worldFromModel );
    return skinGeometry( ids, numIds, poseWorldAsMatrices.begin(), numPoseModel, worldFromModelAsMatrix );
}

hkResult hkDebugDisplayHandler::updateGeometryTransformEx( hkUint64 id, const hkQsTransform& transform )
{
    hkMatrix4 transformAsMatrix;
    transformAsMatrix.set( transform );
    return updateGeometryTransformEx( id, transformAsMatrix );
}

hkResult hkDebugDisplayHandler::displayTransformedLine( hkUint64 id, const hkTransform& transform, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag )
{
    hkQsTransform t;
    t.setFromTransform( transform );
    return displayTransformedLine( id, t, start, end, color, tag );
}

hkResult hkDebugDisplayHandler::displayTransformedLine( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag )
{
    hkVector4 startWorld;
    hkVector4 endWorld;
    startWorld.setTransformedPos( transform, start );
    endWorld.setTransformedPos( transform, end );
    return displayLine( id, startWorld, endWorld, color, tag );
}

hkResult hkDebugDisplayHandler::displayLitTriangle( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag )
{
    hkVector4   ab; ab.setSub( b, a );
    hkVector4   ac; ac.setSub( c, a );
    hkVector4   n; n.setCross( ab, ac );
    if ( n.normalizeIfNotZero<3>() )
    {
        hkSimdReal zero9; zero9.setFromFloat( 0.9f );
        hkVector4   light0; light0.set( 0.5f, 2, 1 ); light0.normalize<3>(); light0.mul( zero9 );
        hkVector4   light1; light1.set( -3, -2, -1 ); light1.normalize<3>(); light1.mul( zero9 );
        hkVector4   colorBase; colorBase.set( hkColor::getRedAsFloat( color ), hkColor::getGreenAsFloat( color ), hkColor::getBlueAsFloat( color ) );
        hkVector4   color0; color0.set( 0.75f, 0.75f, 0.75f, 0 ); color0.mul( colorBase );
        hkVector4   color1; color1.set( 0, 0.05f, 0.08f, 0 ); color1.mul( colorBase );
        hkVector4   one = hkVector4::getConstant<HK_QUADREAL_1>();
        hkSimdReal  half = hkSimdReal::getConstant<HK_QUADREAL_INV_2>();
        hkVector4   finalColor;
        finalColor.setMul( light0.dot<3>( n )*half + half, color0 );
        finalColor.addMul( light1.dot<3>( n )*half + half, color1 );
        hkSimdReal rps; rps.setFromFloat( 0.05f );
        finalColor.addMul( rps, colorBase );
        finalColor.setMax( finalColor, hkVector4::getZero() );
        finalColor.setMin( finalColor, one );
        color = hkColor::rgbFromFloats( finalColor( 0 ), finalColor( 1 ), finalColor( 2 ), hkColor::getAlphaAsFloat( color ) );
        return displayTriangle( id, a, b, c, color, tag );
    }
    return HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayFrame( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb* color, int tag )
{
    hkQsTransform t;
    t.setFromTransform( transform );
    return displayFrame( id, t, size, color, tag );
}

hkResult hkDebugDisplayHandler::displayFrame( hkUint64 id, const hkQsTransform& transform, hkReal size, const hkColor::Argb* color, int tag )
{
    hkVector4 ZERO;
    hkVector4 X;
    hkVector4 Y;
    hkVector4 Z;

    hkVector4 vec; vec.setZero();
    ZERO.setTransformedPos( transform, vec );
    vec.set( size, 0, 0, 0 );
    X.setTransformedPos( transform, vec );
    vec.set( 0, size, 0, 0 );
    Y.setTransformedPos( transform, vec );
    vec.set( 0, 0, size, 0 );
    Z.setTransformedPos( transform, vec );

    hkVector4 dirX; dirX.setSub( X, ZERO );
    hkVector4 dirY; dirY.setSub( Y, ZERO );
    hkVector4 dirZ; dirZ.setSub( Z, ZERO );

    bool succeeded = true;
    succeeded &= ( displayArrow( id, ZERO, dirX, color ? *color : hkColor::RED, tag ).isSuccess() );
    succeeded &= ( displayArrow( id, ZERO, dirY, color ? *color : hkColor::GREEN, tag ).isSuccess() );
    succeeded &= ( displayArrow( id, ZERO, dirZ, color ? *color : hkColor::BLUE, tag ).isSuccess() );
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayTetra( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb& color, int tag )
{
    hkVector4 ZERO;
    hkVector4 X;
    hkVector4 Y;
    hkVector4 Z;

    hkVector4 vec; vec.setZero();
    ZERO.setTransformedPos( transform, vec );
    vec.set( size, 0, 0, 0 );
    X.setTransformedPos( transform, vec );
    vec.set( 0, size, 0, 0 );
    Y.setTransformedPos( transform, vec );
    vec.set( 0, 0, size, 0 );
    Z.setTransformedPos( transform, vec );

    bool succeeded = true;
    succeeded &= ( displayLine( id, ZERO, X, color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, ZERO, Y, color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, ZERO, Z, color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, X, Y, color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, X, Z, color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, Y, Z, color, tag ).isSuccess() );
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayTransformedStar( hkUint64 id, const hkTransform& transform, hkVector4Parameter position, hkReal scale, hkColor::Argb color, int tag )
{
    hkVector4 positionWorld;
    positionWorld.setTransformedPos( transform, position );
    return displayStar( id, positionWorld, color, scale, tag );
}

hkResult hkDebugDisplayHandler::displayTransformedStar( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter position, hkReal scale, hkColor::Argb color, int tag )
{
    hkVector4 positionWorld;
    positionWorld.setTransformedPos( transform, position );
    return displayStar( id, positionWorld, color, scale, tag );
}

hkResult hkDebugDisplayHandler::displayPlane(
    hkUint64 id,
    hkVector4Parameter plane,
    hkVector4Parameter offset,
    hkReal scale,
    hkColor::Argb color,
    int tag )
{
    hkVector4 pos;

    pos.setAddMul( offset, plane, -plane.getComponent<3>() );
    hkVector4Parameter normal = plane;

    int major = normal.getIndexOfMaxAbsComponent<3>();
    hkVector4 binorm; binorm.setZero();
    binorm( ( major + 1 ) % 3 ) = 1;

    binorm.setCross( normal, binorm );
    binorm.normalize<3, HK_ACC_12_BIT, HK_SQRT_SET_ZERO>();
    hkSimdReal s; s.setFromFloat( scale );
    binorm.mul( s );

    hkVector4 tangent;
    tangent.setCross( binorm, normal );
    tangent.normalize<3, HK_ACC_12_BIT, HK_SQRT_SET_ZERO>();
    tangent.mul( s );

    //Draw the plane
    bool succeeded = true;
    for ( int e = 0; e < 4; e++ )
    {
        hkVector4 pt1 = pos;
        ( ( ( e + 0 ) % 4 ) / 2 ) ? pt1.sub( tangent ) : pt1.add( tangent );
        ( ( ( e + 1 ) % 4 ) / 2 ) ? pt1.sub( binorm ) : pt1.add( binorm );

        hkVector4 pt2 = pos;
        ( ( ( e + 1 ) % 4 ) / 2 ) ? pt2.sub( tangent ) : pt2.add( tangent );
        ( ( ( e + 2 ) % 4 ) / 2 ) ? pt2.sub( binorm ) : pt2.add( binorm );

        succeeded &= ( displayLine( id, pt1, pt2, color, tag ).isSuccess() );
    }

    //Draw the normal
    hkVector4 scaledNormal; scaledNormal.setMul( s, normal );
    succeeded &= ( displayArrow( id, pos, scaledNormal, color, tag ).isSuccess() );
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayModelSpacePose(
    hkUint64 id,
    int numTransforms,
    const hkInt16* parentIndices,
    const hkQsTransform* modelSpacePose,
    const hkQsTransform& worldFromModel,
    hkColor::Argb color,
    int tag )
{
    const hkInt16 numBones = static_cast< hkInt16 > ( numTransforms );

    bool succeeded = true;
    for ( hkInt16 i = 0; i < numBones; i++ )
    {
        hkVector4 p1;

        p1.setTransformedPos( worldFromModel, modelSpacePose[i].getTranslation() );

        hkReal boneLen = 1.0f;

        const hkInt16 parent = parentIndices[i];

        // Display connections between child and parent
        if ( parent == -1 )
        {
            hkVector4 p2;
            p2.set( 0, 0, 0 );

            p2.setTransformedPos( worldFromModel, p2 );
            succeeded &= ( displayLine( id, p1, p2, color, tag ).isSuccess() );
        }
        else
        {
            hkVector4 p2;
            p2.setTransformedPos( worldFromModel, modelSpacePose[parent].getTranslation() );
            succeeded &= ( displayLine( id, p1, p2, color, tag ).isSuccess() );

            hkVector4 bone; bone.setSub( p1, p2 );
            boneLen = bone.length<3>().getReal();
            boneLen = ( boneLen > 10.0f ) ? 10.0f : boneLen;
            boneLen = ( boneLen < 0.1f ) ? 0.1f : boneLen;
        }


        const hkVector4 worldPosition = p1;
        hkQuaternion worldRotation; worldRotation.setMul(
            worldFromModel.getRotation(),
            modelSpacePose[i].getRotation() );

        // Display local axis
        {
            hkVector4 boneLocalFrame;
            const hkReal boneAxisSize = 0.25f;

            hkVector4 p2;
            p1 = worldPosition;
            boneLocalFrame.set( boneAxisSize * boneLen, 0, 0, 0 );
            p2.setRotatedDir( worldRotation, boneLocalFrame );
            p2.add( p1 );
            succeeded &= ( displayLine( id, p1, p2, 0x7fff0000, tag ).isSuccess() );
            p1 = worldPosition;
            boneLocalFrame.set( 0, boneAxisSize * boneLen, 0 );
            p2.setRotatedDir( worldRotation, boneLocalFrame );
            p2.add( p1 );
            succeeded &= ( displayLine( id, p1, p2, 0x7f00ff00, tag ).isSuccess() );
            p1 = worldPosition;
            boneLocalFrame.set( 0, 0, boneAxisSize * boneLen );
            p2.setRotatedDir( worldRotation, boneLocalFrame );
            p2.add( p1 );
            succeeded &= ( displayLine( id, p1, p2, 0x7f0000ff, tag ).isSuccess() );
        }
    }

    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayLocalSpacePose(
    hkUint64 id,
    int numTransforms,
    const hkInt16* parentIndices,
    const hkQsTransform* localSpacePose,
    const hkQsTransform& worldFromModel,
    hkColor::Argb color,
    int tag )
{
    // Transform to model space (from hkaSkeletonUtils::transformLocalPoseToModelPose)
    hkLocalBuffer<hkQsTransform> modelSpacePose( numTransforms );
    {
        for ( int i = 0; i < numTransforms; i++ )
        {
            const int parentId = parentIndices[i];
            HK_ASSERT_NO_MSG( 0x675efb2b, ( parentIndices[i] == -1 ) || ( parentIndices[i] < i ) );
            const hkQsTransform& worldFromParent = ( -1 != parentId ) ? modelSpacePose[parentId] : hkQsTransform::getIdentity();
            modelSpacePose[i].setMul( worldFromParent, localSpacePose[i] );
        }
    }

    return displayModelSpacePose(
        id,
        numTransforms,
        parentIndices,
        modelSpacePose.begin(),
        worldFromModel,
        color,
        tag );
}

hkResult hkDebugDisplayHandler::displayRay( hkUint64 id, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag )
{
    hkVector4 end;
    end.setAdd( start, dir );
    return displayLine( id, start, end, color, tag );
}

hkResult hkDebugDisplayHandler::displayTransformedRay( hkUint64 id, const hkTransform& transform, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag )
{
    hkQsTransform t;
    t.setFromTransform( transform );
    return displayTransformedRay( id, t, start, dir, color, tag );
}

hkResult hkDebugDisplayHandler::displayTransformedRay( hkUint64 id, const hkQsTransform& transform, hkVector4Parameter start, hkVector4Parameter dir, hkColor::Argb color, int tag )
{
    hkVector4 startWorld;
    hkVector4 directionWorld;
    startWorld.setTransformedPos( transform, start );
    directionWorld.setRotatedDir( transform.getRotation(), dir );
    return displayRay( id, startWorld, directionWorld, color, tag );
}

hkResult hkDebugDisplayHandler::displayAabb( hkUint64 id, const hkAabb& aabb, hkColor::Argb color, int tag )
{
    hkVector4 lines[24];

    lines[0].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );
    lines[1].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );

    lines[2].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );
    lines[3].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );

    lines[4].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );
    lines[5].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );

    lines[6].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );
    lines[7].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );

    lines[8].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );
    lines[9].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );

    lines[10].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );
    lines[11].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );

    lines[12].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );
    lines[13].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );

    lines[14].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );
    lines[15].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );

    lines[16].set( aabb.m_max( 0 ), aabb.m_max( 1 ), aabb.m_min( 2 ) );
    lines[17].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );

    lines[18].set( aabb.m_min( 0 ), aabb.m_max( 1 ), aabb.m_max( 2 ) );
    lines[19].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );

    lines[20].set( aabb.m_min( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );
    lines[21].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );

    lines[22].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_max( 2 ) );
    lines[23].set( aabb.m_max( 0 ), aabb.m_min( 1 ), aabb.m_min( 2 ) );

    
    bool succeeded = true;
    for ( int i = 0; i < 24; i += 2 )
    {
        succeeded &= ( displayLine( id, lines[i], lines[i + 1], color, tag ).isSuccess() );
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayTransformedAabb( hkUint64 id, const hkTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag )
{
    hkVector4   vertices[8];
    for ( int i = 0; i < 8; ++i )
    {
        hkVector4   localVertex;
        localVertex.set( i & 1 ? aabb.m_max.getComponent<0>() : aabb.m_min.getComponent<0>(),
            i & 2 ? aabb.m_max.getComponent<1>() : aabb.m_min.getComponent<1>(),
            i & 4 ? aabb.m_max.getComponent<2>() : aabb.m_min.getComponent<2>(),
            hkSimdReal::getConstant<HK_QUADREAL_0>() );
        vertices[i]._setTransformedPos( transform, localVertex );
    }

    
    bool succeeded = true;
    succeeded &= ( displayLine( id, vertices[0], vertices[1], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[1], vertices[3], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[3], vertices[2], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[2], vertices[0], color, tag ).isSuccess() );

    succeeded &= ( displayLine( id, vertices[4], vertices[5], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[5], vertices[7], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[7], vertices[6], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[6], vertices[4], color, tag ).isSuccess() );

    succeeded &= ( displayLine( id, vertices[0], vertices[4], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[1], vertices[5], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[2], vertices[6], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, vertices[3], vertices[7], color, tag ).isSuccess() );
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displayTransformedAabb( hkUint64 id, const hkQsTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag )
{
    hkTransform t;
    transform.copyToTransformNoScale( t );
    return displayTransformedAabb( id, t, aabb, color, tag );
}

hkResult hkDebugDisplayHandler::displayFrustum( hkUint64 id, const hkFrustum& frustum, hkColor::Argb color, int tag )
{
    hkVector4 points[8];

    frustum.calculatePlaneIntersectionPoints( points );

    

    // caps
    bool succeeded = true;
    succeeded &= ( displayLine( id, points[0], points[1], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[1], points[2], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[2], points[3], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[3], points[0], color, tag ).isSuccess() );

    succeeded &= ( displayLine( id, points[4], points[5], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[5], points[6], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[6], points[7], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[7], points[4], color, tag ).isSuccess() );

    succeeded &= ( displayLine( id, points[0], points[4], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[1], points[5], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[2], points[6], color, tag ).isSuccess() );
    succeeded &= ( displayLine( id, points[3], points[7], color, tag ).isSuccess() );
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::displaySphere( hkUint64 id, hkVector4Parameter center, hkReal radius, hkColor::Argb color, int tag )
{
    hkSphere s;
    s.setPositionAndRadius( center, hkSimdReal::fromFloat( radius ) );

    hkDisplaySphere sphere( s, 2 );
    hkInplaceArray<hkDisplayGeometry*, 1> spheres;
    spheres.pushBackUnchecked( &sphere );

    return displayGeometry( id, spheres, hkTransform::getIdentity(), color, tag );
}

hkResult hkDebugDisplayHandler::displayLines(
    hkUint64 id,
    const hkStridedVertices &starts,
    const hkStridedVertices &ends,
    int numLines,
    hkColor::Argb color,
    int tag )
{
    HK_ASSERT_NO_MSG( 0x441c4fd2, numLines >= 0 && starts.getSize() >= numLines && ends.getSize() >= numLines );

    hkVector4 start;
    hkVector4 end;
    bool succeeded = true;
    for ( int i = 0; i < numLines; ++i )
    {
        starts.getVertex( i, start );
        ends.getVertex( i, end );
        succeeded &= ( displayLine( id, start, end, color, tag ).isSuccess() );
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandler::addGeometry( hkUint64 id, hkDisplayGeometry* geometry, const hkTransform& transform, int tag )
{
    hkInplaceArray<hkDisplayGeometry*, 1> arrayOfGeometries;
    arrayOfGeometries.pushBackUnchecked( geometry );
    return addGeometry( id, arrayOfGeometries, transform, tag );
}

hkResult hkDebugDisplayHandler::addGeometry( hkUint64 id, hkGeometry* geometry, const hkTransform& transform, int tag )
{
    hkDisplayGeometry dispGeometry( geometry );
    dispGeometry.setTransform( transform );
    return addGeometry( id, &dispGeometry, transform, tag );
}

hkResult hkDebugDisplayHandler::addGeometryEx(
    hkUint64 id,
    hkDisplayGeometry* geometry,
    const hkTransform& transform,
    const hkColor::Argb color,
    hkDisplayGeometryFlags flags,
    int tag )
{
    hkInplaceArray<hkDisplayGeometry*, 1> arrayOfGeometries;
    arrayOfGeometries.pushBackUnchecked( geometry );
    return addGeometryEx( id, arrayOfGeometries, transform, color, flags, tag );
}

hkResult hkDebugDisplayHandler::addGeometryEx(
    hkUint64 id,
    hkGeometry* geometry,
    const hkTransform& transform,
    const hkColor::Argb color,
    hkDisplayGeometryFlags flags,
    int tag )
{
    hkDisplayGeometry dispGeometry( geometry );
    dispGeometry.setTransform( transform );
    return addGeometryEx( id, &dispGeometry, transform, color, flags, tag );
}

hkResult hkDebugDisplayHandler::displayGeometry( hkUint64 id, hkDisplayGeometry* geometry, const hkTransform& transform, hkColor::Argb color, int tag )
{
    hkInplaceArray<hkDisplayGeometry*, 1> arrayOfGeometries;
    arrayOfGeometries.pushBackUnchecked( geometry );
    return displayGeometry( id, arrayOfGeometries, transform, color, tag );
}

hkResult hkDebugDisplayHandler::displayGeometry( hkUint64 id, hkGeometry* geometry, const hkTransform& transform, hkColor::Argb color, int tag )
{
    hkDisplayGeometry dispGeometry( geometry );
    hkResult res = displayGeometry( id, &dispGeometry, transform, color, tag );
    dispGeometry.releaseGeometry();     
    return res;
}

hkResult hkDebugDisplayHandler::displaySpline( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag )
{
    if ( ( b - a ).cross( c - a ).lengthSquared<3>().getReal() < hkReal(1.e-6) )
    {
        display2Points( id, a, c, color, tag );
    }
    else
    {
        const hkVector4 t0 = ( a + b ) * hkSimdReal_Inv2;
        const hkVector4 t1 = ( b + c ) * hkSimdReal_Inv2;
        const hkVector4 t2 = ( t0 + t1 )*hkSimdReal_Inv2;
        displaySpline( id, a, t0, t2, color, tag );
        displaySpline( id, t2, t1, c, color, tag );
    }
    return HK_SUCCESS;
}

hkResult hkDebugDisplayHandlerUtils::displayStarUsing2PointBatches( hkDebugDisplayHandler& handler, hkUint64 id, hkVector4Parameter position, hkColor::Argb color, hkReal scale, int tag )
{
    bool success = true;
    auto func = [&handler, &success]( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag )
    {
        success &= ( handler.display2Points( id, a, b, color, tag ).isSuccess() );
    };
    displayStarUsing2PointBatchesFunc( func, id, position, color, scale, tag );
    return success ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkDebugDisplayHandlerUtils::displayArrowUsing2PointBatches( hkDebugDisplayHandler& handler, hkUint64 id, hkVector4Parameter from, hkVector4Parameter dir, hkColor::Argb color, int tag )
{
    bool success = true;
    auto func = [&handler, &success]( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag )
    {
        success &= ( handler.display2Points( id, a, b, color, tag ).isSuccess() );
    };
    displayArrowUsing2PointBatchesFunc( func, id, from, dir, color, tag );
    return success ? HK_SUCCESS : HK_FAILURE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
