// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#ifndef HK_VISUALIZE_DEBUG_DISPLAY_H
#define HK_VISUALIZE_DEBUG_DISPLAY_H

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Visualize/Process/hkDebugDisplayProcess.h>



struct debugRenderNow;

/// Callbacks used with debugRenderNow
struct HK_EXPORT_COMMON debugRenderNowCallbacks
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,debugRenderNowCallbacks);
    debugRenderNowCallbacks()
        :       m_tweakee( HK_NULL)
                ,m_tweakeeClass( HK_NULL)
                , m_hideMembers(HK_NULL)
                , m_doRenderCallback(false)
    {

    }

    virtual ~debugRenderNowCallbacks() {}

    void setTweakee(_In_opt_ void* tweakee, _In_opt_ const hkReflect::Type* tweakeeClass)
    {
        m_tweakee = tweakee;
        m_tweakeeClass = tweakeeClass;
    }

    virtual void handleKey(unsigned char key) {}

    virtual void render(_Inout_ debugRenderNow* r) {}

    typedef hkBool (HK_CALL *HideMemberFunc)(_In_ const hkReflect::Type* m);

    void* m_tweakee;
    const hkReflect::Type* m_tweakeeClass;
    HideMemberFunc m_hideMembers;
    hkBool m_doRenderCallback;
};

/// Provides a framework for drawing immediate mode display information now.
/// This api is currently only supported in the Havok demo framework.
struct HK_EXPORT_COMMON debugRenderNow : public hkBaseObject
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,debugRenderNow);
    debugRenderNow()
        : m_title(HK_NULL)
        , m_clearDebug(true)
        , m_wait(true)
        , m_cameraSet(false)
    {
    }

        // The real work is done in the destructor after the options have been set.
    virtual ~debugRenderNow() { if (s_callbacks) s_callbacks->render(this); };

    debugRenderNow& setTitle(_In_z_ const char* title) { m_title = title; return *this; }

    debugRenderNow& setCamera( hkVector4Parameter to )
    {
        m_cameraSet = true;
        m_cameraTo = to;
        return *this;
    }

        // clear the debug display after rendering
    debugRenderNow& clearDebug(hkBool c) { m_clearDebug = c; return *this; }
    debugRenderNow& wait(hkBool w) { m_wait = w; return *this; }
    debugRenderNow& setCallbacks(_In_opt_ struct debugRenderNowCallbacks* c) { s_callbacks = c; return *this; }
    const char* m_title;
    hkBool m_clearDebug;
    hkBool m_wait;
    static struct HK_EXPORT_COMMON debugRenderNowCallbacks* s_callbacks;
    hkBool m_cameraSet;
    hkVector4 m_cameraTo;
};

/// Helper function for using the debugRenderNow framework.
extern void HK_CALL debugRenderImmediate(int wait, int clear);

class hkDisplayGeometry;
struct hkFrustum;

/// A helper singleton, which forwards debug display calls to any registered handlers.
/// This is the best place to make debug display calls to the Havok demos and vdb.
/// You may use the macros or access the singleton directly.
/// This is a thread safe singleton
class HK_EXPORT_COMMON hkDebugDisplay : public hkDebugDisplayHandler
{
    public:

        HK_DECLARE_CLASS(hkDebugDisplay, New);
        HK_DECLARE_CLASS_Singleton(hkDebugDisplay);

        hkDebugDisplay();
        ~hkDebugDisplay();

            /// Add a display viewer to the debug display. All
            /// debug points and lines will be sent to this display
            /// handler until it is removed.
        void addDebugDisplayHandler(_In_ hkDebugDisplayHandler* debugDisplay);

            /// Remove a display viewer from the debug display.
            /// No more debug points and lines will be sent to the
            /// display handler.
        void removeDebugDisplayHandler(_In_ hkDebugDisplayHandler* debugDisplay);

            /// Removes all display viewer from the debug display.
            /// No debug points and lines will be displayed after
            /// this function is called.
        void clear();

        //
        // hkDebugDisplayHandler interface
        //

        virtual hkResult setDisplayOptions( Options& options ) HK_OVERRIDE;
        virtual hkResult addGeometry(
            hkUint64 id,
            const hkArrayBase<hkDisplayGeometry*>& geometries,
            const hkTransform& transform,
            int tag = 0 ) HK_OVERRIDE;
        using hkDebugDisplayHandler::addGeometry;
        virtual hkResult addGeometryEx(
            hkUint64 id,
            const hkArrayBase<hkDisplayGeometry*>& geometries,
            const hkTransform& transform,
            const hkColor::Argb color,
            hkDisplayGeometryFlags flags = hkDisplayGeometryFlags::DEFAULT,
            int tag = 0 ) HK_OVERRIDE;
        using hkDebugDisplayHandler::addGeometryEx;
        virtual hkResult addGeometryInstance(
            hkUint64 id,
            hkUint64 originalGeomId,
            const hkTransform& transform,
            int tag = 0 ) HK_OVERRIDE;
        virtual hkResult setGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits ) HK_OVERRIDE;
        virtual hkResult clearGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits ) HK_OVERRIDE;
        virtual hkResult setGeometryColor( hkUint64 id, hkColor::Argb color ) HK_OVERRIDE;
        virtual hkResult setGeometryAlpha( hkUint64 id, hkReal alpha ) HK_OVERRIDE;
        virtual hkResult updateGeometryTransform( hkUint64 id, const hkTransform& transform ) HK_OVERRIDE;
        virtual hkResult updateGeometryTransformEx( hkUint64 id, const hkMatrix4& transform ) HK_OVERRIDE;
        using hkDebugDisplayHandler::updateGeometryTransformEx;
        virtual hkResult updateGeometryVerts( hkUint64 id, const hkArray<hkVector4>& newVerts ) HK_OVERRIDE;
        virtual hkResult removeGeometry( hkUint64 id ) HK_OVERRIDE;
        virtual hkResult updateCamera(
            const char* name,
            hkVector4Parameter from,
            hkVector4Parameter to,
            hkVector4Parameter up,
            hkReal nearPlane,
            hkReal farPlane,
            hkReal fov ) HK_OVERRIDE;
        virtual hkResult removeCamera( const char* name ) HK_OVERRIDE;
        virtual hkResult displayPoint( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayPointEx( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, hk1PointDisplayStyle::Enum style = hk1PointDisplayStyle::DOT, hkReal scale = 0, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult display2Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult display2PointsEx( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, hk2PointDisplayStyle::Enum style = hk2PointDisplayStyle::LINE_SEGMENT, hkReal thickness = 0, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult display3Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        
        virtual hkResult displayText( hkUint64 id, const char* text, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult display3dText( hkUint64 id, const char* text, hkVector4Parameter pos, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayPoint2d( hkUint64 id, hkVector4Parameter position, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayLine2d( hkUint64 id, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayTriangle2d( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayText2d( hkUint64 id, const char* text, hkVector4Parameter pos, hkReal sizeScale, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayGeometry( hkUint64 id, const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        using hkDebugDisplayHandler::displayGeometry;
        virtual hkResult flushDisplay( bool clearDisplayGeometry = true ) HK_OVERRIDE;
        virtual hkResult endFrame() HK_OVERRIDE;

        //
        // Display handler aux functions, overwritten for perf.
        //

        using hkDebugDisplayHandler::displayFrame;
        virtual hkResult displayFrame( hkUint64 id, const hkQsTransform& transform, hkReal size, const hkColor::Argb* color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayTetra( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb& color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayPlane( hkUint64 id, hkVector4Parameter plane, hkVector4Parameter offset, hkReal scale, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayModelSpacePose( hkUint64 id, int numTransforms, const hkInt16* parentIndices, const hkQsTransform* modelSpacePose, const hkQsTransform& worldFromModel, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayAabb( hkUint64 id, const hkAabb& aabb, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        using hkDebugDisplayHandler::displayTransformedAabb;
        virtual hkResult displayTransformedAabb( hkUint64 id, const hkTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayFrustum( hkUint64 id, const hkFrustum& frustum, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;
        virtual hkResult displayLines( hkUint64 id, const hkStridedVertices &starts, const hkStridedVertices &ends, int numLines, hkColor::Argb color, int tag = 0 ) HK_OVERRIDE;

        /// Get the from position of the last set camera (the name argument is ignored for now).
        hkVector4 getCameraFrom( const char* name );

        /// Get the forward vector of the last set camera (the name argument is ignored for now).
        hkVector4 getCameraFwd( const char* name );

        /// Get current debug display handlers
        const hkArray< hkRefPtr<hkDebugDisplayHandler> >& getDebugDisplayHandlers() { return m_debugDisplayHandlers;  };

    protected:

        hkArray< hkRefPtr<hkDebugDisplayHandler> > m_debugDisplayHandlers;
        mutable class hkCriticalSection* m_arrayLock;

        hkVector4 m_cameraFrom;
        hkVector4 m_cameraFwd;
};

#if defined(HK_DISABLE_DEBUG_DISPLAY)
#   define HK_ON_DEBUG_DISPLAY(code)
#else
#   define HK_ON_DEBUG_DISPLAY(code) code
#endif

// Macros for displaying debug points and lines.
// These can be compiled out if HK_DISABLE_DEBUG_DISPLAY is defined.
// NOTE: Please refer to the hkDebugDisplayHandler class definition for
// information about these macros and their parameters.  These calls are
// essentially short-hand versions of calls to the hkDisplayHandler singleton,
// please read its class description.




#define DEBUG_DISPLAY_PROCESS_TAG HK_ON_DEBUG_DISPLAY(hkDebugDisplayProcess::getProcessTagStatic())

#define HK_ADD_DEBUG_DISPLAY_HANDLER(handler) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().addDebugDisplayHandler(handler))
#define HK_REMOVE_DEBUG_DISPLAY_HANDLER(handler) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().removeDebugDisplayHandler(handler))
#define HK_DISPLAY_END_FRAME() HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().endFrame())

#define HK_SET_OBJECT_COLOR(id, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().setGeometryColor(id, color))
#define HK_ADD_GEOMETRY(geometries, transform, id) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().addGeometry(id, geometries, transform, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_UPDATE_GEOMETRY(transform, id) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().updateGeometryTransform(id, transform))
#define HK_GET_CAMERA_FROM(name) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().getCameraFrom(name))
#define HK_GET_CAMERA_FWD(name) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().getCameraFwd(name))
#define HK_REMOVE_GEOMETRY(id) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().removeGeometry(id))
#define HK_DISPLAY_GEOMETRY(geometry, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayGeometry(0, geometry, hkTransform::getIdentity(), color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_GEOMETRY_WITH_TRANSFORM(geometry, transform, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayGeometry(0, geometry, transform, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_UPDATE_CAMERA(from, to, up, nearPlane, farPlane, fov, name) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().updateCamera(name, from, to, up, nearPlane, farPlane, fov))
#define HK_DISPLAY_POINT(position, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayPoint(0, position, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_THICK_POINT(position, radius, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayPointEx(0, position, color, hk1PointDisplayStyle::DOT, radius, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_LINE(start, end, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayLine(0, start, end, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_LINES(starts, ends, numLines, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayLines(0, starts, ends, numLines, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_LINE_MODEL(worldFromModel, start, end, color ) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTransformedLine(0, worldFromModel, start, end, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_SPHERE(center, radius, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displaySphere(0, center, radius, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_CAPSULE(center, radius, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayCapsule(0, center, radius, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_RAY(start, dir, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayRay(0, start, dir, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_RAY_MODEL(worldFromModel, start, dir, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTransformedRay(0, worldFromModel, start, dir, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_ARROW(start, direction, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayArrow(0, start, direction, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_STAR(pos, scale, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayStar(0, pos, color, scale, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_STAR_MODEL(worldFromModel, pos, scale, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTransformedStar(0, worldFromModel, pos, scale, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_PLANE(plane, offset, scale, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayPlane(0, plane, offset, scale, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_MODEL_SPACE_POSE(numTransforms, parentIndices, modelSpacePose, worldFromModel, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayModelSpacePose(0, numTransforms, parentIndices, modelSpacePose, worldFromModel, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_LOCAL_SPACE_POSE(numTransforms, parentIndices, localSpacePose, worldFromModel, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayLocalSpacePose(0, numTransforms, parentIndices, localSpacePose, worldFromModel, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_TEXT(text, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayText(0, text, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_3D_TEXT(text, pos, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().display3dText(0, text, pos, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_FRAME(worldFromLocal, size) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayFrame(0, worldFromLocal, size, HK_NULL, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_FRAME_WITH_COLOR(worldFromLocal, size, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayFrame(0, worldFromLocal, size, &(color), DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_TETRA(worldFromLocal, size, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTetra(0, worldFromLocal, size, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_BOUNDING_BOX(aabb, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayAabb(0, aabb, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_TRANSFORMED_BOUNDING_BOX(transform, aabb, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTransformedAabb(0, transform, aabb, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_TRIANGLE(a, b, c, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayTriangle(0, a,b,c, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_LIT_TRIANGLE(a, b, c, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayLitTriangle(0, a,b,c, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_FRUSTUM(frustum, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displayFrustum(0, frustum, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_SPLINE(a, b, c, color) HK_ON_DEBUG_DISPLAY(hkDebugDisplay::getInstance().displaySpline(0, a, b, c, color, DEBUG_DISPLAY_PROCESS_TAG))
#define HK_DISPLAY_INTERPOLATING_SPLINE(a, b, c, color) HK_DISPLAY_SPLINE(a, a * -hkSimdReal_Inv2 + b * hkSimdReal_2 - c * hkSimdReal_Inv2, c, color)

#endif // HK_VISUALIZE_DEBUG_DISPLAY_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
