// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Types/Geometry/Plane/hkPlane.h>
#include <Common/Base/Types/Geometry/Frustum/hkFrustum.h>
#include <Common/Base/Object/hkSingleton.h>

#define CALL_HANDLERS_AND_RETURN( FUNC, ... ) \
HK_MULTILINE_MACRO_BEGIN \
    bool succeeded = true; \
    m_arrayLock->enter(); \
    for ( int i = 0; i < m_debugDisplayHandlers.getSize(); i++ ) \
    { \
        succeeded &= ( m_debugDisplayHandlers[i]->FUNC( __VA_ARGS__ ).isSuccess() ); \
    } \
    m_arrayLock->leave(); \
    return succeeded ? HK_SUCCESS : HK_FAILURE; \
HK_MULTILINE_MACRO_END

debugRenderNowCallbacks* debugRenderNow::s_callbacks = HK_NULL;

hkDebugDisplay::hkDebugDisplay()
{
    m_arrayLock = new hkCriticalSection( 1000 ); // usually no contention
}

hkDebugDisplay::~hkDebugDisplay()
{
    delete m_arrayLock;
}

void hkDebugDisplay::addDebugDisplayHandler( hkDebugDisplayHandler* debugDisplay )
{
    m_arrayLock->enter();
    m_debugDisplayHandlers.pushBack( debugDisplay );
    m_arrayLock->leave();
}

void hkDebugDisplay::removeDebugDisplayHandler( hkDebugDisplayHandler* debugDisplay )
{
    m_arrayLock->enter();
    int index = m_debugDisplayHandlers.indexOf( debugDisplay );
    if ( index >= 0 )
    {
        m_debugDisplayHandlers.removeAt( index );
    }
    m_arrayLock->leave();
}

void hkDebugDisplay::clear()
{
    m_arrayLock->enter();
    m_debugDisplayHandlers.clear();
    m_arrayLock->leave();
}

hkResult hkDebugDisplay::setDisplayOptions( Options& options )
{
    CALL_HANDLERS_AND_RETURN( setDisplayOptions, options );
}

hkResult hkDebugDisplay::addGeometry(
    hkUint64 id,
    const hkArrayBase<hkDisplayGeometry*>& geometries,
    const hkTransform& transform,
    int tag )
{
    CALL_HANDLERS_AND_RETURN( addGeometry, id, geometries, transform, tag );
}

hkResult hkDebugDisplay::addGeometryEx(
    hkUint64 id,
    const hkArrayBase<hkDisplayGeometry*>& geometries,
    const hkTransform& transform,
    const hkColor::Argb color,
    hkDisplayGeometryFlags flags,
    int tag )
{
    CALL_HANDLERS_AND_RETURN( addGeometryEx, id, geometries, transform, color, flags, tag );
}

hkResult hkDebugDisplay::addGeometryInstance(
    hkUint64 id,
    hkUint64 originalGeomId,
    const hkTransform& transform,
    int tag )
{
    CALL_HANDLERS_AND_RETURN( addGeometryInstance, id, originalGeomId, transform, tag );
}

hkResult hkDebugDisplay::setGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits )
{
    CALL_HANDLERS_AND_RETURN( setGeometryFlagBits, id, flagBits );
}

hkResult hkDebugDisplay::clearGeometryFlagBits( hkUint64 id, hkDisplayGeometryFlags flagBits )
{
    CALL_HANDLERS_AND_RETURN( clearGeometryFlagBits, id, flagBits );
}

hkResult hkDebugDisplay::setGeometryColor( hkUint64 id, hkColor::Argb color )
{
    CALL_HANDLERS_AND_RETURN( setGeometryColor, id, color );
}

hkResult hkDebugDisplay::setGeometryAlpha( hkUint64 id, hkReal alpha )
{
    CALL_HANDLERS_AND_RETURN( setGeometryAlpha, id, alpha );
}

hkResult hkDebugDisplay::updateGeometryTransform( hkUint64 id, const hkTransform& transform )
{
    CALL_HANDLERS_AND_RETURN( updateGeometryTransform, id, transform );
}

hkResult hkDebugDisplay::updateGeometryTransformEx( hkUint64 id, const hkMatrix4& transform )
{
    CALL_HANDLERS_AND_RETURN( updateGeometryTransformEx, id, transform );
}

hkResult hkDebugDisplay::updateGeometryVerts( hkUint64 id, const hkArray<hkVector4>& newVerts )
{
    CALL_HANDLERS_AND_RETURN( updateGeometryVerts, id, newVerts );
}

hkResult hkDebugDisplay::removeGeometry( hkUint64 id )
{
    CALL_HANDLERS_AND_RETURN( removeGeometry, id );
}

hkResult hkDebugDisplay::updateCamera(
    const char* name,
    hkVector4Parameter from,
    hkVector4Parameter to,
    hkVector4Parameter up,
    hkReal nearPlane,
    hkReal farPlane,
    hkReal fov )
{
    m_cameraFrom = from;
    m_cameraFwd.setSub( to, from );
    m_cameraFwd.normalize<3, HK_ACC_23_BIT, HK_SQRT_SET_ZERO>();
    CALL_HANDLERS_AND_RETURN( updateCamera, name, from, to, up, nearPlane, farPlane, fov );
}

hkVector4 hkDebugDisplay::getCameraFrom( const char* name )
{
    return m_cameraFrom;
}

hkVector4 hkDebugDisplay::getCameraFwd( const char* name )
{
    return m_cameraFwd;
}

hkResult hkDebugDisplay::removeCamera( const char* name )
{
    CALL_HANDLERS_AND_RETURN( removeCamera, name );
}

hkResult hkDebugDisplay::displayPoint( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayPoint, id, a, color, tag );
}

hkResult hkDebugDisplay::displayPointEx( hkUint64 id, hkVector4Parameter a, hkColor::Argb color, hk1PointDisplayStyle::Enum style, hkReal scale, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayPointEx, id, a, color, style, scale, tag );
}

hkResult hkDebugDisplay::display2Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( display2Points, id, a, b, color, tag );
}

hkResult hkDebugDisplay::display2PointsEx( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, hk2PointDisplayStyle::Enum style, hkReal thickness, int tag )
{
    CALL_HANDLERS_AND_RETURN( display2PointsEx, id, a, b, color, style, thickness, tag );
}

hkResult hkDebugDisplay::display3Points( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( display3Points, id, a, b, c, color, tag );
}

hkResult hkDebugDisplay::displayText( hkUint64 id, const char* text, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayText, id, text, color, tag );
}

hkResult hkDebugDisplay::display3dText( hkUint64 id, const char* text, hkVector4Parameter pos, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( display3dText, id, text, pos, color, tag );
}

hkResult hkDebugDisplay::displayPoint2d( hkUint64 id, hkVector4Parameter position, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayPoint2d, id, position, color, tag );
}

hkResult hkDebugDisplay::displayLine2d( hkUint64 id, hkVector4Parameter start, hkVector4Parameter end, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayLine2d, id, start, end, color, tag );
}

hkResult hkDebugDisplay::displayTriangle2d( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayTriangle2d, id, a, b, c, color, tag );
}

hkResult hkDebugDisplay::displayText2d( hkUint64 id, const char* text, hkVector4Parameter pos, hkReal sizeScale, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayText2d, id, text, pos, sizeScale, color, tag );
}

hkResult hkDebugDisplay::displayGeometry( hkUint64 id, const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayGeometry, id, geometries, transform, color, tag );
}

hkResult hkDebugDisplay::flushDisplay( bool clearDisplayGeometry )
{
    CALL_HANDLERS_AND_RETURN( flushDisplay, clearDisplayGeometry );
}

hkResult hkDebugDisplay::displayFrame( hkUint64 id, const hkQsTransform& transform, hkReal size, const hkColor::Argb* color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayFrame, id, transform, size, color, tag );
}

hkResult hkDebugDisplay::displayTetra( hkUint64 id, const hkTransform& transform, hkReal size, const hkColor::Argb& color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayTetra, id, transform, size, color, tag );
}

hkResult hkDebugDisplay::displayPlane( hkUint64 id, hkVector4Parameter plane, hkVector4Parameter offset, hkReal scale, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayPlane, id, plane, offset, scale, color, tag );
}

hkResult hkDebugDisplay::displayModelSpacePose( hkUint64 id, int numTransforms, const hkInt16* parentIndices, const hkQsTransform* modelSpacePose, const hkQsTransform& worldFromModel, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayModelSpacePose, id, numTransforms, parentIndices, modelSpacePose, worldFromModel, color, tag );
}

hkResult hkDebugDisplay::displayAabb( hkUint64 id, const hkAabb& aabb, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayAabb, id, aabb, color, tag );
}

hkResult hkDebugDisplay::displayTransformedAabb( hkUint64 id, const hkTransform& transform, const hkAabb& aabb, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayTransformedAabb, id, transform, aabb, color, tag );
}

hkResult hkDebugDisplay::displayFrustum( hkUint64 id, const hkFrustum& frustum, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayFrustum, id, frustum, color, tag );
}

hkResult hkDebugDisplay::displayLines( hkUint64 id, const hkStridedVertices &starts, const hkStridedVertices &ends, int numLines, hkColor::Argb color, int tag )
{
    CALL_HANDLERS_AND_RETURN( displayLines, id, starts, ends, numLines, color, tag );
}

hkResult hkDebugDisplay::endFrame()
{
    hkCriticalSectionLock arrayLock(m_arrayLock);
    for (int i = 0; i < m_debugDisplayHandlers.getSize(); i++)
    {
        HK_RETURN_IF_FAILED( m_debugDisplayHandlers[i]->endFrame() );
    }
    return HK_SUCCESS;
}

HK_SINGLETON_IMPLEMENTATION( hkDebugDisplay );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
