// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Visualize/hkVisualize.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Visualize/UnitTest/Serialize/VdbSerialize.h>

#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#include <Common/Visualize/hkServerObjectHandler.h>

using namespace hkSerialize;

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
#define ON_HK_SERIALIZE(x) x
#else
#define ON_HK_SERIALIZE(x)
#endif

//#define TEST_TCP_IP_THROUGHPUT
#ifdef TEST_TCP_IP_THROUGHPUT
//#define TEST_TCP_IP_THROUGHPUT_ADD_FLUSH_DELAY
#include <chrono>
#include <thread>
#include <future>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>
#endif

namespace
{
    void flush( hkStreamWriter& writer )
    {
        writer.flush();
#ifdef TEST_TCP_IP_THROUGHPUT_ADD_FLUSH_DELAY
        // If testing throughput, give TCP/IP a chance to push stuff through.
        // This assumes that we don't normally experience penalty associated with this
        // as the two will be running at different intervals in different processes normally.
        using namespace std::chrono_literals;
        std::this_thread::sleep_for( 1ms );
#endif
    }

    // Test the serialization/deserialization of VDB objects.
    // For implementation notes and requirements, look at hkVdbReflect::Var.
    hkReal Serialize_vdbTypeOnly(
        hkReflect::Var var,
        hkVdbSerialize::TypeSave& saver,
        hkVdbSerialize::TypeLoad& loader,
#ifndef HK_VDB_USE_HK_TYPE_SERIALIZE
        hkVdbReflect::TypeInfo& serverTypeInfoOut,
        const hkReflect::Type*& clientTypeOut,
#endif
        hkStreamWriter& writer,
        hkStreamReader& reader,
        SerializeVdbObjectExtraFails::Enum failOn,
        SerializeVdbObjectStats* statsOut )
    {
        hkStopwatch stopwatch;
        hkReal totalTime = 0;

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE

        // "warm up" the saver
        stopwatch.start();
#ifdef TEST_TCP_IP_THROUGHPUT
        // COM-4220
        hkArray<char> contentsBuffer; contentsBuffer.reserve( 1024 ); // reasonable reserve?
        hkResult result = saver.contentsVar( var, &contentsBuffer );
        writer.write( contentsBuffer.begin(), contentsBuffer.getSize() );
#else
        hkResult result = saver.contentsVar( var, &writer );
#endif
        stopwatch.stop();
        HK_TEST( result == HK_SUCCESS );
        totalTime += stopwatch.getElapsedSeconds();
        if ( statsOut ) statsOut->m_serializeType += stopwatch.getElapsedSeconds();
        stopwatch.reset();

#else

        // Serialize type
        const hkReflect::Type* serverType = var.getType();
        {
            stopwatch.start();
            hkResult result = hkVdbSerialize::serializeType(
                writer,
                saver,
                serverType,
                serverTypeInfoOut );
            stopwatch.stop();
            HK_TEST( result.isSuccess() );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_serializeType += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#endif

        flush( writer );

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE

        // "warm up" the loader
        stopwatch.start();
        hkReflect::Var loadedVar = loader.toVar( &reader );
        stopwatch.stop();
        HK_TEST( loadedVar );
        totalTime += stopwatch.getElapsedSeconds();
        if ( statsOut ) statsOut->m_deserializeType += stopwatch.getElapsedSeconds();
        stopwatch.reset();
        loadedVar.destroy();

#else

        // Deserialize type
        {
            stopwatch.start();
            clientTypeOut = hkVdbSerialize::deserializeType(
                reader,
                loader );
            stopwatch.stop();
            HK_TEST( clientTypeOut );
            HK_TEST( clientTypeOut->getAlignOf() == serverType->getAlignOf() );
            HK_TEST( clientTypeOut->getSizeOf() == serverType->getSizeOf() );
            HK_TEST(
                ( ( failOn & SerializeVdbObjectExtraFails::UNSUPPORTED_SUB_TYPES ) == 0 ) ||
                ( clientTypeOut->getNumDataFields() == serverType->getNumDataFields() ) );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_deserializeType += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#endif

        return totalTime;
    }

    hkReal Serialize_vdbObjectOnly(
        hkReflect::Var var,
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
        hkVdbSerialize::TypeSave& saver,
        hkVdbSerialize::TypeLoad& loader,
#else
        hkVdbReflect::TypeInfo& serverTypeInfo,
        const hkReflect::Type* clientType,
#endif
        hkStreamWriter& writer,
        hkStreamReader& reader,
        SerializeVdbObjectExtraFails::Enum failOn,
        SerializeVdbObjectStats* statsOut )
    {
        hkStopwatch stopwatch;
        hkReal totalTime = 0;

        // Capture supported field values
        hkArray<hkVdbReflect::Var> serverFieldValues;
        {
            hkVdbReflect::Var serverObject = var;
            for ( hkVdbReflect::Var::iterator iter = serverObject.begin();
                iter != serverObject.end();
                ++iter )
            {
                hkVdbReflect::Var serverFieldValue = ( *iter );
                serverFieldValues.pushBack( serverFieldValue );
            }
            HK_TEST(
                ( ( failOn & SerializeVdbObjectExtraFails::EMPTY_TYPE ) == 0 ) ||
                serverFieldValues.getSize() );
        }

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE

        // Serialize object
        {
            stopwatch.start();
            // Vdb packets must know their size, so writing to an array and then copying is a necessary step
            hkArray<char> contentsBuffer; contentsBuffer.reserve( 1024 ); // reasonable reserve?
            hkResult result = saver.contentsVar( var, &contentsBuffer );
            writer.write( contentsBuffer.begin(), contentsBuffer.getSize() );
            stopwatch.stop();
            HK_TEST( result == HK_SUCCESS );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_serializeObject += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#else

        // Compute object sizes/offsets
        hkArray<hkUint16> objectLocalOffsets;
        {
            stopwatch.start();
            hkUint32 objectSize = hkVdbSerialize::computeObjectSize(
                var,
                serverTypeInfo,
                objectLocalOffsets );
            stopwatch.stop();
            HK_TEST( objectSize != 0 );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_computeObjectSize += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

        // Serialize object
        {
            stopwatch.start();
            hkResult result = hkVdbSerialize::serializeObject(
                writer,
                var,
                serverTypeInfo,
                objectLocalOffsets );
            stopwatch.stop();
            HK_TEST( result.isSuccess() );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_serializeObject += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#endif

        flush( writer );

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE

        // Deserialize object
        hkVdbReflect::Var clientObject;
        {
            stopwatch.start();
            clientObject = loader.toVar( &reader );
            stopwatch.stop();
            HK_TEST( clientObject );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_deserializeObject += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#else

        // Deserialize object
        hkVdbReflect::Var clientObject;
        {
            stopwatch.start();
            clientObject = hkVdbSerialize::deserializeObject(
                reader,
                clientType );
            stopwatch.stop();
            HK_TEST( clientObject );
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_deserializeObject += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

#endif

        // Check iteration on properties
        {
            int i = 0;
            for ( hkVdbReflect::Var::iterator iter = clientObject.begin();
                iter != clientObject.end();
                ++iter )
            {
                HK_TEST( i < serverFieldValues.getSize() );
                hkVdbReflect::Var serverFieldValue = serverFieldValues[i];
                hkVdbReflect::Var clientFieldValue = ( *iter );
                HK_TEST(
                    serverFieldValue.equals( clientFieldValue ) ||
                    ( ( failOn & SerializeVdbObjectExtraFails::UNSUPPORTED_SUB_TYPES ) == 0 ) );
                hkStringBuf b; clientFieldValue.toString( b );
                HK_TEST( b.cString() );
                i++;
            }
        }

        // Check cleanup of client object and type
        {
            stopwatch.start();
            clientObject.destroy();
            stopwatch.stop();
            totalTime += stopwatch.getElapsedSeconds();
            if ( statsOut ) statsOut->m_destroyObject += stopwatch.getElapsedSeconds();
            stopwatch.reset();
        }

        return totalTime;
    }
}

hkReal Serialize_vdbObject(
    hkReflect::Var var,
    hkVdbSerialize::TypeSave& saver,
    hkVdbSerialize::TypeLoad& loader,
    SerializeVdbObjectExtraFails::Enum failOn,
    SerializeVdbObjectStats* statsOut,
    int num )
{
#ifdef TEST_TCP_IP_THROUGHPUT
    hkRefPtr<hkSocket> serverSocket = hkRefNew<hkSocket>( hkSocket::create() );
    hkRefPtr<hkSocket> serverToClientSocket;
    hkRefPtr<hkSocket> clientSocket = hkRefNew<hkSocket>( hkSocket::create() );
    auto poll = [&serverSocket, &serverToClientSocket]()
    {
        hkMemoryRouter router;
        hkMemorySystem::getInstance().threadInit( router, "lamda" );
        hkBaseSystem::initThread( &router );

        do
        {
            serverToClientSocket = serverSocket->pollForNewClient();
        } while ( !serverToClientSocket );

        hkBaseSystem::quitThread();
        hkMemorySystem::getInstance().threadQuit( router );
    };
    serverSocket->listen( 25005 );
    std::future<void> result( std::async( poll ) );
    clientSocket->connect( "localhost", 25005 );
    result.get();
    // Note: ostream wraps in a buffered stream writer with an MTU guess
    hkVdbOStream ostream( &serverToClientSocket->getWriter() );
    hkStreamWriter& writer = *ostream.getStreamWriter();
    hkStreamReader& reader = clientSocket->getReader();
#else

    // The memory stream reader doesn't track the array so it can't handle a resize.
    // We also don't want mem allocs skewing perf metrics (when we are taking those).
    // So we set buffer to be just big enough for our max text.
    // Sizes from our tests:
    // Running VdbSerialize_main unit test...
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 952
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 668
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 2856
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 610
    // Running NpWorldInspectorTest_main unit test...
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 7426
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 3008
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1373
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1908
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 180
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 189
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 116
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1688
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1528
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1460
    //    UnitTest\Serialize\VdbSerialize.cpp( 390 ) : Report: buffer size : 1476

    hkLocalArray<char> buffer( 8 * 1024 );
    hkArrayStreamWriter writer( &buffer, hkArrayStreamWriter::ARRAY_BORROW );
    hkMemoryStreamReader reader( buffer.begin(), buffer.getCapacity(), hkMemoryStreamReader::MEMORY_INPLACE );
#endif
    hkStopwatch stopwatch;
    hkReal totalTime = 0;

    // Process type
#ifndef HK_VDB_USE_HK_TYPE_SERIALIZE
    hkVdbReflect::TypeInfo serverTypeInfo;
    const hkReflect::Type* clientType;
#endif
    {
        totalTime += Serialize_vdbTypeOnly(
            var,
            saver,
            loader,
#ifndef HK_VDB_USE_HK_TYPE_SERIALIZE
            serverTypeInfo,
            clientType,
#endif
            writer,
            reader,
            failOn,
            statsOut );
    }

    // Process object
    for ( int i = 0; i < num; i++ )
    {
        totalTime += Serialize_vdbObjectOnly(
            var,
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
            saver,
            loader,
#else
            serverTypeInfo,
            clientType,
#endif
            writer,
            reader,
            failOn,
            statsOut );
    }

    // Check cleanup of client object and type
#ifndef HK_VDB_USE_HK_TYPE_SERIALIZE
    {
        stopwatch.start();
        hk::DeleteTypeInfo::deleteType( const_cast< hkReflect::Type* >( clientType ) );
        stopwatch.stop();
        totalTime += stopwatch.getElapsedSeconds();
        if ( statsOut ) statsOut->m_destroyType += stopwatch.getElapsedSeconds();
        stopwatch.reset();
    }
#endif

    // HK_REPORT2(0x730f7e08, "buffer size: " << buffer.getSize() );

    return totalTime;
}

int VdbSerialize_main()
{
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
    hkSerialize::Save saver;
    saver.withMultiBundle();
    hkSerialize::Load loader;
#else
    hkVdbSerialize::TypeSave saver;
    hkVdbSerialize::TypeLoad loader;
#endif

    // Basic tests
    {
#define TEST_TYPE( TYPE ) \
        TYPE TYPE##Object; \
        Serialize_vdbObject( &(TYPE##Object), saver, loader, SerializeVdbObjectExtraFails::ALL )

        TEST_TYPE( TestSimpleRecord );
        TEST_TYPE( TestSimpleExRecord );
        TEST_TYPE( TestSimpleContainerRecord );
        TEST_TYPE( TestComplexRecord );
    }

    // Perf stats
    if ( 0 )
    {
#define TEST_PERF_OF_TYPE( TYPE ) \
        SerializeVdbObjectStats TYPE##Stats( "100000 " HK_PP_STRINGIFY_NAME(TYPE) "(s)" ); \
        TYPE TYPE##Object; \
        Serialize_vdbObject( &(TYPE##Object), saver, loader, SerializeVdbObjectExtraFails::ALL, &(TYPE##Stats), 100000 ); \
        { hkStringBuf buf; (TYPE##Stats).printTo( buf ); HK_REPORT2(0x35a6e968, buf.cString() ); }

        HK_REPORT2(0x6dbcbcc3, "" );
        TEST_PERF_OF_TYPE( TestSimpleRecord );
        TEST_PERF_OF_TYPE( TestSimpleExRecord );
        TEST_PERF_OF_TYPE( TestSimpleContainerRecord );
        TEST_PERF_OF_TYPE( TestComplexRecord );
    }
    return 0;
}

//
// test registration
//
HK_TEST_REGISTER( VdbSerialize_main, "Slow", "Common/Test/UnitTest/Visualize/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
