// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// These defines are for debugging/development and should not be used by clients
//#define HK_VDB_DISABLE_OBJECT_CMD_WRITING
//#define HK_VDB_DISABLE_OBJECT_CMD_READING
//#define HK_VDB_DISABLE_OBJECT_CMD_CACHING
//#define HK_VDB_DISABLE_OBJECT_CMD_SIGNALLING
//#define HK_VDB_ENABLE_OBJECT_CMD_PROFILING

#define HK_VDB_DISABLE_OBJECT_CMD_CONSISTENCY_CHECKS






#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
#include <Common/Base/Serialize/hkSerialize.h>
#else
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#endif






//#define HK_USE_HASH_MAP_FOR_VDB_IDS
#ifndef HK_USE_HASH_MAP_FOR_VDB_IDS
#   include <Common/Base/Container/PointerMap/hkMap.h>
#   include <Common/Base/Container/Set/hkSet.h>
#   define hkVdbIdMap hkMap
#   define hkVdbIdSet hkSet
#   define HK_SOH_ON_HASH_MAP( x )
#   define HK_SOH_ON_DEP_MAP( x ) x
#else
#   include <Common/Base/Container/Hash/hkHashSet.h>
#   define hkVdbIdMap hkHashMap
#   define hkVdbIdSet hkHashSet
#   define HK_SOH_ON_HASH_MAP( x )
#   define HK_SOH_ON_DEP_MAP( x ) x
#endif

#include <Common/Base/Container/LocalArray/hkLocalArray.h>

//////////////////////////////////////////////////////////////////////////
/// This namespace is used for functionality similar to hkReflect.
/// However, it operates under some important limitations.
/// These limitations are known by hkVdbSerialize functions
/// and relied upon for important optimizations and properties to facilitate
/// high-throughput, cross-platform sending of object data to the visual debugger.
/// It is not typically necessary to interact with this namespace directly.
/// Use hkServerObjectHandler instead.
//////////////////////////////////////////////////////////////////////////
namespace hkVdbReflect
{
    typedef hkReflect::Type Type;

    /// Defines information computed for a Type which
    /// is used to serialize objects of that type.
    struct HK_EXPORT_COMMON TypeInfo
    {
        HK_DECLARE_CLASS( TypeInfo, New );
        hkArray<hkUint16> m_fieldOffsets;
        hkArray<const Type*> m_fieldTypes;
    };

    /// A Var to be used with hkVdbSerialize.
    /// This Var offers a iterator over limited/supported fields.
    struct HK_EXPORT_COMMON Var : public hkReflect::Var
    {
        HK_DECLARE_CLASS( Var, New, EmptyCtor );
        Var( hkReflect::Var var ) : hkReflect::Var( var ) {}
        HK_INLINE hkBool32 isValid() const;
        HK_INLINE int numProperties() const;
        HK_INLINE hkBool32 hasProperty( const char* name ) const;
        HK_INLINE Var findProperty( const char* name ) const;

        struct HK_EXPORT_COMMON _Iterator
        {
            HK_INLINE _Iterator() : m_recordType( HK_NULL ) {}
            HK_INLINE const _Iterator& operator++();
            HK_INLINE const _Iterator operator++( int ); // Note: unoptimizable
            
            
            HK_INLINE const _Iterator operator+( int ) const;
            HK_INLINE const _Iterator& operator+=( int );
            
            
            HK_INLINE bool operator==( const _Iterator& iter ) const;
            HK_INLINE bool operator!=( const _Iterator& iter ) const { return !operator==( iter ); }
            HK_INLINE bool operator<( const _Iterator& iter ) const;
            HK_INLINE bool operator>( const _Iterator& iter ) const;
            HK_INLINE bool operator<=( const _Iterator& iter ) const { return operator==( iter ) || operator<( iter ); }
            HK_INLINE bool operator>=( const _Iterator& iter ) const { return operator==( iter ) || operator>( iter ); }
            HK_INLINE Var operator*();
            HK_INLINE const Var operator*() const { return const_cast< _Iterator* >( this )->operator*(); }
            HK_INLINE const char* name() const;
        private:
            HK_INLINE _Iterator( hkReflect::Var var );
            hkReflect::Var m_var;
            mutable const hkReflect::RecordType* m_recordType;
            mutable hkArrayView<const hkReflect::FieldDecl>::const_iterator m_fieldIt;
            mutable hkArrayView<const hkReflect::FieldDecl>::const_iterator m_fieldItEnd;
            friend struct hkVdbReflect::Var;
        };
        typedef _Iterator iterator;
        typedef const _Iterator const_iterator;

        /// Returns an iterator over supported sub-vars.
        /// (currently only forward iteration is supported)
        HK_INLINE iterator begin();
        HK_INLINE const_iterator begin() const;

        /// Returns an iterator which is one passed the end of the valid iteration.
        /// Use iter != end() as an exit condition for iteration.
        HK_INLINE iterator end();
        HK_INLINE const_iterator end() const;

        /// Compare one var to another.
        int compare( const hkVdbReflect::Var& v ) const;

        /// Returns true if this var equals the provided var.
        HK_INLINE bool equals( const hkVdbReflect::Var& v ) const { return compare( v ) == 0; }

        /// Returns true if this var is less than the provided var.
        HK_INLINE bool lessThan( const hkVdbReflect::Var& v ) const { return compare( v ) < 0; }
    };
}

//////////////////////////////////////////////////////////////////////////
/// Attributes used with hkVdbReflect::Vars.
/// It is not typically necessary to interact with this namespace directly.
/// Use hkServerObjectHandler instead.
//////////////////////////////////////////////////////////////////////////
namespace hkVdbAttr
{
    /// Attached to deserialized hkVdbReflect::Var fields to indicate
    /// their offset index.
    HK_DECLARE_VALUE_ATTR( FieldOffsetIndex, int, Runtime );
}

//////////////////////////////////////////////////////////////////////////
/// Serialization
/// The functions below should be used to serialize objects and their types to the vdb.
/// It ensures the following:
/// 1) The vdb client doesn't need to have types compiled-in.
/// 2) The through-put on objects is very fast (approaching mem-copy speed).
/// It doesn't support:
/// 1) Versioning
/// 2) Objects with pointers (except strings), you can use isTypeValid and isSubTypeValid for more info.
/// It is not typically necessary to interact with this namespace directly.
/// Use hkServerObjectHandler instead.
//////////////////////////////////////////////////////////////////////////
namespace hkVdbSerialize
{
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
    class HK_EXPORT_COMMON TypeSave : public hkSerialize::Save { public: TypeSave(); };
    class HK_EXPORT_COMMON TypeLoad : public hkSerialize::Load { public: TypeLoad(); };
#else
    class TypeSave : public hkSerialize::TagfileWriteFormat { public: HK_EXPORT_COMMON TypeSave(); };
    class TypeLoad : public hkSerialize::TagfileReadFormat { public: HK_EXPORT_COMMON TypeLoad(); };
#endif

    //
    // Server
    //

    
    /// Returns true if the record type is completely supported by hkVdbSerialize.
    /// Even if false is returned, objects of the type may be serialized,
    /// it just means some fields may not be available.
    /// The only hard-requirement is that type is a record type.
    
    bool isTypeValid( const hkVdbReflect::Type* recordType );

    /// Returns true if a sub-type (field type) is valid (see isTypeValid).
    bool isSubTypeValid( const hkVdbReflect::Type* type );

    /// Serialize type information to writer using saver object.
    /// The saver provided to this method expects to be used one-for-one
    /// with the loader provided to the deserializeType method.
    /// Information about the type is outputted.
    hkResult serializeType(
        hkStreamWriter& writer,
        TypeSave& saver,
        const hkReflect::Type* type,
        hkVdbReflect::TypeInfo& typeInfoOut,
        hkCriticalSection* writeLock = HK_NULL );

    /// Given the object we want to serialize and it's type info,
    /// returns the size to write the object as well as localOffsets
    /// which must be provided to serializeObject.
    hkUint32 computeObjectSize(
        hkReflect::Var object,
        const hkVdbReflect::TypeInfo& typeInfo,
        hkArray<hkUint16>& localOffsetsOut );

    /// computeObjectSize must be called first. The localOffsets
    /// outputted from that method must be provided to this call.
    /// Serializes the object using the writer.
    hkResult serializeObject(
        hkStreamWriter& writer,
        hkReflect::Var object,
        const hkVdbReflect::TypeInfo& typeInfo,
        hkArrayView<hkUint16> localOffsets,
        hkCriticalSection* writeLock = HK_NULL );
    

    //
    // Client
    //

    /// Deserialize type information from reader using loader object.
    /// The loader provided to this method expects to be used one-for-one
    /// with the saver provided to the serializeType method.
    /// Returns a type suitable for use with objects serialized with this type.
    const hkVdbReflect::Type* deserializeType(
        hkStreamReader& reader,
        TypeLoad& loader,
        hkMemoryAllocator* allocator = HK_NULL,
        hkCriticalSection* readLock = HK_NULL,
        const hkReflect::Type** serverTypeOut = HK_NULL );

    /// Returns the total object size. The type sizeof isn't sufficient as we
    /// allocated space for some pointed-to data after the object.
    hkUint32 getObjectSize(
        hkVdbReflect::Var clientObject );

    /// Deserializes the object using the reader.
    /// The type provided must be from deserializeType().
    hkVdbReflect::Var deserializeObject(
        hkStreamReader& reader,
        const hkVdbReflect::Type* clientType,
        hkMemoryAllocator* allocator = HK_NULL,
        hkCriticalSection* readLock = HK_NULL );

    /// Reserializes an object that was created from deserializeObject,
    /// using the writer.
    hkResult reserializeObject(
        hkStreamWriter& writer,
        hkVdbReflect::Var clientObject,
        hkCriticalSection* writeLock = HK_NULL );
}


/// Flags for objects to indicate special properties.
struct HK_EXPORT_COMMON hkObjectFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// The object has no properties.
        NONE = 0,

        /// The object will be visible in the object inspection tree.
        VISIBLE = ( 1 << 0 ),

        /// The object will be queryable by the object inspection search filter.
        QUERYABLE = ( 1 << 1 ),

        // Experimental flag for an object which exists for one frame.
        // DO NOT USE, not well tested or performant.
        TRANSIENT = ( 1 << 2 ),

        /// The object is a root level container for other objects.
        /// For example, a physics world, a character proxy list, etc.
        /// Objects marked as a root level container should not be stored as children of objects not also
        /// marked as root object containers (nodes that are not containers should not have container children).
        /// The vdb uses this for some optimizations and tree behaviors.
        ROOT_OBJECT_CONTAINER = ( 1 << 3 ),

        /// In most cases, the default gives expected behavior.
        DEFAULT = ( VISIBLE | QUERYABLE )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkObjectFlags, hkObjectFlags::Bits );
};



/// Flags for object connections.
struct HK_EXPORT_COMMON hkConnectivityFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// The connection has no flags.
        NONE = 0,

        /// Connections from other objects *to* this object.
        INCOMING_CONNECTIONS = ( 1 << 0 ),

        /// Connections *from* this object to other objects.
        OUTGOING_CONNECTIONS = ( 1 << 1 ),

        /// All connections for this object.
        ALL = ( INCOMING_CONNECTIONS | OUTGOING_CONNECTIONS )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkConnectivityFlags, hkConnectivityFlags::Bits );
};


/// Interface for a registry of object types.
template <typename TYPE_ID_STORAGE>
class hkObjectTypeRegistry
{
public:
    typedef TYPE_ID_STORAGE TypeId;

    /// Indicates a type id which is not valid in the system.
    /// Note: don't change InvalidTypeId value as it coincides with related class values.
    static const TYPE_ID_STORAGE InvalidTypeId = TYPE_ID_STORAGE( 0 );

    /// Returns if the particular type id has been registered.
    virtual bool isObjectTypeRegistered( TypeId typeId ) const = 0;

    /// Register an object type for a particular id.
    /// The id must be unique for all registered types.
    virtual bool registerObjectType( TypeId typeId, const hkReflect::Type* type ) = 0;
};

/// Interface for a registry of objects.
template <typename OBJECT_ID_STORAGE>
class hkObjectRegistry
{
public:
    typedef OBJECT_ID_STORAGE ObjectId;

    /// Indicates an object id which is not valid in the system.
    /// Note: don't change InvalidObjectId value as it coincides with related class values.
    static const OBJECT_ID_STORAGE InvalidObjectId = OBJECT_ID_STORAGE( 0 );

    
    /// Returns if the particular object id has been added.
    virtual bool hasObject( ObjectId id ) const = 0;

    /// Add objects with provided ids and flags to the registry.
    /// The ids must be unique for all objects added to the registry.
    /// Returns the number of objects successfully added.
    /// Note: usually batching adds into fewer calls will perform better.
    virtual int addObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects, hkObjectFlags flags = hkObjectFlags::DEFAULT, int tag = 0 ) = 0;

    /// Update previously added objects.
    /// Returns the number of objects successfully updated.
    /// Note: usually batching updates into fewer calls will perform better.
    virtual int updateObjects( const hkArrayView<ObjectId>& ids, const hkArrayView<hkReflect::Var>& objects ) = 0;

    /// Remove previously added objects.
    /// Returns the number of objects successfully removed.
    /// Note: usually batching removals into fewer calls will perform better.
    virtual int removeObjects( const hkArrayView<ObjectId>& ids ) = 0;
    

    /// *** Performance warning, batching objects for adding is faster.
    /// Add an object with provided id and flags to the registry.
    /// The id must be unique for all objects added to the registry.
    /// Returns true if the object was successfully added.
    HK_INLINE hkBool32 addObject(
        ObjectId id,
        hkReflect::Var object,
        hkObjectFlags flags = hkObjectFlags::DEFAULT,
        int tag = 0 )
    {
        return addObjects(
            hkArrayViewT::fromSingleObject( id ),
            hkArrayViewT::fromSingleObject( object ),
            flags,
            tag );
    }

    /// *** Performance warning, batching objects for update is faster.
    /// Update previously added object.
    /// Returns true if the object was successfully updated.
    HK_INLINE hkBool32 updateObject( ObjectId id, hkReflect::Var object )
    {
        return updateObjects(
            hkArrayViewT::fromSingleObject( id ),
            hkArrayViewT::fromSingleObject( object ) );
    }

    /// *** Performance warning, batching objects for removal is faster.
    /// Remove previously added object.
    /// Returns true if the object was successfully removed.
    HK_INLINE hkBool32 removeObject( ObjectId id )
    {
        return removeObjects(
            hkArrayViewT::fromSingleObject( id ) );
    }
};

/// Interface for a directed graph (ids that can be connected to one another).
template <typename CONNECTION_ID_STORAGE>
class hkConnectionGraph
{
public:
    typedef CONNECTION_ID_STORAGE ConnectionId;

    /// Indicates the root object id.
    /// It forms the top of a hierarchical/tree display for the graph (even though multiple parents
    /// are allowed/possible).
    /// Note: don't change RootConnectionId value as it coincides with related class values.
    static const CONNECTION_ID_STORAGE RootConnectionId = CONNECTION_ID_STORAGE( 0 );

    /// Indicates a connection id which is not valid in the system.
    /// Note: don't change RootConnectionId value as it coincides with related class values.
    static const CONNECTION_ID_STORAGE InvalidConnectionId = CONNECTION_ID_STORAGE( -1 );

    
    /// Returns if the particular connection exists from -> to.
    virtual bool isConnected( ConnectionId from, ConnectionId to ) = 0;

    /// Connects fromId ->* toIds.
    /// Returns the number of connections successfully made.
    /// Note: usually batching connections into fewer calls will perform better.
    virtual int connect( ConnectionId fromId, const hkArrayView<ConnectionId>& toIds, int tag = 0 ) = 0;

    /// Disconnects fromId ->* toIds.
    /// Returns the number of disconnections successfully made.
    /// Note: usually batching disconnections into fewer calls will perform better.
    virtual int disconnect( ConnectionId fromId, const hkArrayView<ConnectionId>& toIds ) = 0;

    /// Disconnects id from its connections as specified by provided flags.
    /// Returns the number of disconnections successfully made.
    /// Note: usually batching disconnections into fewer calls will perform better.
    virtual int disconnect( ConnectionId id, hkConnectivityFlags flags = hkConnectivityFlags::ALL ) = 0;
    

    /// *** Performance warning, batching connections is faster.
    /// Connects fromId -> toId.
    /// Returns true if the connection was successfully made.
    HK_INLINE hkBool32 connectOne( ConnectionId fromId, ConnectionId toId, int tag = 0 )
    {
        return connect(
            fromId,
            hkArrayViewT::fromSingleObject( toId ),
            tag );
    }

    /// *** Performance warning, batching disconnections is faster.
    /// Disconnects fromId -> toId.
    /// Returns true if the disconnection was successfully made.
    HK_INLINE hkBool32 disconnectOne( ConnectionId fromId, ConnectionId toId )
    {
        return disconnect(
            fromId,
            hkArrayViewT::fromSingleObject( toId ) );
    }

    /// Disconnects ids from their connections as specified by provided flags.
    /// Returns the number of disconnections successfully made.
    /// Note: usually batching disconnections into fewer calls will perform better.
    HK_INLINE_VIRTUAL virtual int disconnect( const hkArrayView<ConnectionId>& ids, hkConnectivityFlags flags = hkConnectivityFlags::ALL )
    {
        int numDisconnected = 0;
        for ( int i = 0; i < ids.getSize(); i++ )
        {
            ConnectionId id = ids[i];
            numDisconnected += disconnect( id, flags );
        }
        return numDisconnected;
    }
};


/// Interface for a registry of connected objects.
template <typename OBJECT_ID_STORAGE, typename CONNECTION_ID_STORAGE>
class hkConnectedObjectRegistry :
    public hkObjectRegistry<OBJECT_ID_STORAGE>,
    public hkConnectionGraph<CONNECTION_ID_STORAGE>
{};

/// Helper registry for managing unique ids from any type of unique key.
template<typename K, typename I = hkUint32, bool RefCount = true>
class hkObjectIdRegistry
{
public:
    typedef K KeyType;
    typedef I IdType;

    HK_DECLARE_CLASS( hkObjectIdRegistry, New );
    hkObjectIdRegistry() : m_nextId( 1 ) {}

    /// Get or create an id for the provided unique key.
    /// If the registry is a ref-counting registry, then a ref-count is
    /// maintained for each call to this function for the key.
    /// Returns the unique id and outputs if it was created.
    IdType getOrCreateId( const KeyType& key, hkBool32& wasIdCreatedOut )
    { return getOrCreateIdInternal( key, wasIdCreatedOut ); }

    /// Free a previously created id.
    /// If the registry is a ref-counting registry then it will only free
    /// the id if this call has occurred the same number of times as
    /// getOrCreateId has for the particular key.
    IdType freeId( const KeyType& key, hkBool32& wasIdFreedOut );

    /// Reserve space for some number of keys/ids.
    void reserve( int numIds ) { m_keyToIdMap.reserve( numIds ); }

protected:
    IdType getOrCreateIdInternal( const KeyType& key, hkBool32& wasIdCreatedOut, hkUint32 initRefCount = 1 );

    struct IdPlusRefCount
    {
        IdType m_id;
        hkUint32 m_refCount;
    };

    IdType m_nextId;
    hkArray<IdType> m_idPool;
    hkHashMap<KeyType, IdPlusRefCount> m_keyToIdMap;
};

/// Helper registry for managing unique ids from any type of unique key for
/// "transient" objects; objects which are difficult to track the lifetime of.
template<typename K, typename I = hkUint32>
class hkTransientObjectIdRegistry : protected hkObjectIdRegistry<K, I, true>
{
public:
    typedef K KeyType;
    typedef I IdType;

    HK_DECLARE_CLASS( hkTransientObjectIdRegistry, New );
    hkTransientObjectIdRegistry() : hkObjectIdRegistry<K, I, true>(), m_step( 0 ) {}

    /// Gets or creates and id for the unique key.
    /// This must be called for every unique key between calls to step.
    IdType referenceId( const KeyType& key, hkBool32& wasIdCreatedOut )
    {
        return hkObjectIdRegistry<K, I, true>::getOrCreateIdInternal( key, wasIdCreatedOut, m_step + 1 );
    }

    /// Indicate a new frame has occurred.
    /// Optionally iterate over freedIds.
    template<typename Functor>
    void step( Functor freedIdsFunctor );
    void step() { step( [] (IdType) {} ); }

    /// Reserve space for some number of keys/ids.
    void reserve( int numIds )
    {
        hkObjectIdRegistry<K, I, true>::reserve( numIds );
    }

protected:
    using hkObjectIdRegistry<K, I, true>::m_idPool;
    using hkObjectIdRegistry<K, I, true>::m_keyToIdMap;
    hkUint32 m_step;
};

#include <Common/Visualize/Serialize/hkVdbSerialize.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
