// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Serialize/hkVdbSerialize.h>

#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>

using namespace hkVdbCmdUtils;

namespace
{
#define HK_ASSERT_UNSUPPORTED_IMPL_OP \
    HK_ASSERT( \
        0x22441263, \
        false, \
        "Operation is unsupported on Vdb object" )

    using namespace hkSerialize;
    using namespace hkReflect;
    using namespace hkReflect::Detail;

    void assertKnownType( const ValueType* type )
    {
#ifdef HK_DEBUG
        if ( const BoolType* boolType = type->asBool() ) {}
        else if ( const IntType* intType = type->asInteger() ) {}
        else if ( const FloatType* floatType = type->asFloat() ) {}
        else if ( const StringType* strType = type->asString() ) {}
        else
        {
            HK_ASSERT( 0x22441172, false, "Unknown value type" );
        }
#endif
    }
    void assertKnownType( const CompoundType* type )
    {
#ifdef HK_DEBUG
        if ( const PointerType* ptrType = type->asPointer() ) {}
        else if ( const ArrayType* arrayType = type->asArray() ) {}
        else if ( const RecordType* recordType = type->asRecord() ) {}
        else
        {
            HK_ASSERT( 0x22441172, false, "Unknown compound type" );
        }
#endif
    }
    void assertKnownType( const VoidType* type )
    {
#ifdef HK_DEBUG
        if ( type->getKind() == KIND_VOID ) {}
        else
        {
            HK_ASSERT( 0x22441172, false, "Unknown void type" );
        }
#endif
    }
    void assertKnownType( const OpaqueType* type )
    {
#ifdef HK_DEBUG
        if ( type->getKind() == KIND_OPAQUE ) {}
        else
        {
            HK_ASSERT( 0x22441172, false, "Unknown opaque type" );
        }
#endif
    }
    template<typename T>
    int compareInternal( const T& a, const T& b )
    {
        // Note: records might have fields which were skipped (unsupported) and therefore
        // will not pass hkReflect::Var::compare().
        auto ai = a.begin();
        auto bi = b.begin();
        while ( ( ai != a.end() ) && ( bi != b.end() ) )
        {
            hkVdbReflect::Var ao = ( *ai );
            hkVdbReflect::Var bo = ( *bi );
            if ( const int elementCmp = ao.compare( bo ) )
            {
                // First non-zero elementCmp is the result.
                return elementCmp;
            }
            ++ai;
            ++bi;
        }

        if ( bi != b.end() )
        {
            //a has got fewer fields than b
            return -1;
        }
        else if ( ai != a.end() )
        {
            // b has got fewer fields than a
            return 1;
        }
        else
        {
            return 0;
        }
    }

//#pragma region hkVdbReflect Impls

    struct VdbObjectAllocImpl : public AllocationImpl
    {
        HK_DECLARE_CLASS( VdbObjectAllocImpl, NoNew );
        VdbObjectAllocImpl() { /* keep clang happy */ }
        static const VdbObjectAllocImpl s_instance;

        virtual Var allocate( const Type* type ) const HK_OVERRIDE
        {
            HK_ASSERT(
                0x22441257,
                false,
                "Our vdb objects contain a dynamically sized section at the end, this cannot be known from the Type alone.  "
                "Use hkVdbReflect::Var::deserializeObject()." );
            return Var();
        }
        virtual Var allocateForClone( const Var& src, const Type* type ) const HK_OVERRIDE
        {
            return allocate( type );
        }
        virtual void deallocate( void* target, const Type* type ) const HK_OVERRIDE
        {
            
            
            target = hkAddByteOffset( target, -HK_NEXT_MULTIPLE_OF( 16, s32 ) );
            int size = *reinterpret_cast< int* >( target );
            hkMemHeapAllocator()->blockFree( target, size );
        }
    };
    const VdbObjectAllocImpl VdbObjectAllocImpl::s_instance;

    namespace VdbObjectImplUtils
    {
        template<typename T>
        T getResolvedPtr( const void* ptrAddress, const Type* type )
        {
            if ( FieldDecl fieldDecl = type->isField() )
            {
                const Type* context = fieldDecl.getDeclContext();
                HK_ASSERT(
                    0x22441333,
                    context &&
                    TypeDetail::localAddressOptional<Opt::ALLOC_IMPL>( context ) &&
                    *TypeDetail::localAddressOptional<Opt::ALLOC_IMPL>( context ) == hkReflect::Detail::OptionStorage<hkReflect::Opt::ALLOC_IMPL>::Storage( &VdbObjectAllocImpl::s_instance ),
                    "Decl context is not a type we allocated" );

                int offsetIdx = type->getValueAttribute<hkVdbAttr::FieldOffsetIndex>( -1 );
                HK_ASSERT(
                    0x22441334,
                    offsetIdx != -1,
                    "Missing offset index attr" );

                const void* vdbObjectStart = hkAddByteOffset( ptrAddress, -fieldDecl.getOffset() );
                const hkUint16* offetsTable =
                    reinterpret_cast< const hkUint16* >(
                        hkAddByteOffset( vdbObjectStart, context->getSizeOf() ) );
                const hkUint16 offset = offetsTable[offsetIdx];
                return reinterpret_cast< T >( hkAddByteOffset( vdbObjectStart, offset ) );
            }
            HK_ASSERT( 0x22441335, false, "Only field types are supported" );
            return HK_NULL;
        }
    };

    
    
    
    
    struct VdbObjectCompoundImpl : public CompoundImpl
    {
        HK_DECLARE_CLASS( VdbObjectCompoundImpl, NoNew );
        VdbObjectCompoundImpl() { /* keep clang happy */ }
        static const VdbObjectCompoundImpl s_instance;
        virtual VarIter iterBegin( const void* comAddr, const CompoundType* comType ) const HK_OVERRIDE
        {
            CompoundVar var( comAddr, comType );
            hkVdbReflect::Var safeVar( var );
            hkVdbReflect::Var::iterator first = safeVar.begin();
            if ( first != safeVar.end() )
            {
                return VarIter( var, *first, 0 );
            }
            else
            {
                return VarIter( var );
            }
        }
        virtual void iterNext( VarIter& prev ) const HK_OVERRIDE
        {
            if ( prev.isValid() )
            {
                int next = int( prev.m_data ) + 1;
                hkVdbReflect::Var safeVar( prev.m_compound );
                hkVdbReflect::Var::iterator first = safeVar.begin();
                first += next;
                if ( first < safeVar.end() )
                {
                    prev.m_cur = *first;
                    prev.m_data = next;
                    return;
                }
            }
            prev.setInvalid();
        }
    };
    const VdbObjectCompoundImpl VdbObjectCompoundImpl::s_instance;

    struct VdbObjectStringImpl : public StringImpl
    {
        HK_DECLARE_CLASS( VdbObjectStringImpl, NoNew );
        VdbObjectStringImpl() { /* keep clang happy */ }
        static const VdbObjectStringImpl s_instance;

        virtual hkResult getValue( _In_ const void* addr, _In_ const StringType* type, StringValue* val ) const HK_OVERRIDE
        {
            *val = VdbObjectImplUtils::getResolvedPtr<const char*>( addr, type );
            return HK_SUCCESS;
        }
        // Unsupported operations
        virtual hkResult setValue( void* addr, const StringType* type, StringValue newValue ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; return HK_FAILURE; }
        virtual void clearAllocs( void* string, const StringType* type ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
    };
    const VdbObjectStringImpl VdbObjectStringImpl::s_instance;

#if 0
    struct VdbObjectArrayImpl : public ArrayImpl
    {
        HK_DECLARE_CLASS( VdbObjectArrayImpl, NoNew );
        VdbObjectArrayImpl() { /* keep clang happy */ }
        static const VdbObjectArrayImpl s_instance;

        virtual ArrayValue getValue( const void* addr, const ArrayType* type ) const HK_OVERRIDE
        {
            ArrayValue resolvedVal = VdbObjectImplUtils::getResolvedPtr<const char*>( addr, type );
            return resolvedVal;
        }
        // Unsupported operations
        virtual hkResult setValue( void* arrAddr, const ArrayType* arrType, const ArrayValue& val ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; return HK_FAILURE; }
        virtual hkResult setNumElements( void* arrAddr, const ArrayType* arrType, int nelem ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
        virtual hkResult spliceInto( void* arrAddr, const ArrayType* arrType, int index, int numToDel, const ArrayValue& val ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
        virtual AllocResult allocateElements( void* arrAddr, const ArrayType* arrType, QualType elemType, int n ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
        virtual void clearAllocs( void* arrAddr, const ArrayType* type ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
        virtual bool insert( Var var, const VarIter& hint ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
        virtual bool remove( const VarIter& var ) const HK_OVERRIDE { HK_ASSERT_UNSUPPORTED_IMPL_OP; }
    };
    const VdbObjectArrayImpl VdbObjectArrayImpl::s_instance;
#endif

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
    struct VdbObjectTypeReg : public TypeReg
    {
        HK_DECLARE_CLASS( VdbObjectTypeReg, NoNew );
        VdbObjectTypeReg() { /* keep clang happy */ }
        static const VdbObjectTypeReg s_instance;

        virtual const Type* typeFromName( const char* name ) const
        {
            if ( TypeReg* builtin = hkReflect::getTypeReg() )
            {
                return builtin->typeFromName( name );
            }
            return HK_NULL;
        }
        virtual const Type* typeFromType( const Type* type ) const
        {
            if ( TypeReg* builtin = hkReflect::getTypeReg() )
            {
                const Type* child = HK_NULL;
                const Type* parent = type;
                while ( parent )
                {
                    // Check for builtin
                    if ( const Type* builtinParent = builtin->typeFromType( parent ) )
                    {
                        // We don't want the VDB to need rebuilding after a change to a type.
                        // We only even access builtin types so that inheritance checks or hkDynCasts
                        // work against known types...
                        // Ideally, we want to pass those hkDynCast even if there's a version mismatch
                        // but we *don't* want the actual builtin type to be used because the data
                        // is a memcpy.
                        
                        if ( parent->getValueAttribute<hk::Version>( 0 ) == builtinParent->getValueAttribute<hk::Version>( 0 ) )
                        {
                            // Process our provided type
                            if ( parent == type )
                            {
                                HK_ASSERT_NO_MSG( 0x22441351, !child );
                                type = builtinParent;
                            }
                            // Or process an ancestor
                            else
                            {
                                HK_ASSERT_NO_MSG( 0x22441352, child );
                                TypeDetail::setParent( const_cast< Type* >( child ), builtinParent );
                                break;
                            }
                        }
                    }

                    // Advance
                    child = parent;
                    parent = child->getParent();
                }
            }
            return type;
        }
    };
    const VdbObjectTypeReg VdbObjectTypeReg::s_instance;
#endif

//#pragma endregion

//#pragma region hkVdbReflect Utils

    struct VdbObjectTypeBuilder : public TypeVisitor<VdbObjectTypeBuilder, const Type*, hkVdbReflect::TypeInfo*, hkMemoryAllocator*, int>
    {
        VdbObjectTypeBuilder( const Type* type, hkBool32 isSubType ) :
            m_type( type ),
            m_isSubType( isSubType ),
            m_hasUnsupportedMembers( false ),
            m_localOffsetIdx( 0 )
        {
            HK_ASSERT( 0x22441412, m_type, "This class expects a type" );
        }

        bool verify()
        {
            return ( startVisit( HK_NULL, HK_NULL ) && !m_hasUnsupportedMembers );
        }

        bool buildInfo( hkVdbReflect::TypeInfo& serverTypeInfoOut )
        {
            serverTypeInfoOut.m_fieldOffsets.clear();
            serverTypeInfoOut.m_fieldTypes.clear();
            return ( startVisit( &serverTypeInfoOut, HK_NULL ) && !m_hasUnsupportedMembers );
        }

        const Type* buildType( hkMemoryAllocator* allocator )
        {
            return startVisit( HK_NULL, allocator );
        }

        const Type* startVisit( hkVdbReflect::TypeInfo* serverInfoOut, hkMemoryAllocator* allocator )
        {
            HK_ASSERT(
                0x22441329,
                ( serverInfoOut == HK_NULL ) ||
                ( serverInfoOut != HK_NULL ) ^ ( allocator != HK_NULL ),
                "Should be verifying, building server type info, or building client types" );

            if ( m_type->asRecord() || m_isSubType )
            {
                return dispatch( m_type, serverInfoOut, allocator, 0 );
            }
            else
            {
                return HK_NULL;
            }
        }

        // Subtypes: BoolType, IntType, FloatType, StringType
        const Type* visit( const ValueType* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            assertKnownType( type );

            // Strings are supported with a special impl
            if ( const StringType* stringType = type->asString() )
            {
                return typeFromStringType( stringType, typeInfo, allocator, memberOffset );
            }
            // We support the other simple types
            else
            {
                return typeFromSimpleType( type, typeInfo, allocator );
            }
        }

        // Subtypes: PointerType, ArrayType, RecordType
        const Type* visit( const CompoundType* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            assertKnownType( type );

            // Arrays are supported with a special impl
            if ( const ArrayType* arrayType = type->asArray() )
            {
                return typeFromArrayType( arrayType, typeInfo, allocator, memberOffset );
            }
            // Records are supported with a special impl
            else if ( const RecordType* recordType = type->asRecord() )
            {
                return typeFromRecordType( recordType, typeInfo, allocator, memberOffset );
            }
            // We don't support pointers
            else
            {
                return opaqueTypeFromSimpleType( type, typeInfo, allocator );
            }
        }

        const Type* visit( const VoidType* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            // Not sure what this means in context of visiting a record var, see if it hits
            HK_ASSERT_NOT_IMPLEMENTED(0x75a2b2b5 );
            assertKnownType( type );
            return typeFromSimpleType( type, typeInfo, allocator );
        }

        const Type* visit( const OpaqueType* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            assertKnownType( type );
            return typeFromSimpleType( type, typeInfo, allocator );
        }

        const Type* typeFromStringType( const StringType* stringType, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            // We don't support passed one level of indirection
            if ( memberOffset < 0 )
            {
                return opaqueTypeFromSimpleType( stringType, typeInfo, allocator );
            }

            if ( allocator )
            {
                TypeBuilder builder;
                {
                    // Begin with a shallow clone
                    builder.beginShallowClone( stringType );

                    // Add our custom string impl
                    builder.setItem<Opt::IMPL>( &VdbObjectStringImpl::s_instance );

                    // Add the offset index as an attribute
                    hkVdbAttr::FieldOffsetIndex* idxAttr = new hkVdbAttr::FieldOffsetIndex();
                    idxAttr->m_value = m_localOffsetIdx++;
                    builder.addAttribute( idxAttr );

                    // Add delete info so this type can be cleaned up
                    builder.addDeleteInfo();
                }

                Type* allocatedType = builder.allocate( allocator );
                typeInfo->m_fieldTypes.pushBack( allocatedType );
                return allocatedType;
            }
            else
            {
                if ( typeInfo )
                {
                    HK_ASSERT_NO_MSG( 0x22441332, m_localOffsetIdx == typeInfo->m_fieldOffsets.getSize() );
                    typeInfo->m_fieldOffsets.pushBack( hkUint16( memberOffset ) );
                    typeInfo->m_fieldTypes.pushBack( stringType );
                    m_localOffsetIdx++;
                }
                return stringType;
            }
        }

        const Type* typeFromArrayType( const ArrayType* arrayType, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            // We don't support passed one level of indirection
            if ( memberOffset < 0 )
            {
                return opaqueTypeFromSimpleType( arrayType, typeInfo, allocator );
            }

            // See if our data is a pointer to some other memory
            if ( arrayType->getFixedCount() == 0 )
            {
                // Not currently supported
                
                return opaqueTypeFromSimpleType( arrayType, typeInfo, allocator );
            }
            // Otherwise we are a fixed (in place) array
            else
            {
                // See if our subtype is a simple one (or combination of simple types), otherwise, we don't support it
                const Type* arraySubType = dispatch( arrayType->getSubType(), typeInfo, allocator, memberOffset );

                if ( allocator )
                {
                    if ( arraySubType )
                    {
                        TypeBuilder builder;
                        {
                            // Begin with a shallow clone
                            builder.beginShallowClone( arrayType );

                            // Add delete info so this type can be cleaned up
                            builder.addDeleteInfo();

                            // Add our subtype
                            builder.setItem<Opt::SUBTYPE>( arraySubType );
                        }

                        Type* allocatedType = builder.allocate( allocator );
                        typeInfo->m_fieldTypes.pushBack( allocatedType );
                        return allocatedType;
                    }
                    else
                    {
                        return opaqueTypeFromSimpleType( arrayType, typeInfo, allocator );
                    }
                }
                else
                {
                    m_hasUnsupportedMembers |= ( arraySubType == HK_NULL );
                    return arrayType;
                }
            }
        }

        const Type* typeFromRecordType( const RecordType* recordType, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator, int memberOffset )
        {
            if ( allocator )
            {
                TypeBuilder builder;
                {
                    // Begin with a shallow clone
                    builder.beginShallowClone( recordType );

                    // If we are the root type set our special attributes so that we can be fully cleaned up
                    if ( recordType == m_type )
                    {
                        HK_ASSERT( 0x22441338, !typeInfo, "We should not yet have a type info" );

                        // This ALLOC_IMPL knows how to deallocate the dynamically sized section of our vdb
                        // object where we place supported pointer data.
                        builder.setItem<Opt::ALLOC_IMPL>( &VdbObjectAllocImpl::s_instance );

                        // Custom delete function to delete recursively added types
                        typeInfo = new hkVdbReflect::TypeInfo();
                        builder.addDeleteInfo( deallocVdbObjectType, hkUlong( typeInfo ) );
                    }
                    else
                    {
                        // Add delete info so this type can be cleaned up
                        builder.addDeleteInfo();
                    }

                    // Override parent
                    if ( const Type* parentType = recordType->getParentRecord() )
                    {
                        parentType = dispatch( parentType, typeInfo, allocator, memberOffset );
                        builder.overrideParent( parentType );
                    }

                    // Clear fields, we'll be adding them back *potentially* with a new type
                    // (see COM-4215)
                    builder.setItem<Opt::DECLS>( HK_NULL );
                    hkArrayView<const DataFieldDecl> dataFields = recordType->getDataFields();
                    for ( int i = 0; i < dataFields.getSize(); i++ )
                    {
                        const DataFieldDecl fieldDecl = dataFields[i];
                        const char* name = fieldDecl.getName();
                        int offset = fieldDecl.getOffset();
                        Decl::DeclFlags flags = fieldDecl.getFlags();
                        const Type* fieldType = dispatch( fieldDecl.getType()->getParent(), typeInfo, allocator, memberOffset + offset );
                        if ( !fieldType )
                        {
                            fieldType = opaqueTypeFromSimpleType( fieldDecl.getType(), typeInfo, allocator );
                        }
                        builder.addMember( name, offset, flags, fieldType );
                    }

                    // Ensure that we only iterate over valid fields
                    builder.setItem<Opt::IMPL>( &VdbObjectCompoundImpl::s_instance );
                }

                const Type* allocatedType = builder.allocate( allocator );
                HK_ASSERT_NO_MSG( 0x22441337, allocatedType && ( allocatedType->getSizeOf() == recordType->getSizeOf() ) );
                if ( recordType == m_type )
                {
                    // We just use this storage for the size
                    typeInfo->m_fieldOffsets.setSize( 2 );
                    *reinterpret_cast< int* >( typeInfo->m_fieldOffsets.begin() ) = builder.getTotalNeededMemory();
                }
                else
                {
                    typeInfo->m_fieldTypes.pushBack( allocatedType );
                }

                return allocatedType;
            }
            else
            {
                // Dispatch on parent fields
                if ( const RecordType* parentRecordType = recordType->getParentRecord() )
                {
                    dispatch( parentRecordType, typeInfo, allocator, memberOffset );
                }

                // Process our fields
                hkArrayView<const DataFieldDecl> dataFields = recordType->getDataFields();
                for ( int i = 0; i < dataFields.getSize(); i++ )
                {
                    const DataFieldDecl fieldDecl = dataFields[i];
                    int offset = fieldDecl.getOffset();
                    const Type* fieldType = fieldDecl.getType()->getParent();
                    m_hasUnsupportedMembers |= ( dispatch( fieldType, typeInfo, allocator, memberOffset + offset ) == HK_NULL );
                }

                return recordType;
            }
        }

        const Type* typeFromSimpleType( const Type* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator )
        {
            if ( allocator )
            {
                TypeBuilder builder;
                {
                    // Begin with a shallow clone
                    builder.beginShallowClone( type );

                    // Add delete info so this type can be cleaned up
                    builder.addDeleteInfo();
                }

                // Even for simple types this is needed since we will be setting the DECL_CONTEXT on the returned type.
                Type* allocatedType = builder.allocate( allocator );
                typeInfo->m_fieldTypes.pushBack( allocatedType );
                return allocatedType;
            }
            else
            {
                return type;
            }
        }

        const Type* opaqueTypeFromSimpleType( const Type* type, hkVdbReflect::TypeInfo* typeInfo, hkMemoryAllocator* allocator )
        {
            
            // Even for opaque types this is needed since we will be setting the DECL_CONTEXT on the returned type.
            return typeFromSimpleType(
                hkReflect::getType<hkReflect::Detail::Opaque>(),
                typeInfo,
                allocator );
        }

        static void HK_CALL deallocVdbObjectType( Type* type, hkMemoryAllocator* allocator, hkUlong data )
        {
            // Cleanup sub-types
            hkVdbReflect::TypeInfo* typeInfo = ( hkVdbReflect::TypeInfo* ) data;
            for ( int i = 0; i < typeInfo->m_fieldTypes.getSize(); i++ )
            {
                const Type* subType = typeInfo->m_fieldTypes[i];
                hk::DeleteTypeInfo::deleteType( const_cast<Type*>( subType ) );
            }

            // Cleanup ourselves
            int typeSize = *reinterpret_cast< int* >( typeInfo->m_fieldOffsets.begin() );
            TypeBuilder::deallocateType( type, allocator, typeSize );

            // Delete the info storage
            delete typeInfo;
        }

        const Type* m_type;
        hkBool32 m_isSubType;
        hkBool32 m_hasUnsupportedMembers;
        hkInt32 m_localOffsetIdx;
    };

    struct VdbObjectSerializer
    {
        VdbObjectSerializer( Var var, const hkVdbReflect::TypeInfo& info ) :
            m_var( var ),
            m_type( HK_NULL ),
            m_typeInfo( &info )
        {}

        VdbObjectSerializer( const Type* type ) :
            m_type( type ),
            m_typeInfo( HK_NULL )
        {}

        VdbObjectSerializer( Var var ) :
            m_var( var ),
            m_type( HK_NULL ),
            m_typeInfo( HK_NULL )
        {}

        hkInt32 computeNeededSize( hkArray<hkUint16>& localOffsetsOut )
        {
            if ( m_typeInfo->m_fieldOffsets.isEmpty() )
            {
                // Fast path
                localOffsetsOut.clear();
                // Space for size + sizeof
                return s32 + m_var.getType()->getSizeOf();
            }
            else
            {
                // Collect size/alignment info
                hkLocalArray<hkInt32> sizes( 10 );
                hkLocalArray<hkInt32> alignments( 10 );
                auto func = [this, &sizes, &alignments]( Var var )
                {
                    hkInt32 neededSize; computeNeededSizeFor( var, neededSize );
                    // Note: we don't care about the root var's alignment and size
                    if ( m_var != var )
                    {
                        sizes.pushBack( neededSize );
                        alignments.pushBack( var.getType()->getAlignOf() );
                    }
                };
                iterateOffsetVars( func );

                // Determine offsets
                localOffsetsOut.setSize( sizes.getSize() + 1 );
                hkInt32 sizeOfOffsetsTable = ( sizeof( hkUint16 ) * sizes.getSize() );
                hkInt32 localOffset = ( m_var.getType()->getSizeOf() + sizeOfOffsetsTable );
                for ( int i = 0; i < sizes.getSize(); i++ )
                {
                    hkInt32 size = sizes[i];
                    hkInt32 alignment = alignments[i];
                    localOffset = HK_NEXT_MULTIPLE_OF( alignment, localOffset );
                    localOffsetsOut[i] = hkUint16( localOffset );
                    localOffset += size;
                }

                // We keep this just for use in serialize
                localOffsetsOut[sizes.getSize()] = hkUint16( localOffset );

                // Return total size (space for size + data size with offsets).
                // We send the object unpacked; this is a cost/benefit on slightly more mem
                // over the network for improved writing/reading speed (no unpacking).
                return s32 + localOffset;
            }
        }

        hkResult serialize(
            hkStreamWriter& writer,
            hkCriticalSection* writeLock,
            hkArrayView<hkUint16> localOffsets )
        {
            hkVdbOStream ostream( &writer );
            if ( localOffsets.isEmpty() )
            {
                // Fast path
                if ( writeLock ) writeLock->enter();
                ostream.write32( m_var.getType()->getSizeOf() );
                ostream.writeRaw( m_var.getAddress(), m_var.getType()->getSizeOf() );
                if ( writeLock ) writeLock->leave();
            }
            else
            {
                HK_ASSERT(
                    0x22441330,
                    m_typeInfo &&
                    m_typeInfo->m_fieldOffsets.getSize() &&
                    ( m_typeInfo->m_fieldOffsets.getSize() == ( localOffsets.getSize() - 1 ) ) &&
                    ( localOffsets[localOffsets.getSize() - 1] != 0 ),
                    "Typeinfo is inconsistent with localOffsets provided" );

                // Size of data
                hkUint16 totalSize = localOffsets.back();
                ostream.write32( totalSize );
                localOffsets = localOffsets.rtrim( 1 );

                // Write objects
                hkInt32 localOffsetsIdx = 0;
                hkInt32 localWriteOffset = 0;
                auto func = [this, &localOffsetsIdx, &localWriteOffset, &ostream, &writeLock, &localOffsets]( Var var )
                {
                    if ( writeLock ) writeLock->enter();

                    // Start with padding if needed.
                    // We send the object unpacked; this is a cost/benefit on slightly more mem
                    // over the network for improved writing/reading speed (no unpacking).
                    if ( localWriteOffset != 0 )
                    {
                        hkUint16 requiredLocalOffset = localOffsets[localOffsetsIdx];
                        while ( localWriteOffset < requiredLocalOffset ) { ostream.write8( 0 ); localWriteOffset++; }
                        localOffsetsIdx++;
                    }

                    // Write the var's data
                    hkInt32 neededSize;
                    const void* dataAddress = computeNeededSizeFor( var, neededSize );
                    ostream.writeRaw( dataAddress, neededSize );

                    // Write our offsets table if this is the first time through
                    if ( localWriteOffset == 0 )
                    {
                        hkInt32 offsetsTableSize = ( sizeof( hkUint16 ) * localOffsets.getSize() );
                        ostream.writeArray16u( localOffsets.begin(), localOffsets.getSize() );
                        localWriteOffset = ( neededSize + offsetsTableSize );
                    }
                    else
                    {
                        localWriteOffset += neededSize;
                    }

                    if ( writeLock ) writeLock->leave();
                };
                iterateOffsetVars( func );
                HK_ASSERT( 0x22441331, localOffsetsIdx == localOffsets.getSize(), "Unexpected local offset iteration" );
                HK_ASSERT( 0x22441336, localWriteOffset == totalSize, "Did not reach expected write offset" );
            }
            return HK_SUCCESS;
        }

        hkResult reserialize(
            hkStreamWriter& writer,
            hkCriticalSection* writeLock )
        {
            hkVdbOStream ostream( &writer );
            int objectSize = getDeserializedObjectSize();
            if ( writeLock ) writeLock->enter();
            ostream.write32( objectSize );
            ostream.writeRaw( m_var.getAddress(), objectSize );
            if ( writeLock ) writeLock->leave();
            return HK_SUCCESS;
        }

        hkVdbReflect::Var deserialize(
            hkStreamReader& reader,
            hkMemoryAllocator* allocator,
            hkCriticalSection* readLock )
        {
            // To support this, we'd need to store the allocator* in the object so we can deallocate with the same*
            HK_ASSERT(
                0x22441345,
                ( allocator == hkMemHeapAllocator() ),
                "We do not yet support non-heap allocators" );

            hkVdbIStream istream( &reader );
            if ( readLock ) readLock->enter();
            int objectSize = istream.read32();
            // We reserve s32 to store size of allocation
            int totalSize = HK_NEXT_MULTIPLE_OF( 16, s32 ) + objectSize;
            void* object = allocator->blockAlloc( totalSize );
            int* totalSizeStorage = reinterpret_cast< int* >( object );
            *totalSizeStorage = totalSize;
            object = hkAddByteOffset( object, HK_NEXT_MULTIPLE_OF( 16, s32 ) );
            istream.readRaw( object, objectSize );
            if ( readLock ) readLock->leave();
            return Var( object, m_type );
        }

        inline int getDeserializedObjectSize()
        {
            void* memStart = hkAddByteOffset( m_var.getAddress(), -HK_NEXT_MULTIPLE_OF( 16, s32 ) );
            int memSize = *reinterpret_cast< int* >( memStart );
            int objectSize = memSize - HK_NEXT_MULTIPLE_OF( 16, s32 );
            return objectSize;
        }

        inline int getDeserializedNeededSize()
        {
            int objectSize = getDeserializedObjectSize();
            return s32 + objectSize;
        }

    protected:

        template<typename Functor>
        void iterateOffsetVars( Functor& func )
        {
            HK_ASSERT(
                0x22441251,
                m_typeInfo &&
                m_typeInfo->m_fieldOffsets.getSize() == m_typeInfo->m_fieldTypes.getSize(),
                "Iterating requires valid type info" );

            const Type* type = m_var.getType();
            if ( type->asRecord() )
            {
                // Do us first
                func( m_var );

                // Now do special vars
                for ( int i = 0; i < m_typeInfo->m_fieldOffsets.getSize(); i++ )
                {
                    hkUint16 fieldOffset = m_typeInfo->m_fieldOffsets[i];
                    const Type* fieldType = m_typeInfo->m_fieldTypes[i];
                    HK_ASSERT( 0x22441252, fieldType, "Must provide type for field" );

                    void* fieldAddress = hkAddByteOffset( m_var.getAddress(), fieldOffset );
                    Var var( fieldAddress, fieldType );
                    func( var );
                }
            }
        }

        const void* computeNeededSizeFor( Var var, hkInt32& sizeOut )
        {
            if ( StringVar svar = var )
            {
                static const char* emptystr = "";
                const char* strval = svar.getValue();
                int strlen = strval ? hkString::strLen( strval ) : 0;
                sizeOut = ( strlen + 1 ) * sizeof( const char );
                return strval ? strval : emptystr;
            }
            
            
            
            
            
            
            
            else if ( RecordVar rvar = var )
            {
                sizeOut = rvar.getType()->getSizeOf();
                return rvar.getAddress();
            }
            
            
            
            
            
            
            
            else
            {
                HK_ASSERT( 0x22441306, false, "This type is not supported by local offset system" );
                sizeOut = 0;
                return HK_NULL;
            }
        }

        Var m_var;
        const Type* m_type;
        const hkVdbReflect::TypeInfo* m_typeInfo;
    };

//#pragma endregion
};

int hkVdbReflect::Var::compare( const hkVdbReflect::Var& b ) const
{
    if ( m_type->asRecord() )
    {
        // Use our special iterator that skips unsupported types.
        return compareInternal<hkVdbReflect::Var>( *this, b );
    }
    else if ( m_type->asArray() )
    {
        // Use the default VarIter.
        hkReflect::ArrayVar aArray = *this;
        hkReflect::ArrayVar bArray = b;
        if ( aArray && bArray )
        {
            return compareInternal<hkReflect::ArrayVar>( aArray, bArray );
        }
    }

    // Note: comparisons against opaque succeed since we stub
    // these out when going across the network
    if ( m_type->asOpaque() || b.getType()->asOpaque() )
    {
        return 0;
    }
    else
    {
        return hkReflect::Var::compare( b );
    }
}

//#pragma region hkVdbSerialize

#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
hkVdbSerialize::TypeSave::TypeSave()
{
    withTarget( HK_NULL );
    withMultiBundle();
}

hkVdbSerialize::TypeLoad::TypeLoad()
{
    withIgnoreInvalid();
    withTypeReg( &VdbObjectTypeReg::s_instance );
}
#else
hkVdbSerialize::TypeSave::TypeSave() :
    hkSerialize::TagfileWriteFormat( (char*) HK_NULL )
{
    enableMultiBundle();
}

hkVdbSerialize::TypeLoad::TypeLoad()
{}
#endif

//#pragma region hkVdbSerialize (server)

bool hkVdbSerialize::isTypeValid( const hkVdbReflect::Type* recordType )
{
    return ( recordType && VdbObjectTypeBuilder( recordType, false ).verify() );
}
bool hkVdbSerialize::isSubTypeValid( const hkVdbReflect::Type* type )
{
    return ( type && VdbObjectTypeBuilder( type, true ).verify() );
}
hkResult hkVdbSerialize::serializeType(
    hkStreamWriter& writer,
    hkVdbSerialize::TypeSave& saver,
    const hkReflect::Type* type,
    hkVdbReflect::TypeInfo& typeInfoOut,
    hkCriticalSection* writeLock )
{
    if ( !type )
    {
        HK_ASSERT( 0x22441411, false, "No type provided to serialize" );
        return HK_FAILURE;
    }

    // hkSerialize::Save/BundleBuilder does not support non-seek-tell supported writers
    // see COM-4220
    hkArray<char> nonSeekTellSupportedBackBuffer;
    hkIo::WriteBuffer typeInformationWriteBuffer;
    if ( writer.seekTellSupported() )
    {
        typeInformationWriteBuffer.attach( &writer );
    }
    else
    {
        nonSeekTellSupportedBackBuffer.reserve( 1000 );
        typeInformationWriteBuffer.attach( &nonSeekTellSupportedBackBuffer );
    }

    // Write our type to the buffer
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
    saver.contentsPtr( type, typeInformationWriteBuffer );
#else
    // BundleBuilder needs to be destroyed to flush writes.
    {
        struct NoPointersIdFromVar : public BundleBuilder
        {
            NoPointersIdFromVar( WriteFormat* format, hkIo::WriteBuffer* buf ) : BundleBuilder( format, buf ) {}
            virtual VarId pointer( const PointerVar& ptr ) HK_OVERRIDE { return 0; }
        };

        NoPointersIdFromVar bb( &saver, &typeInformationWriteBuffer );
        bb.add( type );
    }
#endif

    // If successful, build info and write to output
    if ( typeInformationWriteBuffer.tell() )
    {
        if ( !VdbObjectTypeBuilder( type, false ).buildInfo( typeInfoOut ) )
        {
            HK_WARN( 0x22441323, "Type " << type->getName() << " has properties that will be ignored by the VDB" );
        }

        HK_ASSERT_NO_MSG(
            0x22441322,
            ( typeInfoOut.m_fieldOffsets.getSize() == typeInfoOut.m_fieldTypes.getSize() ) );

        if ( writeLock ) writeLock->enter();
        {
            // Finalize the buffer size, flushing writes to the underlying stream
            typeInformationWriteBuffer.detach();
            if ( nonSeekTellSupportedBackBuffer.getSize() )
            {
                writer.write( nonSeekTellSupportedBackBuffer.begin(), nonSeekTellSupportedBackBuffer.getSize() );
            }
        }
        if ( writeLock ) writeLock->leave();

        return HK_SUCCESS;
    }
    else
    {
        HK_WARN( 0x22441327, "Type " << type->getName() << " could not be serialized" );
        return HK_FAILURE;
    }
}
hkUint32 hkVdbSerialize::computeObjectSize(
    hkReflect::Var object,
    const hkVdbReflect::TypeInfo& typeInfo,
    hkArray<hkUint16>& localOffsetsOut )
{
    return VdbObjectSerializer( object, typeInfo ).computeNeededSize( localOffsetsOut );
}
hkResult hkVdbSerialize::serializeObject(
    hkStreamWriter& writer,
    hkReflect::Var object,
    const hkVdbReflect::TypeInfo& typeInfo,
    hkArrayView<hkUint16> localOffsets,
    hkCriticalSection* writeLock )
{
    return VdbObjectSerializer( object, typeInfo ).serialize( writer, writeLock, localOffsets );
}

//#pragma endregion

//#pragma region hkVdbSerialize (client)

const hkVdbReflect::Type* hkVdbSerialize::deserializeType(
    hkStreamReader& reader,
    hkVdbSerialize::TypeLoad& loader,
    hkMemoryAllocator* allocator,
    hkCriticalSection* readLock,
    const hkReflect::Type** serverTypeOut )
{
    hkIo::ReadBuffer readBuffer( &reader );

    // Read our type bundle
#ifdef HK_VDB_USE_HK_TYPE_SERIALIZE
    if ( readLock ) readLock->enter();
    hkReflect::Var var = loader.toVar( readBuffer );
    if ( readLock ) readLock->leave();
#else
    if ( readLock ) readLock->enter();
    hkViewPtr<Bundle> bundle = loader.read( readBuffer );
    if ( readLock ) readLock->leave();
    if ( !bundle )
    {
        HK_ASSERT( 0x22441324, false, "Could not read data for type" );
        return HK_NULL;
    }

    // Get items from our bundle.
    // The first var is always HK_NULL as a reserve, so we should have two vars.
    hkArray<Bundle::Item> items;
    bundle->getItems( items );
    if ( ( items.getSize() != 2 ) || ( items[0].var().getAddress() != HK_NULL ) )
    {
        HK_ASSERT( 0x22441325, false, "Data for type is not in expected format" );
        return HK_NULL;
    }

    // Get our type var.
    hkReflect::Var var = items[1].var();
#endif
    if ( !var )
    {
        HK_ASSERT( 0x22441324, false, "Could not read type" );
        return HK_NULL;
    }

    // Get our runtime type.
    Type* type = hkDynCast( var );
    if ( !type )
    {
        HK_ASSERT( 0x22441326, false, "Runtime type is not available" );
        return HK_NULL;
    }

    // Build the client type
    if ( type )
    {
        if ( serverTypeOut ) *serverTypeOut = type;
        return VdbObjectTypeBuilder( type, false ).buildType( allocator ? allocator : hkMemHeapAllocator() );
    }
    else
    {
        return HK_NULL;
    }
}
hkUint32 hkVdbSerialize::getObjectSize(
    hkVdbReflect::Var clientObject )
{
    return VdbObjectSerializer( clientObject ).getDeserializedNeededSize();
}
hkVdbReflect::Var hkVdbSerialize::deserializeObject(
    hkStreamReader& reader,
    const Type* clientType,
    hkMemoryAllocator* allocator,
    hkCriticalSection* readLock )
{
    return VdbObjectSerializer( clientType ).deserialize( reader, allocator ? allocator : hkMemHeapAllocator(), readLock );
}
hkResult hkVdbSerialize::reserializeObject(
    hkStreamWriter& writer,
    hkVdbReflect::Var clientObject,
    hkCriticalSection* writeLock )
{
    return VdbObjectSerializer( clientObject ).reserialize( writer, writeLock );
}

//#pragma endregion

//#pragma endregion

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
