// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

class hkTriangleIndices;
class hkDisplayGeometry;
class hkAabb;

/// This utility class extends the hkOArchive to provide a endian
/// safe stream interface for serializing all of the basic data
/// types associated with the visual debugger.
class HK_EXPORT_COMMON hkVdbOStream : public hkOArchive
{
public:
    HK_DECLARE_CLASS( hkVdbOStream, New );

    /// Create with the specified hkStreamWriter.
    /// Note: hkVdbOStream is not safe to use with non-blocking streams.
    hkVdbOStream( _Inout_ hkStreamWriter* sb, bool ensureBuffered = false, bool byteswap = HK_ENDIAN_BIG );

    /// Create with a stream from the hkStreamWriter factory.
    /// The file is truncated.
    hkVdbOStream( _In_z_ const char* filename, bool byteswap = HK_ENDIAN_BIG );

    /// Create with a memory block. 'mem' must exist for the lifetime of this object.
    hkVdbOStream( _Out_writes_bytes_( memSize ) void* mem, int memSize, bool byteswap = HK_ENDIAN_BIG );

    /// Creates with an expanding memory block. 'arr' must exist for the lifetime of this object.
    hkVdbOStream( hkArray<char>& arr, bool byteswap = HK_ENDIAN_BIG );

    /// Serialize an id that persists over more than one frame.
    void writePersistentId( hkInt64 id );

    /// Serialize an id that persists over more than one frame.
    void writePersistentId( hkUint64 id );

    /// Serialize an hkVector4.
    void writeQuadVector4( const hkVector4& v );

    /// Serialize an hkTransform.
    void writeTransform( const hkTransform& t );

    /// Serialize an hkMatrix4.
    void writeMatrix( const hkMatrix4& m );

    /// Serialize an hkGeometry::Triangle.
    void writeTriangle( const hkGeometry::Triangle& ti );

    /// Serialize an hkGeometry.
    void writeGeometry( const hkGeometry& g );

    /// Serialize an hkArray of hkVector4s
    void writeVectorArray( const hkArrayBase<hkVector4>& v );

    /// Serialize an hkDisplayGeometry.
    void writeDisplayGeometry( _Inout_ hkDisplayGeometry* dg );

    /// Serialize an hkAabb.
    void writeAabb( const hkAabb& aabb );

    /// Before serializing an arbitrary display geom, compute the bytes it will take
    /// up so that we don't have to buffer it on the Server before making a packet out of it.
    static hkUint32 computeBytesRequired( const hkGeometry& g );
    static hkUint32 computeBytesRequired( const hkArrayBase<hkVector4>& v );
    static hkUint32 computeBytesRequired( _Inout_ hkDisplayGeometry* dg );
};

/// An hkVdbOStream meant to be used locally (on the stack).
/// It's optimized for this case to reduce ctor/dtor overhead.
struct HK_EXPORT_COMMON hkVdbLocalOStream : public hkVdbOStream
{
    HK_DECLARE_CLASS( hkVdbLocalOStream, New );
    hkVdbLocalOStream( hkArray<char>& buffer, bool byteswap = HK_ENDIAN_BIG ) :
        hkVdbOStream( ( hkStreamWriter* ) HK_NULL, byteswap ),
        m_inplaceWriter( &buffer, hkArrayStreamWriter::ARRAY_BORROW )
    {
        setStreamWriter( &m_inplaceWriter );
    }

protected:
    hkArrayStreamWriter m_inplaceWriter;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
