// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshBody.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Visualize/Shape/hkDisplayAABB.h>
#include <Common/Visualize/Shape/hkDisplayBox.h>
#include <Common/Visualize/Shape/hkDisplayCapsule.h>
#include <Common/Visualize/Shape/hkDisplayCone.h>
#include <Common/Visualize/Shape/hkDisplayCylinder.h>
#include <Common/Visualize/Shape/hkDisplayMesh.h>
#include <Common/Visualize/Shape/hkDisplayPlane.h>
#include <Common/Visualize/Shape/hkDisplaySemiCircle.h>
#include <Common/Visualize/Shape/hkDisplaySphere.h>
#include <Common/Visualize/Shape/hkDisplayTaperedCapsule.h>
#include <Common/Visualize/Shape/hkDisplayWireframe.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>

hkVdbIStream::hkVdbIStream( _Inout_ hkStreamReader* sb, bool bs )
    : hkIArchive( sb, bs )
{}

hkVdbIStream::hkVdbIStream( _In_z_ const char* filename, bool bs )
    : hkIArchive( filename, bs )
{}

hkVdbIStream::hkVdbIStream( _In_bytecount_( memSize ) const void* mem, int memSize, bool byteswap )
    : hkIArchive( mem, memSize, byteswap )
{}

void hkVdbIStream::readQuadVector4( hkVector4& v )
{
    readArrayFloat32( &v( 0 ), 3 );
    v( 3 ) = 0.0f;  // initialize element that is not transmitted
}

void hkVdbIStream::readTransform( hkTransform& t )
{
    //  readFloats(&t.getRotation().getColumn(0)(0), 16);

    //*
    // get the position
    hkVector4 position;
    readArrayFloat32( &position( 0 ), 3 );
    position( 3 ) = 1.0f;

    // get the orientation
    hkQuaternion orientation;
    readArrayFloat32( const_cast<hkReal*>( &orientation( 0 ) ), 4 );

    // setup the transform
    t.setTranslation( position );
    t.setRotation( orientation );
    //*/
}

void hkVdbIStream::readMatrix( hkMatrix4& m )
{
    readArrayFloat32( &m.getColumn( 0 )( 0 ), sizeof( hkMatrix4 ) / sizeof( hkReal ) );
}

void hkVdbIStream::readTriangle( hkGeometry::Triangle& ti )
{
    ti.m_a = read32();
    ti.m_b = read32();
    ti.m_c = read32();
}

void hkVdbIStream::readVectorArray( hkArray<hkVector4>& v )
{
    // extract the array of vertices
    int numVertices = read32();

    int startIndex = v.getSize();
    v.setSize( startIndex + numVertices );
    for ( int i = 0; i < numVertices; i++ )
    {
        readQuadVector4( v[startIndex + i] );
    }
}

void hkVdbIStream::readGeometry( hkGeometry& g )
{
    readVectorArray( g.m_vertices );

    // extract the array of triangle indicies
    {
        int numTriangles = read32();

        int startIndex = g.m_triangles.getSize();
        g.m_triangles.setSize( startIndex + numTriangles );
        for ( int i = 0; i < numTriangles; i++ )
        {
            readTriangle( g.m_triangles[startIndex + i] );
        }
    }
}

void hkVdbIStream::readDisplayGeometry( _Outref_result_maybenull_ hkDisplayGeometry*& dg )
{
    // grab the type information
    char t_asChar = read8();
    hkDisplayGeometryType t = ( hkDisplayGeometryType ) t_asChar;

    // for grabbing the transform
    hkTransform transform;

    // grab the specific information pertaining to this geometry
    switch ( t )
    {
        case HK_DISPLAY_ANY:
        {
            readTransform( transform );
            // read in the geometry
            hkGeometry* geometry = new hkGeometry();
            readGeometry( *geometry );

            dg = new hkDisplayGeometry( geometry );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_SPHERE:
        {
            readTransform( transform );
            // read in the sphere and the tesselation resolution
            hkReal radius = readFloat32();

            hkVector4 position;
            readQuadVector4( position );
            hkSphere sphere( position, radius );

            int subdivisions = read32();
            read32();   //There used to be two tesselation parameters.

            dg = new hkDisplaySphere( sphere, subdivisions );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_CAPSULE:
        {
            readTransform( transform );
            hkReal radius = readFloat32();

            hkVector4 top;
            readQuadVector4( top );
            hkVector4 bottom;
            readQuadVector4( bottom );

            int numSides = read32();
            int numHeightSegments = read32();

            dg = new hkDisplayCapsule( top, bottom, radius, numSides, numHeightSegments );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_TAPERED_CAPSULE:
        {
            readTransform( transform );
            hkReal bottomRadius = readFloat32();
            hkReal topRadius = readFloat32();

            hkVector4 top;
            readQuadVector4( top );
            hkVector4 bottom;
            readQuadVector4( bottom );

            int numSides = read32();
            int numHeightSegments = read32();

            dg = new hkDisplayTaperedCapsule( top, bottom, topRadius, bottomRadius, numSides, numHeightSegments );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_CYLINDER:
        {
            readTransform( transform );
            hkReal radius = readFloat32();

            hkVector4 top;
            readQuadVector4( top );
            hkVector4 bottom;
            readQuadVector4( bottom );

            int numSides = read32();
            int numHeightSegments = read32();

            dg = new hkDisplayCylinder( top, bottom, radius, numSides, numHeightSegments );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_BOX:
        {
            readTransform( transform );
            hkVector4 halfExtents; readQuadVector4( halfExtents );

            dg = new hkDisplayBox( halfExtents );
            dg->setTransform( transform );
        }
        break;

        case HK_DISPLAY_AABB:
        {
            // no transform for this type
            hkVector4 minExtent;
            hkVector4 maxExtent;
            readQuadVector4( minExtent );
            readQuadVector4( maxExtent );
            dg = new hkDisplayAABB( minExtent, maxExtent );
        }
        break;

        case HK_DISPLAY_CONE:
        {
            hkVector4 position;
            hkVector4 axis;
            hkReal angle;
            hkReal height;
            int numSegments;

            readQuadVector4( position );
            readQuadVector4( axis );
            angle = readFloat32();
            height = readFloat32();
            numSegments = read32();

            dg = new hkDisplayCone( angle, height, numSegments, axis, position );
        }
        break;

        case HK_DISPLAY_SEMICIRCLE:
        {
            //hkDisplaySemiCircle* dsemi = static_cast<hkDisplaySemiCircle*>(dg);
            hkVector4 center;
            hkVector4 normal;
            hkVector4 perp;
            hkReal radius;
            hkReal thetaMin;
            hkReal thetaMax;
            int numSegments;

            readQuadVector4( center );
            readQuadVector4( normal );
            readQuadVector4( perp );
            radius = readFloat32();
            thetaMin = readFloat32();
            thetaMax = readFloat32();
            numSegments = read32();

            dg = new hkDisplaySemiCircle( center, normal, perp,
                thetaMin, thetaMax, radius,
                numSegments );
        }
        break;

        case HK_DISPLAY_PLANE:
        {
            //hkDisplayPlane* dplane = static_cast<hkDisplayPlane*>(dg);
            hkVector4 center;
            hkVector4 normal;
            hkVector4 perpToNormal;
            hkReal extent;

            readQuadVector4( center );
            readQuadVector4( normal );
            readQuadVector4( perpToNormal );
            extent = readFloat32(); //XX should be 3 extents but need to change protocol to add it.
            hkVector4 extent3; extent3.setXYZ( extent );
            dg = new hkDisplayPlane( normal, perpToNormal, center, extent3 );
        }
        break;

        case HK_DISPLAY_MESH:
        {
            hkArray<char> meshAsTagfile;
            {
                int tagfileSize = read32();
                meshAsTagfile.setSize( tagfileSize );
                readRaw( meshAsTagfile.begin(), tagfileSize );
            }

            hkObjectResource* resource = hkSerializeUtil::loadOnHeap( meshAsTagfile.begin(), meshAsTagfile.getSize() );
            if ( resource )
            {
                hkMeshBody* meshBody = resource->getContents<hkMemoryMeshBody>();
                meshBody->addReference();
                resource->removeReference();

                dg = new hkDisplayMesh( meshBody );
                meshBody->removeReference();
            }
            else
            {
                dg = HK_NULL;
            }

        }
        break;

        case HK_DISPLAY_WIREFRAME:
        {
            readTransform( transform );

            hkDisplayWireframe* wf = new hkDisplayWireframe();
            // read in the lines
            readVectorArray( wf->m_lines );

            dg = wf;
            dg->setTransform( transform );
        }
        break;

        default:
            HK_ASSERT( 0x4739a00e, 0, "Display stream corrupt or unsupported display geometry received" );
    }
}

void hkVdbIStream::readAabb( hkAabb& aabb )
{
    readQuadVector4( aabb.m_min );
    readQuadVector4( aabb.m_max );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
