// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Visualize/hkVisualize.h>

#include <Common/Base/System/Stopwatch/hkSystemClock.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamParser.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Visualize/hkProcessContext.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Visualize/Process/hkStatisticsProcess.h>
#include <Common/Visualize/Serialize/hkVdbOStream.h>
#include <Common/Visualize/hkProcessSerializeUtils.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Visualize/hkCommonProcessContext.h>

using namespace hkVdbCmdUtils;

int hkStatisticsProcess::s_tag = 0;
hkUint32 hkStatisticsProcess::s_nextSessionId = 1;

_Ret_notnull_ hkProcess* HK_CALL hkStatisticsProcess::create(const hkArray<hkProcessContext*>& contexts)
{
    return new hkStatisticsProcess( contexts ); // doesn't require a context (the monitors are global)
}

void HK_CALL hkStatisticsProcess::registerProcess()
{
    s_tag = hkProcessFactory::getInstance().registerProcess( getName(), create );
}

hkStatisticsProcess::hkStatisticsProcess( const hkArray<hkProcessContext*>& contexts )
    : m_context(nullptr)
{
    for(hkProcessContext* context : contexts)
    {
        if(hkString::strCmp(HK_COMMON_PROCESS_CONTEXT_TYPE_STRING, context->getType()) == 0)
        {
            m_context = static_cast<hkCommonProcessContext*>(context);
            break;
        }
    }

    m_sessionId = s_nextSessionId++;
}

hkStatisticsProcess::~hkStatisticsProcess()
{}

void hkStatisticsProcess::step( hkReal frameTimeInMs )
{
    if ( !m_outStream )
        return; // nothing to write to

    // see if we have any per thread timers:
    hkArrayView<hkTimerData> timers;
    if(m_context)
    {
        timers = m_context->m_monitorStreams;
    }

    hkTimerData localTimers;
    if ( timers.getSize() == 0 )
    {
        // send main thread timer data
        hkMonitorStream& stream = hkMonitorStream::getInstance();
        localTimers.m_streamBegin = stream.getStart();
        localTimers.m_streamEnd = stream.getEnd();
        timers = hkArrayView<hkTimerData>( &localTimers, 1 );
    }

    writeOutStatsBuffers( timers, hkVdbCmdType::SEND_PERF_STATS );
    sendMemoryStats();
}

void hkStatisticsProcess::writeOutStatsBuffers( hkArrayView<hkTimerData> timers, hkVdbCmdType::Enum commandType )
{
    hkUlong totalMonLen = 0;
    int numStreams = timers.getSize();
    for ( int thr = 0; thr < numStreams; ++thr )
    {
        totalMonLen += timers[thr].m_streamEnd - timers[thr].m_streamBegin;
    }

    if ( totalMonLen == 0 )
    {
        return;
    }

    // Send stat maps
    {
        m_stringMapBuffer.setSize( 0 );
        m_typeMapBuffer.setSize( 0 );

        hkMonitorStream::CommandStreamConfig config; // local config
        hkMonitorStreamParserUtil::extractMapsFromStreams( config, timers, m_stringMap, m_typeMap );

        if ( m_stringMap.m_map.getSize() == 0 )
            return;

        if ( hkProcessSerializeUtils::varToBuffer( m_format, &m_stringMap, m_stringMapBuffer ).isFailure() )
            return;

        if ( hkProcessSerializeUtils::varToBuffer( m_format, &m_typeMap, m_typeMapBuffer ).isFailure() )
            return;

        int strBufSize = m_stringMapBuffer.getSize();
        int typeBufSize = m_typeMapBuffer.getSize();

        const int mapsPacketSize = s8 /*command id*/ + s32 /*session*/ +
            ( strBufSize + s32 /*len int*/ ) + ( typeBufSize + s32 /*len int*/ );

        m_outStream->write32( mapsPacketSize );
        m_outStream->write8u( hkVdbCmdType::SEND_STATS_MAPS );

        // m_sessionId so that we know on other end when a new reader is needed for types etcc
        m_outStream->write32u( m_sessionId );

        // The string map
        m_outStream->write32u( strBufSize );
        if ( strBufSize > 0 )
            m_outStream->writeRaw( m_stringMapBuffer.begin(), strBufSize );

        // The type map
        m_outStream->write32u( typeBufSize );
        if ( typeBufSize > 0 )
            m_outStream->writeRaw( m_typeMapBuffer.begin(), typeBufSize );
    }

    // Send the rest
    const int packetSize = s8 /*command id*/ + s8 + s8 /*config*/ + sF /*float*/ +
        ( ( int ) totalMonLen + ( s32 * numStreams /*stream len*/ ) + s32 /*num streams*/ );

    m_outStream->write32( packetSize );
    m_outStream->write8u( hkUint8( commandType ) );

    // config
    #if HK_ENABLE_64BIT_TIMERS
    m_outStream->write8u( 64 );
#else
    m_outStream->write8u( 32 );
#endif

#if HK_ENABLE_EXTRA_TIMER
    m_outStream->write8u( 2 );
#else
    m_outStream->write8u( 1 );
#endif

    m_outStream->writeFloat32( 1e3f / hkSystemClock::getTicksPerSecond() );

    // The large data stream(s):
    // num streams * [streamlen, stream data]
    m_outStream->write32u( numStreams );
    for ( int si = 0; si < numStreams; ++si )
    {
        hkUint32 monLen = ( hkUint32 ) ( timers[si].m_streamEnd - timers[si].m_streamBegin );
        m_outStream->write32u( monLen );
        if ( monLen > 0 )
            m_outStream->writeRaw( timers[si].m_streamBegin, monLen );
    }
}

void hkStatisticsProcess::sendMemoryStats()
{
    hkMemorySystem::MemoryStatistics memStats;
    hkMemorySystem::getInstance().getMemoryStatistics( memStats );
    const int numEntries = memStats.m_entries.getSize();

    if ( numEntries )
    {
        int numCharacters = 0;
        for ( int i = 0; i < numEntries; i++ )
        {
            // +1 for null terminator
            numCharacters += ( hkString::strLen( memStats.m_entries[i].m_allocatorName ) + 1 );
        }

        const int packetSize = s8 + s16 + ( s8 * numCharacters ) + ( s64 * 6 * numEntries );
        m_outStream->write32( packetSize );

        m_outStream->write8u( hkVdbCmdType::SEND_MEM_STATS );
        m_outStream->write16u( hkUint16( memStats.m_entries.getSize() ) );
        for ( int i = 0; i < memStats.m_entries.getSize(); i++ )
        {
            hkMemorySystem::MemoryStatistics::Entry& entry = memStats.m_entries[i];
            {
                m_outStream->writeString( entry.m_allocatorName );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_allocated ) );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_inUse ) );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_peakInUse ) );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_available ) );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_totalAvailable ) );
                m_outStream->write64( hkInt64( entry.m_allocatorStats.m_largestBlock ) );
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
