// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkProcess.h>
#include <Common/Base/Container/Hash/hkHashMap.h>

//////////////////////////////////////////////////////////////////////////
// This class is only used in limited circumstances where normal object
// inspection limitations (as described in hkVdbSerialize) are unacceptable.
// It should not be used and may be going away in the near future.
//////////////////////////////////////////////////////////////////////////

class hkProcessContext;
class hkRemoteObjectProcess;
namespace hkSerialize
{
    class Save;
    class Load;
}




    /// Interface to receive a notification when objects are sent to the server from
    /// a client.
class HK_EXPORT_COMMON hkRemoteObjectServerSideListener
{
    public:
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkRemoteObjectServerSideListener);
        virtual ~hkRemoteObjectServerSideListener() {}

            /// Callback raised when an object is received from a client.
        virtual void receiveObjectCallback( _Inout_ hkReferencedObject* object ) = 0;
};

    /// Interface to receive a notification when objects are sent to a client from the
    /// server. Also used to send objects to the server from a client.
class HK_EXPORT_COMMON hkRemoteObjectClientSideListener
{
    public:
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkRemoteObjectClientSideListener);
        virtual ~hkRemoteObjectClientSideListener() {}

            /// Declaration for function type used to send objects from the client to the
            /// server. This matches hkVdbSendObject which is used in the hkVdbClient.
        typedef void (HK_CALL *SendObject)(_Inout_ hkReferencedObject* object);

            /// Initializes the listener with a SendObject function. The function
            /// should be hkVdbSendObject in the VdbClient library.
        hkRemoteObjectClientSideListener( SendObject sendFunc ): m_sendFunc(sendFunc) {}

            /// Callback raised when an object is received from the VDB server.
        virtual void receiveObjectCallback(_Inout_ hkReferencedObject* object ) = 0;
            /// Sends and object via the SendObject function used to initialize the listener.
        void sendObject(_Inout_ hkReferencedObject* object );

    private:
            /// Function to send an object.
        SendObject m_sendFunc;
};

    /// The hkRemoteObjectProcess sends hkReferencedObjects between the VDB server and it's
    /// clients. Objects are sent bidirectionally as tagfiles and loaded onto the heap after
    /// receipt. Notifications of new objects are sent via hkRemoteObjectServerSideListeners
    /// which are be attached to the process.
class HK_EXPORT_COMMON hkRemoteObjectProcess : public hkReferencedObject, public hkProcess
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_VDB);

            /// Creates the process.
        hkRemoteObjectProcess();
            /// Destructs the process.
        virtual ~hkRemoteObjectProcess();

            /// Adds a server side listener to the process.
        void addListener(_In_ hkRemoteObjectServerSideListener* listener );
            /// Removes a server side listener from the process.
        void removeListener(_In_ hkRemoteObjectServerSideListener* listener );

            /// Determines how the data is serialized for sendObject()
        enum SendObjectType
        {
                /// Write data to a packfile
            SEND_OBJECT_PACKFILE,
                /// Write data to a tagfile
            SEND_OBJECT_TAGFILE
        };

            /// Sends an object from the server to all clients using the hkRemoteObjectProcess protocol.
        static void sendObject( _Inout_opt_ hkVdbOStream* stream, hkSerialize::Save& serializer, const hkReflect::Var& object, int tag );
        static void writeObjectDataOnly( _Inout_opt_ hkVdbOStream* stream, hkSerialize::Save& serializer, const hkReflect::Var& object, int tag );

            /// hkProcess interface
        virtual void init() HK_OVERRIDE;
            /// hkProcess interface
        virtual void getConsumableCommands(_Outref_result_bytebuffer_maybenull_(numCommands) hkUint8*& commands, _Out_ int& numCommands ) HK_OVERRIDE;
            /// hkProcess interface
        virtual void consumeCommand( hkUint8 command ) HK_OVERRIDE;
            /// hkProcess interface
        virtual int getProcessTag() const HK_OVERRIDE { return s_tag; }

            /// Creates the instances of the process. Invoked by the hkProcessFactory.
        static _Ret_notnull_ hkProcess* HK_CALL create(const hkArray<hkProcessContext*>& contexts);
            /// Registers the process with hkProcessFactory.
        static void HK_CALL registerProcess();
            /// Gets the name of the process
        static inline _Ret_z_ const char* HK_CALL getName() { return "RemoteObject"; }

    protected:

            /// Array of listeners that are notified when objects are received.
        hkArray<hkRemoteObjectServerSideListener*> m_listeners;

            /// Map of viewer tag to deserializer
        hkHashMap<int, hkSerialize::Load*> m_deserializers;

            /// Unique ID for the process. Assigned when the process is registered.
        static int s_tag;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
