// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/VisualDebugger/Viewer/hkxSceneViewer.h>
#include <Common/GeometryUtilities/Mesh/Converters/SceneDataToMesh/hkSceneDataToMeshConverter.h>
#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshSystem.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Visualize/Shape/hkDisplayMesh.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>

int hkxSceneViewer::s_tag = 0;

_Ret_maybenull_
hkProcess* HK_CALL hkxSceneViewer::create( const hkArray<hkProcessContext*>& contexts )
{
    hkxSceneDataContext* context = HK_NULL;

    for ( hkInt32 i = 0; i < contexts.getSize(); ++i )
    {
        if ( hkString::strCmp( contexts[i]->getType(), HK_SCENE_DATA_CONTEXT_TYPE_STRING ) == 0 )
        {
            context = static_cast<hkxSceneDataContext*>(contexts[i]);
        }
    }

    return (context != HK_NULL ? new hkxSceneViewer( context ) : HK_NULL );
}

void HK_CALL hkxSceneViewer::registerViewer()
{
    s_tag = hkProcessFactory::getInstance().registerProcess( getName(), create );
}

hkxSceneViewer::hkxSceneViewer(_Inout_ hkxSceneDataContext* context)
    : m_context( context )
{
    HK_ASSERT(0x5d5b8242, m_context != HK_NULL, "Cannot initialize viewer with null context." );

    m_context->addListener( this );
}

hkxSceneViewer::~hkxSceneViewer()
{
    for ( hkInt32 i = 0; i < m_context->getScenes().getSize(); ++i )
    {
        sceneRemovedCallback( m_context->getScenes()[i] );
    }

    // clear out the texture paths
    if ( m_displayHandler )
    {
        if ( hkDeprecatedDebugDisplayHandlerInterface* depInterface = m_displayHandler->getDeprecatedInterface() )
        {
            depInterface->clearTextureSearchPaths();
        }
    }

    m_context->removeListener( this );
}

void hkxSceneViewer::sceneAddedCallback(_Inout_ hkxScene* scene)
{
    // Initialize memory mesh system to hold onto converted data
    hkMemoryMeshSystem meshSystem;

    hkArray< hkRefPtr< hkxNode > > nodesWithMeshes;
    hkxSceneUtils::findAllMeshNodes( scene, scene->m_rootNode, nodesWithMeshes );

    // Only create dummy nodes if the root node is HK_NULL.  This can happen if the scene nodes were
    // pruned but skins and meshes were left in.
    if ( scene->m_rootNode == HK_NULL )
    {
        for ( hkInt32 i = 0; i < scene->m_meshes.getSize(); ++i )
        {
            hkxNode* dummyNode = new hkxNode();
            {
                dummyNode->m_object.set( scene->m_meshes[i].val() );
                dummyNode->m_keyFrames.pushBack( hkMatrix4::getIdentity() );
            }

            nodesWithMeshes.pushBack( dummyNode );
        }
    }

    // Add all meshes to the debug display
    for ( hkInt32 i = 0; i < nodesWithMeshes.getSize(); ++i )
    {
        // Get the node
        hkxNode* node = nodesWithMeshes[i];

        // Get the mesh associated with the node
        hkxMesh* mesh = hkxSceneUtils::getMeshFromNode( nodesWithMeshes[i] );

        // Convert mesh to hkMeshShape
        hkMeshShape* genericMeshShape = hkSceneDataToMeshConverter::convert( &meshSystem, HK_NULL, hkMatrix4::getIdentity(), mesh, m_context->getAllowTextureMipmap() );

        // Wrap in hkMeshBody
        hkMeshBody* genericMeshBody = meshSystem.createBody( genericMeshShape, nodesWithMeshes[i]->m_keyFrames[0], HK_NULL );
        genericMeshShape->removeReference();

        // Add to the debug display handler
        if ( isLocalViewer() || !m_displayHandler->getDeprecatedInterface() )
        {
            hkDisplayMesh* geometry = new hkDisplayMesh( genericMeshBody );
            m_displayHandler->addGeometry( hkUint64( node ), geometry, hkTransform::getIdentity(), s_tag );
            geometry->removeReference();
        }
        else
        {
            m_displayHandler->getDeprecatedInterface()->addGeometryHash(
                genericMeshBody,
                hkUint64( mesh ),
                hkUint64( node ),
                s_tag );
        }
        genericMeshBody->removeReference();
    }
}

void hkxSceneViewer::sceneRemovedCallback(_Inout_ hkxScene* scene)
{
    hkArray< hkRefPtr< hkxNode > > nodesWithMeshes;
    hkxSceneUtils::findAllMeshNodes( scene, scene->m_rootNode, nodesWithMeshes );

    // Only create dummy nodes if the root node is HK_NULL.  This can happen if the scene nodes were
    // pruned but skins and meshes were left in.
    if ( scene->m_rootNode == HK_NULL )
    {
        for ( hkInt32 i = 0; i < scene->m_meshes.getSize(); ++i )
        {
            hkxNode* dummyNode = new hkxNode();
            {
                dummyNode->m_object.set( scene->m_meshes[i].val() );
                dummyNode->m_keyFrames.pushBack( hkMatrix4::getIdentity() );
            }

            nodesWithMeshes.pushBack( dummyNode );
        }
    }

    // Remove all meshes from the debug display
    for ( hkInt32 i = 0; i < nodesWithMeshes.getSize(); ++i )
    {
        // Get the node
        hkxNode* node = nodesWithMeshes[i];

        // Remove from the debug display handler
        m_displayHandler->removeGeometry( hkUint64( node ) );
    }
}

void hkxSceneViewer::init()
{
    if ( m_context )
    {
        if ( m_displayHandler )
        {
            if ( hkDeprecatedDebugDisplayHandlerInterface* depInterface = m_displayHandler->getDeprecatedInterface() )
            {
                const hkArray<const char*>& searchPaths = m_context->getTextureSearchPaths();
                for ( int texturePathsIter = 0; texturePathsIter < searchPaths.getSize(); texturePathsIter++ )
                {
                    depInterface->addTextureSearchPath( searchPaths[texturePathsIter] );
                }
            }
        }

        for ( hkInt32 i = 0; i < m_context->getScenes().getSize(); ++i )
        {
            sceneAddedCallback( m_context->getScenes()[i] );
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
