// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Math/Matrix/hkMatrixDecomposition.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

class hkxNode;

/// Scene utilities
class hkxSceneUtils
{
    public:
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkxSceneUtils);
            /// Options taken by "hkxSceneUtils::transformScene"
        struct HK_EXPORT_COMMON SceneTransformOptions
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SCENE_DATA, hkxSceneUtils::SceneTransformOptions );

                /// Should we apply this transformation to scene graph nodes
            hkBool m_applyToNodes;

                /// Should we apply this transformation to vertex buffers
            hkBool m_applyToBuffers;

                /// Should we apply this transformation to float channels
            hkBool m_applyToFloatChannels;

                /// Should we apply this transformation to lights
            hkBool m_applyToLights;

                /// Should we apply this transformation to cameras
            hkBool m_applyToCameras;

                /// Should we flip index buffer winding
            hkBool m_flipWinding;

                /// The transform to apply
            hkMatrix3 m_transformMatrix;
        };

        /// Node collector interface
        struct HK_EXPORT_COMMON NodeCollector
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SCENE_DATA, hkxSceneUtils::NodeCollector );

            HK_ALWAYS_INLINE virtual ~NodeCollector(){} // silence gcc warnings

            /// Called to add a node to the collector
            virtual void addNode(hkxNode* nodeIn) = 0;
        };

            /// Given a scene and the options specified in the SceneTransformOption struct, it goes
            /// through nodes, attributes, meshes, etc., applying the specified transform to the scene.
            /// Useful for scaling scenes as well as for transforming coordinate systems
        static HK_EXPORT_COMMON void HK_CALL transformScene( class hkxScene& scene, const SceneTransformOptions& opts );

            /// Extracts environment data from an hkxScene; used mostly for backwards compatibility as previously
            /// environment information was stored in the hkxScene object. The variables extracted are:
            /// "asset" (ex: "car"),
            /// "assetPath" (ex: "c:/temp/car.max"),
            /// "assetFolder" (ex: "c:/temp/"),
            /// "modeller" (ex: "3ds max 8.0.0"),
            /// "selected" (ex: "chassis")
        static HK_EXPORT_COMMON void HK_CALL fillEnvironmentFromScene (const class hkxScene& scene, class hkxEnvironment& environment);

            /// Reorders the nodes in an hkxScene by their names. It does so alphabetically, regardless of case.
        static HK_EXPORT_COMMON void HK_CALL reorderNodesAlphabetically ( class hkxScene& scene );

            /// Checks the contents of the node, and if it's a mesh it returns it, or if it's a skin binding, it returns the mesh associated
            /// with it.
        static HK_EXPORT_COMMON _Ret_maybenull_ class hkxMesh* HK_CALL getMeshFromNode(_In_opt_ const class hkxNode* aNode);

            // Reverse lookup. Handy to get a name for a mesh
        static HK_EXPORT_COMMON _Ret_maybenull_ const hkxNode* HK_CALL findFirstNodeUsingMesh(_In_opt_ const hkxNode* rootNode, _In_ const class hkxMesh* aMesh);

            /// finds the first mesh in the scene
        static HK_EXPORT_COMMON _Ret_maybenull_ hkxNode* HK_CALL findFirstMeshNode(_In_ const hkxScene* scene);

            /// finds all nodes under node, which have a mesh attached, which is probably visible in the Havok demos.
            /// Also if the mesh name can be found in extraNodesToFind, the node will be returned.
        static HK_EXPORT_COMMON void HK_CALL findAllGraphicsNodes(bool collectShapes, bool ignorePShapes, const hkStringMap<int>& extraNodesToFind, _In_opt_ hkxNode* node, hkArray<hkxNode*>& nodesOut);
        static HK_EXPORT_COMMON void HK_CALL findAllGraphicsNodes(bool collectShapes, bool ignorePShapes, const hkMap<hkUuid, int>& extraNodesToFind, _In_opt_ hkxNode* node, hkArray<hkxNode*>& nodesOut);

            /// finds all nodes under node
        static HK_EXPORT_COMMON void HK_CALL findAllNodes(_Inout_opt_ hkxNode* node, hkArray< hkRefPtr<hkxNode> >& nodes );

            /// Finds all nodes under node, which have a mesh attached and optionally compute its corresponding worldFromLocal transform.
            /// This function gets called recursively and hence for the root we need to pass in the identity for the rootTransform.
        static HK_EXPORT_COMMON void HK_CALL findAllMeshNodes(_In_ const hkxScene* scene,
                                                _Inout_opt_ hkxNode* node,
                                                hkArray< hkRefPtr<hkxNode> >& nodes,
                                                _In_opt_ const hkMatrix4* rootTransform = HK_NULL,
                                                _Inout_opt_ hkArray<hkMatrix4>* worldFromLocalTransforms = HK_NULL);

        /// Compute the world Aabb of the scene
        static HK_EXPORT_COMMON void HK_CALL calcSceneAabb(_In_ const hkxScene* scene, hkAabb& aabbOut);

            // Helper function used by the createskin filter which allows for *translations* as well as rotations
        static HK_EXPORT_COMMON void HK_CALL transformVertexBuffer( const hkTransform& tr, class hkxVertexBuffer& vbuffer);

        /// Merges the source scene into the destination scene
        static HK_EXPORT_COMMON void HK_CALL mergeScenes(_Inout_ hkxScene* dstScene, _Inout_ hkxScene* srcScene);

        /// Removes all unused skin bindings
        static HK_EXPORT_COMMON void HK_CALL removeUnusedSkinBindings(_Inout_ hkxScene* scene);

        /// Removes all unused meshes
        static HK_EXPORT_COMMON void HK_CALL removeUnusedMeshes(_Inout_ hkxScene* scene);

        /// Removes all duplicate materials
        static HK_EXPORT_COMMON void HK_CALL removeDuplicateMaterials(_Inout_ hkxScene* scene);

        /// Removes all unused materials
        static HK_EXPORT_COMMON void HK_CALL removeUnusedMaterials(_Inout_ hkxScene* scene);

        /// Removes all duplicate textures
        static HK_EXPORT_COMMON void HK_CALL removeDuplicateTextures(_Inout_ hkxScene* scene);

        /// Removes all unused textures
        static HK_EXPORT_COMMON void HK_CALL removeUnusedTextures(_Inout_ hkxScene* scene);

        // Flips the winding of the triangles in the index buffer.
        static HK_EXPORT_COMMON void HK_CALL flipWinding( class hkxIndexBuffer &ibuffer );

        /// Collects a sub-set of nodes
        static HK_EXPORT_COMMON void HK_CALL collectNodes(_In_opt_ hkxNode* rootNodeIn, _Inout_ NodeCollector* collectorIn);

    public:

            // Contains useful information about the transform
        struct HK_EXPORT_COMMON TransformInfo
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SCENE_DATA, hkxSceneUtils::TransformInfo );

                // The transform as a Matrix4
            hkMatrix3 m_transform;

                // The inverse of the transform
            hkMatrix3 m_inverse;

                // The inverse of the transform, transposed
            hkMatrix3 m_inverseTranspose;

                // The transform decomposed
            hkMatrixDecomposition::Decomposition m_decomposition;
        };

    private:

            // Transforms a node and its children. It also transform node attributes
        static HK_EXPORT_COMMON void HK_CALL transformNode( const TransformInfo& transformInfo, class hkxNode& node);

        static HK_EXPORT_COMMON void HK_CALL transformSkinBinding( const TransformInfo& transformInfo, class hkxSkinBinding& skinBinding);

        static HK_EXPORT_COMMON void HK_CALL transformVertexBuffer( const TransformInfo& transformInfo, class hkxVertexBuffer& vbuffer);

        static HK_EXPORT_COMMON void HK_CALL transformFloatChannel( const TransformInfo& transformInfo, class hkxVertexFloatDataChannel& floatChannel);

        static HK_EXPORT_COMMON void HK_CALL transformLight( const TransformInfo& transformInfo, class hkxLight& light);

        static HK_EXPORT_COMMON void HK_CALL transformCamera( const TransformInfo& transformInfo, class hkxCamera& camera);

        static HK_EXPORT_COMMON void HK_CALL transformSpline( const TransformInfo& transformInfo, class hkxSpline& spline);

            // Called by transformNode
        static HK_EXPORT_COMMON void HK_CALL transformAttributeGroup ( const TransformInfo& transformInfo, struct hkxAttributeGroup& attributeGroup);

            // Called by transformAttributeGroup
        static HK_EXPORT_COMMON void HK_CALL transformAnimatedFloat (const TransformInfo& transformInfo, class hkxAnimatedFloat& animatedFloat);
        static HK_EXPORT_COMMON void HK_CALL transformAnimatedQuaternion (const TransformInfo& transformInfo, class hkxAnimatedQuaternion& animatedQuaternion);
        static HK_EXPORT_COMMON void HK_CALL transformAnimatedMatrix (const TransformInfo& transformInfo, class hkxAnimatedMatrix& animatedMatrix);
        static HK_EXPORT_COMMON void HK_CALL transformAnimatedVector (const TransformInfo& transformInfo, class hkxAnimatedVector& animatedVector);

            // Transforms a fullMatrix4, reused in different places
        static HK_EXPORT_COMMON void HK_CALL transformMatrix4 (const TransformInfo& transformInfo, hkMatrix4& matrix4);

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
