// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>
#include <Common/SceneData/Mesh/Channels/hkxBlendShapeCollectionChannel.h>
#include <Common/SceneData/Mesh/Channels/hkxVertexFloatDataChannel.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <Common/Base/Types/Uuid/hkUuidMapOperations.cxx>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>

#define DEBUG_LOG_IDENTIFIER "common.hkx.scene"
#include <Common/Base/System/Log/hkLog.hxx>

// perform the given transform on the selected scene elements
void hkxSceneUtils::transformScene( hkxScene& scene, const SceneTransformOptions& opts )
{
    // Construct the "transformInfo" object
    TransformInfo transformInfo;
    {
        // The 4x4 matrix
        transformInfo.m_transform = opts.m_transformMatrix;

        // Its inverse
        if( transformInfo.m_inverse.setInverse( transformInfo.m_transform ).isFailure() )
        {
            HK_WARN_ALWAYS ( 0xabba45e4, "Inversion failed. Check the Matrix is not singular" );
            return;
        }

        // The inverse, transposed (for normals)
        transformInfo.m_inverseTranspose = transformInfo.m_inverse;
        transformInfo.m_inverseTranspose.transpose();

        // Its decomposition
        hkMatrixDecomposition::decomposeMatrix(transformInfo.m_transform, transformInfo.m_decomposition);
    }

    HK_REPORT_SECTION_BEGIN(0x5e4345e4, "Transform Scene" );

    // nodes
    if( opts.m_applyToNodes && scene.m_rootNode )
    {
        // transform node and node's children
        hkxSceneUtils::transformNode( transformInfo, *scene.m_rootNode);

        Log_Info( "Processed all nodes in the scene." );
    }

    // skin bindings
    if( opts.m_applyToNodes && scene.m_skinBindings.getSize() )
    {
        // iterate through all binding in the scene
        for( int cs = 0; cs < scene.m_skinBindings.getSize(); ++cs )
        {
            hkxSkinBinding* skin= scene.m_skinBindings[cs];
            hkxSceneUtils::transformSkinBinding( transformInfo, *skin);
        }

        Log_Info( "Processed {} skin bindings.", scene.m_skinBindings.getSize() );
    }

    // mesh buffers (vertex and/or index)
    if( ( opts.m_applyToBuffers || opts.m_flipWinding ) && scene.m_meshes.getSize() )
    {
        // find the buffers
        for( int cm = 0; cm < scene.m_meshes.getSize(); ++cm )
        {
            hkxMesh* curMesh = scene.m_meshes[cm];
            const int blendShapeChannelIndex = curMesh->findUserChannelByClassType<hkxBlendShapeCollectionChannel>();

            for( int cs = 0; cs < curMesh->m_sections.getSize(); ++cs )
            {
                hkxMeshSection* curSection = curMesh->m_sections[cs];

                // buffers?
                if( opts.m_applyToBuffers )
                {
                    // transform the vertex buffer (and the blend shape's buffer)
                    hkxSceneUtils::transformVertexBuffer( transformInfo, *curSection->m_vertexBuffer);

                    hkxBlendShapeCollectionChannel* blendShapeChannel = HK_NULL;
                    if (blendShapeChannelIndex >= 0 && blendShapeChannelIndex < curSection->m_userChannels.getSize())
                    {
                        blendShapeChannel = hkDynCast(curSection->m_userChannels[blendShapeChannelIndex]);
                    }
                    if (blendShapeChannel)
                    {
                        hkxSceneUtils::transformVertexBuffer(transformInfo, blendShapeChannel->m_vertData);
                    }
                }

                // winding?
                if( opts.m_flipWinding )
                {
                    for( int cib = 0; cib < curSection->m_indexBuffers.getSize(); ++cib )
                    {
                        // flip the triangle winding
                        hkxSceneUtils::flipWinding( *curSection->m_indexBuffers[cib] );
                    }
                }
            }
        }

        Log_Info( "Processed {} meshes.", scene.m_meshes.getSize() );
    }

    // vertex float channels
    if ( opts.m_applyToFloatChannels )
    {
        for( int cm = 0; cm < scene.m_meshes.getSize(); ++cm )
        {
            hkxMesh* curMesh = scene.m_meshes[cm];

            // find the float channels
            for (int ci=0; ci<curMesh->m_userChannelInfos.getSize(); ++ci)
            {
                HK_ASSERT_NO_MSG(0x610862a4, curMesh->m_userChannelInfos[ci]);
                const hkxMesh::UserChannelInfo& info = *curMesh->m_userChannelInfos[ci];

                if(info.m_className == hkReflect::getName<hkxVertexFloatDataChannel>())
                {
                    // transform this vertex float channel in each mesh section
                    for (int si=0; si<curMesh->m_sections.getSize(); ++si)
                    {
                        hkxMeshSection* section = curMesh->m_sections[si];
                        hkxVertexFloatDataChannel* floatChannel = static_cast<hkxVertexFloatDataChannel*>( section->m_userChannels[ci].val() );
                        hkxSceneUtils::transformFloatChannel( transformInfo, *floatChannel );
                    }
                }
            }
        }
    }
    else if (transformInfo.m_decomposition.m_hasScale)
    {
        // HCL-737 : Warn user if distance/angle channels found and using a scaling transform
        bool ok = true;
        for( int cm = 0; ok && (cm < scene.m_meshes.getSize()); ++cm )
        {
            hkxMesh* curMesh = scene.m_meshes[cm];

            // find the float channels
            for (int ci=0; ok && (ci<curMesh->m_userChannelInfos.getSize()); ++ci)
            {
                HK_ASSERT_NO_MSG(0x37fd0201, curMesh->m_userChannelInfos[ci]);
                const hkxMesh::UserChannelInfo& info = *curMesh->m_userChannelInfos[ci];

                if(info.m_className == hkReflect::getName<hkxVertexFloatDataChannel>())
                {
                    // transform this vertex float channel in each mesh section
                    for (int si=0; ok && si<curMesh->m_sections.getSize(); ++si)
                    {
                        hkxMeshSection* section = curMesh->m_sections[si];
                        hkxVertexFloatDataChannel* floatChannel = static_cast<hkxVertexFloatDataChannel*>( section->m_userChannels[ci].val() );
                        if (floatChannel->m_dimensions!=hkxVertexFloatDataChannel::FLOAT)
                        {
                            HK_WARN_ALWAYS(0xabba8ea1, "Vertex channels found but not transformed - was this intentional?");
                            ok = false;
                        }
                    }
                }
            }
        }
    }

    // cameras
    if( opts.m_applyToCameras && scene.m_cameras.getSize() )
    {
        for( int cc = 0; cc < scene.m_cameras.getSize(); ++cc )
        {
            // transform the cameras
            hkxSceneUtils::transformCamera( transformInfo, *scene.m_cameras[cc]);
        }

        Log_Info( "Processed {} cameras.", scene.m_cameras.getSize() );
    }

    // lights
    if( opts.m_applyToLights && scene.m_lights.getSize() )
    {
        for( int cc = 0; cc < scene.m_lights.getSize(); ++cc )
        {
            // transform the lights
            hkxSceneUtils::transformLight( transformInfo, *scene.m_lights[cc]);
        }

        Log_Info( "Processed {} lights.", scene.m_lights.getSize() );
    }

    // cameras
    if( opts.m_applyToNodes && scene.m_splines.getSize() )
    {
        for( int cc = 0; cc < scene.m_splines.getSize(); ++cc )
        {
            // transform the cameras
            hkxSceneUtils::transformSpline( transformInfo, *scene.m_splines[cc]);
        }

        Log_Info( "Processed {} splines.", scene.m_splines.getSize() );
    }


    // Finally, the scene itself, remembering that transforms in the pipeline in the
    // order Ta -> Tb should be multiplied as Tb * Ta   (EXP-1438)

    hkMatrix3 Ta = scene.m_appliedTransform;  // the incoming transform
    hkMatrix3 Tb = transformInfo.m_transform; // the transform in this filter
    Tb.mul(Ta);
    scene.m_appliedTransform = Tb;
    HK_REPORT_SECTION_END();
}


// transform a node's keys
void hkxSceneUtils::transformNode( const TransformInfo& transformInfo, hkxNode& node)
{
    // recurse into node tree
    for( int c = 0; c < node.m_children.getSize(); ++c )
    {
        hkxSceneUtils::transformNode( transformInfo, *node.m_children[c]);
    }

    // Transform the keyframes
    for( int i = 0; i < node.m_keyFrames.getSize(); i++ )
    {
        transformMatrix4(transformInfo, node.m_keyFrames[i]);
    }

    // Transform any attributes of the node that can be transformed
    for ( int agIndex = 0; agIndex < node.m_attributeGroups.getSize(); ++agIndex)
    {
        hkxAttributeGroup& ag = node.m_attributeGroups[agIndex];

        hkxSceneUtils::transformAttributeGroup (transformInfo, ag);
    }
}

void hkxSceneUtils::transformAttributeGroup(const TransformInfo& transformInfo, hkxAttributeGroup& attributeGroup)
{
    for ( int aIndex = 0; aIndex < attributeGroup.m_attributes.getSize(); ++aIndex)
    {
        hkxAttribute& attribute = attributeGroup.m_attributes[aIndex];

        if (attribute.m_value)
        {
            // we can transform certain known attribute types
            // and can transform based on the hint in those attributes
            // Floats
            if ( hkxAnimatedFloat* f = hkReflect::exactMatchDynCast<hkxAnimatedFloat>(attribute.m_value))
            {
                transformAnimatedFloat (transformInfo, *f);
                continue;
            }

            // Vectors
            if ( hkxAnimatedVector* v = hkReflect::exactMatchDynCast<hkxAnimatedVector>(attribute.m_value))
            {
                transformAnimatedVector (transformInfo, *v);
                continue;
            }

            // Quaternions
            if ( hkxAnimatedQuaternion* q = hkReflect::exactMatchDynCast<hkxAnimatedQuaternion>(attribute.m_value) )
            {
                transformAnimatedQuaternion (transformInfo, *q);
                continue;
            }

            // Matrices
            if ( hkxAnimatedMatrix* m = hkReflect::exactMatchDynCast<hkxAnimatedMatrix>(attribute.m_value) )
            {
                transformAnimatedMatrix (transformInfo, *m);
                continue;

            }
        }
    }
}

void hkxSceneUtils::transformAnimatedFloat (const TransformInfo& transformInfo, hkxAnimatedFloat& animatedFloat)
{
    // Scale : floats representing distances
    const bool shouldScale = (animatedFloat.m_hint & hkxAttribute::HINT_SCALE) != 0;

    // Flip : floats representing angles
    const bool shouldFlip = ((animatedFloat.m_hint & hkxAttribute::HINT_FLIP) && transformInfo.m_decomposition.m_flips) != 0;

    // Floats can only be scaled or flipped
    if ( shouldScale || shouldFlip)
    {
        if (shouldScale && shouldFlip)
        {
            HK_WARN_ALWAYS(0xabba8a03, "Float attribute with both FLIP and SCALE flags... Weird..");
        }

        hkFloat32 scaleFloat = (shouldFlip) ? -1.0f : 1.0f;
        if (shouldScale)
        {
            const hkVector4& scaleVector = transformInfo.m_decomposition.m_scale;
            const hkFloat32 scaleLength = hkFloat32(scaleVector.length<3>().getReal()) * 0.57735026918962576450914878050196f; // 1/ sqrt(3)
            scaleFloat *= scaleLength;

        }

        for (int fi= 0; fi < animatedFloat.m_floats.getSize(); ++fi)
        {
            animatedFloat.m_floats[fi] *= scaleFloat;
        }
    }
}

void hkxSceneUtils::transformAnimatedQuaternion (const TransformInfo& transformInfo,  hkxAnimatedQuaternion& animatedQuaternion)
{
    // Quaternions are always transformed as they always represent rotations
    for (int qi=0; qi < animatedQuaternion.m_quaternions.getSize()/4; qi++)
    {
        hkQuaternion quatRef;
        quatRef.m_vec.load<4,HK_IO_NATIVE_ALIGNED>(&animatedQuaternion.m_quaternions[4*qi]);

        // We rotate the axis of the quaternion by the basis of the transform
        hkVector4 imag = quatRef.getImag();
        imag._setRotatedDir(transformInfo.m_decomposition.m_basis, imag);
        quatRef.setImag(imag);

        // And if the transformation involves a change of handedness, flip the sign
        if (transformInfo.m_decomposition.m_flips)
        {
            quatRef.setRealPart(-quatRef.getRealPart());
        }

        quatRef.m_vec.store<4,HK_IO_NATIVE_ALIGNED>(&animatedQuaternion.m_quaternions[4*qi]);
    }
}

void hkxSceneUtils::transformAnimatedMatrix (const TransformInfo& transformInfo,  hkxAnimatedMatrix& animatedMatrix)
{
    if (animatedMatrix.m_hint & hkxAttribute::HINT_TRANSFORM_AND_SCALE)
    {
        for (int mi= 0; mi < animatedMatrix.m_matrices.getSize()/16; ++mi)
        {
            hkMatrix4 m;
            m.set4x4ColumnMajor(&animatedMatrix.m_matrices[16*mi]);
            transformMatrix4(transformInfo, m);
            m.get4x4ColumnMajor(&animatedMatrix.m_matrices[16*mi]);
        }
    }
}

void hkxSceneUtils::transformAnimatedVector (const TransformInfo& transformInfo,  hkxAnimatedVector& animatedVector)
{
    // Vectors are either just rotated or rotated and scaled
    const bool shouldRotate = (animatedVector.m_hint & hkxAttribute::HINT_TRANSFORM) != 0;
    const bool shouldScale = (animatedVector.m_hint & hkxAttribute::HINT_SCALE) != 0;

    if (!shouldRotate && !shouldScale)
    {
        return;
    }

    hkMatrix3 theTransform; theTransform.setIdentity();
    {
        if (shouldRotate && !shouldScale)
        {
            theTransform = transformInfo.m_decomposition.m_basis;
        }

        if (shouldRotate && shouldScale)
        {
            theTransform = transformInfo.m_transform;
        }

        if (!shouldRotate && shouldScale)
        {
            // uncommon, but...
            const hkVector4& scaleVector = transformInfo.m_decomposition.m_scale;
            hkMatrix3Util::_setDiagonal(scaleVector, theTransform);
        }
    }

    for (int vi= 0; vi < animatedVector.m_vectors.getSize()/4; ++vi)
    {
        hkVector4 vs;
        vs.load<4,HK_IO_NATIVE_ALIGNED>(&animatedVector.m_vectors[4*vi]);
        vs._setRotatedDir(theTransform, vs);
        vs.store<4,HK_IO_NATIVE_ALIGNED>(&animatedVector.m_vectors[4*vi]);
    }
}



// transform the skin binding
void hkxSceneUtils::transformSkinBinding( const TransformInfo& transformInfo, hkxSkinBinding& nodeInfo)
{
    // Transform initBoneTransform Array
    {
        for( int i = 0; i < nodeInfo.m_bindPose.getSize(); i++ )
        {
            transformMatrix4(transformInfo, nodeInfo.m_bindPose[i]);
        }
    }

    // Transform init skin transform
    {
        transformMatrix4 (transformInfo, nodeInfo.m_initSkinTransform);
    }
}


// transform the vertex buffer data
void hkxSceneUtils::transformVertexBuffer( const TransformInfo& transformInfo, hkxVertexBuffer& vb)
{
    // Positions are transformed by the full matrix
    const hkMatrix3& trans = transformInfo.m_transform;

    // Normals are transformed by the inverse of the basis, transposed
    const hkMatrix3& transNormal = transformInfo.m_inverseTranspose;

    const hkxVertexDescription& desc = vb.getVertexDesc();
    const hkxVertexDescription::ElementDecl* posDecl =  desc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0 );
    const hkxVertexDescription::ElementDecl* normDecl =  desc.getElementDecl( hkxVertexDescription::HKX_DU_NORMAL, 0 );
    const hkxVertexDescription::ElementDecl* tangDecl =  desc.getElementDecl( hkxVertexDescription::HKX_DU_TANGENT, 0 );
    const hkxVertexDescription::ElementDecl* binormDecl = desc.getElementDecl( hkxVertexDescription::HKX_DU_BINORMAL, 0 );
    int posStride = 0;
    int normStride = 0;
    int tangStride = 0;
    int binormStride = 0;
    void* posData = HK_NULL;
    void* normData = HK_NULL;
    void* tangData = HK_NULL;
    void* binormData = HK_NULL;
    if (posDecl)
    {
        posData = vb.getVertexDataPtr(*posDecl);
        posStride = posDecl->m_byteStride;
    }
    if (normDecl)
    {
        normData = vb.getVertexDataPtr(*normDecl);
        normStride = normDecl->m_byteStride;
    }
    if (tangDecl)
    {
        tangData = vb.getVertexDataPtr(*tangDecl);
        tangStride = tangDecl->m_byteStride;
    }
    if (binormDecl)
    {
        binormData = vb.getVertexDataPtr(*binormDecl);
        binormStride = binormDecl->m_byteStride;
    }

    int numVerts = vb.getNumVertices();
    for( int i = 0; i < numVerts; i++ )
    {
        if (posDecl)
        {
            hkFloat32* data = (hkFloat32*)posData;
            hkVector4 vector;
            vector.load<3,HK_IO_NATIVE_ALIGNED>(data);
            vector._setRotatedDir( trans, vector );
            vector.setComponent<3>(hkSimdReal_1);
            vector.store<4,HK_IO_NATIVE_ALIGNED>(data);
        }

        if (normDecl)
        {
            hkFloat32* data = (hkFloat32*)normData;
            hkVector4 vector;
            vector.load<3,HK_IO_NATIVE_ALIGNED>(data);
            vector._setRotatedDir( transNormal, vector );
            vector.normalizeIfNotZero<3>();
            vector.zeroComponent<3>();
            vector.store<4,HK_IO_NATIVE_ALIGNED>(data);
        }

        if (tangDecl)
        {
            hkFloat32* data = (hkFloat32*)tangData;
            hkVector4 vector;
            vector.load<3,HK_IO_NATIVE_ALIGNED>(data);
            vector._setRotatedDir( transNormal, vector );
            vector.normalizeIfNotZero<3>();
            vector.zeroComponent<3>();
            vector.store<4,HK_IO_NATIVE_ALIGNED>(data);
        }

        if (binormDecl)
        {
            hkFloat32* data = (hkFloat32*)binormData;
            hkVector4 vector;
            vector.load<3,HK_IO_NATIVE_ALIGNED>(data);
            vector._setRotatedDir( transNormal, vector );
            vector.normalizeIfNotZero<3>();
            vector.zeroComponent<3>();
            vector.store<4,HK_IO_NATIVE_ALIGNED>(data);
        }

        posData = hkAddByteOffset(posData, posStride);
        normData = hkAddByteOffset(normData, normStride);
        tangData = hkAddByteOffset(tangData, tangStride);
        binormData = hkAddByteOffset(binormData, binormStride);
    }

}

void hkxSceneUtils::transformVertexBuffer( const hkTransform& tr, class hkxVertexBuffer& vbuffer)
{
    // Do the rotation part first
    // Now do the translation part
    int numVerts = vbuffer.getNumVertices();
    if (numVerts > 0)
    {
        hkxSceneUtils::TransformInfo transformInfo;
        {
            // The 4x4 matrix
            transformInfo.m_transform = tr.getRotation();

            // Its inverse
            if( transformInfo.m_inverse.setInverse( transformInfo.m_transform ).isFailure() )
            {
                HK_WARN_ALWAYS ( 0xabba45e4, "Inversion failed. Check the Matrix is not singular" );
                return;
            }

            // The inverse, transposed (for normals)
            transformInfo.m_inverseTranspose = transformInfo.m_inverse;
            transformInfo.m_inverseTranspose.transpose();

            // Its decomposition
            hkMatrixDecomposition::decomposeMatrix(transformInfo.m_transform, transformInfo.m_decomposition);
        }

        hkxSceneUtils::transformVertexBuffer(transformInfo, vbuffer);

        // then translate:
        const hkxVertexDescription& desc = vbuffer.getVertexDesc();
        const hkxVertexDescription::ElementDecl* posDecl =  desc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0 );
        if (posDecl)
        {
            hkFloat32* pos = (hkFloat32*)vbuffer.getVertexDataPtr(*posDecl);
            int posStride = posDecl->m_byteStride / sizeof(hkFloat32);
            for (int vi=0; vi < numVerts; ++vi)
            {
                hkVector4 p;
                p.load<3,HK_IO_NATIVE_ALIGNED>(pos);
                p.add( tr.getTranslation() );
                p.setComponent<3>(hkSimdReal_1);
                p.store<4,HK_IO_NATIVE_ALIGNED>(pos);
                pos += posStride;
            }
        }
    }
}

// transform the vertex float channel
void hkxSceneUtils::transformFloatChannel(const TransformInfo& transformInfo, hkxVertexFloatDataChannel& floatChannel)
{
    switch (floatChannel.m_dimensions)
    {
        case hkxVertexFloatDataChannel::DISTANCE:
        {
            const hkVector4& scaleVector = transformInfo.m_decomposition.m_scale;
            const hkFloat32 scaleFloat = hkFloat32(scaleVector.length<3>().getReal()) * 0.57735026918962576450914878050196f; // 1 / sqrt(3)

            floatChannel.m_scaleMin *= scaleFloat;
            floatChannel.m_scaleMax *= scaleFloat;
        }
        break;

        case hkxVertexFloatDataChannel::ANGLE:
        {
            const bool shouldFlip = transformInfo.m_decomposition.m_flips;
            if (shouldFlip)
            {
                const int numVertexFloats = floatChannel.getNumVertexFloats();
                for (int fi=0; fi<numVertexFloats; ++fi)
                {
                    floatChannel.setVertexFloat(fi, floatChannel.getVertexFloat(fi, hkxVertexFloatDataChannel::ORIGINAL)*-1.0f);
                }
            }
        }
        break;

        case hkxVertexFloatDataChannel::FLOAT:
        break;
    }
}


// transform the light
void hkxSceneUtils::transformLight( const TransformInfo& transformInfo, hkxLight& light)
{
    light.m_position._setRotatedDir( transformInfo.m_transform, light.m_position );
    light.m_direction._setRotatedDir( transformInfo.m_decomposition.m_basis, light.m_direction );
}


// transform the camera
void hkxSceneUtils::transformCamera( const TransformInfo& transformInfo, hkxCamera& camera)
{
    camera.m_from._setRotatedDir( transformInfo.m_transform, camera.m_from );
    camera.m_focus._setRotatedDir( transformInfo.m_transform, camera.m_focus );
    camera.m_up._setRotatedDir( transformInfo.m_decomposition.m_basis, camera.m_up );

    // Get a single float value for the scale in the transform,
    // and scale the clipping planes by it
    const hkVector4& scaleVector = transformInfo.m_decomposition.m_scale;
    const hkReal scaleLength = scaleVector.length<3>().getReal() * hkReal(0.57735026918962576450914878050196f); // 1 / sqrt(3)
    camera.m_near *= scaleLength;
    camera.m_far *= scaleLength;

    // Change handness of the camera if required
    if (transformInfo.m_decomposition.m_flips)
    {
        camera.m_leftHanded = !camera.m_leftHanded;
    }

}


// transform the spline
void hkxSceneUtils::transformSpline( const TransformInfo& transformInfo, hkxSpline& spline)
{
    for (int i=0; i < spline.m_controlPoints.getSize(); ++i)
    {
        hkxSpline::ControlPoint& cp = spline.m_controlPoints[i];
        cp.m_tangentIn._setRotatedDir( transformInfo.m_transform, cp.m_tangentIn );
        cp.m_tangentOut._setRotatedDir( transformInfo.m_transform, cp.m_tangentOut );
        cp.m_position._setRotatedDir( transformInfo.m_transform, cp.m_position);
    }
}

// flip the triangle winding
void hkxSceneUtils::flipWinding( hkxIndexBuffer &ibuffer )
{
    // 16 bit indices
    int numI = ibuffer.m_indices16.getSize() | ibuffer.m_indices32.getSize();
    if( ibuffer.m_indices16.getSize())
    {
        switch ( ibuffer.m_indexType )
            {
            case hkxIndexBuffer::INDEX_TYPE_TRI_LIST:
            {
                for( int i = 0; i < numI; i += 3 )
                {
                    hkAlgorithm::swap( ibuffer.m_indices16[i+0], ibuffer.m_indices16[i+2] );
                }
                break;
            }
            case hkxIndexBuffer::INDEX_TYPE_TRI_STRIP:
            {
                // winding implicit off the first tri
                hkAlgorithm::swap( ibuffer.m_indices16[1], ibuffer.m_indices16[2] );
                break;
            }
            // fan not handled
            default: break;
        }
    }

    // 32 bit indices
    else if (ibuffer.m_indices32.getSize())
    {
        switch ( ibuffer.m_indexType )
        {
            case hkxIndexBuffer::INDEX_TYPE_TRI_LIST:
                {
                    for( int i = 0; i < numI; i += 3 )
                    {
                        hkAlgorithm::swap( ibuffer.m_indices32[i+0], ibuffer.m_indices32[i+2] );
                    }
                    break;
                }
            case hkxIndexBuffer::INDEX_TYPE_TRI_STRIP:
                {
                    // winding implicit off the first tri
                    hkAlgorithm::swap( ibuffer.m_indices32[1], ibuffer.m_indices32[2] );
                    break;
                }
                // fan not handled
            default: break;
        }
    }
    

}

// Transforms a fullMatrix4, reused in different places
void hkxSceneUtils::transformMatrix4 (const TransformInfo& transformInfo, hkMatrix4& matrix4)
{
    // We split the matrix4 into a matrix3 and a translation
    hkMatrix3 matrix3;
    {
        // Take it from the matrix4
        matrix3.setCols(matrix4.getColumn<0>(), matrix4.getColumn<1>(), matrix4.getColumn<2>());

        // Change of basis (t * m * t^-1)
        hkMatrix3 temp;
        temp.setMul(matrix3, transformInfo.m_inverse);
        matrix3.setMul(transformInfo.m_transform, temp);
    }

    hkVector4 translation;
    {
        translation = matrix4.getColumn<3>();

        translation._setRotatedDir(transformInfo.m_transform, translation);
        translation.setW(hkSimdReal_1);
    }

    // We put it back together
    matrix4.setCols(matrix3.getColumn<0>(), matrix3.getColumn<1>(), matrix3.getColumn<2>(), translation);
}

_Ret_maybenull_
static hkxNode* _findFirstSelected(_In_opt_ hkxNode* node )
{
    if( !node )
    {
        return HK_NULL;
    }

    for( int i=0; i<node->m_children.getSize(); ++i )
    {
        hkxNode* child = node->m_children[i];
        if( child->m_selected )
        {
            return child;
        }
        else
        {
            hkxNode* descendant = _findFirstSelected( child );
            if( descendant != HK_NULL )
            {
                return descendant;
            }
        }
    }
    return HK_NULL;
}

void hkxSceneUtils::fillEnvironmentFromScene(const hkxScene& scene, hkxEnvironment& environment)
{
    if (scene.m_modeller)
    {
        environment.setVariable("modeller", scene.m_modeller);
    }

    if (scene.m_asset)
    {
        // 'scene.m_asset' may contain a path to a file, or a path to a folder.
        // The GUI will only allow you to select a folder.
        hkStringBuf assetInput(scene.m_asset);
        bool trailingSlash = assetInput.endsWith("/") || assetInput.endsWith("\\");
        assetInput.pathNormalize();
        // pathNormalize removes trailing slashes. As these may be vital to correct later
        // string concatenations we must preserve a trailing slash.
        if (trailingSlash)
        {
            assetInput.append("/");
        }

        int lastSlash = assetInput.lastIndexOf('/');
        int lastDot = assetInput.lastIndexOf('.');
        bool isFolder = lastDot < lastSlash || lastDot == -1;

        // Full path (either to folder or file)
        {
            hkStringBuf assetPath(assetInput);
            assetPath.replace('/', '\\', hkStringBuf::REPLACE_ALL);
            environment.setVariable("assetPath", assetPath);
        }

        // Find the asset name, i.e. remove the path and file extension.
        {
            hkStringBuf assetName(assetInput);
            if (isFolder)
            {
                assetName = "untitled";
            }
            else
            {
                assetName.pathBasename();
                int extensionIndex = hkString::lastIndexOf(assetName, '.');
                // Do not slice if no dot is found or if the dot is the first character.
                if (extensionIndex > 0)
                    assetName.slice(0, extensionIndex);
            }

            environment.setVariable("asset", assetName);
        }

        // Folder
        {
            // Find parent folder of scene.m_asset.
            // If scene.m_assets ends with a slash, use it as-is instead for backwards compatibility.
            hkStringBuf fullPath(assetInput);
            if (!fullPath.endsWith("/"))
            {
                fullPath.pathDirname();
                fullPath.append("/");
            }

            fullPath.replace('/', '\\', hkStringBuf::REPLACE_ALL);
            environment.setVariable("assetFolder", fullPath);
        }
    }

    // EXP-631
    {
        hkxNode* firstSelected = _findFirstSelected(scene.m_rootNode);

        if (firstSelected && firstSelected->m_name)
        {
            environment.setVariable("selected", firstSelected->m_name);
        }
    }


}

static bool _nodeCompare(_In_opt_ const hkxNode* one, _In_opt_ const hkxNode* two)
{
    if (!one || !one->m_name) return true;
    if (!two || !two->m_name) return false;

    return (hkString::strCasecmp(one->m_name, two->m_name)<0);
}

static void _reorderChildrenOfNodeRecursive(_Inout_opt_ hkxNode* node)
{
    if (!node) return;

    // Recurse
    {
        for (int i=0; i<node->m_children.getSize(); i++)
        {
            _reorderChildrenOfNodeRecursive(node->m_children[i]);
        }
    }

    // Reorder
    {
        hkAlgorithm::quickSort(node->m_children.begin(), node->m_children.getSize(), _nodeCompare);
    }
}

/*static*/ void hkxSceneUtils::reorderNodesAlphabetically ( class hkxScene& scene )
{
    _reorderChildrenOfNodeRecursive(scene.m_rootNode);
}

_Ret_maybenull_
/*static*/ hkxMesh* hkxSceneUtils::getMeshFromNode(_In_opt_ const hkxNode* node)
{
    if (!node) return HK_NULL;
    if (!node->m_object) return HK_NULL;

    if (hkxMesh* m = node->hasA<hkxMesh>())
    {
        return m;
    }
    else if (hkxSkinBinding* s = node->hasA<hkxSkinBinding>())
    {
        return s->m_mesh;
    }

    return HK_NULL;
}

_Ret_maybenull_
const hkxNode* HK_CALL hkxSceneUtils::findFirstNodeUsingMesh(_In_opt_ const hkxNode* node, _In_ const hkxMesh* aMesh)
{
    if (node)
    {
        {
            hkxMesh* nodeMesh = getMeshFromNode( node );
            if (nodeMesh == aMesh)
            {
                return node;
            }
        }

        for (int ni=0; ni < node->m_children.getSize(); ++ni)
        {
            const hkxNode* foundNode = findFirstNodeUsingMesh( node->m_children[ni], aMesh );
            if (foundNode)
            {
                return foundNode;
            }
        }
    }

    return HK_NULL;

}




void HK_CALL hkxSceneUtils::findAllGraphicsNodes(bool collectShapes, bool ignorePShapes, const hkStringMap<int>& extraNodesToFind, _In_opt_ hkxNode* node, hkArray<hkxNode*>& nodesOut)
{
    if ( !node )
    {
        return;
    }

    if ( node->findAttributeGroupByName("hkRigidBody") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }
    if ( node->findAttributeGroupByName("hkdBody") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }
    if ( node->findAttributeGroupByName("hkdShape") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }

    if ( extraNodesToFind.hasKey( node->m_name ) )
    {
        ignorePShapes = false;
        collectShapes = true;
    }

    if ( node->findAttributeGroupByName("hkShape") )
    {
        if ( ignorePShapes )
        {
            return;
        }
        collectShapes = true;
    }

    //bool graphicsFound = false;
    if ( collectShapes && !ignorePShapes )
    {
        hkxMesh* mesh = hkxSceneUtils::getMeshFromNode( node );
        if ( mesh )
        {
            nodesOut.pushBack(node);
        }
    }


    {
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            hkxNode* child = node->m_children[c];
            findAllGraphicsNodes( collectShapes, ignorePShapes, extraNodesToFind, child, nodesOut);
        }
    }
}

void HK_CALL hkxSceneUtils::findAllGraphicsNodes(bool collectShapes, bool ignorePShapes, const hkMap<hkUuid, int>& extraNodesToFind, _In_opt_ hkxNode* node, hkArray<hkxNode*>& nodesOut)
{
    if ( !node )
    {
        return;
    }

    if ( node->findAttributeGroupByName("hkRigidBody") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }
    if ( node->findAttributeGroupByName("hkdBody") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }
    if ( node->findAttributeGroupByName("hkdShape") )
    {
        ignorePShapes = false;
        collectShapes = true;
    }

    if ( extraNodesToFind.hasKey( node->m_uuid ) )
    {
        ignorePShapes = false;
        collectShapes = true;
    }

    if ( node->findAttributeGroupByName("hkShape") )
    {
        if ( ignorePShapes )
        {
            return;
        }
        collectShapes = true;
    }

    //bool graphicsFound = false;
    if ( collectShapes && !ignorePShapes )
    {
        hkxMesh* mesh = hkxSceneUtils::getMeshFromNode( node );
        if ( mesh )
        {
            nodesOut.pushBack(node);
        }
    }


    {
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            hkxNode* child = node->m_children[c];
            findAllGraphicsNodes( collectShapes, ignorePShapes, extraNodesToFind, child, nodesOut);
        }
    }
}

void HK_CALL hkxSceneUtils::findAllNodes(_Inout_opt_ hkxNode* node, hkArray< hkRefPtr<hkxNode> >& nodes)
{
    if ( !node )
    {
        return;
    }

    nodes.pushBack( node );
    {
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            findAllNodes( node->m_children[c], nodes);
        }
    }
}

//
//  Collects a sub-set of nodes

void HK_CALL hkxSceneUtils::collectNodes(_In_opt_ hkxNode* rootNodeIn, _Inout_ NodeCollector* collectorIn)
{
    if ( !rootNodeIn )
    {
        return;
    }

    // Collect the root node
    collectorIn->addNode(rootNodeIn);

    // Recurse
    for (int c = rootNodeIn->m_children.getSize() - 1; c >= 0; c--)
    {
        collectNodes(rootNodeIn->m_children[c], collectorIn);
    }
}

void HK_CALL hkxSceneUtils::findAllMeshNodes(_In_ const hkxScene* scene,
                                                _Inout_opt_ hkxNode* node,
                                                hkArray< hkRefPtr<hkxNode> >& nodes,
                                                _In_opt_ const hkMatrix4* rootTransform,
                                                _Inout_opt_ hkArray<hkMatrix4>* worldFromLocalTransforms )
{
    if ( !node )
    {
        node = scene->m_rootNode;

        if (!node)
            return;
    }

    bool computeTransforms = ( rootTransform != HK_NULL ) && ( worldFromLocalTransforms != HK_NULL );

    hkMatrix4 worldFromLocal;
    if ( computeTransforms )
    {
        worldFromLocal.setMul( *rootTransform, node->m_keyFrames[0] );
    }

    hkxMesh* mesh = hkxSceneUtils::getMeshFromNode( node );
    if ( mesh )
    {
        nodes.pushBack( node );

        if ( computeTransforms )
        {
            (*worldFromLocalTransforms).pushBack( worldFromLocal );
        }
    }

    {
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            findAllMeshNodes(scene, node->m_children[c], nodes, (computeTransforms ? &worldFromLocal : HK_NULL), worldFromLocalTransforms);
        }
    }
}

//
//  Compute the world Aabb of the scene

void HK_CALL hkxSceneUtils::calcSceneAabb(_In_ const hkxScene* scene, hkAabb& aabbOut)
{
    hkArray< hkRefPtr<hkxNode> > meshNodes;
    hkArray< hkMatrix4 > meshNodeWorldFromLocalTransforms;

    // First, get all the graphics nodes and their transform
    hkMatrix4 worldTransform; worldTransform.setIdentity();
    findAllMeshNodes( scene, scene->m_rootNode, meshNodes, &worldTransform, &meshNodeWorldFromLocalTransforms );

    // Compute aabb
    aabbOut.setEmpty();
    hkArray<hkVector4> mshPos;
    for (int m = meshNodes.getSize() - 1 ; m >= 0 ; m--)
    {
        hkxMesh* mesh = hkxSceneUtils::getMeshFromNode(meshNodes[m]);
        mshPos.setSize(0);

        mesh->collectVertexPositions(mshPos);

        // transform pos
        for (int v = mshPos.getSize() - 1 ; v >= 0 ; v--)
        {
            meshNodeWorldFromLocalTransforms[m].transformPosition(mshPos[v], mshPos[v]);
        }

        hkAabb meshAabb;
        hkAabbUtil::calcAabb(mshPos.begin(), mshPos.getSize(), meshAabb);

        aabbOut.includeAabb(meshAabb);
    }
}

_Ret_maybenull_ hkxNode* HK_CALL hkxSceneUtils::findFirstMeshNode(_In_ const hkxScene* scene)
{
    hkxNode* node = scene->m_rootNode;
    // Go searching for the a node
    for (int i=0;i<node->m_children.getSize();i++)
    {
        hkxNode* subNode = node->m_children[i];

        // Find if its the right kind of node
        if (subNode->hasA<hkxMesh>())
        {
            return subNode;
        }
    }
    return HK_NULL;
}

//
//  Merges the source scene into the destination scene

void HK_CALL hkxSceneUtils::mergeScenes(_Inout_ hkxScene* dstScene, _Inout_ hkxScene* srcScene)
{
    dstScene->m_sceneLength = hkMath::max2(srcScene->m_sceneLength, dstScene->m_sceneLength);
    dstScene->m_numFrames   = hkMath::max2(srcScene->m_numFrames, dstScene->m_numFrames);
    dstScene->m_asset       = HK_NULL;  // Reset the name, we have merged more than one asset

    // Compute the transform
    hkMatrix3 dstSceneFromWorld;    dstSceneFromWorld.setInverse(dstScene->m_appliedTransform);
    hkMatrix3 dstSceneFromSrcScene; dstSceneFromSrcScene.setMul(dstSceneFromWorld, srcScene->m_appliedTransform);

    // Transform the source scene into the destination scene
    if ( !dstSceneFromSrcScene.isApproximatelyEqual(hkMatrix3::getIdentity(), 1.0e-3f) )
    {
        hkxSceneUtils::SceneTransformOptions opts;
        opts.m_applyToNodes             = true;
        opts.m_applyToBuffers           = true;
        opts.m_applyToFloatChannels     = true;
        opts.m_applyToLights            = true;
        opts.m_applyToCameras           = true;
        opts.m_flipWinding              = false;
        opts.m_transformMatrix          = dstSceneFromSrcScene;
        hkxSceneUtils::transformScene(*srcScene, opts);
    }

    // Add all the children of the source root node to the destination root node
    hkxNode* dstRoot = dstScene->m_rootNode;
    hkxNode* srcRoot = srcScene->m_rootNode;
    if ( dstRoot && srcRoot )
    {
        // Ignore duplicate nodes
        for (int sci = 0; sci < srcRoot->m_children.getSize(); sci++)
        {
            hkxNode* srcChild = srcRoot->m_children[sci];

            // See if we can locate this child
            int dci = dstRoot->m_children.getSize() - 1;
            for (; dci >= 0; dci--)
            {
                hkxNode* dstChild = dstRoot->m_children[dci];

                if ( srcChild->m_uuid == dstChild->m_uuid )
                {
                    break;  // Found a child with the same name, stop!
                }
            }

            if ( dci < 0 )
            {
                // No duplicate found, append!
                dstRoot->m_children.pushBack(srcChild);
            }
        }
    }

    // Append everything else
    dstScene->m_selectionSets   .append(srcScene->m_selectionSets);
    dstScene->m_cameras         .append(srcScene->m_cameras);
    dstScene->m_lights          .append(srcScene->m_lights);
    dstScene->m_meshes          .append(srcScene->m_meshes);
    dstScene->m_materials       .append(srcScene->m_materials);
    dstScene->m_inplaceTextures .append(srcScene->m_inplaceTextures);
    dstScene->m_externalTextures.append(srcScene->m_externalTextures);
    dstScene->m_skinBindings    .append(srcScene->m_skinBindings);
    dstScene->m_splines         .append(srcScene->m_splines);

    // Remove unused geometry
    removeUnusedSkinBindings(dstScene);
    removeUnusedMeshes(dstScene);

    // Remove duplicates
    removeDuplicateTextures(dstScene);
    removeDuplicateMaterials(dstScene);

    // Remove other unused stuff
    removeUnusedMaterials(dstScene);
    removeUnusedTextures(dstScene);
}

//
//  Utility functions

namespace hkxSceneUtilsImpl
{
    //
    //  Returns true if a texture is used within the given scene

    template <typename T>
    static bool HK_CALL isTextureUsed(_In_ const hkxScene* scene, _In_ const T* tex)
    {
        for (int mi = scene->m_materials.getSize() - 1; mi >= 0; mi--)
        {
            const hkxMaterial* mtl = scene->m_materials[mi];

            for (int ti = mtl->m_stages.getSize() - 1; ti >= 0; ti--)
            {
                const hkxMaterial::TextureStage& texStage = mtl->m_stages[ti];

                if ( texStage.m_texture.val() == tex )
                {
                    return true;
                }
            }
        }

        return false;
    }

    //
    //  Returns true if a material is used within the given scene

    static bool HK_CALL isMtlUsed(_In_ const hkxScene* scene, _In_ const hkxMaterial* mtl)
    {
        int meshIdx = scene->m_meshes.getSize() - 1;
        for (; meshIdx >= 0; meshIdx--)
        {
            const hkxMesh* mesh = scene->m_meshes[meshIdx];

            for (int si = mesh->m_sections.getSize() - 1; si >= 0; si--)
            {
                const hkxMeshSection* meshSection = mesh->m_sections[si];
                if ( meshSection->m_material == mtl )
                {
                    return true;    // Found a mesh section that uses the material, stop!
                }
            }
        }

        return false;
    }
}

//
//  Removes all unused skin bindings

void HK_CALL hkxSceneUtils::removeUnusedSkinBindings(_Inout_ hkxScene* scene)
{
    // Get all nodes
    hkxNode* rootNode = scene->m_rootNode;
    hkArray< hkRefPtr<hkxNode> > allNodes;
    findAllNodes(rootNode, allNodes);

    // Remove any non-referenced skin bindings
    for (int k = scene->m_skinBindings.getSize() - 1; k >= 0; k--)
    {
        hkxSkinBinding* skinBinding = scene->m_skinBindings[k];

        // Search for nodes that reference it
        int ni = allNodes.getSize() - 1;
        for (; ni >= 0; ni--)
        {
            hkxNode* node = allNodes[ni];
            if ( node->m_object.val() == skinBinding )
            {
                break;
            }
        }

        if ( ni < 0 )
        {
            // Skin binding is not referenced, remove!
            scene->m_skinBindings.removeAt(k);
        }
    }
}

//
//  Removes all unused meshes

void HK_CALL hkxSceneUtils::removeUnusedMeshes(_Inout_ hkxScene* scene)
{
    // Get all nodes
    hkxNode* rootNode = scene->m_rootNode;
    hkArray< hkRefPtr<hkxNode> > allNodes;
    findAllNodes(rootNode, allNodes);

    // Remove any non-referenced meshes
    for (int k = scene->m_meshes.getSize() - 1; k >= 0; k--)
    {
        hkxMesh* mesh = scene->m_meshes[k];

        // Search for nodes that reference it
        int ni = allNodes.getSize() - 1;
        for (; ni >= 0; ni--)
        {
            hkxNode* node = allNodes[ni];
            if ( node->m_object.val() == mesh )
            {
                break;
            }
        }

        if ( ni < 0 )
        {
            // Mesh is not referenced, remove!
            scene->m_meshes.removeAt(k);
        }
    }
}

//
//  Removes all duplicate materials

void HK_CALL hkxSceneUtils::removeDuplicateMaterials(_Inout_ hkxScene* scene)
{
    hkArray<hkxMaterial*> uniqueMtls;
    for (int meshIdx = scene->m_meshes.getSize() - 1; meshIdx >= 0; meshIdx--)
    {
        hkxMesh* mesh = scene->m_meshes[meshIdx];

        for (int si = mesh->m_sections.getSize() - 1; si >= 0; si--)
        {
            hkxMeshSection* section = mesh->m_sections[si];
            hkxMaterial* mtl        = section->m_material;

            // See if we already have it
            int ui = uniqueMtls.getSize() - 1;
            for (; ui >= 0; ui--)
            {
                if ( mtl->equals(uniqueMtls[ui]) )
                {
                    // Found another duplicate material, replace!
                    section->m_material = uniqueMtls[ui];
                    break;
                }
            }

            if ( ui < 0 )
            {
                // Unique material, add!
                uniqueMtls.pushBack(mtl);
            }
        }
    }
}

//
//  Removes all unused materials

void HK_CALL hkxSceneUtils::removeUnusedMaterials(_Inout_ hkxScene* scene)
{
    for (int mtlIdx = scene->m_materials.getSize() - 1; mtlIdx >= 0; mtlIdx--)
    {
        hkxMaterial* mtl = scene->m_materials[mtlIdx];

        if ( !hkxSceneUtilsImpl::isMtlUsed(scene, mtl) )
        {
            // Material is not referenced, stop!
            scene->m_materials.removeAt(mtlIdx);
        }
    }
}

//
//  Removes all duplicate textures

void HK_CALL hkxSceneUtils::removeDuplicateTextures(_Inout_ hkxScene* scene)
{
    hkArray<hkxTextureInplace*> uniqueInplaceTex;
    hkArray<hkxTextureFile*> uniqueFileTex;

    // Iterate over all materials
    for (int mi = scene->m_materials.getSize() - 1; mi >= 0; mi--)
    {
        hkxMaterial* mtl = scene->m_materials[mi];

        for (int ti = mtl->m_stages.getSize() - 1; ti >= 0; ti--)
        {
            hkxMaterial::TextureStage& texStage = mtl->m_stages[ti];

            if(hkxTextureInplace* texInplace = hkDynCast<hkxTextureInplace>(texStage.m_texture))
            {
                // See if we already have an identical one
                int ui = uniqueInplaceTex.getSize() - 1;
                for (; ui >= 0; ui--)
                {
                    if ( texInplace->equals(uniqueInplaceTex[ui]) )
                    {
                        // Found another identical texture, replace!
                        texStage.m_texture = uniqueInplaceTex[ui];
                        break;
                    }
                }

                if ( ui < 0 )
                {
                    // This is a unique texture, save!
                    uniqueInplaceTex.pushBack(texInplace);
                }
            }
            else if( hkxTextureFile* texFile = hkDynCast<hkxTextureFile>(texStage.m_texture) )
            {
                // See if we already have an identical one
                int ui = uniqueFileTex.getSize() - 1;
                for (; ui >= 0; ui--)
                {
                    if ( texFile->equals(uniqueFileTex[ui]) )
                    {
                        // Found another identical texture, replace!
                        texStage.m_texture = uniqueFileTex[ui];
                        break;
                    }
                }

                if ( ui < 0 )
                {
                    // This is a unique texture, save!
                    uniqueFileTex.pushBack(texFile);
                }
            }
        }
    }
}

//
//  Removes all unused textures

void HK_CALL hkxSceneUtils::removeUnusedTextures(_Inout_ hkxScene* scene)
{
    for (int ti = scene->m_inplaceTextures.getSize() - 1; ti >= 0; ti--)
    {
        hkxTextureInplace* tex = scene->m_inplaceTextures[ti];

        if ( !hkxSceneUtilsImpl::isTextureUsed(scene, tex) )
        {
            scene->m_inplaceTextures.removeAt(ti);
        }
    }

    for (int ti = scene->m_externalTextures.getSize() - 1; ti >= 0; ti--)
    {
        hkxTextureFile* tex = scene->m_externalTextures[ti];

        if ( !hkxSceneUtilsImpl::isTextureUsed(scene, tex) )
        {
            scene->m_externalTextures.removeAt(ti);
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
