// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Mesh/hkxVertexDescription.h>
#include <Common/SceneData/Mesh/hkxVertexUtil.h>

hkUlong hkxVertexUtil::getAlignment(hkUlong addr)
{
    int align = 0;

    // Find lsb
    while ((addr) && ((addr & 0x1) == 0))
    {
        addr = addr >> 1;
        align++;
    }

    return ((hkUlong)1 << align);
}

hkUlong hkxVertexUtil::getAlignment(hkxVertexDescription::DataUsage usage, const hkxVertexBuffer& buffer)
{
    const hkxVertexDescription& desc =  buffer.getVertexDesc();
    const hkxVertexDescription::ElementDecl* ed = desc.getElementDecl(usage, 0);
    if (ed)
    {
        hkUlong baseAddr = reinterpret_cast<hkUlong>( buffer.getVertexDataPtr( *ed ) );
        hkUlong stride = ed->m_byteStride;
        return getAlignment( baseAddr + stride );
    }
    return 0;
}

hkxVertexDescription::DataType hkxVertexUtil::getBoneIndicesType( const hkRefPtr< hkxVertexBuffer >& vertexBuffer )
{
    const hkxVertexDescription& desc = vertexBuffer->getVertexDesc();

    // find the bone index desc
    const hkxVertexDescription::ElementDecl* boneDecl =
        desc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0);

    return boneDecl->m_type;
}

bool hkxVertexUtil::reformatBoneIndicesTo16Bit( hkRefPtr< hkxVertexBuffer >& vertexBuffer )
{
    int numVerts = vertexBuffer->getNumVertices();

    const hkxVertexDescription& origDesc = vertexBuffer->getVertexDesc();
    hkxVertexDescription desc;
    for ( int i = 0;  i < origDesc.m_decls.getSize();  i++ )
    {
        desc.m_decls.pushBack( origDesc.m_decls[i] );
    }

    // Find the bone index desc
    hkxVertexDescription::ElementDecl* boneIdxDecl = desc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0);
    if ( boneIdxDecl->m_type == hkxVertexDescription::HKX_DT_INT16 )
    {
        // Nothing to do, the bone indices are already 16 bit wide
        return false;
    }
    else
    {
        if ( boneIdxDecl->m_type == hkxVertexDescription::HKX_DT_UINT8 )
        {
            // Change the new buffer's bone indices width to 16 bit
            boneIdxDecl->m_type = hkxVertexDescription::HKX_DT_INT16;
        }
        else
        {
            // Unknown bit width of source bone indices!
            HK_WARN_ALWAYS(0xabba9a3b, "hkxVertexUtil::reformatBoneIndicesTo16Bit encountered an unexpected "
                                       "bone index bit width in the source vertex buffer!" );
            return false;
        }
    }

    // Proceed with re-allocation of the buffer and conversion of its bone indices from 8 to 16 bit

    HK_ASSERT_NO_MSG(0x6f74c0a5, boneIdxDecl->m_type == hkxVertexDescription::HKX_DT_INT16 );

    hkxVertexBuffer* newVertexBuffer = new hkxVertexBuffer();
    newVertexBuffer->setNumVertices( numVerts, desc );

    // Do a full memberwise copy of the buffer's contents here, changing the bone index bit width as we go
    {
        const hkxVertexBuffer& sourceVertBuf = *(vertexBuffer.val());
        const hkxVertexDescription& sourceVertDecl = sourceVertBuf.getVertexDesc();

        const hkxVertexDescription::ElementDecl* sourceBoneWeightDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0);
        int sourceBoneWeightStride = sourceBoneWeightDecl->m_byteStride;
        const hkUint8* sourceBoneWeights = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceBoneWeightDecl) );

        const hkxVertexDescription::ElementDecl* sourceBoneIndexDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0);
        int sourceBoneIndexStride = sourceBoneIndexDecl->m_byteStride;
        const hkUint8* sourceBoneIndices = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceBoneIndexDecl) );

        const hkxVertexDescription::ElementDecl* sourcePosDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
        int sourcePosStride = sourcePosDecl->m_byteStride;
        const hkUint8* sourcePositions = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourcePosDecl) );

        const hkxVertexDescription::ElementDecl* sourceNrmDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
        int sourceNrmStride = sourceNrmDecl->m_byteStride;
        const hkUint8* sourceNormals = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceNrmDecl) );

        const hkxVertexDescription::ElementDecl* sourceTanDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
        int sourceTanStride = sourceTanDecl->m_byteStride;
        const hkUint8* sourceTangents = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceTanDecl) );

        const hkxVertexDescription::ElementDecl* sourceBitangentsDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);
        int sourceBitanStride = sourceBitangentsDecl->m_byteStride;
        const hkUint8* sourceBitangents = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceBitangentsDecl) );

        const hkxVertexDescription::ElementDecl* sourceTexDecl =
            sourceVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_TEXCOORD, 0);
        int sourceTexStride = sourceTexDecl->m_byteStride;
        const hkUint8* sourceTexCoords = (hkUint8*)( sourceVertBuf.getVertexDataPtr(*sourceTexDecl) );


        hkxVertexBuffer& destVertBuf = *newVertexBuffer;
        const hkxVertexDescription& destVertDecl = destVertBuf.getVertexDesc();

        const hkxVertexDescription::ElementDecl* destBoneWeightDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0);
        int destBoneWeightStride = destBoneWeightDecl->m_byteStride;
        hkUint8* destBoneWeights = (hkUint8*)( destVertBuf.getVertexDataPtr(*destBoneWeightDecl) );

        const hkxVertexDescription::ElementDecl* destBoneIndexDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0);
        int destBoneIndexStride = destBoneIndexDecl->m_byteStride;
        const hkUint8* destBoneIndices = (hkUint8*)( destVertBuf.getVertexDataPtr(*destBoneIndexDecl) );

        const hkxVertexDescription::ElementDecl* destPosDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
        int destPosStride = destPosDecl->m_byteStride;
        const hkUint8* destPositions = (hkUint8*)( destVertBuf.getVertexDataPtr(*destPosDecl) );

        const hkxVertexDescription::ElementDecl* destNrmDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
        int destNrmStride = destNrmDecl->m_byteStride;
        const hkUint8* destNormals = (hkUint8*)( destVertBuf.getVertexDataPtr(*destNrmDecl) );

        const hkxVertexDescription::ElementDecl* destTanDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
        int destTanStride = destTanDecl->m_byteStride;
        const hkUint8* destTangents = (hkUint8*)( destVertBuf.getVertexDataPtr(*destTanDecl) );

        const hkxVertexDescription::ElementDecl* destBitangentsDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);
        int destBitanStride = destBitangentsDecl->m_byteStride;
        const hkUint8* destBitangents = (hkUint8*)( destVertBuf.getVertexDataPtr(*destBitangentsDecl) );

        const hkxVertexDescription::ElementDecl* destTexDecl =
            destVertDecl.getElementDecl(hkxVertexDescription::HKX_DU_TEXCOORD, 0);
        int destTexStride = destTexDecl->m_byteStride;
        const hkUint8* destTexCoords = (hkUint8*)( destVertBuf.getVertexDataPtr(*destTexDecl) );

        for ( int i = 0;  i < numVerts;  i++ )
        {
            const hkUint8* sourceIndices = (const hkUint8*)( sourceBoneIndices + i*sourceBoneIndexStride );
            const hkUint8* sourceWeights = (const hkUint8*)( sourceBoneWeights + i*sourceBoneWeightStride );
            const hkFloat32* sourcePos = (const hkFloat32*)( sourcePositions + i*sourcePosStride );
            const hkFloat32* sourceNrm = (const hkFloat32*)( sourceNormals + i*sourceNrmStride );
            const hkFloat32* sourceTan = (const hkFloat32*)( sourceTangents + i*sourceTanStride );
            const hkFloat32* sourceBitan = (const hkFloat32*)( sourceBitangents + i*sourceBitanStride );
            const hkFloat32* sourceTex = (const hkFloat32*)( sourceTexCoords + i*sourceTexStride );

            hkUint16* destIndices = (hkUint16*)( destBoneIndices + i*destBoneIndexStride );
            hkUint8* destWeights = (hkUint8*)( destBoneWeights + i*destBoneWeightStride );
            hkFloat32* destPos = (hkFloat32*)( destPositions + i*destPosStride );
            hkFloat32* destNrm = (hkFloat32*)( destNormals + i*destNrmStride );
            hkFloat32* destTan = (hkFloat32*)( destTangents + i*destTanStride );
            hkFloat32* destBitan = (hkFloat32*)( destBitangents + i*destBitanStride );
            hkFloat32* destTex = (hkFloat32*)( destTexCoords + i*destTexStride );

            for (int sbs=0; sbs < 4; ++sbs) // Assumes 4 bones per vertex!
            {
                destIndices[sbs] = sourceIndices[sbs];
                destWeights[sbs] = sourceWeights[sbs];
            }

            for (int c=0; c < 3; c++)
            {
                destPos[c] = sourcePos[c];
                destNrm[c] = sourceNrm[c];
                destTan[c] = sourceTan[c];
                destBitan[c] = sourceBitan[c];
            }

            destTex[0] = sourceTex[0];
            destTex[1] = sourceTex[1];
        }
    }

    vertexBuffer = newVertexBuffer;
    newVertexBuffer->removeReference();
    return true;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
