// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>

hkxMeshSection::hkxMeshSection(const hkxMeshSection& other)
    : hkReferencedObject(other)
{
    m_vertexBuffer = other.m_vertexBuffer;

    for(int i = 0; i < other.m_indexBuffers.getSize(); i++)
    {
        m_indexBuffers.pushBack(other.m_indexBuffers[i]);
    }

    m_material = other.m_material;

    for(int i = 0; i < other.m_userChannels.getSize(); i++)
    {
        m_userChannels.pushBack(other.m_userChannels[i]);
    }

    for(int i = 0; i < other.m_vertexAnimations.getSize(); i++)
    {
        m_vertexAnimations.pushBack(other.m_vertexAnimations[i]);
    }

    m_linearKeyFrameHints = other.m_linearKeyFrameHints;

    m_boneMatrixMap.setSize( other.m_boneMatrixMap.getSize() );
    for(int i = 0; i < other.m_boneMatrixMap.getSize(); i++)
    {
        m_boneMatrixMap[i] = other.m_boneMatrixMap[i];
    }

}

/// Returns the total number of triangles in all index buffers
hkUint32 hkxMeshSection::getNumTriangles () const
{
    hkUint32 nTriangles = 0;

    for (int n=0; n<m_indexBuffers.getSize(); n++)
    {
        hkxIndexBuffer* ibuffer = m_indexBuffers[n];

        nTriangles += ibuffer->getNumTriangles();
    }

    return nTriangles;
}

/// Explore the index buffers for the indices of the triIndex'th triangle
void hkxMeshSection::getTriangleIndices (hkUint32 triIndex, hkUint32& indexAOut, hkUint32& indexBOut, hkUint32& indexCOut) const
{
    hkUint32 nTriangles = 0;
    for (int n=0; n<m_indexBuffers.getSize(); n++)
    {
        hkxIndexBuffer* ibuffer = m_indexBuffers[n];
        hkUint32 nBuffer = ibuffer->getNumTriangles();

        if ( triIndex < nTriangles + nBuffer )
        {
            ibuffer->getTriangleIndices( triIndex-nTriangles, indexAOut, indexBOut, indexCOut );
            return;
        }

        nTriangles += nBuffer;
    }
}

void hkxMeshSection::collectVertexPositions (hkArray<hkVector4>& verticesInOut) const
{
    hkxVertexBuffer* vertices = m_vertexBuffer;
    if (vertices )
    {
        const hkxVertexDescription& vDesc = vertices->getVertexDesc();
        const hkxVertexDescription::ElementDecl* vdecl = vDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
        int numVerts = vertices->getNumVertices();
        if (vdecl && (numVerts > 0))
        {
            void* data = vertices->getVertexDataPtr(*vdecl);
            hkUint32 posStride = vdecl->m_byteStride;

            int bufOffset = verticesInOut.getSize();
            verticesInOut.setSize( bufOffset + numVerts );
            for (int vi=0; vi < numVerts; ++vi)
            {
                verticesInOut[vi + bufOffset].load<4,HK_IO_NATIVE_ALIGNED>((const hkFloat32*)data);
                data = hkAddByteOffset(data, posStride);
            }
        }
    }
}

void hkxMeshSection::appendGeometry (struct hkGeometry& geometryInOut, int materialIndex) const
{
    collectVertexPositions(geometryInOut.m_vertices);

    // Now, check the index buffer
    for (int ib=0; ib < m_indexBuffers.getSize(); ++ib)
    {
        hkxIndexBuffer* ibuffer = m_indexBuffers[ib];

        const int numIndices = ibuffer->m_indices16.getSize() | ibuffer->m_indices32.getSize();
        bool smallIndices = ibuffer->m_indices16.getSize() > 0;
        int index = 0;

        while (index < numIndices)
        {
            hkGeometry::Triangle newTriangle;

            newTriangle.m_material = materialIndex;

            switch (ibuffer->m_indexType)
            {
            case hkxIndexBuffer::INDEX_TYPE_TRI_LIST:
                {

                    newTriangle.m_a = smallIndices ? ibuffer->m_indices16[index] : ibuffer->m_indices32[index];
                    newTriangle.m_b = smallIndices ? ibuffer->m_indices16[index+1] : ibuffer->m_indices32[index+1];
                    newTriangle.m_c = smallIndices ? ibuffer->m_indices16[index+2] : ibuffer->m_indices32[index+2];

                    index += 3;
                    break;
                }
            case hkxIndexBuffer::INDEX_TYPE_TRI_STRIP:
                {
                    if (index<2)
                    {
                        index++;
                        continue;
                    }

                    if (index==2)
                    {
                        newTriangle.m_a = smallIndices ? ibuffer->m_indices16[0] : ibuffer->m_indices32[0];
                        newTriangle.m_b = smallIndices ? ibuffer->m_indices16[1] : ibuffer->m_indices32[1];
                        newTriangle.m_c = smallIndices ? ibuffer->m_indices16[2] : ibuffer->m_indices32[2];

                        index ++;
                        break;
                    }

                    const hkGeometry::Triangle &previousTriangle = geometryInOut.m_triangles[geometryInOut.m_triangles.getSize()-1];

                    newTriangle.m_a = previousTriangle.m_c;
                    newTriangle.m_b = previousTriangle.m_b;
                    newTriangle.m_c = smallIndices? ibuffer->m_indices16[index] : ibuffer->m_indices32[index];

                    index ++;
                    break;
                }
            default:
                {
                    HK_WARN_ALWAYS(0xabbaa883, "Unsupported index buffer type - Ignoring");
                    index = numIndices;
                    continue;
                }
            }

            geometryInOut.m_triangles.pushBack(newTriangle);

        }
    }
}

_Ret_notnull_ hkxMeshSection* hkxMeshSection::copy( const CopyOptions& options ) const
{
    hkxMeshSection* m = new hkxMeshSection();

    if (options.m_shareVertexBuffers)
    {
        m->m_vertexBuffer = m_vertexBuffer;
    }
    else
    {
        m->m_vertexBuffer.setAndDontIncrementRefCount(new hkxVertexBuffer());
        m->m_vertexBuffer->setNumVertices( m_vertexBuffer->getNumVertices(), m_vertexBuffer->getVertexDesc() );
        m->m_vertexBuffer->copy( *m_vertexBuffer, false );
    }

    if (options.m_shareIndexBuffers)
    {
        m->m_indexBuffers = m_indexBuffers;
    }
    else
    {
        m->m_indexBuffers.setSize(m_indexBuffers.getSize());
        for (int ib=0; ib < m_indexBuffers.getSize(); ++ib)
        {
            m->m_indexBuffers[ib].setAndDontIncrementRefCount(new hkxIndexBuffer());
            m->m_indexBuffers[ib]->m_indexType = m_indexBuffers[ib]->m_indexType;
            m->m_indexBuffers[ib]->m_indices16 = m_indexBuffers[ib]->m_indices16;
            m->m_indexBuffers[ib]->m_indices32 = m_indexBuffers[ib]->m_indices32;
            m->m_indexBuffers[ib]->m_vertexBaseOffset = m_indexBuffers[ib]->m_vertexBaseOffset;
            m->m_indexBuffers[ib]->m_length = m_indexBuffers[ib]->m_length;
        }
    }

    if (options.m_shareMaterials)
    {
        m->m_material = m_material;
    }
    else
    {
        HK_ERROR(0x40f767ff, "not impl..");
    }

    if (options.m_shareUserChannels)
    {
        m->m_userChannels = m_userChannels;
    }
    else
    {
        HK_ERROR(0x6b5c6bca, "not impl..");
    }

    if (options.m_shareVertexAnims)
    {
        m->m_vertexAnimations = m_vertexAnimations;
        m->m_linearKeyFrameHints = m_linearKeyFrameHints;
    }
    else
    {
        HK_WARN_ALWAYS(0xabba85dc, "vertex anim copy not impl..");
    }

    if (options.m_shareBoneMappings)
    {
        m->m_boneMatrixMap.setSize( m_boneMatrixMap.getSize() );
        for (int mi=0; mi < m_boneMatrixMap.getSize(); ++mi)
        {
            m->m_boneMatrixMap[mi] = m_boneMatrixMap[mi];
        }
    }
    else
    {
        HK_ERROR(0x3d183622, "not impl..");
    }

    return m;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
