// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/SceneData/hkSceneData.h>

#include <Common/GeometryUtilities/Mesh/Converters/SceneDataToMesh/hkSceneDataToMeshConverter.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshShape.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshSystem.h>
#include <Common/GeometryUtilities/Mesh/Skin/hkSkinnedMeshShapeBuilder.h>
#include <Common/GeometryUtilities/Mesh/Utils/MeshSectionMergeUtil/hkMeshSectionMergeUtil.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Base/Types/Uuid/hkUuidMapOperations.cxx>
#include <Common/Base/Container/PointerMap/hkMap.hxx>

// this
#include <Common/SceneData/Mesh/MemoryMeshFactory/hkxMemoryMeshFactory.h>

hkxMemoryMeshFactory::hkxMemoryMeshFactory(_In_ hkMemoryMeshSystem* meshSystem)
{
    m_meshSystem = meshSystem;
}

hkxMemoryMeshFactory::~hkxMemoryMeshFactory()
{
    m_meshSystem = HK_NULL;
}

_Ret_maybenull_
hkMeshSystem* hkxMemoryMeshFactory::getMeshSystem()
{
    return m_meshSystem;
}

void hkxMemoryMeshFactory::extractShapes(_Inout_ hkRootLevelContainer* rootLevelContainer, hkStringMap<int>& extraGraphicsNodes, hkStringMap<hkMeshShape*>& shapesOut)
{
    extractShapes( rootLevelContainer, extraGraphicsNodes, m_meshSystem, shapesOut );
}

void hkxMemoryMeshFactory::extractShapes(_Inout_ hkRootLevelContainer* rootLevelContainer, hkMap<hkUuid, int>& extraGraphicsNodes, hkMap<hkUuid, hkMeshShape*>& shapesOut )
{
    extractShapes( rootLevelContainer, extraGraphicsNodes, m_meshSystem, shapesOut );
}

/// Use to accumulate all rigid body nodes and physics systems from the list of \a rootLevelContainers.
/// This function depends on the serialization type, so it is likely that this function has to be reimplemented for each game engines
/// This implementation was moved here from the hkdAssetProcessingUtil
void hkxMemoryMeshFactory::extractShapes(_In_ const hkRootLevelContainer* rootLevelContainer, hkStringMap<int>& extraGraphicsNodes, _Inout_ hkMemoryMeshSystem *meshSystem, hkStringMap<hkMeshShape*>& shapesOut)
{
    //
    //  Find all the physics nodes with the original hkxNode data
    //

    hkxScene* scene = rootLevelContainer->findObject<hkxScene>();
    hkArray<hkxNode*> nodes;
    hkxSceneUtils::findAllGraphicsNodes(false, false, extraGraphicsNodes, scene->m_rootNode, nodes);

    // Keep a map of already converted objects
    typedef hkPointerMap<const hkReferencedObject*, hkMeshShape*> MeshMap;
    MeshMap knownMeshShapes;

    // Add all the newly found rigid bodies
    for (int n = 0; n < nodes.getSize(); n++)
    {
        hkxNode* node = nodes[n];
        const char* nodeName = node->m_name.cString();
        if ( shapesOut.getWithDefault( nodeName, HK_NULL ) )
        {
            HK_WARN_ALWAYS( 0xabba43de, "Object: '" << nodeName <<"' exists twice, deleting second version" );
            continue;
        }

        // Get its mesh shape
        hkMeshShape* meshShape = HK_NULL;
        {
            // Get the node and its graphics. Check if we've already converted it
            hkxNode* sceneNode          = node;
            const hkRefVariant& nodeObj = sceneNode->m_object;
            const hkReferencedObject* k = nodeObj.val();
            MeshMap::Iterator it        = knownMeshShapes.findKey(k);

            if ( knownMeshShapes.isValid(it) )
            {
                // Already converted, reuse!
                meshShape = knownMeshShapes.getValue(it);
                meshShape->addReference();
            }
            else
            {
                // New, convert now!
                meshShape = hkSceneDataToMeshConverter::convert(meshSystem, HK_NULL, scene, sceneNode, hkSceneDataToMeshConverter::SPACE_ONLY_USING_SCALE_SKEW);
                meshShape->setName(nodeName);
                knownMeshShapes.insert(k, meshShape);
            }
        }

        if (!hkMeshSectionMergeUtil::hasUniqueMaterials(meshShape))
        {
            // We may want to merge materials -> to improve performance etc
            

            const bool mergeMaterialSections = false;

            if (mergeMaterialSections)
            {
                hkMeshShape* mergedShape = hkMeshSectionMergeUtil::mergeShapeSectionsByMaterial(meshSystem, meshShape);
                if (mergedShape)
                {
                    meshShape->removeReference();
                    meshShape = mergedShape;
                }
                else
                {
                    HK_WARN(0x8d7292bb, "Unable to merge sections with the same material");
                }

            }
            else
            {
                HK_WARN(0x24234a21, "Mesh shape '" << nodeName << "' has multiple sections with the same material: impacts graphics performance");
            }
        }

        if ( meshShape != HK_NULL)
        {
            shapesOut.insert( nodeName, meshShape);
        }
    }
}

void hkxMemoryMeshFactory::extractShapes(_In_ const hkRootLevelContainer* rootLevelContainer, hkMap<hkUuid, int>& extraGraphicsNodes, _Inout_ hkMemoryMeshSystem *meshSystem, hkMap<hkUuid, hkMeshShape*>& shapesOut)
{
    //
    //  Find all the physics nodes with the original hkxNode data
    //

    hkxScene* scene = rootLevelContainer->findObject<hkxScene>();
    hkArray<hkxNode*> nodes;
    hkxSceneUtils::findAllGraphicsNodes(false, false, extraGraphicsNodes, scene->m_rootNode, nodes);

    // Keep a map of already converted objects
    typedef hkPointerMap<const hkReferencedObject*, hkMeshShape*> MeshMap;
    MeshMap knownMeshShapes;

    // Add all the newly found rigid bodies
    for (int n = 0; n < nodes.getSize(); n++)
    {
        hkxNode* node = nodes[n];
        const hkUuid& nodeUuid = node->m_uuid;

        if ( shapesOut.getWithDefault( nodeUuid, HK_NULL ) )
        {
            HK_WARN_ALWAYS( 0xabba43de, "Object: '" << node->m_name.cString() <<"' exists twice, deleting second version" );
            continue;
        }

        // Get its mesh shape
        hkMeshShape* meshShape = HK_NULL;
        {
            // Get the node and its graphics. Check if we've already converted it
            hkxNode* sceneNode          = node;
            const hkRefVariant& nodeObj = sceneNode->m_object;
            const hkReferencedObject* k = nodeObj.val();
            MeshMap::Iterator it        = knownMeshShapes.findKey(k);

            if ( knownMeshShapes.isValid(it) )
            {
                // Already converted, reuse!
                meshShape = knownMeshShapes.getValue(it);
                meshShape->addReference();
            }
            else
            {
                // New, convert now!
                meshShape = hkSceneDataToMeshConverter::convert(meshSystem, HK_NULL, scene, sceneNode, hkSceneDataToMeshConverter::SPACE_ONLY_USING_SCALE_SKEW);
                meshShape->setName(node->m_name.cString());
                knownMeshShapes.insert(k, meshShape);
            }
        }

        if (!hkMeshSectionMergeUtil::hasUniqueMaterials(meshShape))
        {
            // We may want to merge materials -> to improve performance etc
            

            const bool mergeMaterialSections = false;

            if (mergeMaterialSections)
            {
                hkMeshShape* mergedShape = hkMeshSectionMergeUtil::mergeShapeSectionsByMaterial(meshSystem, meshShape);
                if (mergedShape)
                {
                    meshShape->removeReference();
                    meshShape = mergedShape;
                }
                else
                {
                    HK_WARN(0x8d7292bb, "Unable to merge sections with the same material");
                }

            }
            else
            {
                HK_WARN(0x24234a21, "Mesh shape '" << node->m_name << "' has multiple sections with the same material: impacts graphics performance");
            }
        }

        if ( meshShape != HK_NULL)
        {
            shapesOut.insert( nodeUuid, meshShape);
        }
    }
}

// Nothing to do here
void hkxMemoryMeshFactory::storeShapes(_Inout_ hkRootLevelContainer* rootLevelContainer, hkArray<const hkMeshShape*>& shapes )
{
    // Add a reference to all shapes, to have the same behavior as the Vision mesh factory
    hkReferencedObject::addReferences(shapes.begin(), shapes.getSize());
}

const hkMatrix3& hkxMemoryMeshFactory::getTransform() const
{
    return hkMatrix3::getIdentity();
}

void hkxMemoryMeshFactory::setTransform( const hkMatrix3& transform )
{
}

//
//  Creates a skinned mesh shape from the given mesh shapes and transforms. A single bone will drive each of the provided shapes.
//  Currently, the implementation returns HK_NULL.

_Ret_notnull_
hkSkinnedMeshShape* hkxMemoryMeshFactory::createSkinnedMesh(_In_z_ const char* skinnedMeshName, _In_reads_(numMeshes) const hkMeshShape** meshShapesIn, _In_reads_(numMeshes) const hkQTransform* transformsIn, int numMeshes, int maxBonesPerSection)
{
    // If you implement this, make sure to implement and call removeShapes properly, so you don't end-up with both the skinned and original graphics
    // inside the asset file.

    // Create skinned mesh shape
    hkStorageSkinnedMeshShape* skinnedMeshShape = new hkStorageSkinnedMeshShape;

    // Set name
    skinnedMeshShape->setName( skinnedMeshName );

    // Build the skinned mesh shape from the given meshes
    hkSkinnedMeshBuilder builder( skinnedMeshShape, m_meshSystem, maxBonesPerSection );
    for ( int i = 0; i < numMeshes; i++ )
    {
        builder.addMesh( meshShapesIn[i], transformsIn[i], 1 );
    }
    builder.build();

    return skinnedMeshShape;
}

//
//  Removes all the given shapes from the root level container

void hkxMemoryMeshFactory::removeShapes(_Inout_opt_ hkRootLevelContainer* rootLevelContainer, hkArray<const hkMeshShape*>& shapes)
{}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
