// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Attributes/hkxAttributeGroup.h>

hkxAttributeGroup& hkxAttributeGroup::operator=( const hkxAttributeGroup& other )
{
    m_name = other.m_name;
    m_attributes.setSize(0);
    if (other.m_attributes.getSize())
        m_attributes.insertAt(0, other.m_attributes.begin(), other.m_attributes.getSize());

    return *this;
}

hkResult hkxAttributeGroup::getBoolValue(_In_z_ const char* name, bool warnIfNotFound, hkBool& boolOut) const
{
    // Bool
    {
        hkxSparselyAnimatedBool* data = findBoolAttributeByName(name);
        if (data)
        {
            boolOut=data->m_bools[0];
            return HK_SUCCESS;
        }
    }

    // Int
    {
        hkxSparselyAnimatedInt* data = findIntAttributeByName(name);
        if (data)
        {
            boolOut = (data->m_ints[0] != 0);
            return HK_SUCCESS;
        }
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Bool attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getIntValue(_In_z_ const char* name, bool warnIfNotFound, int& intOut) const
{
    // Int
    {
        hkxSparselyAnimatedInt* data = findIntAttributeByName(name);
        if (data)
        {
            intOut = data->m_ints[0];
            return HK_SUCCESS;
        }
    }

    // Bool
    {
        hkxSparselyAnimatedBool* data = findBoolAttributeByName(name);
        if (data)
        {
            intOut=data->m_bools[0] ? 1 : 0;
            return HK_SUCCESS;
        }
    }

    // Enum
    {
        hkxSparselyAnimatedEnum* data = findEnumAttributeByName(name);
        if (data)
        {
            intOut = data->m_ints[0];
            return HK_SUCCESS;
        }
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Integer attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getIntValue (_In_z_ const char* name, bool warnIfNotFound, hkUint32& intOut) const
{
    // We treat the same as ints
    return getIntValue(name, warnIfNotFound, (int&) (intOut));
}

hkResult hkxAttributeGroup::getStringValue(_In_z_ const char* name, bool warnIfNotFound, _Outref_ const char*& stringOut) const
{
    // String
    {
        hkxSparselyAnimatedString* data = findStringAttributeByName(name);
        if (data)
        {
            stringOut = data->m_strings[0].cString();
            return HK_SUCCESS;
        }
    }

    // Enum
    {
        hkxSparselyAnimatedEnum* data = findEnumAttributeByName(name);
        if (data)
        {
            int intValue = data->m_ints[0];
            data->m_enum->getNameOfValue(intValue, &stringOut);

            return HK_SUCCESS;
        }
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "String attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getFloatValue(_In_z_ const char* name, bool warnIfNotFound, float& floatOut) const
{
    hkxAnimatedFloat* data = findFloatAttributeByName(name);
    if (data)
    {
        floatOut = data->m_floats[0];
        return HK_SUCCESS;
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Float attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getVectorValue(_In_z_ const char* name, bool warnIfNotFound, hkVector4& vectorOut) const
{
    hkxAnimatedVector* data = findVectorAttributeByName(name);
    if (data)
    {
        vectorOut.load<4,HK_IO_NATIVE_ALIGNED>(&data->m_vectors[0]);
        return HK_SUCCESS;
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Float attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getQuaternionValue(_In_z_ const char* name, bool warnIfNotFound, hkQuaternion& quaternionOut) const
{
    hkxAnimatedQuaternion* data = findQuaternionAttributeByName(name);
    if (data)
    {
        quaternionOut.m_vec.load<4,HK_IO_NATIVE_ALIGNED>(&data->m_quaternions[0]);
        return HK_SUCCESS;
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Quaternion attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

hkResult hkxAttributeGroup::getMatrixValue(_In_z_ const char* name, bool warnIfNotFound, hkMatrix4& matrixOut) const
{
    hkxAnimatedMatrix* data = findMatrixAttributeByName(name);
    if (data)
    {
        matrixOut.set4x4ColumnMajor(&data->m_matrices[0]);
        return HK_SUCCESS;
    }

    if (warnIfNotFound)
    {
        HK_WARN_ALWAYS (0xabbaab81, "Matrix attribute "<<name<<" not found in "<<m_name<<" attribute group");
    }

    return HK_FAILURE;
}

int hkxAttributeGroup::findAttributeIndexByName(_In_z_ const char* name ) const
{
    for (int i=0; i < m_attributes.getSize(); ++i)
    {
        if (hkString::strCasecmp(m_attributes[i].m_name, name) == 0 )
        {
            return i;
        }
    }
    return -1;
}

hkRefVariant hkxAttributeGroup::findAttributeVariantByName(_In_z_ const char* name ) const
{
    const int index = findAttributeIndexByName( name );

    hkVariant v;

    return (index<0) ? hkRefVariant(v) : m_attributes[index].m_value;
}

_Ret_maybenull_
hkReferencedObject* hkxAttributeGroup::findAttributeObjectByName(_In_z_ const char* name, _In_opt_ const hkReflect::Type* type) const
{
    hkRefVariant var = findAttributeVariantByName(name );

    // compare class by name so that it deals with serialized classes etc better (for instance in the filters)
    if (var && (!type || var.getType()->extendsOrEquals(type)) )
    {
        return var;
    }
    return HK_NULL;
}

_Ret_maybenull_
class hkxSparselyAnimatedBool* hkxAttributeGroup::findBoolAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxSparselyAnimatedBool*> (findAttributeObjectByName(name, hkReflect::getType<hkxSparselyAnimatedBool>()));

}

_Ret_maybenull_
class hkxSparselyAnimatedInt* hkxAttributeGroup::findIntAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxSparselyAnimatedInt*> (findAttributeObjectByName(name, hkReflect::getType<hkxSparselyAnimatedInt>()));

}

_Ret_maybenull_
class hkxSparselyAnimatedEnum* hkxAttributeGroup::findEnumAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxSparselyAnimatedEnum*> (findAttributeObjectByName(name, hkReflect::getType<hkxSparselyAnimatedEnum>()));

}

_Ret_maybenull_
class hkxSparselyAnimatedString* hkxAttributeGroup::findStringAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxSparselyAnimatedString*> (findAttributeObjectByName(name, hkReflect::getType<hkxSparselyAnimatedString>()));

}

_Ret_maybenull_
class hkxAnimatedFloat* hkxAttributeGroup::findFloatAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxAnimatedFloat*> (findAttributeObjectByName(name, hkReflect::getType<hkxAnimatedFloat>()));

}

_Ret_maybenull_
class hkxAnimatedVector* hkxAttributeGroup::findVectorAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxAnimatedVector*> (findAttributeObjectByName(name, hkReflect::getType<hkxAnimatedVector>()));
}

_Ret_maybenull_
class hkxAnimatedQuaternion* hkxAttributeGroup::findQuaternionAttributeByName (_In_z_ const char* name) const
{
    return static_cast<hkxAnimatedQuaternion*> (findAttributeObjectByName(name, hkReflect::getType<hkxAnimatedQuaternion>()));

}

_Ret_maybenull_
class hkxAnimatedMatrix* hkxAttributeGroup::findMatrixAttributeByName(_In_z_ const char* name) const
{
    return static_cast<hkxAnimatedMatrix*> (findAttributeObjectByName(name, hkReflect::getType<hkxAnimatedMatrix>()));

}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
